/* -------------------------------------------------------------------------- */

#include <stdio.h>
#include <string.h>
#include "ima.h"

/* -------------------------------------------------------------------------- */
/*     wav */
typedef struct riff_head {
	unsigned int sign;
	unsigned int size;
	unsigned int type;
} riff_head;

typedef struct riff_chunk {
	unsigned int type;
	unsigned int size;
} riff_chunk;

typedef struct wave_format {
	unsigned short tag;
	unsigned short nch;
	unsigned int freq;
	unsigned int rate;
	unsigned short align;
	unsigned short bits;
} wave_format;
/* -------------------------------------------------------------------------- */

/*  */
void encode(char *inname, char *outname)
{
	FILE *in, *out;
	riff_head head;
	riff_chunk chunk;
	wave_format format;
	size_t remain, step;
	char buf[256], ima[64];
	ima_state state[2];

	memset(&format, 0, sizeof(format));

	if(!(in = fopen(inname, "rb")))
	{
		printf("%s: can't open\n", inname);
	}
	else
	{
		if(!(out = fopen(outname, "wb")))
		{
			printf("%s: can't create\n", outname);
		}
		else
		{
			if( ! ((fread(&head, sizeof(head), 1, in)) &&
				(head.sign == 'FFIR') &&
				(head.type == 'EVAW')) )
			{
				printf("%s: not waveform file\n", inname);
			}
			else
			{
				while(fread(&chunk, sizeof(chunk), 1, in))
				{
					remain = chunk.size;

					if(chunk.type == ' tmf')
					{
						if(remain >= sizeof(format))
							remain -= fread(&format, 1, sizeof(format), in);
					}
					
					else if(chunk.type == 'atad')  /*   data   LE      */
					{
						if( ! ((format.tag  == 1)     && 
						  	   (format.freq == 44100) &&
							   (format.bits == 16)    &&
							   (format.nch  == 2 || format.nch  == 1) )) /*       */
						{
							printf("%s: unsupported format. "
								"required: PCM/44100/16bit/stereo or PCM/44100/16bit/mono\n", inname);
						}
						else
						{
							puts("encoding stream...");
							memset(&state, 0, sizeof(state));
							
							step = remain;
							while(remain)
							{
								if(step > sizeof(buf))
									step = sizeof(buf);

								if(!(step = fread(buf, 1, step, in)))
									break;
									
								/*         */
								if (format.nch  == 2)
									ima_encode_stereo(state, ima, buf, step);
								else 	
									ima_encode_mono(state, ima, buf, step);
									
								fwrite(ima, 1, step/4, out);

								remain -= step;
								step = sizeof(buf);
							}

							puts("finished encoding");
						}
					}

					fseek(in, (long)remain, SEEK_CUR);
				}
			}
			fclose(out);
		}
		fclose(in);
	}

	puts("");
}
/* -------------------------------------------------------------------------- */

/*   */
void decode2(char *inname, char *outname)
{
	FILE *in, *out;
	char buf[64], pcm[256];
	size_t step;
	ima_state state[2];
	riff_head head;
	riff_chunk chunk;
	wave_format format;
	size_t len;

	if(!(in = fopen(inname, "rb")))
	{
		printf("%s: can't open\n", inname);
	}
	else
	{
		if(!(out = fopen(outname, "wb")))
		{
			printf("%s: can't create\n", outname);
		}
		else
		{
			fseek(in, 0, SEEK_END);
			len = ftell(in);

			puts("writing wavefile header...");

			head.sign = 'FFIR';
			head.type = 'EVAW';
			head.size = sizeof(head) + 
				sizeof(chunk)*2 + 
				sizeof(format) + 
				len*4;
			fwrite(&head, sizeof(head), 1, out);

			chunk.type = ' tmf';
			chunk.size = sizeof(format);
			fwrite(&chunk, sizeof(chunk), 1, out);

			format.tag = 1;
			format.freq = 44100;
			format.bits = 16;
			format.nch = 2;
			format.align = format.bits * format.nch / 8;
			format.rate = format.freq * format.align;
			fwrite(&format, sizeof(format), 1, out);

			chunk.type = 'atad';
			chunk.size = (unsigned)len*4;
			fwrite(&chunk, sizeof(chunk), 1, out);

			puts("decoding stereo stream...");
		
			memset(&state, 0, sizeof(state));

			rewind(in);
			while((step = fread(buf, 1, sizeof(buf), in)))
			{
				ima_decode_stereo(state, pcm, buf, step);
				fwrite(pcm, 1, step*4, out);
			}

			puts("decoding finished");
			
			fclose(out);
		}
		fclose(in);
	}

	puts("");
}

// 
void decode1(char *inname, char *outname)
{
	FILE *in, *out;
	char buf[64], pcm[256];
	size_t step;
	ima_state state[2];
	riff_head head;
	riff_chunk chunk;
	wave_format format;
	size_t len;

	if(!(in = fopen(inname, "rb")))
	{
		printf("%s: can't open\n", inname);
	}
	else
	{
		if(!(out = fopen(outname, "wb")))
		{
			printf("%s: can't create\n", outname);
		}
		else
		{
			fseek(in, 0, SEEK_END);
			len = ftell(in);

			puts("writing wavefile header...");

			head.sign = 'FFIR';
			head.type = 'EVAW';
			head.size = sizeof(head) + 
				sizeof(chunk)*2 + 
				sizeof(format) + 
				len*4;
			fwrite(&head, sizeof(head), 1, out);

			chunk.type = ' tmf';
			chunk.size = sizeof(format);
			fwrite(&chunk, sizeof(chunk), 1, out);

			format.tag = 1;  // PCM
			format.freq = 44100;
			format.bits = 16;
			format.nch = 1; // mono
			format.align = format.bits * format.nch / 8;
			format.rate = format.freq * format.align;
			fwrite(&format, sizeof(format), 1, out);

			chunk.type = 'atad';
			chunk.size = (unsigned)len*2;  //  
			fwrite(&chunk, sizeof(chunk), 1, out);

			puts("decoding mono stream...");
		
			memset(&state, 0, sizeof(state));

			rewind(in);
			while((step = fread(buf, 1, sizeof(buf), in)))
			{
				ima_decode_mono(state, pcm, buf, step);
				fwrite(pcm, 1, step*4, out);
			}

			puts("decoding finished");
			
			fclose(out);
		}
		fclose(in);
	}

	puts("");
}



/*       */
void add_ext(char *str, char *ext)
{
	char *temp;

	if((temp = strrchr(str, '\\')))
		str = temp+1;
	if((temp = strrchr(str, '/')))
		str = temp+1;
	if(strchr(str, '.'))
		return;
	strcat(str, ext);
}
/*       */
void del_ext(char *str)
{
	char *temp;

	if((temp = strrchr(str, '\\')))
		str = temp+1;
	if((temp = strrchr(str, '/')))
		str = temp+1;
	if((temp = strrchr(str, '.')))
		*temp = 0;
}
/* -------------------------------------------------------------------------- */

/*  - */
int main(int argc, char **argv)
{
	static char in[1024] = "";
	static char out[1024] = "";
	int decode = 0 , mono = 0;
	int showhelp = 0, n;
	char *opt;

	puts("\nIMA ADPCM encoder-decoder for AZBK sound system\n(c) MAXIOL 2021\n(c) redsh, 2010\n");

    /*     */
	for(n = 1; n < argc; ++n)
	{
		opt = argv[n];
		if((*opt == '-') || (*opt == '/'))
		{
			opt++;
			if( (strcmpi(opt, "o") == 0) ||
				(strcmpi(opt, "-out") == 0) )
			{
				strcpy(out, argv[++n]);
			}
			else if( (strcmpi(opt, "d") == 0) ||
				(strcmpi(opt, "-decode") == 0) )
			{
				decode = 1;
			}
			else if( (strcmpi(opt, "m") == 0) ||
				(strcmpi(opt, "-mono") == 0) )
			{
				mono = 1;
			}
			else if( (strcmpi(opt, "h") == 0) ||
				(strcmpi(opt, "?") == 0) ||
				(strcmpi(opt, "-help") == 0) )
			{
				showhelp = 1;
			}
			else
			{
				printf("%s: unknown option\n", opt);
				showhelp = 1;
			}
		}

		else
		{
			strcpy(in, opt);
		}
	}

	if( (!showhelp) && (strcmpi(in, "") == 0) )
	{
		puts("no input file specifed\n");
		showhelp = 1;
	}

	if(showhelp)
	{	puts("supported format for encoding: required: PCM/44100/16bit/stereo or PCM/44100/16bit/mono\n");		
		puts("Usage: imaed <inputfile> [options] ...");
		puts("-o || --out      - output file");
		puts("-d || --decode   - decode ima file");
		puts("-m || --mono     - decode mono ima file");
		puts("-h || --help     - show help message\n");
		return 1;
	}

	if(!decode)
	{
		if(strcmpi(out, "") == 0)
		{
			strcpy(out, in);
			del_ext(out);
			add_ext(out, ".ima");
		}
		add_ext(in, ".wav");
		encode(in, out);
	}

	else
	{
		if(strcmpi(out, "") == 0)
		{
			strcpy(out, in);
			del_ext(out);
			add_ext(out, ".wav");
		}
		add_ext(in, ".ima");
		if (mono==0)
			decode2(in, out);
		else   
			decode1(in, out);
	}

	return 0;
}

/* -------------------------------------------------------------------------- */


