// BoulderMap.cpp: implementation of the CBoulderMap class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Boulder.h"
#include "BoulderMap.h"

#include "BlockSpace.h"
#include "BlockGround.h"
#include "BlockArmor.h"
#include "BlockGold.h"
#include "BlockStone.h"
#include "BlockMonster.h"
#include "BlockBoulder.h"
#include "BlockBoom.h"

#include "FileExeption.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

extern CBoulderApp theApp;

CString CBoulderMap::m_strFindFileName;
BYTE	CBoulderMap::m_totalLevelNum;
int		CBoulderMap::m_curLevelIndex;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CBoulderMap::CBoulderMap ()
{
	m_levelNum = -1;
	m_pMap = NULL;

	m_nGold = 0;
	m_nTotalGold = 0;
	m_nMonsters = 0;
	m_nTotalMonsters = 0;
	m_nExplosion = 0;
}

//////////////////////////////////////////////////////////////////////
CBoulderMap::CBoulderMap(int levelNum, int fill)
{
	m_levelNum = levelNum;
	m_pMap = NULL;

	m_nGold = 0;
	m_nMonsters = 0;
	m_nExplosion = 0;

	m_pMap = new CMapBlock** [MAP_HEIGHT];

	for (int i = 0; i < MAP_HEIGHT; i++)
		m_pMap[i] = new CMapBlock* [MAP_WIDTH];

	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x < MAP_WIDTH; x++)
			m_pMap[y][x] = CreateNewBlock (fill);

	m_nTotalGold = GetGoldNumber ();
	m_nTotalMonsters = GetMonstersNumber ();
}

//////////////////////////////////////////////////////////////////////
CBoulderMap::CBoulderMap(int levelNum, CString strFileName)
{
	m_levelNum = 0;
	m_pMap = NULL;
	
	m_nGold = 0;
	m_nMonsters = 0;
	m_nExplosion = 0;

	LoadMap (levelNum, strFileName);

	m_nTotalGold = GetGoldNumber ();
	m_nTotalMonsters = GetMonstersNumber ();
}

//////////////////////////////////////////////////////////////////////
CBoulderMap::CBoulderMap (CBoulderMap* pMap)
{
	m_levelNum = pMap->GetLevelNumber ();
	m_pMap = NULL;

	m_nGold = 0;
	m_nMonsters = 0;
	m_nExplosion = 0;

	m_pMap = new CMapBlock** [MAP_HEIGHT];

	for (int i = 0; i < MAP_HEIGHT; i++)
		m_pMap[i] = new CMapBlock* [MAP_WIDTH];

	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x < MAP_WIDTH; x++)
		{
			CMapBlock* pBlock = pMap->GetMapItemIndirect (CPoint (x, y)); 
			int type = pBlock->GetBlockType ();
			if (type != BLOCK_MONSTER)
				m_pMap[y][x] = CreateNewBlock (type);
			else
				m_pMap[y][x] = CreateNewBlock (((CBlockMonster*)pBlock)->GetMonsterType ());

		}

	m_nTotalGold = GetGoldNumber ();
	m_nTotalMonsters = GetMonstersNumber ();
}

//////////////////////////////////////////////////////////////////////
CBoulderMap::~CBoulderMap()
{
	DestroyMap ();	
}

//////////////////////////////////////////////////////////////////////
CMapBlock*	CBoulderMap::GetMapItem (CPoint ptPos)
{
	if (ptPos.x < MAP_WIDTH - 1 
		&& m_pMap[ptPos.y][ptPos.x + 1]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x + 1])->GetMovement () == MOVEMENT_LEFT
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x + 1])->GetMovement () == MOVEMENT_DROP_LEFT
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x + 1])->GetMovement () == MOVEMENT_LEFT_ONGROUP
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x + 1])->GetMovement () == VK_LEFT)
			return m_pMap[ptPos.y][ptPos.x + 1];

	if (ptPos.x > 0 
		&& m_pMap[ptPos.y][ptPos.x - 1]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x - 1])->GetMovement () == MOVEMENT_RIGHT
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x - 1])->GetMovement () == MOVEMENT_DROP_RIGHT
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x - 1])->GetMovement () == MOVEMENT_RIGHT_ONGROUP
			||((CBlockDinamic*)m_pMap[ptPos.y][ptPos.x - 1])->GetMovement () == VK_RIGHT)
			return m_pMap[ptPos.y][ptPos.x - 1];

	if (ptPos.y < MAP_HEIGHT - 1
		&& m_pMap[ptPos.y + 1][ptPos.x]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y + 1][ptPos.x])->GetMovement () == MOVEMENT_UP
			||((CBlockDinamic*)m_pMap[ptPos.y + 1][ptPos.x])->GetMovement () == MOVEMENT_UP_ONGROUP
			||((CBlockDinamic*)m_pMap[ptPos.y + 1][ptPos.x])->GetMovement () == VK_UP)
			return m_pMap[ptPos.y + 1][ptPos.x];

	if (ptPos.y > 0
		&& m_pMap[ptPos.y - 1][ptPos.x]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y - 1][ptPos.x])->GetMovement () == MOVEMENT_DOWN
			||((CBlockDinamic*)m_pMap[ptPos.y - 1][ptPos.x])->GetMovement () == VK_DOWN)
			return m_pMap[ptPos.y - 1][ptPos.x];

	if (ptPos.x < MAP_WIDTH - 1 && ptPos.y > 0
		&& m_pMap[ptPos.y - 1][ptPos.x + 1]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y - 1][ptPos.x + 1])->GetMovement () == MOVEMENT_DROP_LEFT)
			return m_pMap[ptPos.y - 1][ptPos.x + 1];

	if (ptPos.x > 0 && ptPos.y > 0
		&& m_pMap[ptPos.y - 1][ptPos.x - 1]->GetAbstractType () == ABSTRACT_DINAMIC)
		if (((CBlockDinamic*)m_pMap[ptPos.y - 1][ptPos.x - 1])->GetMovement () == MOVEMENT_DROP_RIGHT)
			return m_pMap[ptPos.y - 1][ptPos.x - 1];
	
	return m_pMap[ptPos.y][ptPos.x];
}

//////////////////////////////////////////////////////////////////////
CMapBlock*	CBoulderMap::GetMapItemIndirect (CPoint ptPos)
{
	return m_pMap[ptPos.y][ptPos.x];
}

//////////////////////////////////////////////////////////////////////
CMapBlock* CBoulderMap::CreateNewBlock (int type)
{
	switch (type)
	{
	case BLOCK_SPACE:
		return new CBlockSpace;
	case BLOCK_GROUND:
		return new CBlockGround;
	case BLOCK_ARMOR:
		return new CBlockArmor;
	case BLOCK_GOLD:
		m_nGold++;
		m_nTotalGold++;
		return new CBlockGold;
	case BLOCK_STONE:
		return new CBlockStone;
	case BLOCK_BOULDER:
		return new CBlockBoulder;
	case BLOCK_MONSTER1:
	case BLOCK_MONSTER2:
		m_nMonsters++;
		m_nTotalMonsters++;
		return new CBlockMonster (type);
	case BLOCK_EXPLOSION:
		m_nExplosion++;
		return new CBlockBoom;
	}

	ASSERT (FALSE);
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DeleteBlock (CMapBlock* block)
{
	if (block->GetBlockType () == BLOCK_GOLD)
		m_nGold--;

	if (block->GetBlockType () == BLOCK_MONSTER)
		m_nMonsters--;

	if (block->GetBlockType () == BLOCK_EXPLOSION)
		m_nExplosion--;

	delete block;
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DeleteBlock (CPoint ptPos)
{
	DeleteBlock (GetMapItemIndirect (ptPos));
}

//////////////////////////////////////////////////////////////////////
int CBoulderMap::ConvertFileIndex (BYTE index)
{
	if (index >= 0 && index <=4)
		return BLOCK_SPACE + index;

	switch (index)
	{
	case 8:
		return BLOCK_BOULDER;
	case 20:
		return BLOCK_MONSTER1;
	case 24:
		return BLOCK_MONSTER2;
	}

	ASSERT (FALSE);
}

//////////////////////////////////////////////////////////////////////
BYTE CBoulderMap::ConvertMapIndex (int index)
{
	if (index >= BLOCK_SPACE && index <=BLOCK_SPACE + 4)
		return BLOCK_SPACE + index;

	switch (index)
	{
	case BLOCK_BOULDER:
		return 8;
	case BLOCK_MONSTER1:
		return 20;
	case BLOCK_MONSTER2:
		return 24;
	}

	ASSERT (FALSE);
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DrawMap (CDC *pDc, BOOL bDrawAllMap)
{
	for (int y = 0; y < MAP_HEIGHT; y ++)
		for (int x = 0; x < MAP_WIDTH; x++)
		{
			m_pMap[y][x]->DrawBlock (pDc, CPoint (x, y), bDrawAllMap);
			
			if (m_pMap[y][x]->IsNeedRedraw ())
				m_pMap[y][x]->DrawBlock (pDc, CPoint (x, y), TRUE);

			m_pMap[y][x]->SetRedrawFlag (FALSE);
		}
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DrawMap (CDC *pDc, CPoint ptPos)
{
	m_pMap[ptPos.y][ptPos.x]->DrawBlock (pDc, ptPos, TRUE);
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::ChangeMapPointer (CPoint ptPos, int type)
{
	m_pMap[ptPos.y][ptPos.x] = CreateNewBlock (type);
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::ChangeMapPointer (CPoint ptPos, CMapBlock* pBlock)
{
	m_pMap[ptPos.y][ptPos.x] = pBlock;
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::ChangeMapItem (CPoint ptPos, int type)
{
	ChangeMapItem (ptPos, CreateNewBlock (type));
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::ChangeMapItem (CPoint ptPos, CMapBlock* pBlock)
{
	DeleteBlock (m_pMap[ptPos.y][ptPos.x]);
	m_pMap[ptPos.y][ptPos.x] = pBlock;
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::ChangeMapItem (CMapBlock* pBlock, int type)
{
	for (int y = 0; y < MAP_HEIGHT; y ++)
		for (int x = 0; x < MAP_WIDTH; x++)
			if (m_pMap[y][x] == pBlock)
				ChangeMapItem (CPoint (x, y), type);
}

//////////////////////////////////////////////////////////////////////
CMapBlock* CBoulderMap::SetBoulderPos (CPoint ptPos)
{
	DeleteBlock(GetBoulder ());
	
	CMapBlock* pBoulder = CreateNewBlock (BLOCK_BOULDER);
	ChangeMapItem (CPoint (ptPos.x, ptPos.y), pBoulder);
	
	return pBoulder;
}

//////////////////////////////////////////////////////////////////////
CMapBlock* CBoulderMap::GetBoulder ()
{
	CPoint ptPos = GetBoulderPos ();
	if (ptPos.x == - 1 && ptPos.y == - 1)
		return NULL;
	
	return GetMapItem (ptPos);
}

//////////////////////////////////////////////////////////////////////
CPoint CBoulderMap::GetBoulderPos ()
{
	for (int y = 0; y < MAP_HEIGHT; y ++)
		for (int x = 0; x < MAP_WIDTH; x++)
			if (m_pMap[y][x]->GetBlockType () == BLOCK_BOULDER)
				return CPoint (x, y);

	return CPoint (-1, -1);
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::LoadMap (int levelNum, CString strFileName)
{
	CString strError;
	// Open file & check format
	CFile file;
	if (!file.Open (strFileName, CFile::modeRead))
	{
		strError.LoadString (IDS_FILE_CANT_OPEN);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	CString strLabel;
	strLabel.LoadString (IDS_BOULDER_MAP_LABEL);

	CString header;
	file.Seek (4, SEEK_SET);
	if (file.Read (header.GetBuffer (6), 6) != 6 || header != strLabel)
	{
		file.Close ();
		strError.LoadString (IDS_FILE_WRONG_FORMAT);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	header.ReleaseBuffer ();

	BYTE totalLevelNum;
	file.Read (&totalLevelNum, 1);

	if (!totalLevelNum)
	{
		file.Close ();
		strError.LoadString (IDS_FILE_NO_LEVELS);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	// if format correct -> find specified level
	int pos = 0;
	BYTE curLevelNum = 0;
	BYTE *pBuff = new BYTE[MAP_WIDTH * MAP_HEIGHT * sizeof (BYTE)];

	BOOL bNotFound = TRUE;
	for (int level = 0; level < totalLevelNum; level++)
	{
		file.Read (&curLevelNum, 1);
		
		if (curLevelNum == levelNum)
		{
			file.Read (pBuff, MAP_WIDTH * MAP_HEIGHT * sizeof (BYTE)); 
			file.Close ();
			bNotFound = FALSE;
			break;
		}

		file.Seek (MAP_WIDTH * MAP_HEIGHT, SEEK_CUR);
	}

	if (bNotFound)
	{
		delete pBuff;	
		file.Close ();
		strError.LoadString (IDS_FILE_NO_SPECIFIED_LEVEL);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	if (m_pMap)
		DestroyMap ();
	
	// if level found -> init CMapBlock
	m_pMap = new CMapBlock** [MAP_HEIGHT];

	for (int i = 0; i < MAP_HEIGHT; i++)
		m_pMap[i] = new CMapBlock* [MAP_WIDTH];

	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x < MAP_WIDTH; x++)
			m_pMap[y][x] = CreateNewBlock (ConvertFileIndex (pBuff[x + y * MAP_WIDTH]));

	delete pBuff;

	m_levelNum = levelNum;
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::SaveMap (CString strFileName)
{
	CString strError;
	
	if (!m_pMap)
	{
		strError.LoadString (IDS_FILE_NO_DATA);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}
	
	// Open file for writing & check format
	CFile file;
	if (!file.Open (strFileName, CFile::modeReadWrite))
		if (!CreateEmptyMapFile (strFileName))
			if (!file.Open (strFileName, CFile::modeReadWrite))
			{
				strError.LoadString (IDS_FILE_NOT_EXISTS);
				throw CFileExeption ((LPTSTR)(LPCSTR)strError);
			}

	CString strLabel;
	strLabel.LoadString (IDS_BOULDER_MAP_LABEL);

	CString header;
	file.Seek (4, SEEK_SET);
	if (file.Read (header.GetBuffer (6), 6) != 6 || header != strLabel)
	{
		file.Close ();
		strError.LoadString (IDS_FILE_WRONG_FORMAT);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	header.ReleaseBuffer ();

	BYTE totalLevelNum;
	file.Read (&totalLevelNum, 1);

	BOOL bNotFound = TRUE;

	if (totalLevelNum)
	{
		// if file not empty -> find specified level
		int pos = 0;
		BYTE curLevelNum = 0;
		
		for (int level = 0; level < totalLevelNum; level++)
		{
			file.Read (&curLevelNum, 1);
			
			if (curLevelNum == m_levelNum)
			{
				file.Seek (-1, SEEK_CUR);
				bNotFound = FALSE;
				break;
			}

			file.Seek (MAP_WIDTH * MAP_HEIGHT, SEEK_CUR);
		}
	}

	BYTE *pBuff = new BYTE[MAP_WIDTH * MAP_HEIGHT * sizeof (BYTE)];
	
	int pos = 0;
	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x < MAP_WIDTH; x++)
		{
			CBlockMonster* pBlock = (CBlockMonster*)m_pMap[y][x];
			int type = pBlock->GetBlockType ();

			if (type == BLOCK_MONSTER)
				type = pBlock->GetMonsterType ();

			pBuff[pos++] = ConvertMapIndex (type);
		}
	
	file.Write (&m_levelNum, 1);
	file.Write (pBuff, MAP_WIDTH * MAP_HEIGHT * sizeof (BYTE));
	delete pBuff;
	
	file.Seek (10, SEEK_SET);

	if (bNotFound)
		totalLevelNum++;

	file.Write (&totalLevelNum, 1);
	file.Close ();
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DeleteMap (int levelNum, CString strFileName)
{
	CString strTmpFileName;
	strTmpFileName.LoadString (IDS_TEMP_MAP); 
	CreateEmptyMapFile (strTmpFileName);
	CBoulderMap map;

	BYTE nLevel;
	BOOL bError = FindFirstLevel (strFileName, &nLevel);

	while (bError)
	{
		if (levelNum != nLevel)
		{
			map.LoadMap (nLevel, strFileName);
			map.SaveMap (strTmpFileName);
		}
		
		bError = FindNextLevel (&nLevel);
	}

	CFile::Remove (strFileName);
	CFile::Rename (strTmpFileName, strFileName);
}

//////////////////////////////////////////////////////////////////////
BOOL CBoulderMap::CreateEmptyMapFile (CString strFileName)
{
	CFile file;
	if (!file.Open (strFileName, CFile::modeCreate|CFile::modeWrite))
		return FALSE;
	
	CString header;
	header.LoadString (IDS_BOULDER_MAP_LABEL);
	int crc = 0;
	file.Write (&crc, sizeof(int));
	file.Write (header, 6);

	BYTE totalLevelNum = 0;
	file.Write (&totalLevelNum, 1);
	file.Close ();

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
BOOL CBoulderMap::FindFirstLevel (CString name, BYTE* nLevel)
{
	CString strError;

	// Open file & check format
	CFile file;
	if (!file.Open (name, CFile::modeRead))
	{
		strError.LoadString (IDS_FILE_CANT_OPEN);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	CString strLabel;
	strLabel.LoadString (IDS_BOULDER_MAP_LABEL);
	
	CString header;
	file.Seek (4, SEEK_SET);
	if (file.Read (header.GetBuffer (6), 6) != 6 || header != strLabel)
	{
		file.Close ();
		strError.LoadString (IDS_FILE_WRONG_FORMAT);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	header.ReleaseBuffer ();

	
	file.Read (&m_totalLevelNum, 1);

	m_strFindFileName = name;

	// if format correct -> find specified level
	m_curLevelIndex = 1;

	if (m_totalLevelNum)
	{
		file.Read (nLevel, 1);
		file.Close ();
		return TRUE;
	}
		
	file.Close ();
	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
BOOL CBoulderMap::FindNextLevel (BYTE* nLevel)
{
	CString strError;

	CFile file;
	if (!file.Open (m_strFindFileName, CFile::modeRead))
	{
		strError.LoadString (IDS_FILE_CANT_OPEN);
		throw CFileExeption ((LPTSTR)(LPCSTR)strError);
	}

	file.Seek (11 + (MAP_WIDTH * MAP_HEIGHT * sizeof (BYTE) + 1) * m_curLevelIndex, SEEK_SET);

	if (m_curLevelIndex < m_totalLevelNum)
	{
		file.Read (nLevel, 1);
		file.Close ();

		m_curLevelIndex++;
		return TRUE;
	}

	file.Close ();
	return FALSE;
}

//////////////////////////////////////////////////////////////////////
void CBoulderMap::DestroyMap ()
{
	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x < MAP_WIDTH; x++)
			delete m_pMap[y][x];

	for (int i = 0; i < MAP_HEIGHT; i++)
		delete []m_pMap[i];

	delete []m_pMap;

	m_pMap = NULL;
}


