// MapEditor.cpp : implementation file
//

#include "stdafx.h"
#include "boulder.h"
#include "MapEditor.h"
#include "BlockDinamic.h"
#include "BlockMonster.h"

#include "AddLevelDlg.h"
#include "FileExeption.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CMapEditor

IMPLEMENT_DYNCREATE(CMapEditor, CView)

CMapEditor::CMapEditor()
{
	m_pMap = NULL;

	m_bModifed = FALSE;
	m_bEnableAnimation = TRUE;

	m_ptCursorPos = CPoint (0, 0);

	m_bKeyPressed = FALSE;
	m_bLButtonPressed = FALSE;
}

CMapEditor::~CMapEditor()
{
	delete m_pMap;
}


BEGIN_MESSAGE_MAP(CMapEditor, CView)
	//{{AFX_MSG_MAP(CMapEditor)
	ON_WM_CREATE()
	ON_WM_ERASEBKGND()
	ON_WM_TIMER()
	ON_WM_KEYDOWN()
	ON_WM_KEYUP()
	ON_WM_SHOWWINDOW()
	ON_COMMAND(ID_EDIT_ADDNEWLEVEL, OnEditAddnewlevel)
	ON_COMMAND(ID_EDIT_DELETELEVEL, OnEditDeletelevel)
	ON_COMMAND(ID_EDIT_SHOWTOOLWINDOW, OnEditShowtoolwindow)
	ON_UPDATE_COMMAND_UI(ID_EDIT_SHOWTOOLWINDOW, OnUpdateEditShowtoolwindow)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_CAPTURECHANGED()
	ON_WM_MOUSEMOVE()
	ON_WM_SETCURSOR()
	ON_COMMAND(ID_EDIT_UNDO, OnEditUndo)
	ON_UPDATE_COMMAND_UI(ID_EDIT_UNDO, OnUpdateEditUndo)
	ON_COMMAND(ID_EDIT_REDO, OnEditRedo)
	ON_UPDATE_COMMAND_UI(ID_EDIT_REDO, OnUpdateEditRedo)
	ON_COMMAND(ID_EDIT_SWITCHTOEDITWINDOW, OnEditSwitchtoeditwindow)
	ON_COMMAND(ID_EDIT_SAVELEVEL, OnEditSavelevel)
	//}}AFX_MSG_MAP
	ON_CBN_SELCHANGE (IDC_EB_COMBO, OnEbCombo)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CMapEditor drawing

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnDraw(CDC* pDC)
{
	CDocument* pDoc = GetDocument();

	if (m_pMap)
	{
		CDC memDC;
		memDC.CreateCompatibleDC (pDC);

		CBitmap* oldBm;
		oldBm = memDC.SelectObject (&m_bmBackBuff);

		BITMAP bm;
		m_bmBackBuff.GetBitmap (&bm);
		
		m_pMap->DrawMap (&memDC, TRUE);
		
		if (GetFocus () == this)
		{
			if (m_bLButtonPressed && m_editBar.GetSelectedTool () == EDIT_TOOL_LINE)
				DrawLineCursor (&memDC, m_ptClickPos, m_ptMovePos);
			else
				DrawCursor (&memDC, m_ptCursorPos);
		}

		pDC->BitBlt (0, 0, bm.bmWidth, bm.bmHeight, &memDC, 0, 0, SRCCOPY);

		memDC.SelectObject (oldBm);
	}
}


/////////////////////////////////////////////////////////////////////////////
void CMapEditor::DrawCursor (CDC* pDC, CPoint ptPos)
{
	pDC->PatBlt (ptPos.x * BLOCK_WIDTH, ptPos.y * BLOCK_HEIGHT, BLOCK_WIDTH, BLOCK_HEIGHT, DSTINVERT);
}

/////////////////////////////////////////////////////////////////////////////
// CMapEditor diagnostics

#ifdef _DEBUG
void CMapEditor::AssertValid() const
{
	CView::AssertValid();
}

void CMapEditor::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CMapEditor message handlers

int CMapEditor::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	CRect rc;
	GetClientRect (&rc);

	CClientDC dc(this);
	m_bmBackBuff.CreateCompatibleBitmap (&dc, rc.Width (), rc.Height ());

	CFrameWnd* pParent = (CFrameWnd*)GetParent ();
	m_editBar.Create (pParent, IDD_EDIT_BAR, CBRS_TOOLTIPS|CBRS_ALIGN_ANY, IDD_EDIT_BAR);

	pParent->EnableDocking (CBRS_ALIGN_ANY);
	m_editBar.EnableDocking (0);
	m_editBar.ModifyStyle (WS_CAPTION, 0);

	pParent->FloatControlBar(&m_editBar, CPoint (100, 100));
	m_editBar.InitEditBar (this);
	
	CString strMapName;
	strMapName.LoadString (IDS_DEFUALT_EDITOR_MAP_NAME);
	SetFileName (strMapName);
	
	if (TestExistanceOfMapFile ())
	{
		BYTE iLevel = 0;
		CBoulderMap::FindFirstLevel (m_strFileName, &iLevel);
	
		SelectLevel (iLevel, m_strFileName);
	}

	InitLevelList (m_editBar.GetLevelList ());

	StartTimer ();
	
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::OnEraseBkgnd(CDC* pDC) 
{
	return TRUE;		
	//return CView::OnEraseBkgnd(pDC);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::SetFileName (CString strFileName)
{
	m_strFileName = strFileName;
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::InitLevelList (CComboBox* pCombo)
{
	try
	{
		CString strLevelName, strLevel;
		strLevel.LoadString (IDS_LEVEL_NUM);

		BYTE nLevel;
		BOOL bError = CBoulderMap::FindFirstLevel (m_strFileName, &nLevel);

		while (bError)
		{
			strLevelName.Format (strLevel, nLevel + 1);
			pCombo->AddString (strLevelName);
			pCombo->SetItemData (pCombo->FindString(0, strLevelName), nLevel);

			bError = CBoulderMap::FindNextLevel (&nLevel);
		}
	}
	catch (CFileExeption error)
	{
		AfxMessageBox (error);
	}
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::SaveModifedLevel ()
{
	if (!m_pMap)
		return TRUE;
	
	if (m_bModifed)
	{
		CString str, strf;
		str.LoadString (IDS_SAVE_EDIT_CHANGES);
		strf.Format (str, m_pMap->GetLevelNumber () + 1);

		StopTimer ();
		BOOL bAnswer = AfxMessageBox (strf, MB_YESNOCANCEL|MB_ICONWARNING);
		StartTimer ();
		
		if (bAnswer == IDCANCEL)
			return FALSE;

		if (bAnswer == IDYES)
		{
			bAnswer = IDRETRY;

			while (bAnswer == IDRETRY)
			{
				try
				{
					m_pMap->SaveMap (m_strFileName);
					bAnswer = IDOK;
				}
				catch (CFileExeption error)
				{
					bAnswer = AfxMessageBox (error, MB_RETRYCANCEL|MB_ICONERROR);

					if (bAnswer == IDCANCEL)
						return FALSE;
				}
			}
		}
	}
	m_bModifed = FALSE;

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::SelectLevel (int nLevel, CString strMapName)
{
	if (m_pMap)
		if (m_pMap->GetLevelNumber () == nLevel)
			return;

	if (!SaveModifedLevel ())
		return;

	if (!m_pMap)
		m_pMap = new CBoulderMap ();

	try
	{
		m_pMap->LoadMap (nLevel, strMapName);
	}
	catch (CFileExeption error)
	{
		AfxMessageBox (error, MB_OK|MB_ICONERROR);
	}

	m_ptCursorPos = CPoint (0, 0);
	m_oUndoRedo.Reset ();
}

//////////////////////////////////////////////////////////////////////
void CMapEditor::IncreaseState ()
{
	for (int y = 0; y < MAP_HEIGHT; y ++)
		for (int x = 0; x < MAP_WIDTH; x++)
		{
			int type = (m_pMap->GetMapItemIndirect (CPoint (x, y)))->GetAbstractType ();
			if (type == ABSTRACT_DINAMIC || type == ABSTRACT_STATIC)
				((CBlockDinamic*)m_pMap->GetMapItemIndirect (CPoint (x, y)))->IncreaseState ();
		}
}

//////////////////////////////////////////////////////////////////////
void CMapEditor::OnTimer(UINT nIDEvent) 
{
	if (m_pMap && m_bEnableAnimation)
		IncreaseState ();
	
	Invalidate (FALSE);
	UpdateWindow ();

	CView::OnTimer(nIDEvent);
}

//////////////////////////////////////////////////////////////////////
void CMapEditor::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (nChar >= 0x30 && nChar <= 0x37)
	{
		m_oUndoRedo.Update (m_pMap);

		m_bKeyPressed = TRUE;
		m_editKeyPressed = nChar;

		SetMapBlockInMap (m_ptCursorPos, m_editBar.IndexToBlock (nChar - 0x30));
		m_editBar.SetSelectedBlock (nChar - 0x30);
	}
	else
	{
		switch (nChar)
		{
		case VK_RIGHT: 
			if (m_ptCursorPos.x < MAP_WIDTH - 1)
				m_ptCursorPos.x++;
			break;
		case VK_LEFT:
			if (m_ptCursorPos.x > 0)
				m_ptCursorPos.x--;
			break;
		case VK_DOWN:
			if (m_ptCursorPos.y < MAP_HEIGHT - 1)
				m_ptCursorPos.y++;
			break;
		case VK_UP:
			if (m_ptCursorPos.y > 0)
				m_ptCursorPos.y--;
			break;
		}

		if (m_bKeyPressed)
		{
			SetMapBlockInMap (m_ptCursorPos, m_editBar.IndexToBlock (m_editKeyPressed - 0x30));
			Invalidate (FALSE);
			UpdateWindow ();
		}
	}

	CView::OnKeyDown(nChar, nRepCnt, nFlags);
}

//////////////////////////////////////////////////////////////////////
void CMapEditor::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (nChar >= 0x30 && nChar <= 0x37)
		m_bKeyPressed = FALSE;

	CView::OnKeyUp(nChar, nRepCnt, nFlags);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::StartTimer ()
{
	SetTimer (TIMER_EDITOR, EDITOR_INTERNAL_TIMER_DELAY, NULL);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::StopTimer ()
{
	KillTimer (TIMER_EDITOR);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnShowWindow(BOOL bShow, UINT nStatus) 
{
	CView::OnShowWindow(bShow, nStatus);
	
	CFrameWnd* pParent = (CFrameWnd*)GetParent ();
	pParent->ShowControlBar (&m_editBar, bShow, FALSE);

	if (bShow)
		StartTimer ();
	else
		StopTimer ();
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEbCombo ()
{
	if (m_editBar.GetSelectedLevel () != -1)
		SelectLevel (m_editBar.GetSelectedLevel (), m_strFileName); 
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditAddnewlevel() 
{
	CAddLevelDlg dlg;
	dlg.m_nLevel = m_pMap->GetLevelNumber () + 1;
	if (dlg.DoModal () == IDCANCEL)
		return;

	AddNewLevel (dlg.m_nLevel - 1, dlg.m_iFill);

	m_oUndoRedo.Reset ();
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditDeletelevel() 
{
	if (m_editBar.GetSelectedLevel () != -1)
		DeleteLevel (m_editBar.GetSelectedLevel ());

	m_oUndoRedo.Reset ();
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditShowtoolwindow() 
{
	CFrameWnd* pParent = (CFrameWnd*)GetParent ();
	pParent->ShowControlBar (&m_editBar, !(m_editBar.IsWindowVisible ()), FALSE);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnUpdateEditShowtoolwindow(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck (m_editBar.IsWindowVisible ());
	pCmdUI->Enable (IsWindowVisible ());
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::OnCmdMsg(UINT nID, int nCode, void* pExtra, AFX_CMDHANDLERINFO* pHandlerInfo) 
{
	if (IsWindow (m_editBar.m_hWnd) &&
		m_editBar.IsWindowVisible () &&
		m_editBar.OnCmdMsg (nID, nCode, pExtra, pHandlerInfo))
		return TRUE;
	else
		return CView::OnCmdMsg(nID, nCode, pExtra, pHandlerInfo);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnLButtonDown(UINT nFlags, CPoint point) 
{
	if (m_editBar.GetSelectedBlock () == -1)
		return;
	
	m_oUndoRedo.Update (m_pMap);

	switch (m_editBar.GetSelectedTool ())
	{
	case EDIT_TOOL_NORMAL:
		SetMapBlockInView (point, m_editBar.GetSelectedBlock ());
		SetCapture ();
		m_bLButtonPressed = TRUE;
		break;
	case EDIT_TOOL_LINE:
		m_ptClickPos = ViewToMap (point);
		m_ptMovePos = m_ptClickPos;
		SetCapture ();
		m_bLButtonPressed = TRUE;
		break;
	case EDIT_TOOL_PAINT:
		PaintMapInView (point, m_editBar.GetSelectedBlock (), GetMapBlock (ViewToMap (point)));
		break;
	case EDIT_TOOL_FILL:
		FillMap (m_editBar.GetSelectedBlock ());
		break;
	}
	
	Invalidate (FALSE);
	UpdateWindow ();

	CView::OnLButtonDown(nFlags, point);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnLButtonUp(UINT nFlags, CPoint point) 
{
	ReleaseCapture ();
	
	//CView::OnLButtonUp(nFlags, point);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnMouseMove(UINT nFlags, CPoint point) 
{
	if (m_bLButtonPressed)
	{
		switch (m_editBar.GetSelectedTool ())
		{
		case EDIT_TOOL_NORMAL:
			SetMapBlockInView (point, m_editBar.GetSelectedBlock ());
			break;
		case EDIT_TOOL_LINE:
			m_ptMovePos = ViewToMap (point);
			break;
		}
	}
	
	Invalidate (FALSE);
	UpdateWindow ();
			
	CView::OnMouseMove(nFlags, point);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnCaptureChanged(CWnd *pWnd) 
{
	m_bLButtonPressed = FALSE;

	switch (m_editBar.GetSelectedTool ())
	{
	case EDIT_TOOL_LINE:
		DrawLineInMap (m_ptClickPos, m_ptMovePos);
		break;
	}

	Invalidate (FALSE);
	UpdateWindow ();

	CView::OnCaptureChanged(pWnd);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::SetMapBlockInView (CPoint ptView, int block)
{
	SetMapBlockInMap (ViewToMap (ptView), block);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::SetMapBlockInMap (CPoint ptMap, int block)
{
	m_bModifed = TRUE;

	if (ptMap.x >= 0 && ptMap.x < MAP_WIDTH 
		&& ptMap.y >=0 && ptMap.y < MAP_HEIGHT)
		m_pMap->ChangeMapItem (ptMap, block);
}

/////////////////////////////////////////////////////////////////////////////
int CMapEditor::GetMapBlock (CPoint ptMap)
{
	if (ptMap.x >= 0 && ptMap.x < MAP_WIDTH 
		&& ptMap.y >=0 && ptMap.y < MAP_HEIGHT)
		{
		int type = m_pMap->GetMapItemIndirect (ptMap)->GetBlockType ();
		
		if (type == BLOCK_MONSTER)
			return ((CBlockMonster*)m_pMap->GetMapItemIndirect(ptMap))->GetMonsterType ();
		else 
			return type;
		}

	return BLOCK_NULL;
}

/////////////////////////////////////////////////////////////////////////////
void CALLBACK LineDDAProc (int x, int y, LPARAM lParam)
{
	if (x >= 0 && x < MAP_WIDTH && y >=0 && y < MAP_HEIGHT)
	{
		CURSOR_PARAM* pParam = (CURSOR_PARAM*)lParam;
		pParam->pMap->ChangeMapItem (CPoint (x , y), pParam->pBar->GetSelectedBlock ());
	}
}
  
/////////////////////////////////////////////////////////////////////////////
void CMapEditor::DrawLineInMap (CPoint beg, CPoint end)
{
	m_bModifed = TRUE;

	CURSOR_PARAM param;
	param.pMap = m_pMap;
	param.pBar = &m_editBar;

	::LineDDA (beg.x, beg.y, end.x, end.y, LineDDAProc, (LPARAM)&param);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::PaintMapInView (CPoint ptView, int block, int wayBlock)
{
	PaintMapInMap (ViewToMap (ptView), block, wayBlock);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::PaintMapInMap (CPoint ptMap, int block, int wayBlock)
{
	if (block == wayBlock)
		return;
	
	SetMapBlockInMap (ptMap, block);

	if ((GetMapBlock (CPoint (ptMap.x + 1, ptMap.y)) != BLOCK_NULL)
		&& GetMapBlock (CPoint (ptMap.x + 1, ptMap.y)) == wayBlock)
		PaintMapInMap (CPoint (ptMap.x + 1, ptMap.y), block, wayBlock);

	if ((GetMapBlock (CPoint (ptMap.x - 1, ptMap.y)) != BLOCK_NULL)
		&& GetMapBlock (CPoint (ptMap.x - 1, ptMap.y)) == wayBlock)
		PaintMapInMap (CPoint (ptMap.x - 1, ptMap.y), block, wayBlock);

	if ((GetMapBlock (CPoint (ptMap.x, ptMap.y + 1)) != BLOCK_NULL)
		&& GetMapBlock (CPoint (ptMap.x, ptMap.y + 1)) == wayBlock)
		PaintMapInMap (CPoint (ptMap.x, ptMap.y + 1), block, wayBlock);

	if ((GetMapBlock (CPoint (ptMap.x, ptMap.y - 1)) != BLOCK_NULL)
		&& GetMapBlock (CPoint (ptMap.x, ptMap.y - 1)) == wayBlock)
		PaintMapInMap (CPoint (ptMap.x, ptMap.y - 1), block, wayBlock);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::FillMap (int block)
{
	for (int y = 0; y < MAP_HEIGHT; y++)
		for (int x = 0; x <MAP_WIDTH; x++)
			SetMapBlockInMap (CPoint (x , y), block);
}

/////////////////////////////////////////////////////////////////////////////
void CALLBACK CursorDDAProc (int x, int y, LPARAM lpDC)
{
	CDC* pDC = (CDC*)lpDC;
	
	CPoint ptMap(x * BLOCK_WIDTH, y * BLOCK_HEIGHT);
	CRect rc = CRect (ptMap, CPoint (ptMap.x + BLOCK_WIDTH, ptMap.y + BLOCK_HEIGHT));
	pDC->InvertRect (&rc);
}
  
/////////////////////////////////////////////////////////////////////////////
void CMapEditor::DrawLineCursor (CDC* pMemDC, CPoint beg, CPoint end)
{ 
	::LineDDA (beg.x, beg.y, end.x, end.y, CursorDDAProc, (LPARAM)pMemDC);
}

/////////////////////////////////////////////////////////////////////////////
CPoint CMapEditor::ViewToMap (CPoint ptView)
{
	return CPoint (ptView.x / BLOCK_WIDTH, ptView.y / BLOCK_HEIGHT);
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	if (m_editBar.GetSelectedBlock () == BLOCK_NULL)
	{
		SetCursor (AfxGetApp ()->LoadCursor (IDC_NOTDEFINED));
		return TRUE;
	}
	
	switch (m_editBar.GetSelectedTool ())
	{
	case EDIT_TOOL_NORMAL:
		SetCursor (AfxGetApp ()->LoadCursor (IDC_NORMAL));
		return TRUE;
	case EDIT_TOOL_LINE:
		SetCursor (AfxGetApp ()->LoadCursor (IDC_LINE));
		return TRUE;
	case EDIT_TOOL_PAINT:
		SetCursor (AfxGetApp ()->LoadCursor (IDC_PAINT));
		return TRUE;
	case EDIT_TOOL_FILL:
		SetCursor (AfxGetApp ()->LoadCursor (IDC_FILL));
		return TRUE;
	}
		
	return CView::OnSetCursor(pWnd, nHitTest, message);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditUndo() 
{
	m_oUndoRedo.Undo (m_pMap);	
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnUpdateEditUndo(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (m_oUndoRedo.CanUndo ());
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditRedo() 
{
	m_oUndoRedo.Redo (m_pMap);
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnUpdateEditRedo(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable (m_oUndoRedo.CanRedo ());
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditSwitchtoeditwindow() 
{
	CWnd* pCurFocus = GetFocus ();
	if (pCurFocus == SetFocus ())
		m_editBar.SetFocus ();
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::AddNewLevel (int iLevel, int fill)
{
	if (!SaveModifedLevel ())
		return FALSE;
	
	CString strLevelName;
	
	BYTE nLevel;
	try
	{
		BOOL bError = CBoulderMap::FindFirstLevel (m_strFileName, &nLevel);

		while (bError)
		{
			if (iLevel == nLevel)
			{
				StopTimer ();
				int iAnswer = AfxMessageBox (IDS_LEVEL_EXISTS, MB_YESNO | MB_ICONSTOP);
				StartTimer ();
				
				if (iAnswer == IDYES)
					break;

				return FALSE;
			}

			bError = CBoulderMap::FindNextLevel (&nLevel);
		}
	}
	catch (CFileExeption error)
	{
		AfxMessageBox (error);
		return FALSE;
	}	

	if (iLevel != nLevel)
	{
		CString strLevel;
		strLevel.LoadString (IDS_LEVEL_NUM);
		strLevelName.Format (strLevel, iLevel + 1);
		m_editBar.GetLevelList ()->AddString (strLevelName);
		m_editBar.GetLevelList ()->SetItemData (m_editBar.GetLevelList ()->FindString(0, strLevelName), iLevel);
	}

	delete m_pMap;
	m_pMap = new CBoulderMap (iLevel, fill);
	m_pMap->SetLevelNumber (iLevel);

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::DeleteLevel (int iLevel)
{
	CString str;
	str.LoadString (IDS_DELETE_ALL_LEVELS);
	if (m_editBar.GetLevelList ()->GetCount () == 1)
	{
		AfxMessageBox (str, MB_OK|MB_ICONSTOP);
		return FALSE;
	}

	CString str1;
	str.LoadString (IDS_DELETE_LEVEL);
	str1.Format (str, iLevel + 1);

	if (AfxMessageBox (str1, MB_YESNO|MB_ICONQUESTION) == IDYES)
	{
		CBoulderMap::DeleteMap (iLevel, "boulder.mps");
		
		for (int i =0; i < m_editBar.GetLevelList ()->GetCount (); i++)
		{
			if (m_editBar.GetLevelList ()->GetItemData (i) == iLevel)
			{
				m_editBar.GetLevelList ()->DeleteString (i);
				break;
			}
		}

		m_pMap->LoadMap (m_editBar.GetLevelList ()->GetItemData (0), "boulder.mps");

		return TRUE;
	}

	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
void CMapEditor::OnEditSavelevel() 
{
	m_pMap->SaveMap (m_strFileName);
	m_bModifed = FALSE;
	
	CWnd* pWnd = GetFocus ();
	AfxMessageBox (IDS_LEVEL_IS_SAVED, MB_OK);
	pWnd->SetFocus ();
}

/////////////////////////////////////////////////////////////////////////////
BOOL CMapEditor::TestExistanceOfMapFile ()
{
	CFileStatus status;
	BOOL bExists;
	bExists =  CFile::GetStatus (m_strFileName, status);
	
	if (!bExists)
	{
		CBoulderMap::CreateEmptyMapFile (m_strFileName);
		m_pMap = new CBoulderMap (0, BLOCK_GROUND);
		m_pMap->SetLevelNumber (0);
		m_pMap->SaveMap (m_strFileName);
	}

	return bExists;
}
