/*
 * Game debugging tool for DUNGEON
 *
 * COPYRIGHT 1980, 1990, INFOCOM COMPUTERS AND COMMUNICATIONS, CAMBRIDGE MA.
 * ALL RIGHTS RESERVED, COMMERCIAL USAGE STRICTLY PROHIBITED
 * WRITTEN BY R. M. SUPNIK
 */

#include "glk.h"

#include "miscfort.h"
#include "commons.h"
#include "funcs.h"

#ifdef HAVE_GDT

/*
 * Translated for use in dungeon-glk by Paul Goyette
 *
 * This translation is hereby placed into the public domain.
 */

#include <stdio.h> /* for snprintf */

/*
 * Declarations
 */
#define	valid1(a1, l1)		((a1 > 0) && (a1 <= l1))
#define valid2(a1, a2, l1)	(valid1(a1, l1) && valid1(a2, l1) && \
					(a1 <= a2))
#define valid3(a1, l1, a2, l2)	(valid1(a1, l1) && valid1(a2, l2))

/*
 * The debugger
 */
int gdt_(void)
{
static struct dbgcmd_t {
	char command[4];
	char cmd_sml[4];
	int  arg_typ;
	char *descr;
} dbgcmd[] = {
	{ "DR", "dr", 2, "Display ROOMS" }, { "DO", "do", 2, "Display OBJS" },
	{ "DA", "da", 2, "Display ADVS"  }, { "DC", "dc", 2, "Display CEVENT" },
	{ "DX", "dx", 2, "Display EXITS" }, { "DH", "dh", 0, "Display HACKS" },
	{ "DL", "dl", 0, "Display lngth" }, { "DV", "dv", 2, "Display VILLS" },
	{ "DF", "df", 2, "Display FINDX" }, { "DS", "ds", 0, "Display state" },
	{ "AF", "af", 1, "Alter FINDX"   }, { "HE", "he", 0, "This helptext" },
	{ "NR", "nr", 0, "No robber"     }, { "NT", "nt", 0, "No troll" },
	{ "NC", "nc", 0, "No cyclop"     }, { "ND", "nd", 0, "No dying" },
	{ "RR", "rr", 0, "Restore robber"}, { "RT", "rt", 0, "Restore troll" },
	{ "RC", "rc", 0, "Restore cyclop"}, { "RD", "rd", 0, "Restore death" },
	{ "TK", "tk", 1, "Take"          }, { "EX", "ex", 0, "Exit debugger" },
	{ "AR", "ar", 3, "Alter ROOMS"   }, { "AO", "ao", 3, "Alter OBJS" },
	{ "AA", "aa", 3, "Alter ADVS"    }, { "AC", "ac", 3, "Alter CEVENT" },
	{ "AX", "ax", 1, "Alter EXITS"   }, { "AV", "av", 3, "Alter VILLS" },
	{ "D2", "d2", 2, "Display ROOM2" }, { "DN", "dn", 2, "Display switch" },
	{ "AN", "an", 1, "Alter switch"  }, { "DM", "dm", 2, "Display RTEXT" },
	{ "DT", "dt", 1, "Display TEXT"  }, { "AH", "ah", 0, "Alter HERE" },
	{ "DP", "dp", 0, "Display parser"}, { "PD", "pd", 0, "Program detail" },
	{ "DZ", "dz", 0, "Display puzzle"}, { "AZ", "az", 1, "Alter puzzle" }
};

#define	MAX_DBGCMD (sizeof(dbgcmd) / sizeof(dbgcmd[0]))

	char *cmdbuf, buf[80], outbuf[80];
	integer *dest_array;
	int i, j, k, cmdlen;
	integer arg1, arg2;
	int f_base, f_max, n_sw, n_fl;
	integer c__101 = 101;	/* Constant for Cyclops Room */
	char *f_type;

	if (!misc_1.gdtflg) {
		glk_put_string("You are not an authorized user.\n");
		return 0;
	}

	while (1) {
		/* Here to get next command. */
		glk_put_string("GDT> ");
		cmdlen = getline_null_(buf, sizeof(buf) - 1);

		cmdbuf = buf;
		if (cmdlen == 0)
			continue;

		for (i = 0; i < MAX_DBGCMD; i++) {
			if (s_cmp(cmdbuf, dbgcmd[i].command, cmdlen, 2) == 0)
				break;
			if (s_cmp(cmdbuf, dbgcmd[i].cmd_sml, cmdlen, 2) == 0)
				break;
		}
		if ( i >= MAX_DBGCMD) {
			snprintf(outbuf, sizeof(outbuf),
				"Unrecognized GDT command %s\n", cmdbuf);
			glk_put_string(outbuf);
			continue;
		}

		switch (dbgcmd[i].arg_typ) {
		    case 0:	/* No args		*/
			arg1 = 0;
			arg2 = 0;
			break;
		    case 1:	/* Entry number		*/
			glk_put_string("Entry: ");
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg1 = str_to_num(buf);
			arg2 = 0;
			break;
		    case 2:	/* Lower/Upper limits	*/
			glk_put_string("Lower limit: ");
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg1 = str_to_num(buf);
			glk_put_string("Upper limit: ");
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			break;
		    case 3:	/* Row/Column		*/
			glk_put_string("Index: ");
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg1 = str_to_num(buf);
			glk_put_string("Column: ");
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			break;
		    default:
			weeprintf("Internal error - bad arg_type: %d\n",
				dbgcmd[i].arg_typ);
			return 0;
		}

		/*
		 * Handle the various commands
		 */
		switch (i) {
		    case 0:	/* DR - Display Rooms */
			if (!valid2(arg1, arg2, rooms_1.rlnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("RM#  DESC1  EXITS ACTION  VALUE  FLAGS\n");
			for (i = arg1; i <= arg2; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%3d  %5d  %5d  %5d  %5d 0x%04x\n", i,
					rooms_1.rdesc1[i - 1],
					rooms_1.rexit[i - 1],
					rooms_1.ractio[i - 1],
					rooms_1.rval[i - 1],
					rooms_1.rflag[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		   case 22:	/* AR - Alter Room Entry */
			if (!valid1(arg1, rooms_1.rlnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			dest_array = NULL;
			switch (arg2) {
			    case 1:
				dest_array = &rooms_1.rdesc1[0];
				break;
			    case 2:
				dest_array = &rooms_1.rexit[0];
				break;
			    case 3:
				dest_array = &rooms_1.ractio[0];
				break;
			    case 4:
				dest_array = &rooms_1.rval[0];
				break;
			    case 5:
				dest_array = &rooms_1.rflag[0];
				break;
			}
			if (dest_array == NULL) {
				glk_put_string("Invalid field number.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				dest_array[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			dest_array[arg1 - 1] = arg2;
			break;

		    case 1:	/* DO - Display Objects */
			if (!valid2(arg1, arg2, objcts_1.olnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("OB#  DESC1  DESC2  DESCO  ACT FLAGS1 "
				       "FLAGS2  FV TV\n");
			glk_put_string("      SIZE  CAPAC   ROOM  ADV    CAN "
				       "  READ\n");
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%3d  %5d  %5d  %5d  %3d 0x%04x "
					"0x%04x  %2d %2d\n", i,
					objcts_1.odesc1[i - 1],
					objcts_1.odesc2[i - 1],
					objcts_1.odesco[i - 1],
					objcts_1.oactio[i - 1],
					objcts_1.oflag1[i - 1],
					objcts_1.oflag2[i - 1],
					objcts_1.ofval[i - 1],
					objcts_1.otval[i - 1]);
				glk_put_string(outbuf);
				snprintf(outbuf, sizeof(outbuf),
					"     %5d  %5d  %5d  %3d  %5d  %5d\n",
					objcts_1.osize[i - 1],
					objcts_1.ocapac[i - 1],
					objcts_1.oroom[i - 1],
					objcts_1.oadv[i - 1],
					objcts_1.ocan[i - 1],
					objcts_1.oread[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 23:	/* AO - Alter Object Entry */
			if (!valid1(arg1, objcts_1.olnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			dest_array = NULL;
			switch (arg2) {
			    case 1:
				dest_array = &objcts_1.odesc1[0];
				break;
			    case 2:
				dest_array = &objcts_1.odesc2[0];
				break;
			    case 3:
				dest_array = &objcts_1.odesco[0];
				break;
			    case 4:
				dest_array = &objcts_1.oactio[0];
				break;
			    case 5:
				dest_array = &objcts_1.oflag1[0];
				break;
			    case 6:
				dest_array = &objcts_1.oflag2[0];
				break;
			    case 7:
				dest_array = &objcts_1.ofval[0];
				break;
			    case 8:
				dest_array = &objcts_1.otval[0];
				break;
			    case 9:
				dest_array = &objcts_1.osize[0];
				break;
			    case 10:
				dest_array = &objcts_1.ocapac[0];
				break;
			    case 11:
				dest_array = &objcts_1.oroom[0];
				break;
			    case 12:
				dest_array = &objcts_1.oadv[0];
				break;
			    case 13:
				dest_array = &objcts_1.ocan[0];
				break;
			    case 14:
				dest_array = &objcts_1.oread[0];
				break;
			}
			if (dest_array == NULL) {
				glk_put_string("Invalid field number.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				dest_array[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			dest_array[arg1 - 1] = arg2;
			break;

		    case 2:	/* DA - Display Adventurers */
			if (!valid2(arg1, arg2, advs_1.alnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("AD#   ROOM  SCORE  VEHIC OBJECT ACTION"
				       "  STREN  FLAGS\n");
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%3d  %5d  %5d  %5d  %5d  %5d  %5d "
					"0x%04x\n", i,
					advs_1.aroom[i - 1],
					advs_1.ascore[i - 1],
					advs_1.avehic[i - 1],
					advs_1.aobj[i - 1],
					advs_1.aactio[i - 1],
					advs_1.astren[i - 1],
					advs_1.aflag[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 24:	/* AA - Alter Advs Entry */
			if (!valid1(arg1, advs_1.alnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			dest_array = NULL;
			switch (arg2) {
			    case 1:
				dest_array = &advs_1.aroom[0];
				break;
			    case 2:
				dest_array = &advs_1.ascore[0];
				break;
			    case 3:
				dest_array = &advs_1.avehic[0];
				break;
			    case 4:
				dest_array = &advs_1.aobj[0];
				break;
			    case 5:
				dest_array = &advs_1.aactio[0];
				break;
			    case 6:
				dest_array = &advs_1.astren[0];
				break;
			    case 7:
				dest_array = &advs_1.aflag[0];
				break;
			}
			if (dest_array == NULL) {
				glk_put_string("Invalid field number.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				dest_array[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			dest_array[arg1 - 1] = arg2;
			break;

		    case 3:	/* DC - Display Clock Events */
			if (!valid2(arg1, arg2, cevent_1.clnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("CL#   TICK ACTION   FLAG CANCEL\n");
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%3d  %5d  %5d  %5d  %5d\n", i,
					cevent_1.ctick[i - 1],
					cevent_1.cactio[i - 1],
					cevent_1.cflag[i - 1],
					cevent_1.ccncel[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 25:	/* AC - Alter Clock Events */
			if (!valid1(arg1, cevent_1.clnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			dest_array = NULL;
			switch (arg2) {
			    case 1:
				dest_array = &cevent_1.ctick[0];
				break;
			    case 2:
				dest_array = &cevent_1.cactio[0];
				break;
			    case 3:
				dest_array = &cevent_1.cflag[0];
				break;
			    case 4:
				dest_array = &cevent_1.ccncel[0];
				break;
			}
			if (dest_array == NULL) {
				glk_put_string("Invalid field number.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				dest_array[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			dest_array[arg1 - 1] = arg2;
			break;

		    case 4:	/* DX - Display exits */
			if (!valid2(arg1, arg2, exits_1.xlnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("    RANGE CONTENTS\n");
			j = 0;
			for (i = arg1; i <= arg2 ; i += 10) {
				j = i + 9;
				if (j > arg2)
					j = arg2;
				snprintf(outbuf, sizeof(outbuf),
					"%4d-%4d", i, j);
				glk_put_string(outbuf);
				for (k = i; k <= j; k++) {
					snprintf(outbuf, sizeof(outbuf),
					    (exits_1.travel[k - 1] >= 0)?
						"  %5d":" 0x%04x",
						exits_1.travel[k - 1] & 0xffff);
					glk_put_string(outbuf);
				}
				glk_put_string("\n");
			}
			glk_set_style(style_Normal);
			break;

		    case 26:	/* AX - Alter Exits */
			if (!valid1(arg1, exits_1.xlnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				exits_1.travel[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			exits_1.travel[arg1 - 1] = arg2;
			break;

		    case 5:	/* DH - Display Hacks */
			glk_set_style(style_Preformatted);
			snprintf(outbuf, sizeof(outbuf),
				"THFPOS=%d, THFFLG=0x%04x, THFACT=0x%04x, "
				"SWDACT=0x%04x, SWDSTA=%d\n",
				hack_1.thfpos, hack_1.thfflg, hack_1.thfact,
				hack_1.swdact, hack_1.swdsta);
			glk_put_string(outbuf);
			glk_set_style(style_Normal);
			break;
			
		    case 6:	/* DL - Display Lengths */
			glk_set_style(style_Preformatted);
			snprintf(outbuf, sizeof(outbuf),
				"RLNT=%5d, XLNT=%5d,  OLNT=%5d,  CLNT=%5d, "
				"  VLNT=%5d\n",
				rooms_1.rlnt, exits_1.xlnt, objcts_1.olnt,
				cevent_1.clnt, vill_1.vlnt);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"ALNT=%5d, MLNT=%5d, R2LNT=%5d, MBASE=%5d, "
				"STRBIT=%5d\n",
				advs_1.alnt, rmsg_1.mlnt, oroom2_1.r2lnt,
				misc_1.mbase, misc_1.strbit);
			glk_put_string(outbuf);
			glk_set_style(style_Normal);
			break;

		    case 7:	/* DV - Display Villains */
			if (!valid2(arg1, arg2, vill_1.vlnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("VL# OBJECT   PROB   OPPS   BEST  "
				       "MELEE\n");
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%3d  %5d  %5d  %5d  %5d  %5d\n", i,
					vill_1.villns[i - 1],
					vill_1.vprob[i - 1],
					vill_1.vopps[i - 1],
					vill_1.vbest[i - 1],
					vill_1.vmelee[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 27:	/* AV - Alter Villains */
			if (!valid1(arg1, vill_1.vlnt)) {
				glk_put_string("Invalid entry.\n");
				break;
			}
			dest_array = NULL;
			switch (arg2) {
			    case 1:
				dest_array = &vill_1.villns[0];
				break;
			    case 2:
				dest_array = &vill_1.vprob[0];
				break;
			    case 3:
				dest_array = &vill_1.vopps[0];
				break;
			    case 4:
				dest_array = &vill_1.vbest[0];
				break;
			    case 5:
				dest_array = &vill_1.vmelee[0];
				break;
			}
			if (dest_array == NULL) {
				glk_put_string("Invalid field number.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				dest_array[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			dest_array[arg1 - 1] = arg2;
			break;

		    case 8:	/* DF - Display Flags */
		    case 29:	/* DN - Display Switches */
			n_fl = (findex_switches - findex_flags) /
				sizeof(logical);
			n_sw = sizeof(findex_1) / sizeof(logical) - n_sw;
			if (i == 8) {
				f_base = 0;
				f_max  = n_fl;
				f_type = "Flag  ";
			} else {
				f_base = n_fl;
				f_max  = n_sw;
				f_type = "Switch";
			}
		
			if (!valid2(arg1, arg2, f_max)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					"%s #%2d = %d\n", f_type, i,
					findex_flags[i - 1 + f_base]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 10:	/* AF - Alter Flags */
		    case 30:	/* AN - Alter Switches */
			n_fl = (findex_switches - findex_flags) /
				sizeof(logical);
			n_sw = sizeof(findex_1) / sizeof(logical) - n_sw;
			if (i == 10) {
				f_base = 0;
				f_max  = n_fl;
				f_type = "Flag  ";
			} else {
				f_base = n_fl;
				f_max  = n_sw;
				f_type = "Switch";
			}
			if (!valid1(arg1, f_max)) {
				glk_put_string("Invalid index.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				findex_flags[arg1 - 1 + f_base]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			findex_flags[arg1 - 1 + f_base] = arg2;
			break;

		    case 28:	/* D2 - Display Room2 List */
			if (!valid2(arg1, arg2, oroom2_1.r2lnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			for (i = arg1; i <= arg2 ; i++) {
				snprintf(outbuf, sizeof(outbuf),
					" #%2d Room= %5d Obj=%5d\n", i,
					oroom2_1.r2[i - 1], oroom2_1.o2[i - 1]);
				glk_put_string(outbuf);
			}
			glk_set_style(style_Normal);
			break;

		    case 9:	/* DS - Display State */
			glk_set_style(style_Preformatted);
			snprintf(outbuf, sizeof(outbuf),
				"Parse vector= %5d %5d %5d %5d %5d\n",
				prsvec_1.prsa, prsvec_1.prso, prsvec_1.prsi,
				prsvec_1.prswon, prsvec_1.prscon);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"Play  vector= %5d %5d %5d\n",
				play_1.winner, play_1.here, play_1.telflg);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"State vector= %5d %5d %5d %5d %5d %5d %5d\n"
				"              %5d %5d %5d %5d\n",
				play_1.moves, play_1.deaths, play_1.rwscor,
				play_1.mxscor, play_1.mxload, play_1.ltshft,
				play_1.bloc, play_1.mungrm, play_1.hs,
				play_1.egscor, play_1.egmxsc);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"Scol  vector=0x%04x %5d %5d\n",
				screen_1.fromdr, screen_1.scolrm,
				screen_1.scolac);
			glk_put_string(outbuf);
			glk_set_style(style_Normal);
			break;

		    case 11:	/* HE - Help */
			glk_set_style(style_Preformatted);
			for (i = 0; i < MAX_DBGCMD; i++) {
				if ((i % 3) == 0)
					glk_put_string("\n");
				snprintf(outbuf, sizeof(outbuf),
					"%2s (%1d) - %-15s   ",
					dbgcmd[i].command,
					dbgcmd[i].arg_typ,
					dbgcmd[i].descr);
				glk_put_string(outbuf);
			}
			glk_put_string("\n");
			glk_set_style(style_Normal);
			break;

		    case 12:	/* NR - No robber */
			hack_1.thfflg=FALSE_;
			hack_1.thfact=FALSE_;
			newsta_(&c__61, &c__0, &c__0, &c__0, &c__0);
			glk_put_string("No robber.\n");
			break;

		    case 13:	/* NT - No troll */
			findex_1.trollf=TRUE_;
			newsta_(&c__19, &c__0, &c__0, &c__0, &c__0);
			glk_put_string("No troll.\n");
			break;

		    case 14:	/* NC - No cyclops */
			findex_1.cyclof=TRUE_;
			newsta_(&c__58, &c__0, &c__0, &c__0, &c__0);
			glk_put_string("No cyclops.\n");
			break;

		    case 15:	/* ND - No death */		
			misc_1.dbgflg=1;
			glk_put_string("No deaths.\n");
			break;

		    case 16:	/* RR - Restore robber */
			hack_1.thfflg=TRUE_;
			glk_put_string("Restored robber.\n");
			break;

		    case 17:	/* RT - Restore troll */
			findex_1.trollf=FALSE_;
			newsta_(&c__19, &c__0, &c__10, &c__0, &c__0);
			glk_put_string("Restored troll.\n");
			break;

		    case 18:	/* RC - Restore Cyclops */
			findex_1.cyclof=FALSE_;
			findex_1.magicf=FALSE_;
			newsta_(&c__58, &c__0, &c__101, &c__0, &c__0);
			glk_put_string("Restored cyclops.\n");
			break;

		    case 19:	/* RD - Mortal Mode */
			misc_1.dbgflg=0;
			glk_put_string("Restored deaths.\n");
			break;

		    case 20:	/* TK - Take */
			if (valid1(arg1, objcts_1.olnt)) {
			newsta_(&arg1, &c__0, &c__0, &c__0, &play_1.winner);
				glk_put_string("Taken.\n");
			} else
				glk_put_string("Invalid object.\n");
			break;

		   case 21:	/* EX - Exit GDT */
			return 0;

		    case 31:	/* DM - Display Messages */
			if (valid2(arg1, arg2, rmsg_1.mlnt)) {
				glk_put_string("Invalid limits.\n");
				break;
			}
			glk_set_style(style_Preformatted);
			glk_put_string("    RANGE CONTENTS");
			j = 0;
			for (i = arg1; i <= arg2 ; i += 10) {
				j = i + 10;
				if (j > arg2)
					j = arg2;
				snprintf(outbuf, sizeof(outbuf),
					"%4d-%4d", i,j);
				glk_put_string(outbuf);
				for (k = i; k <= j; k++) {
					snprintf(outbuf, sizeof(outbuf),
						"  %5d", rmsg_1.rtext[k]);
					glk_put_string(outbuf);
				}
				glk_put_string("\n");
			}
			glk_set_style(style_Normal);
			break;

		    case 32:	/* DT - Display TEXT */
			rspeak_(&arg1);
			break;

		    case 34:	/* DP - Display Parser State */
			glk_set_style(style_Preformatted);
			snprintf(outbuf, sizeof(outbuf),
				"ORPHS= %5d  %5d  %5d  %5d  %5d  \"%s\"  %5d"
				"  %5d\n",
				prssta_1.oflag, prssta_1.oact,
				prssta_1.oprep1, prssta_1.oobj1,
				prssta_1.oprep, prssta_1.oname,
				prssta_1.oprep2, prssta_1.oobj2);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"IT=    %5d\nPV=    %5d\n", prssta_1.lastit,
				prssta_1.act);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"SYN=  0x%04x 0x%04x  %5d  %5d  %5d\n",
				prssta_1.obj1, prssta_1.obj2,
				prssta_1.prep1, prssta_1.prep2,
				prssta_1.vflag);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"       %5d  %5d  %5d  %5d  %5d\n",
				prssta_1.dobj, prssta_1.dfl1,
				prssta_1.dfl2, prssta_1.dfw1,
				prssta_1.dfw2);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"       %5d  %5d  %5d  %5d  %5d\n",
				prssta_1.iobj, prssta_1.ifl1,
				prssta_1.ifl2, prssta_1.ifw1,
				prssta_1.ifw2);
			glk_put_string(outbuf);
			snprintf(outbuf, sizeof(outbuf),
				"BUNCH= %5d  %5d",
				prssta_1.bunlnt, prssta_1.bunsub);
			glk_put_string(outbuf);
			for (j = 0; j < 10; j++) {
				snprintf(outbuf, sizeof(outbuf),
					"  %5d", prssta_1.bunvec[j]);
				glk_put_string(outbuf);
				if (j == 2)
					glk_put_string("\n     ");
			}
			glk_put_string("\n");
			glk_set_style(style_Normal);
			break;

		    case 33:	/* AH - Alter Here */
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				play_1.here);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			play_1.here = arg2;
			break;

		    case 35:	/* PD - Program Detail */
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				misc_1.prsflg);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			misc_1.prsflg = arg2;
			break;

#define PUZZLE_SIZE	(sizeof(puzzle_1.cpvec) / sizeof(puzzle_1.cpvec[0]))

		    case 36:	/* DZ - Display Puzzle Room */
			glk_set_style(style_Preformatted);
			for (i = 0; i < PUZZLE_SIZE; i += 8) {
				for (j = 0; j < 8; j++) {
					snprintf(outbuf, sizeof(outbuf),
						 "%3d  ",
						puzzle_1.cpvec[i + j]);
					glk_put_string(outbuf);
				}
				glk_put_string("\n");
			}
			break;

		    case 37:	/* AZ - Alter Puzzle Room */
			if (!valid1(arg1, PUZZLE_SIZE)) {
				glk_put_string("Invalid puzzle entry.\n");
				break;
			}
			snprintf(outbuf, sizeof(outbuf), "Old: %d, New: ",
				puzzle_1.cpvec[arg1 - 1]);
			glk_put_string(outbuf);
			cmdlen = getline_null_(buf, sizeof(buf) - 1);
			arg2 = str_to_num(buf);
			puzzle_1.cpvec[arg1 - 1] = arg2;
			break;

		    default:
			glk_put_string("Function not implemented!\n");
			break;
		}
	}
	/* NOTREACHED */
	return 0;
}


#else /* HAVE_GDT */

/* No debugger compiled in; display a message and give up. */
int gdt_()
{
    glk_put_string("This version of Dungeon was compiled without the debugging mode.\n");
    return 0;
}
#endif /* HAVE_GDT */

