/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the creditsVerb in this
 * file for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 * Modification History (this file)
 *
 * POLYADV
 * =======
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version
 *                      * Moved general-purpose functions into this file.
 *                      * Corrections to sack_of_holding:  
 *                      * The item which the player is taking won't itself
 *                        be placed in a container. (This used to be possible 
 *                        if the player had been wearing the item before taking
 *                        it)
 *                      * The leftbulk argument is now optional and is
 *                        calculated automatically if not supplied.  
 *                      * The test for the removal of sufficient bulk has
 *                        been corrected.
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *              
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 */

/* getActor(prop)
   Function to determine the actor.  Appropriate values of prop are:
   &verbActor       - actor executing the current verb
   &currentActor    - actor for the purpose of determining locations
   &travelActor     - actor who is travelling.
 */
getActor: function(prop) {
    local actor := global.(prop);
    if(actor = nil) actor := parserGetMe();
    return actor;
}

/*
   toplocation(object)
   This function returns the top-level room in which an object is situated
   (e.g. when the player is in a nested room, but we want to check whether
   NPCs are present in the top-level room).

   If the object has the istoproom property, this function returns the object
   itself, not its location.

 */

toplocation: function(ob)  // DJP - added function (stolen from Dungeon)
                           // DJP - modified to cope with nil argument
                           // DJP - and to support the 'istoproom' property
                           // DJP - for nested rooms which are to be regarded
                           // DJP - as top-level ones.
                           // DJP - also modified to return nil if the
                           // DJP - outermost object is not a proper room.
{
    if( ob = nil ) return nil;
    while(ob.location and not ob.istoproom) ob := ob.location;
    if((not isclass(ob,room)) or isclass(ob,nestedroom)) return nil;
    else return( ob );
}

truetop: function(ob)      // DJP - returns the true top room
{
    if( ob = nil ) return nil;
    while(ob.location) ob := ob.location;
    return( ob );
}

/* DJP: new function to determine the normal location for dropped
        objects  */
droploc: function (actor) {
        local done := nil;
        local l := actor.location;
        while (not done) {
                if(l.roomdroploc) l := l.roomdroploc;
                else if(isclass(l,nestedroom) and l.isdroploc) done := true;
                else if(isclass(l,room) and (not isclass(l,nestedroom)))
                        done := true;
                else if(l = nil)done := true;
                else l := l.location;
        }
        return l;
}

/* DJP - new function to return the first listed item in a contents list */
firstitem: function (list)
{
    local i,obj,tot := length (list);
    for (i:=1; i <= tot; i++) {
        obj := list[i];
        if(obj.isListed) return obj;
    }
    return nil;
}

asknum: function(prompt,errorstring) {
    local numstring,i,l,char,ok,first,last;
    local nums := '0123456789';
    say(prompt);
    numstring := input();
    l := length(numstring);
    // reject zero-length response
    if (l = 0) goto error;
    // strip trailing spaces
    first := 0;
    last := 0;
    for (i := 1; i <= l; i++) {
        if (substr(numstring,i,1) <> ' ') {
            first := i;
            break;
        }
    }
    for (i := l; i >= 1; i--) {
        if (substr(numstring,i,1) <> ' ') {
            last := i;
            break;
        }
    }
    // reject all-spaces response
    if (first = 0)goto error;
    // remove leading and trailing blanks
    numstring := substr(numstring,first,last-first+1);
    ok := true;
    // check the validity of each character
    for (i := 1; i <= l; i++) {
        char := substr(numstring,i,1);
        if (i = 1 and char = '-') continue;
        if (find (nums,char) = nil) {
            ok := nil;
            break;
        }
    }
    // return the appropriate value
    if (ok) return cvtnum(numstring);
    error:
    say(errorstring);
    return nil;
}

nestcontents: function (obj) {
// DJP - return a list consisting of the object itself and its contents at
// all nesting levels.
    local i,o,l;
    local list := [];
    if (obj = nil) return list;
    list += obj;
    l := length(obj.contents);
    for (i := 1; i <= l; i++) {
        o := obj.contents[i];
        if (length(o.contents) > 0) list += nestcontents(o);
        else list += o;
    }
    return list;
}

vertesttake: function (a,dobj) {
/* Procedure to be called during verification, to check that an object can
be taken.  For liquids in containers, the container is checked
instead.  For normal objects, the verDoTake method is called; for 
objects with the noImpliedTake property the verifyRemove method is used
instead.  Finally, any fixed items which have slipped through the net
are caught. }
 */

    local outhideStatus, o := dobj;
    if (isclass(o,contliquid)) o := dobj.mycont;

    if (not o.noImpliedTake) {
        if (o.location <> a) {
            /*
             *   silently check the verification method
             */
            outhideStatus := outhide(true);
            o.verDoTake(a);
            if (outhide(outhideStatus)) {
                /*
                 *   verification failed - run again, showing the message
                 *   this time
                 */
                o.verDoTake(a);
                return nil;
            }
        }
    }
    else {
        if (o.location <> a) {
            /*
             *   silently check the verification method
             */
            outhideStatus := outhide(true);
            o.verifyRemove(a);
            if (outhide(outhideStatus)) {
                /*
                 *   verification failed - run again, showing the message
                 *   this time
                 */
                o.verifyRemove(a);
                return nil;
            }
        }
    }
    /* trap any fixed items which have slipped through the net */
    if (o.isfixed) {"%You% can't do that with "; o.thedesc; ".\n";}
    return (not o.isfixed);
}

testtake: function (a,dobj) {
/* procedure to try to take an object before it is given or
fed to something.  A suitable verification method (e.g. vertesttake,
verifyRemove, or verDoTake) should have been invoked at the verification
stage.   If the object is a liquid in a container, the container is taken
instead.  Items with the noImpliedTake property are exempt and return true.
However, all fixed items return false because they couldn't be given to
anything else. */

    local tflag := a.realtake, o := dobj;
    if (isclass(o,contliquid)) o := dobj.mycont;

    if (not o.noImpliedTake) {
        if (o.location <> a){
            "\n(Trying to take ";o.thedesc; ")\n";
            a.realtake := true; // dotake will use normal methods
                                // for all actors
            o.doTake(a);        // take the object
            a.realtake := tflag;

            if (o.location = a) return true;
            else return nil;
        }
        else return (not o.isfixed);
    }
    else return (not o.isfixed);
}

// DJP - Function to put item(s) in containers automatically
//

// Arguments: sack_of_holding(actor,item,leftbulk)
// 
// actor:    the actor wishing to take the item
// item:     the item to be taken
// leftbulk: the bulk which must be moved into containers from the player's
//           hands before the item can be taken.   This argument is optional.
//           If omitted, it is calculated automatically.

//
// Note that this code was developed independently of other sack-of-holding
// implementations - for example, see
//
// http://www.duke.edu/~srg3/IFprogramming/tads.html
// for a details of a module written by Dan Shiovitz, or
// ftp://ftp.gmd.de/if-archive/programming/tads/examples/sackItem.t
// for a more recent example contributed by Kevin Forchione.
//
// If you want to incorporate this code in your own game, please note that
// item.doTake needs to be modified (see ccr-adv.t) to call this function
// (and possibly clothingItem.doUnwear).
//
// Candidate containers must be listed in 'global.sacklist', must be open, must
// be carried by the player, and must contain enough space to hold the object.
//
// In addition, containers may define an accepts_item(o) method which returns
// true if the item 'o' is suitable, false otherwise.  If this method is
// not defined, it effectively defaults to 'true'.   (This mechanism is
// important in Polyadv, due to the additional checks which most
// containers make for properties like islong, ishuge etc.)
//
// Items which set their 'nosack' property to 'true' will not be automatically
// placed in containers.   This is useful for important items, e.g. lamps and
// weapons, which are best carried in hand.  (Nosack could, of course be
// a method; for example, it might be set to true for a lamp which is lit.)

sack_of_holding: function(actor,o,...) {
    local c,d,i,j,l,m,olist,leftbulk,cleft;

    if(argcount > 2)
        leftbulk := getarg(3);
    else
        leftbulk := addbulk(actor) + o.bulk - actor.maxbulk;

    // Do nothing if the player already has sufficient room.
    if(leftbulk <= 0) return true;

    if (datatype(global.sacklist) <> 7) return nil;
    if (length(global.sacklist) = 0) return nil;
    olist := actor.contents; l := length(global.sacklist);
    m := length(olist);
    // loop over eligible containers
    for (i := 1; i <= l; i++) {
        c := global.sacklist[i];
        // if the container isn't open or isn't held, go to the next one
        if (not (c.isIn(actor) and c.isopen)) continue;
        // bulk left in container
        cleft := c.maxbulk - addbulk(c.contents);
        for (j := 1; j <= m; j++) {
            d := olist[j];
            // ignore the item itself (if removed after being worn)
            if (d = o) continue;
            // ignore items which are worn
            if (d.isworn) continue;
            // don't try to put a container in itself!
            if (d = c) continue;
            // don't try to put a sack-of-holding inside another one
            if (find(global.sacklist,d))continue;
            // also skip objects (e.g. the lamp) with the nosack property
            if (d.nosack) continue;
            // also skip objects which have an accepts_item() method
            // which returns false
            if (defined(c,&accepts_item)) {
                if(not c.accepts_item(d)) continue;
            }
            // check that the container has room for the object
            if (d.bulk > cleft) continue;
            "\n(Putting <<d.thedesc>> in <<c.thedesc>> to make room)\n";
            c.ioPutIn(actor,d);
            if(not (d.location = c)) continue;
            else {
                // Bulk left to remove from player's hands
                leftbulk -= d.bulk;
                // Bulk left in container
                cleft -= d.bulk;
                // If there's now enough space, return true
                if(leftbulk <= 0) return true;
            }
        }
    }
    // If we've reached this point, we've not succeeded.
    return nil;
}

/* Function to move objects from one room to another */
room_move: function (oldroom,newroom) {
    local l, cur, i, tot;
    l := oldroom.contents;
    tot := length( l );
    for (i := 1; i <= tot; i++) {
        cur := l[i];
        /* Move all the non-fixed objects from oldroom to newroom */
        if (not cur.isfixed)
            cur.moveInto(newroom);
    }
}

/* Switch around the brass key */
obj_switch: function(oldkey,newkey) {
    local l; l := oldkey.location;
    oldkey.moveInto(nil); newkey.moveInto(l);
    newkey.moved := oldkey.moved;
}

