/*
** This file is the std.t replacement for "Arrival". As such it contains
** a lot of the basic definitions for the adventure.
**
** Copyright (c) 1998, Stephen Granade. All rights reserved.
*/

#pragma C+

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
win: function;
theEnd: function;
endCredits: function;
scoreRank: function;
init: function;
initRestore: function;
initCommon: function;
getSysInfo: function;
terminate: function;
pardon: function;
darkTravel: function;

/*
**   The die() function is called when the player dies. Since I also have
**   a win() function, all this does is tell the player that he lost, then
**   calls another function which prints the quit/restore/etc. bit.
*/
die: function
{
    if (global.isHTMLRuntime)
        "\b<CENTER><FONT SIZE=\"+1\">You have let the aliens
            win</FONT></CENTER>\b";
    else "\b\(You have let the aliens win\)\b";
    endCredits(nil);
}


/*
**   The win() function is just like the die function, only with a
**   different message.
*/
win: function
{
    if (global.isHTMLRuntime)
        "\b<FONT SIZE=\"+1\"><CENTER>You have defeated the aliens...for
            now</CENTER></FONT>\b";
    else "\b\(You have defeated the aliens...for now\)\b";
    endCredits(true);
}

/*
**   theEnd() function is like win()/die(), only without a specific message.
**   Instead, you pass the message you want printed to this function.
*/
theEnd: function(msg)
{
    if (global.isHTMLRuntime)
        "\b<FONT SIZE=\"+1\"><CENTER><<msg>></CENTER></FONT>\b";
    else "\b\(<<msg>>\)\b";
    endCredits(nil);
}

/*
**   amusingList contains the list of amusing things to be done.
*/
amusingList: object
    listItems = [
'Looking underneath the refrigerator?'
'Looking under the carpet in the living room?'
'Looking at the dirty dishes each time you wash one?'
'Cleaning different things around the house?'
'Turning on the water faucet when Zigurt and Floban are around?'
'Entering the ship&rsquo;s shadow?'
'...more than once?'
'Asking WHAT IS about stuff?'
'Asking WHO IS ZIGURT or FLOBAN before the ship lands? Before the aliens
come out into the backyard?'
'Calling the ship a sheep?'
'Turning the rain stick over in the presence of Zigurt and Floban?'
'Playing with the universal translator?'
    ]
;

/*
**   The endCredits() function tells the player how well he has done
**   and asks if he'd like to start over. If the player won, it also
**   asks him if he'd like to see some amusing things to do.
**   I've modified the usual version found in std.t so that you can
**   type 'Q' to quit, 'R' to restore, 'S' to restart, and 'U' to undo.
*/
endCredits: function(won)
{
    local       len, i, sepString = '';

    scoreRank();
    if (won)
        "\bYou may restore a saved game, start over, quit, see a list of
            amusing things to try, or undo the current command.\n";
    else "\bYou may restore a saved game, start over, quit, or undo
        the current command.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter (R)ESTORE, RE(S)TART, (Q)UIT, ";
	if (won)
	    "(A)MUSING, AUTHOR&rsquo;S (N)OTE, ";
	"or (U)NDO: >";
        resp = upper(input());
        if ( resp == 'RESTORE' || resp == 'R' )
    {
	    resp = askfile( 'File to restore' );
	    if ( resp == nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus( global.score, global.turnsofar );
		abort;
	    }
	}
        else if ( resp == 'RESTART' || resp == 'S' )
	{
	    scoreStatus( 0, 0 );
            restart();
	}
	else if ( resp == 'QUIT' || resp == 'Q' )
        {
	    terminate();
            quit();
	    abort;
        }
    else if ( (resp == 'AUTHOR\'S NOTE' || resp == 'NOTE' || resp == 'N')
        && won ) {
        authors_note.give;
    }
    else if ( (resp == 'AMUSING' || resp == 'A') && won )
    {
        if (global.isHTMLRuntime) {
            "<UL><LH>Have you tried:</LH>";
            sepString = '<LI>';
        }
        else {
            "\bHave you tried:\n";
            sepString = '\n\t';
        }
        len = length(amusingList.listItems);
        for (i = 1; i <= len; i++)
            "<<sepString>><<amusingList.listItems[i]>>";
        if (global.isHTMLRuntime)
            "</UL>";
        else "\n";
    }
	else if (resp == 'UNDO' || resp == 'U' )
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus(global.score, global.turnsofar);
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
    }
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    "You have achieved a score of ";
    if (global.score == 0 || !global.isHTMLRuntime)
        "<<global.score>> ";
    else "<img src=\"Resources/Score<<global.score>>.jpg\"
        alt=\"<<global.score>> \"> ";
    "points out of a possible ";
    if (!global.isHTMLRuntime)
        "<<global.maxscore>> ";
    else "<img src=\"Resources/Score<<global.maxscore>>.jpg\"
        alt=\"<<global.maxscore>> \"> ";
    "in <<global.turnsofar>> turns. ";
}

/*
**   The init() function is run at the very beginning of the game.
**   It prints introductory text, moves the player to the beginning room
**   (which defaults to startroom), and sets up some background activity.
*/
init: function
{
    initCommon();                           // Do the common init stuff
    
    // put introductory text here

    "\bGeez, mom really hacks you off sometimes. I mean, making you do dishes
    while that new show <q>Captain Dangerous</q> is on. Last week,
    Cap&rsquo;n was having to fight off the slavering hordes of Gamma Proxima
    Epsilon Centauri
    Five B, and he and his side-kick Billy Bob had used up the last of their
    ammo keeping the zombie mind troopers at bay. The evil over-leader
    had just shown up with his mucilage-ray when the show ended.\b
    
    And you&rsquo;re stuck cleaning the Pfaltzgraff while Cap&rsquo;n and Billy
    Bob fight for their lives.\b
    
    Life&rsquo;s not fair when you&rsquo;re eight.\b";
    
    setdaemon( turncount, nil );            // start the turn counter daemon
    // I don't like having the player have to eat or sleep, so I removed the
    //  lines that start those daemons
    Me.location = startroom;                // move player to initial location
    startroom.lookAround( true );           // show player where he is
    startroom.isseen = true;                // note that we've seen the room
    scoreStatus(0, 0);                      // initialize the score display

    notify(alien_daemon, &arrival, 12);     // Start up the beginning alien
                                            // daemon
}

/*
**   initRestore() is called if the player restores a saved game from the
**   command line. This allows me to handle restoring saved games at the start
**   gracefully. It also lets me make sure that any leftover sounds are
**   silenced.
*/
initRestore: function(savedGame)
{
    initCommon();               // Do the stuff common to us & to init()

    "<SOUND CANCEL>";           // Turn off any sounds

    mainRestore(savedGame);     // Now restore the saved game
}

/*
**   As suggested in the manual, I've put the code which is common to both
**   init() and initRestore() into this separate function, initCommon().
*/
initCommon: function
{
    "\H+";                                  // Turn on HTML formatting

    clearscreen();                          // Clear the screen

    getSysInfo();                 // Set up the sys-dependent info

    randomize();                  // If you're going to have random events
                                  //  occur, you're going to want to call
                                  //  randomize() so that the random-number
                                  //  generator doesn't do the same thing
                                  //  every game
}

/*
**   getSysInfo updates all the interpreter-specific flags we maintain
**   (i.e. can we print graphics, make sounds, etc.). This is a separate
**   function because the flags must be updated at *every* restore--there's
**   a slim chance that the player was playing using an HTML runtime, saved
**   their game, then restored that game on a non-HTML runtime. Yeah, it's
**   not very likely, but I've done it, and it's easy enough to add.
*/
getSysInfo: function
{
    if (systemInfo(__SYSINFO_SYSINFO) == true) {    // Adjust the graphics stuff
        global.isHTMLRuntime = (systemInfo(__SYSINFO_HTML) == 1);
        global.graphicsOnce = true;
        global.canDoGraphics = (systemInfo(__SYSINFO_JPEG) == 1) ||
            (systemInfo(__SYSINFO_PNG) == 1);// True if we can do PNG or JPEG
    }
    else {
        "\b\(WARNING: This particular TADS interpreter is an earlier version
            than 2.2.4. Because of this, this game will look mighty funky to
            you, or may not run at all. I highly suggest you get an updated
            version of the TADS interpreter. You should be able to find
            one at
            ftp://ftp.gmd.de/if-archive/programming/tads/executables.\)\b";
        global.isHTMLRuntime = nil;         // This runtime's not HTML capable
        global.graphicsOnce = nil;          // runtime, so we can't do any
        global.canDoGraphics = nil;         // graphics
    }

    if (global.isHTMLRuntime) {
        "<ABOUTBOX><center>
        <img src=\"Resources/Logo.jpg\" alt=\"Arrival\"><br>
        <h3>or, Attack of the B-Movie Clich&eacute;s</h3></center><br><br>
        Copyright &copy; 1998, Stephen Granade<br>
        http://www.phy.duke.edu/~sgranade/arrival.html</ABOUTBOX>";
        "<TITLE>Arrival, or Attack of the B-Movie Clich&eacute;s</TITLE>";
    }
}

/*
**   I'm also going to have to modify restoreVerb to call getSysInfo if
**   the restored game is okay.
**   [N.B. This code comes from adv.t]
*/
modify restoreVerb
    restoreGame(actor) = {
        local savefile, flag;
        
        savefile = askfile('File to restore game from');
        if (savefile == nil || savefile == '') {
            "Failed. ";
            return nil;
        }
        else {
            flag = mainRestore(savefile);
            if (flag) {
                getSysInfo();
                "\H+";
            }
            return flag;
        }
    }
;

/*
**   preinit() is called after compiling the game, before it is written
**   to the binary game file.  It performs all the initialization that can
**   be done statically before storing the game in the file, which speeds
**   loading the game, since all this work has been done ahead of time.
**
**   This routine normally puts all lamp objects (those objects with 
**   islamp = true) into the list global.lamplist. However, "Arrival"
**   doesn't have any lamps, so I removed those statements.
**
**   I also put in the call to initSearch(). initSearch() sets up all the
**   search lists (if you use, for example, underHider); more importantly,
**   it sets up the list which contains all floating items. If you don't
**   call initSearch(), then you won't be able to refer to floating items
**   in your game. Forgetting this can give you heartburn.
*/
preinit: function
{
    initSearch();
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
    "Thanks for playing <b>Arrival</b>. ";
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  The default version just says "I beg your pardon?"
 */
pardon: function
{
    "Children should be seen and not heard, huh? ";
}

/*
** Normally the sleepDaemon and eatDaemon would go here, but I don't like
** having the player have to eat or sleep, so I yanked them out.
*/

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
;

/*
**   The "global" object is the dumping ground for any data items that
**   don't fit very well into any other objects.  The properties of this
**   object that are particularly important to the objects and functions
**   are defined here; if you replace this object, but keep other parts
**   of this file, be sure to include the properties defined here.
**
**   Note that I got rid of awakeTime, sleepTime, eatTime, lastMealTime,
**   and lamplist because this game doesn't use them. If you use eating,
**   sleeping, or lamps in your game, make sure to stick them back in.
**
**   I've also added graphicsOn, and graphicsOnce. These flags
**   control if graphics are shown and whether or not images are shown only
**   once. (Note that graphicsOn is actually a function which checks to
**   see if a) we can show graphics, and b) if the player wants graphics
**   shown.
*/
global: object
    turnsofar = 0               // no turns have transpired so far
    score = 0                   // no points have been accumulated yet
    maxscore = 20               // maximum possible score
    verbose = true              // we default to VERBOSE mode
    isHTMLRuntime = true        // Are we running in an HTML runtime?
    graphicsOnce = true         // Only show graphics one time
    canDoGraphics = true        // Set to true if our runtime can handle
                                //  graphics
    graphicsOn = {              // A function to see if graphics are on
        return (self.canDoGraphics &&
            systemInfo(__SYSINFO_PREF_IMAGES) == 1);
    }
    fileList = []               // A list of previously-displayed pix
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
// Note to me: DO NOT FORGET to adjust the version number in version_info
//  (in arr_hints.t)
version: object
    compile_time = __DATE__
    sdesc = {
        if (global.isHTMLRuntime)
            "\b<img src=\"Resources/Arrival Poster.jpg\"
                alt=\"Arrival, or, Attack of the B-Movie Clich&eacute;s\">\n";
        else "\b\(Arrival\)\nor, Attack of the B-Movie Clich&eacute;s\n";
        "Copyright &copy; 1998 by Stephen Granade. All rights reserved.\n
        (Type <q><<link('about', 'about')>></q> for more information)\n
        Version 2 (<<self.compile_time>>)\n
        Developed with TADS, the Text Adventure Development System.\b";
    }
    ldesc = {
        self.sdesc;
        "<SOUND SRC=\"Resources/ArrTheme.mid\" LAYER=\"BACKGROUND\">"; // Play the title track
    }
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
    noun = 'i'
    canThieve = nil             // See leaveRoom() in rec_room (arr_ship.t)
                                // for details
    leftSceneOfCrime = nil      // Player walked away from broken dishes while
                                // mom was around
    whodesc = "You&rsquo;re eight; you like science fiction and macaroni and
        cheese; you hate spinach and washing dishes. "
    verDoWhatis(actor) = { "You aren&rsquo;t really a what. "; }
;

/*
** I don't use darkTravel() or goToSleep(), but TADS expects them to be here.
** Thus I just set them to be empty functions.
*/
darkTravel: function {}
goToSleep: function {}

