#charset "us-ascii"

/* AddVerbs.t 
 *  This file contains code for elements of the game not specific to a particular
 *  location or NPC, such as definitions of new actions, modifications of existing
 *  ones, some class definitions and modifications, handling for various system
 *  commands, topic definitions, and the like.
 */

#include <adv3.h>
#include <en_us.h>
#include <SquareCircle.h>

/* Additional Verb definitions/modifications with associated Class definitions/mods
 */

//#define __TESTING

DefineIAction(Swim)
  execAction()
  {
    gActor.location.swimDesc;
  }
;

VerbRule(Swim)
  'swim' | 'paddle'
  : SwimAction
  verbRule = 'swim/swimming'
;

VerbRule(SwimIn)
  ('swim' | 'paddle') 'in' singleLiteral
  : SwimAction
  verbRule = 'swim/swimming (in what)'
;

modify VerbRule(Break)
  ('break' | 'ruin' | 'destroy' | 'wreck' | 'rip' | 'tear' | 'smash' | 'shatter' ) dobjList
   :
;

modify VerbRule(DigWith)
     ('dig' | 'burrow') ('in' | ) singleDobj 'with' iobjList
     :
;

modify VerbRule(Turn)
     ('turn' | 'twist' | 'rotate' | 'spin' | 'swivel' ) dobjList
     :
;     

modify VerbRule(Kiss)
     ('kiss' | 'pet') singleDobj
     :
;

modify VerbRule(PushTravelDir)
     ('push' | 'pull' | 'drag' | 'move') dobjList ( | 'to' ( | 'the') ) singleDir
     :
;

modify VerbRule(ScrewWith)
     ('screw' | 'fasten') dobjList 'with' singleIobj
     :
;

modify VerbRule(UnscrewWith) 
     ('unscrew' | 'unfasten') dobjList 'with' singleIobj
     :
;

modify VerbRule(StandOn)
     ('stand' | 'climb' | 'clamber') ('on' | 'in') singleDobj
     :
;

modify VerbRule(ClimbDown)
     ((('climb' | 'go' | 'walk') 'down') | 'descend') singleDobj
     :
;

modify VerbRule(ConsultAbout)
     'consult' singleDobj ('on' | 'about') singleTopic
     | 'search' singleDobj 'for' singleTopic
     | (('look' | 'l') 'up' | ('look' | 'l' ) 'for' | 'find' | 'search' 'for' | 'read' 'about')
          singleTopic 'in' singleDobj
     | ('look' | 'l') singleTopic 'up' 'in' singleDobj
     : 
;

modify VerbRule(ConsultWhatAbout)
     (('look' | 'l') ( 'up' | 'for') | 'find' | 'search' 'for' | 'read' 'about')
     singleTopic
     | 'look' singleTopic 'up'
     :
;

DefineTopicTAction(DrawOn, DirectObject)
;


VerbRule(DrawOnWhat)
   [badness 500] ('draw'|'inscribe'|'sketch') singleTopic
    : DrawOnAction
    verbPhrase = 'draw/drawing (what) (on what)'
    omitIobjInDobjQuery = true
    whichMessageTopic = DirectObject
    construct()
    {
       /* set up the empty direct object phrase */
      dobjMatch = new EmptyNounPhraseProd();
      dobjMatch.responseProd = singleNoun;     
    }   
     
;

 /* The following string preparser allows 'note' to be used in response
  * to a prompt such as "What do you want to draw that on?" and have
  * the parser understand that the player wants to draw on the note, rather
  * than starting an unspecified NOTE command.
  */


StringPreParser
  runOrder = 90
  doParsing(str, which)
  {
    if(str.toLower == 'note' && which == rmcAskObject)
      str = 'paper note';
    return str;
  }  
;


VerbRule(DrawOn)
  ('draw'|'inscribe'|'sketch') singleTopic ('on' |'onto'|'upon'|'round') singleDobj
   : DrawOnAction
   verbPhrase = 'draw/drawing (what) (on what)'
   execAction()
    {
      penHeld.checkPreCondition(nil, true);      
      inherited;
    }
;

DefineTAction(DrawWhatOn)
  
;


VerbRule(DrawWhatOn)
  ('draw'|'inscribe'|'sketch') ('on'|'upon'|'onto') singleDobj
   : DrawWhatOnAction
   verbPhrase = 'draw/drawing (what) (on what)'
//   askDobjResponseProd = singleNoun
//     omitIobjInDobjQuery = true
//     construct()
//     {
//         /* set up the empty topic phrase */
//         topicMatch = new EmptyNounPhraseProd();
//         topicMatch.responseProd = aboutTopicPhrase;
//     }

;



DefineLiteralTAction(WriteOn, DirectObject) 
   
;

DefineTAction(WriteWhatOn)
 
;


VerbRule(WriteOnWhat)
   ('write' | 'sign') singleLiteral
    : WriteOnAction
    verbPhrase = 'write/writing (what) (on what)'
    omitIobjInDobjQuery = true
    whichMessageTopic = DirectObject
    construct()
    {
       /* set up the empty direct object phrase */
      dobjMatch = new EmptyNounPhraseProd();
      dobjMatch.responseProd = singleNoun;
    }
;


VerbRule(WriteOn)
  ('write' | 'sign') singleLiteral ('on'|'upon'|'onto') singleDobj
   : WriteOnAction
   verbPhrase = 'write/writing (what) (on what)'
;

VerbRule(WriteWhatOn)
  ('write' | 'sign') ('on'|'upon') singleDobj
   : WriteWhatOnAction
   verbPhrase = 'write/writing (what) (on what)'
//   askDobjResponseProd = singleNoun
//     omitIobjInDobjQuery = true
//     construct()
//     {
//         /* set up the empty topic phrase */
//         topicMatch = new EmptyNounPhraseProd();
//         topicMatch.responseProd = aboutTopicPhrase;
//     }

;

modify Thing
  dobjFor(DrawOn)
  {
    preCond = [objVisible, touchObj, penHeld]
    verify() { illogical(&cannotDrawOnMsg); }
    check()
    {
      if(!pen.isHeldBy(gActor))
      {
        reportFailure('You have nothing to draw with. ');
        exit;
      }
    }
  }
  dobjFor(WriteOn)
  {
    preCond = [objVisible, touchObj, penHeld]
    verify() { illogical(&cannotWriteOnMsg); }
    check()
    {
      if(!pen.isHeldBy(gActor))
      {
        reportFailure('You have nothing to write with. ');
        exit;
      }
    }
  }
  dobjFor(DrawWhatOn)
  {
    preCond = [objVisible, touchObj, penHeld]
    verify { illogical(&cannotDrawOnMsg); }
    action { askForTopic(DrawOn); }
  }
  
  dobjFor(WriteWhatOn)
  {
    preCond = [objVisible, touchObj, penHeld]
    verify { illogical(&cannotWriteOnMsg); }
    action { askForLiteral(WriteOn); }
  }
//  nominalContainer {
//    if(location != nil)
//      return (location.ofKind(ComplexComponent) ? location.location : location);
//    return nothing;
//  }
  
  dobjFor(Open)
  {
    verify() { illogical(&cannotOpenMsg); }
  }
  
  iobjFor(OpenWith)
  {
    verify() { illogical('{You/he} can\'t open anything with {that iobj/him}. '); }
  }
  dobjFor(SetTo)
  {
    preCond = [touchObj]
    verify { illogical('That\'s not something you can set. '); }
  }
  iobjFor(ThrowAt)
  {
    check()
    {
      if(getOutermostRoom != gActor.getOutermostRoom)
      {
        reportFailure('{The iobj/he} is too far away. ');
        exit;
      }
    }
  }
;



modify VerbRule(AskAboutImplicit)
     ('a' | ('ask' | 'a') 'about') singleTopic
     :
;

modify VerbRule(TellAboutImplicit)
     ('t' | ('tell' | 't' ) 'about') singleTopic
     :
;

/* In beta testing it was useful for some testers to use ALL with verbs for which
 * I wanted it disabled in the final release version (e.g. EXAMINE ALL). The following
 * code defines a TOGGLE ALL command that is only compiled if the __TESTING constant
 * is defined (which it isn't for the released versions).
 */


#ifdef __TESTING

DefineSystemAction(ToggleAll)
  execSystemAction()
   {
     libGlobal.restrictAll = !libGlobal.restrictAll;
     "<<versionInfo.name>> is now in <<libGlobal.restrictAll ? 'RESTRICTED' : 'STANDARD'>> ALL mode. ";
   }
;


VerbRule(ToggleAll)
  'toggle' 'all'
  :ToggleAllAction
  verbPhrase = 'toggle/toggling all'
;

#endif // __TESTING

 /*
  *  Some people don't like the exit lister in the status line. In the competition
  *  release of "Square Circle" The TOGGLE EXITS command wass provided to turn it off, 
  *  and back on again. A subsequent discussion on the TADS 3 v-space list indicated
  *  that a different standard, an EXITS ON | OFF | STATUS | LOOK command, might be
  *  incorporated into future of the TADS 3 library, so this is also implemented below,
  *  while TOGGLE EXITS is retained for backward compatibility with the original
  *  comp release.
  */

DefineSystemAction(ToggleExits)
  execSystemAction()
  {
    gameMain.showExitsInStatusline = !gameMain.showExitsInStatusline;
    "Status line exit lister now <<gameMain.showExitsInStatusline ? 'en' :'dis'>>abled. ";
  }
;

VerbRule(ToggleExits)
  'toggle' ('exit' | 'exits') ('lister' |)
  :ToggleExitsAction
  verbPhrase = 'toggle/toggling exit lister' 
;

modify exitLister
  exitsOnOffCommand(stat)
  {
    gameMain.showExitsInStatusline = stat;
    inherited(stat);
  }
  exitsStatusOnly()
  {
    gameMain.showExitsInStatusline = true;
    enableRoomDesc = nil;
    "Exits will now be listed in the status line only. ";
  }
  exitsLookOnly()
  {
    gameMain.showExitsInStatusline = nil;
    enableRoomDesc = true;
    "Exits will now be listed in room descriptions only. ";
  }
;

DefineSystemAction(ExitsStatus)
  execSystemAction()
  {
    if(gExitLister != nil)
      gExitLister.exitsStatusOnly;
    else
      libMessages.commandNotPresent;
  
  }
;

VerbRule(ExitsStatus)
  'exits' 'status' 
  : ExitsStatusAction
  verbPhrase = 'set/setting exit lister to status line only'
;

DefineSystemAction(ExitsLook)
 execSystemAction()
  {
    if(gExitLister != nil)
      gExitLister.exitsLookOnly;
    else
      libMessages.commandNotPresent;
  }
;

VerbRule(ExitsLook)
  'exits' 'look' 
  : ExitsLookAction
  verbPhrase = 'set/setting exit lister to room descriptions only'
;



DefineTIAction(OpenWith)
;

VerbRule(OpenWith)
  'open' dobjList 'with' singleIobj
  : OpenWithAction
  verbPhrase = 'open/opening (what) (with what)'
;

modify Key
  iobjFor(OpenWith) {
   verify() {} 
   action()
   {
     nestedAction(UnlockWith, gDobj, self);
     nestedAction(Open, gDobj);
   }
  }
  
;

modify Openable
  dobjFor(OpenWith) 
  {
    verify() { 
      if(isLocked) logicalRank(140, 'locked');      
    }
  }
;

DefineTIAction(PutRound);

VerbRule(PutRound)
  (('put' | 'wrap' | 'fold') singleDobj ('round' | 'around') singleIobj)
  | ('wrap' singleIobj 'with' singleDobj)
  : PutRoundAction
  verbPhrase = 'put/putting (what) (round what)'
;

DefineTAction(Stretch);

VerbRule(Stretch)
  'stretch' singleDobj
  : StretchAction
  verbPhrase = 'stretch/stretching (what)'
;

DefineTAction(Bend);

VerbRule(Bend)
  ('bend' | 'fold') singleDobj
  : BendAction
  verbPhrase = 'bend/bending (what)'
;


DefineTAction(KnockOn);

VerbRule(KnockOn)
  ('knock' | 'rap' | 'tap' | 'bang') ('on' |'at' |)   singleDobj
  : KnockOnAction
  verbPhrase = 'knock on/knocking (on what)'
;

//VerbRule(Knock)
//  ('knock' | 'rap' | 'tap')  singleDobj
//  : KnockOnAction
//  verbPhrase = 'rap/rapping (what)'
//;

DefineTAction(Shake);

VerbRule(Shake)
  'shake' dobjList
  : ShakeAction
  verbPhrase = 'shake/shaking (what)'
;

DefineTAction(Lift);

VerbRule(Lift)
  ('lift' | 'raise') dobjList
  : LiftAction
  verbPhrase = 'lift/lifting (what)'
;

modify Thing
  dobjFor(Stretch)
  {
    verify() { illogical('It\'s not elastic. '); }
  }
  dobjFor(Bend) asDobjFor(Stretch)
  dobjFor(PutRound)
  {
    verify() { illogical('You can\'t bend {the dobj/him}. '); }
  }
  dobjFor(KnockOn)
  {
    verify() {}
    check() {}
    action()
    {
      "Knocking on {the dobj/him} has no obvious effect. ";
    }
  }
  dobjFor(Shake)
  {
    preCond = [objHeld]
    verify() { }
    check() {}
    action() { "Shaking {the dobj/him} has no obvious effect. "; }    
  }
  dobjFor(Lift)
  {
    preCond = [objHeld]
    verify() { }
    check() { }
    action() { "{You/he} hold{s} {the dobj/him} aloft. "; }
  }
  inflammable = true
;

modify NonPortable
  dobjFor(Shake)
  {
    verify() { illogical('{You/he} can\'t shake {that dobj/him}. '); }
  }
  dobjFor(Lift)
  {
    verify() { illogical('{You/he} can\'t lift {that dobj/him}. '); }
  } 
;

modify Heavy
  dobjFor(Lift)
  {
    preCond = []
    check()
    {
      reportFailure('The dobj/he} {is} too heavy to lift. ');
      exit;
    }
  }
;


modify Container
  dobjFor(Shake)
  {
    action()
    {
      if(contents.length > 0 )
        "Something rattles around inside. ";
      else
        inherited;
    }
  }
  inspected = nil
  notifyInsert(obj, newCont)
  {
    inspected = nil;
    inherited(obj, newCont);
  }
  isObviousContainer = true
;


modify Door
  dobjFor(KnockOn)
  {
    action()
    {
      if(isOpen)
        "Knocking on an open <<name>> seems pretty futile, but you go ahead and do it
         anyway, with just the lack of effect you could have predicted. ";
      else
        "You knock loudly on <<theName>>, but if anyone hears you they take not the
         least bit of notice. ";
    }
  }
  dobjFor(LookBehind)
  {
    action()
    {
      if(isOpen)
        inherited;
      else
      {
        tryImplicitAction(Open, self);
        if(isOpen) 
          mainReport(&nothingBehindOpenDoorMsg);
      }
    }
  }
  dobjFor(LookThrough)
  {
    action()
    {
      if(isOpen)
        thruDesc;
      else
        inherited;
    }    
  }
  thruDesc = "{You/he} can't see much through {the dobj/him}. "
;


class Bendable : Thing
 dobjFor(Bend)
 {
   preCond = [objHeld]
   verify () {}
   action()
   {
     local verb = gAction.getOrigTokenList[1][1];
     "You <<verb.toLower()>> <<theName>>, but it doesn't seem to achieve much. ";
   }
 }
 dobjFor(PutRound)
 {
   preCond = [objHeld]
   verify() {}   
   action()
   {
     "You wrap <<theName>> round {the iobj/him}. When you let go {the dobj/he} unfolds
      itself and falls to the ground, suggesting that this was really a pretty futile
      exercise. ";
      nestedAction(Drop, self);
   }
 }
 iobjFor(TurnWith)
  {
    verify() { }
    check() {
      if(gDobj != steelHandle && !gDobj.ofKind(Handle))
      {
        reportFailure('That probably won\'t be very effective. ');
        exit;
      }
    }
  }

;

VerbRule(GetDown)
  'get' 'down'
  : DownAction
  verbPhrase = 'get down/getting down'
;


VerbRule(TieTo)
  ('tie' | 'fix') singleDobj 'to' singleIobj
  : AttachToAction
  verbPhrase = 'tie/tying (what) to (what)'
;

DefineTIAction(FillWith);

VerbRule(FillWith)
  'fill' singleDobj 'with' singleIobj
  : FillWithAction
  verbPhrase = 'fill/filling (what) with (what)'
;

DefineTAction(Empty);

VerbRule(Empty)
  'empty' singleDobj
  : EmptyAction
  verbPhrase = 'empty/emptying (what) '
;

modify Thing
  dobjFor(Empty)
  {
    preCond = [touchObj]
    verify() { illogical('{You/he} can\'t empty {the dobj/him}. ');  }
  }
;

modify Container
  dobjFor(Empty)
  {
    preCond {
      if(ofKind(Openable))
        return inherited + objOpen;
      else
        return inherited;
    }
    verify()
    {
      if(contents.length()==0)
        { illogicalNow ('{The dobj/he} is already empty. '); }
    }
    action()
    {
      foreach(local obj in contents)
      {
        if(!obj.ofKind(NonPortable))
        {
          obj.moveInto(roomLocation);
          "You remove <<obj.theName>> from <<theName>>\n";
        }
          
      } 
    }
  }
  
;

modify ComplexContainer
 dobjFor(Empty) maybeRemapTo(subContainer != nil, Empty, subContainer)
 dobjFor(PourInto) maybeRemapTo(subContainer != nil, PourInto, subContainer) 
;

VerbRule(ThrowInto)
     ('throw' | 'toss' | 'drop') dobjList
         ('in' | 'into' | 'in' 'to' | 'inside' | 'inside' 'of') singleIobj
     : PutInAction
     verbPhrase = 'throw/throwing (what) (in what)'
     askIobjResponseProd = inSingleNoun
;


modify Thing
  dobjFor(FillWith)
  {
    verify() { illogical('You can\'t fill that. '); }
  }
  iobjFor(FillWith)
  {
    verify() { illogical('You can\'t fill anything with that. '); }
  }
;  

DefineTIAction(RubWith);

VerbRule(RubWith)
     ('rub' | 'file') singleDobj 'with' singleIobj
     : RubWithAction
     askIobjResponseProd = toSingleNoun
     verbPhrase = 'rub/rubbing (what) (with what)'
 ;
 
VerbRule(RubWithWhat)
     ('rub' | 'file') singleDobj
     : RubWithAction
     verbPhrase = 'rub/rubbing (what) (with what)'
     construct()
     {
         /* set up the empty indirect object phrase */
         iobjMatch = new EmptyNounPhraseProd();
         iobjMatch.responseProd = toSingleNoun;
     }
 ;



VerbRule(ClimbOver)
  ('climb' | 'clamber'| 'scrabble') 'over' singleDobj
  : ClimbAction
  verbPhrase = 'climb over/climbing (over what)'
;

modify Thing
  dobjFor(RubWith)
  {
    preCond = [touchObj]
    verify() {}
    check() {}
    action()
     {
       if(!gIobj.ofKind(Hand))    
        "You rub {the dobj/him} with {the iobj/him}, but, quite frankly,
         it's a pretty futile exercise. ";
     }
  }
  iobjFor(RubWith)
  {
    preCond = [touchObj]
    verify() { illogical('You can\'t rub anything with that. '); }
  }
  dobjFor(Rub)
     {
         preCond = [touchObj]
         verify() { }
         action() { askForIobj(RubWith); }
     }
;

class CanRub : object
   iobjFor(RubWith)
   {
     precond = [touchObj]
     verify() 
       {        
         logicalRank(110, 'can rub'); 
       }
     check() {}     
   }
   dobjFor(RubWith)
   {
     verify() { if(gIobj==self) illogicalSelf('You can\'t rub {the dobj/him} with itself. '); }
   }
;

 /*
  *  The addition of INSULT and PRAISE verbs was something of an experiment in
  *  extending the range of TopicEntry types and conversational commands an NPC
  *  can respond to. In Square Circle it's only really implemented with the
  *  receptionist.
  */

DefineTAction(Insult)
 isConversational(issuer) { return true; }
;

VerbRule(Insult)
 ('insult' | 'berate' | 'offend'| 'affront' | 'disparage') singleDobj
 : InsultAction
 verbPhrase = 'insult/insulting (whom)'
;

DefineTAction(Praise)
 isConversational(issuer) { return true; }
;

VerbRule(Praise)
 ('praise' | 'compliment' | 'flatter') singleDobj
 : PraiseAction
 verbPhrase = 'praise/praising (whom)'
;

  /*
   *  The PraiseHer action was a bit of a hack added when I discovered a beta-tester
   *  trying to use commands of the form PRAISE HER LOOKS when they weren't valid.
   *  The PraiseHer action now allows the parser to recognize this type of command
   *  even when the appropriate ConvNode isn't active.
   */
   
DefineLiteralAction(PraiseHer)
  execAction()
  {
    if(!gActor.canSee(receptionist))
    {
      "{You/he} can\'t see her. ";
      return;
    }  
    local praiseWhat = getLiteral().toLower;  
    local topicObj = nil;    
    foreach(local cur in praiseReceptionistNode.specialTopics)
    {
       if(cur.keywordList.indexOf(praiseWhat))
       {
         topicObj = cur;
         break;
       }
    }
    if(topicObj == nil)
      replaceAction(Praise, receptionist);
    else
    {
      topicObj.handleTopic(gActor, nil);
      receptionist.noteConvAction(gActor);      
    }
  }
  isConversational(issuer) { return true; }
;

VerbRule(PraiseHer)
  'praise' ('her'| )  singleLiteral
  : PraiseHerAction
  verbPhrase = 'praise/praising her (what)'  
;


modify Thing
  dobjFor(Insult)
  {
    verify { illogical('Swearing at inanimate objects is puerile. '); }
  }
  dobjFor(Praise)
  {
    verify { illogical('{You/he} {is} wasting {your} breath. '); }
  }
; 

modify Actor
  dobjFor(Insult)
  {
    verify() 
    {
      if(self == gActor) illogical('Insulting yourself is unnecessary. ');
    }
    
    preCond = [canTalkToObj]
    action()    
    {
      /* note that the issuer is targeting us with conversation */ 
             gActor.noteConversation(self); 
  
             /* let the state object handle it */ 
             curState.handleConversation(gActor, insultTopicObj, insultConvType); 
          
     } 
  }
  dobjFor(Praise)
  {
    verify() 
    {
      if(self == gActor) illogical('That would be immodest. ');
    }
    
    preCond = [canTalkToObj]
    action()    
    {
      /* note that the issuer is targeting us with conversation */ 
             gActor.noteConversation(self); 
  
             /* let the state object handle it */ 
             curState.handleConversation(gActor, praiseTopicObj, praiseConvType); 
         } 
     } 

  defaultPraiseResponse(actor) 
  { 
    "\^<<theName>> appears totally unmoved by your flattery. "; 
  }
  defaultInsultResponse(actor) 
  { 
    "\^<<theName>> ignores your rudeness. "; 
  } 
  dobjFor(LookIn)
  {
    verify() { illogical('{You/he} can\'t look in {the dobj/him}. '); }
  }
  dobjFor(Search) asDobjFor(Examine)
;

praiseConvType : ConvType 
  unknownMsgType = 'No-one\'s listening. ' 
  topicListProp = &miscTopics 
  defaultResponseProp = &defaultPraiseResponse 
  defaultResponse(db, other, topic) 
    { db.defaultPraiseResponse(other); } 
;

insultConvType : ConvType 
  unknownMsgType = 'No-one\'s listening. ' 
  topicListProp = &miscTopics 
  defaultResponseProp = &defaultInsultResponse 
  defaultResponse(db, other, topic) 
    { db.defaultInsultResponse(other); } 
;

insultTopicObj : object;
praiseTopicObj : object;

class InsultTopic : YesNoTopic
  matchList = [insultTopicObj]
  includeInList = [&miscTopics]
;

class PraiseTopic: MiscTopic 
  includeInList = [&miscTopics] 
  matchList = [praiseTopicObj] 
;

 /* The THANK verb was offered in response to one reviewer's transcript, supplied
  * After the IF-Comp. It represents a rather simpler way of extending the range
  * of conversational verbs than that I used with PRAISE and INSULT.
  */ 
   

VerbRule(Thank)
  (('thank' | 'thanks') | ('say' ('thanks' | 'thankyou' | 'thank' 'you') 'to')) singleDobj
  : ThankAction
  verbPhrase = 'thank/thanking (whom)'
;

modify Thing
  dobjFor(Thank)
  {
    preCond = [canTalkToObj]
    verify() { illogical(cannotThankMsg); }    
  }
  cannotThankMsg = 'There\'s no point trying to thank {a dobj/him}. '
;

modify Person
  dobjFor(Thank)
  {
    verify() { }
    action()
    {
      if(curState.ofKind(ConversationReadyState))
         curState.enterConversation(gActor, nil);
      if(curState != nil && curState.propDefined(&thankResponse))
         curState.thankResponse;
      else
         thankResponse;
    }
  }
  thankResponse = "Your effusive thanks leave {the dobj/him} totally unmoved. "
;



DefineTIAction(TakeWith)
;

VerbRule(TakeWith)
  ('take' |'grab'|'catch'|'get'| 'pick' 'up'| 'remove') dobjList 'with' singleIobj |
   'pick' dobjList 'up' 'with' singleIobj
  :TakeWithAction
  verbPhrase = 'take/taking (what) (with what)'
;

modify Thing
  dobjFor(TakeWith)
  {
    preCond = [touchObj, objNotWorn, roomToHoldObj]
    verify()  
    { 
      verifyDobjTake; 
      verifyMoveTo(gIobj); 
      if(isIn(gIobj)) illogicalNow('{The dobj/he) is already in {the obj/him} ');             
    }
  }
  iobjFor(TakeWith)
  {
    preCond = [objHeld]
    verify() { illogical('{You/he} can\'t take anything with {that dobj/him}. '); }
  }
//  mainMoveInto(newDest)
//  {
//    if(isIn(cell))
//      lightFitting.underList.removeElement(self);
//    inherited(newDest);   
//  }
  iobjFor(MoveTo) { verify() { nonObvious; } }
;

VerbRule(Swear)
  ('shit' | 'damn' | 'fuck' | 'bugger' | 'darn' | 'bother') ( |'off')
  : SwearAction
  verbPhrase = 'swear/swearing'
;

DefineIAction(Swear)
  execAction()
  {
    "The word <q><<gVerbName>></q> is totally unnecessary in this story. ";
  }
;


DefineTAction(Pick)
;

VerbRule(Pick)
  'pick' singleDobj
  :PickAction
  verbPhrase = 'pick/picking (what)'
;

modify Thing
  dobjFor(Pick) 
    { 
      verify {illogical('It\'s not clear what picking {the dobj/him} would mean. '); } 
      preCond = [objVisible, touchObj]
    }

;

modify PathPassage
  dobjFor(Pick) asDobjFor(TravelVia)
;

DefineTAction(Thank)
;



/* Sundry Customizations of adv3 classes */



modify Lockable
  checkDobjOpen
  {
    if(isLocked) lockStatusObvious = true;
    inherited;
  }
;

modify Surface
  dobjFor(PourOnto)
  {
    preCond = [objVisible]
    verify() {}
  }
;

modify TouchObjCondition
    verifyPreCondition(obj)
    {
        if (sourceObj == nil)
            return;
        if (!sourceObj.canTouch(obj))
        {
            if (gActor.canSee(obj))
            {
                local info;
                info = gActor.bestVisualInfo(obj);
                if (info != nil && info.trans == distant)
                    inaccessible(&tooDistantMsg, obj);
                else
                    logicalRankOrd(80, 'unreachable but visible', 150);
            }
            else
            {
                if (obj.soundPresence && gActor.canHear(obj))
                    inaccessible(&heardButNotSeenMsg, obj);
                else if (obj.smellPresence && gActor.canSmell(obj))
                    inaccessible(&smelledButNotSeenMsg, obj);
                else if (!gActor.isLocationLit())
                    inaccessible(&tooDarkMsg);
                else
                    inaccessible(&mustBeVisibleMsg, obj);
            }
        }
    }
;


modify Floor
  dobjFor(PourOnto)
  {
    preCond = [objVisible]
    verify() {}
  }
  makeTouchable {
    if(gActor.location is in (cellBed, desk, guardChair, sandBox) && gActor.posture==standing)
      tryImplicitAction(GetOffOf, gActor.location);
  }
  actionDobjFeel {
    makeTouchable;
    inherited;
  }
  actorOutOfPrep = 'off'
  basicExamine()
  {
    inherited;
    if(isIn(gActor.getOutermostRoom) && gActor.getOutermostRoom.isSandy)
     " \^<<theName>> is a bit sandy. ";
  }
 
  
;



modify RoomPart
  nothingBehindMsg = '{You/he} cannot look behind {that dobj/him}. '
  cannotWriteOnMsg = &uselessToScribbleOnMsg
  cannotDrawOnMsg = &uselessToScribbleOnMsg
;

modify Platform
  dobjFor(Climb) asDobjFor(StandOn)
;

modify Switch
  dobjFor(Push) asDobjFor(Flip)
;


modify BasicLocation
  cannotGoThatWay()
     {
         /* "you can't go that way" */
         reportFailure(cannotGoMsg);
 
         /* show a list of exits, if appropriate */
         cannotGoShowExits(gActor);
     }
     cannotGoMsg = &cannotGoThatWayMsg
   swimDesc = "There's nothing to swim in here. "  
   isSandy = nil
;

modify libGlobal
  isTextOnlyInterpeter = nil
;


 /*
  * This is how we add a list of custom verbs to the standard Instructions:
  */
  
modify InstructionsAction
  customVerbs = [ 'DRAW A PENTAGRAM ON THE PAPER', 'PUT STRING ROUND PARCEL',
                 'STRETCH RUBBER BAND', 'BEND STICK', 'KNOCK ON DOOR',
                  'GET DOWN', 'FILL BUCKET WITH WATER', 'EMPTY BUCKET',
                  'CLIMB OVER FENCE', 'THROW STONE INTO SEA',
                  'TAKE BULLET WITH TWEEZERS', 'OPEN DOOR WITH KEY',
                  'RUB LAMP WITH CLOTH']
                  
  conversationVerbs = static inherited + 'PRAISE SUSAN' + 'INSULT BOB'
  
  allRequiredVerbsDisclosed = true
;

 /*
  *  This extends the Inventory command to include what the PC has hidden in his/her
  *  hat and/or boots.
  */
  
modify InventoryAction
  execAction()
  {
    inherited;
    inWornInv('Under your hat you have ', hat);
    inWornInv('In your boots you have ', boots);    
  }
  inWornInv(prefix, obj)
  {
   if(obj.isWornBy(gPlayerChar) && obj.contents.length > 0)
    {
      local items = obj.contents.length;
      local i;
      "<<prefix>> ";
      for (i = 1; i <= items; i++  )
      {
         "<<obj.contents[i].aName>>";
         if(i < items - 1)
           ", ";
         if(i == items - 1)
         " and ";
      }
      ". ";
    }
  }
;

 /*
  *  An alternative to part of the following customizations of YellAction and JumpAction
  *  would have been to use roomBeforeAction to trap these actions in the locations
  *  concerned, a method you'll also find in the source. However, the modification
  *  to YellAction also provides a customized standard response, which echoes the word
  *  (e.g. 'yell', 'shout' or 'scream') actually typed by the player. Note that
  *  gVerbName is not a library macro - it's defined in SquareCircle.h
  */
  
modify YellAction
  execAction()
  {    
    if(gActor.isIn(northRoad) && gActor.canSee(guardian))
    {     
      "{You/he} <<gVerbName>>{s} very quietly indeed, so as not to attract the
       attention of the guard. ";
      exit; 
    }
    "{You/he} <<gVerbName>>{s} as loud as {it actor/he} can. ";
  }  
;

modify JumpAction
  execAction()
  {
    if(gActor.isIn(cellBed) || (gActor.isIn(desk) && !desk.location.ofKind(OutdoorRoom)))
      "Jumping up and down on <<gActor.location.theName>> causes you to 
       hit your head on the ceiling, so you speedily desist. ";  
     
    else
       inherited;    
  }

;

class Fragile : Thing
 dobjFor(ThrowAt) 
 {
   check()
   {
     reportFailure(shouldNotThrowMsg);
     exit;
   }
 }
 iobjFor(ThrowAt)
 {
  check()
   {
     reportFailure(shouldNotThrowAtMsg);
     exit;
   }
 }
 shouldNotThrowMsg = '{You/he} might break {the dobj/him} if {it actor/he} do{es}
   that, so {it actor/he} decide{s} against it. '
 shouldNotThrowAtMsg = '{You/he} might break {the iobj/him} if {it actor/he} do{es}
   that, so {it actor/he} decide{s} against it. '  
;


//nothing : object;

modify Actor
  cannotDrawOnMsg = '{The dobj/he} really wouldn\'t appreciate that. '
  cannotWriteOnMsg = (cannotDrawOnMsg)  
;


modify Underside
  verifyPutInInterior()
  {
    if(lexicalParent != nil && gDobj == lexicalParent)
      illogical(&cannotPutUnderSelfMsg);
    inherited;
  }
;


  /* Various additional system actions for providing information to the player */


DefineSystemAction(WalkThrough)
  execAction()
  {
    "<.p>A complete walkthrough for <i>Square Circle</i> is available in
     the file <b>SCWalkThru.txt</b> that should have accompanied this game
     file. In addition to reading this file you can also, if you really want,
     <a href=REPLAY>REPLAY</a> it.\b
     Context-sensitive in-game hints are available with the 
     <a href=HINTS>HINTS</A> command.<.p>";
  }
;

VerbRule(WalkThrough)
  'walkthru' | 'walkthrough'
  : WalkThroughAction
  verbPhrase = 'walk/walking through'
;


DefineSystemAction(Help)
  execAction()
  {
    "For general instructions about playing the game, type
    <a href='INSTRUCTIONS'>INSTRUCTIONS</a>.\n For specific hints on parts of the game you're
    having difficulty with type <a href=HINTS>HINTS</A>.\n For help on non-standard
    interface features type <a href=FEATURES>FEATURES</A>.\n
     For a brief and general introduction to
    this game type <a href=ABOUT>ABOUT</a>.\nTo see the game credits type 
    <a href=CREDITS>CREDITS</a>.\n
    If you <<gameMain.showExitsInStatusline ?
    'hate the status-line exit lister and want to get rid of it' :
    'want to get the status-line exit lister back'>>,
    type <a href='TOGGLE EXITS'>TOGGLE EXITS</a>";
  }
;

VerbRule(Help)
  'help'
  : HelpAction
  verbPhrase = 'help/helping'
;


DefineSystemAction(License)
  execSystemAction()
  {
   "<u>Square Circle</u>\b
    Copyright (c) 2004 Eric Eve\b
   

1. This license grants you the right to use and distribute Square Circle (<q>the game</q>) 
under the conditions outlined below. You cannot use or distribute the game if you do not 
agree with these conditions.\b

2. The game, including any and all associated files as well as all of the output produced 
by the game, and all derivative works, is copyright (c) 2004 by Eric Eve.\b

3. Queries about the license should be addressed to the copyright holder, whose address 
at the time of writing is <a href='mailto:eric.eve@hmc.ox.ac.uk'>eric.eve@hmc.ox.ac.uk.</a>
If contact cannot be established, you 
may be able to locate the copyright holder via the rec.arts.int-fiction Internet newsgroup.\b

4. You are allowed to use the game and make copies for your personal use without restrictions. 
You may also copy and distribute the game in any medium, as long as (a) you make no charge for 
doing so, and (b) the game, all associated files and this license are included and remain 
unmodified.\b

5. You are <b>not allowed</b> to make a charge for distributing the game (either for profit or merely 
to recover your media and distribution costs) whether as a stand-alone product, or as part of 
a compilation or anthology, without written permission.\b

6. NO WARRANTY. The game is licensed and provided <q>as is</q> without warranty of any kind, 
express or implied, including, but not limited to, the implied warranties of merchantability 
and fitness for a particular purpose or a warranty of non-infringement.";

  }
;


VerbRule(License)
  'license' | 'copyright'
  : LicenseAction
  verbPhrase = 'display/displaying license information'
;

modify VerbRule(LookUnder)
  ('look' | 'l' | 'search' | 'peer') 'under' dobjList
  :
;

VerbRule(Features)
  ('interface'|'features')
  : FeaturesAction
  verbRules = 'explain/explaining features'
;

DefineSystemAction(Features)
  execSystemAction()
  {
    featuresMenu.display();
    "Done.";
  }
;

featuresMenu : MenuItem
  title = 'Special Interface Features'
;

+ MenuLongTopicItem
  title = 'Using ALL with verbs (as in EXAMINE ALL)'
  menuContents {
   "Using ALL with verbs like EXAMINE, GIVE TO and EAT can sometimes amount to
   near-cheating, a mimesis-breaking method of provoking the game into providing a data-dump
   on every object in sight, or of by-passing a puzzle by using OPEN ALL WITH KEY till you
   hit on the right lock. It does not seem to be a feature that genuinely enhances the
   game-playing experience in the spirit in which the game is intended to be experienced,
   and so this game disallows ALL with all but a small group of inventory-handling
   commands such as TAKE and DROP.<.p>";
   
#ifdef __TESTING
   "However, since your tastes may differ from mine, and I have no wish to make my tastes
   ruin your enjoyment of my game, if you really hate not being able to EXAMINE ALL or
   SHOW ALL TO BOB, then you can use the TOGGLE ALL command to enable the more liberal
   (standard) use of ALL.<.p>";
#endif
  }
;

+ MenuLongTopicItem
  title = 'Conversation'
  menuContents = "Square Circle uses the standard conversational features built into
   the TADS3 adv3 library, which build on the standard ASK ABOUT/TELL ABOUT system.
   Since the TADS3 extensions to this system may not be familiar, they are explained
   briefly here.\b
   As in many other games, the basic way to engage in conversation with a Non Player
   Character is to use ASK ABOUT or TELL ABOUT; for example ASK BOB ABOUT LIGHTHOUSE
   or TELL BOB ABOUT THE BLUE BALL. Note that repeating the same command may often
   (although not always) elicit further responses to the same topic; you are
   <i>strongly</i> encouraged to try this on any topic of interest in order to
    get the full story.\b
   If you start a conversation with an explicit greeting command such as TALK TO BOB
   or BOB, HELLO, you'll see that you get a list of possible topics you can ask or
   tell Bob about (which may also include suggestions for things to give him, show
   him, or ask him for). To show such a list during the course of the conversation
   you can issue a TOPICS command. Note that the list of suggested topics you see
   may not be exhaustive, and it is generally worth trying other topics that you want to
   pursue even if they're not listed.\b
   To save typing, once a conversation is underway TELL ABOUT and ASK ABOUT can be
   abbreviated to T and A, e.g. A LIGHTHOUSE or T BLUE BALL.\b
   The most unfamiliar part of the conversation interface is that which allows the
   player to respond with a command other than the usual ASK and TELL or a straightforward
   YES or NO. This will generally occur when the Player Character is required to respond
   to a question, challenge or request from a Non Player Character. At this point, you
   will be presented with a list of possible responses to choose from. For example,
   you might see something like this:\b
   <q>When did you start sleeping with my mistress?</q> Bob demands indignantly.\n
   (You could tell him about your affair, lie, deny ever seeing her, or say it was last week)\b
   Here the first of these is a standard TELL ABOUT response, the others are all non-standard.
   To select the response you want to use, type all or part of the text as it appears,
   e.g. you could use TELL HIM ABOUT AFFAIR, LIE, DENY, and LAST WEEK to choose each of
   the four options respectively.\b
   Note that these non-standard options are generally valid for the current turn, where
   they represent possible responses to a question that's just been asked, or a demand
   that's just been made. If you choose, say LIE at this point, SAY LAST WEEK will not
   be a valid command on a subsequent turn.\b
   One exception for this if is if the NPC is not satisfied with your reply and continues
   to press you for one (e.g. you reply TELL BOB ABOUT THE WEATHER at this point and you
   get a response like <q>We were talking about my mistress -- when did you start sleeping
   with her?</q>), in which case the same options will still be available. If in doubt you
   can always use the TOPICS command to see whether the special topics are still currently
   available.\b
   Another point to note about these special responses is that if you enter something that
   doesn't match any of them, the parser will assume it's meant to be an ordinary command and
   will treat it accordingly. Often, this may be what you want (e.g. you wouldn't want the
   list of conversational options to stop you issuing a QUIT, SAVE, EXAMINE or LOOK command);
   it may, however, occasionally give responses that appear puzzling or even frustrating if you 
   don't understand how the parser is interpreting your response (although this has been
    greatly improved in TADS 3.0.8).\b
   
   Finally, you will get more enjoyment from the game if you treat NPCs as characters
   to be conversed with, rather than as puzzles to be solved. You don't need to elicit
   every possible pre-programmed response from every NPC in order to win or to score maximum
   points or to understand what's going on (although, as mentioned above, it will generally
   prove worthwhile to elicit more information on key topics by ASKing or TELLing about
   them multiple time). Simply follow lines of conversation that seem
   important or interesting, and you'll discover all you need to know without treating the
   conversation system as a maze to be exhaustively mapped.\b
   (For further information, see the section on <q>Interacting with other Characters</q>
    from the INSTRUCTIONS command).
   "
;

+ MenuLongTopicItem
  title = 'Non-Standard Commands (Verbs)'
  menuContents {
   "For the most part, this game can be completed using the standard set of commands
   commonly used in Interactive Fiction. But there are a few more commands that are
   either useful or vital:\b
   Firstly, TADS 3 extends the standard conversation system, so you may want to read
   the Conversation chapter.\b
   Various additional <b>System Commands</b>\n";
   
#ifdef __TESTING      
   "TOGGLE ALL -- toggle permitting the use of ALL with most commands.\n ";
#endif      

  "TOGGLE EXITS -- toggle the status line exit lister on and off.\n  
   EXITS ON -- list exits in both the status line and room descriptions.\n
   EXITS OFF -- list exits in neither the status line nor room descriptions.\n
   EXITS STATUS -- list exits in the status line only.\n
   EXITS LOOK -- list exits in room descriptions only.\n
   INSTRUCTIONS -- show an instructions menu for general instructions on playing IF in TADS 3.\n
   FEATURES -- display this menu, explaining non-standard or interface features.\n
   LICENSE -- display the license information.\n
   NOTE -- add an annotation to a transcript.\b
   Some additional <b>Game Commands</b>\n
   CLIMB OVER FENCE -- does what it says\n
   EMPTY BOX -- takes everything out of the box.\n   
   FILL BUCKET WITH WATER -- is effectively equivalent to PUT WATER IN BUCKET,
   but only for situations where the FILL WITH phrasing makes sense.\n
   KNOCK ON DOOR -- does what it says.\n
   OPEN DOOR WITH KEY -- performs UNLOCK DOOR WITH KEY followed by OPEN DOOR.\n
   RUB LAMP WITH CLOTH -- fails to produce a genie, otherwise does what it says.\n
   SHAKE BOX -- does what it says.\n
   TAKE PIN WITH TWEEZERS -- does what it says.\n
   THROW STONE INTO POND -- does what it says (usually with the same effect
    as PUT STONE IN POND).\b
   There are a few more, but the others are either synonyms for more standard
   verbs, or else inessential actions that a player might try but that have no
   real effect on the game. ";   
   }
;

+ MenuLongTopicItem
  title = 'The Status-Line Exits Lister'
  menuContents = "Personally, the list of exits in the status line is a feature I really
   like; it helps players orient themselves and prevents room descriptions having to
   embody a mechanical list of exits. By default, therefore, this is a feature that is
   enabled in this game.\b
   It may be, however, that it's a feature that's not to your taste, in which case you can
   banish the pesky status line exit lister with a simple TOGGLE EXITS command. If you
   later change your mind you can use TOGGLE EXITS to summon it back again.\b
   More generally, you can use the following commands to control listing of exits:\n
   EXITS ON -- list exits in both the status line and room descriptions.\n
   EXITS OFF -- list exits in neither the status line nor room descriptions.\n
   EXITS STATUS -- list exits in the status line only.\n
   EXITS LOOK -- list exits in room descriptions only."
;

+ MenuLongTopicItem
  title = 'The NOTE Command (for transcripts)'
  menuContents = "If you want to make a transcript while playing a TADS 3 game you
   can use the SCRIPT command (with SCRIPT OFF ending the transcript).\b
   To make notes in the course of the transcript you can use the NOTE command,
   e.g.\b
   <b>>SEARCH SAND</b>\n
   In the sand you find nothing but mpre sand.\b
   <b>>NOTE Typo: mpre </b>\n
   Noted.\b
   If you use this command when you're not actually recording a transcript, the 
   parser will advise you of the fact (since it's probably pretty pointless
   to be making notes that are never recorded). "
;


modify PluralProd
  getVerifyKeepers(results)
    { return results.subset({x: x.allowAction}); }
;
 

/* modify CloseAction to make it close objects in order of size, smallest
 * first, so that contained objects are closed before their containers
 */
 
modify CloseAction
  doActionMain()
     {  
       dobjList_ = dobjList_.sort(nil, {a,b: a.obj_.bulk - b.obj_.bulk});
       inherited;
     }
;


modify Chair
  dobjFor(Take)
  {
    check()
    {
      foreach(obj in gActor.contents)
      {
        if(obj.ofKind(Chair))
        {
          "Your hands are too full to carry another bulky object like <<theName>>. ";
          exit;
        }
      }
    }
  }
;  
  
  
replace VerbRule(GetOffOf)
     'get' ('off' | 'off' 'of' | 'down' 'from') singleDobj
     : GetOffOfAction
     verbPhrase = 'get/getting (off what)'
     askDobjResponseProd = singleNoun
 ;
  
modify Topic
  construct(name_)
  {
    vocabWords = name_;
    name = name_;
    inherited();
  }
  nameWords = (name != nil ? name : vocabWords)
  aName = (abstract ? nameWords : Thing.aNameFrom(nameWords))
  theName = (abstract ? nameWords : Thing.theNameFrom(nameWords)) 
  abstract = nil; 
; 
 
function topicText()
  {
    local obj;
    
    if(gDobj != nil && (gActionIs(GiveTo) || gActionIs(ShowTo)))
      obj = gDobj;
    else if(gTopic != nil)
      obj = gTopic.getBestMatch;
    else if(gAction.ofKind(EventAction))
      return 'orders';
    else
      return gAction.tokenList[1][1];
    if(obj != nil)
      return obj.aName;
    else
      return gTopic.getTopicText.toLower;
  }
 
function listContains(lst, strg)
{
  strg = strg.toLower;
  foreach(local cur in lst)
  {
    if(dataType(cur) == TypeSString && cur.toLower == strg)
      return true;
    if(dataType(cur) == TypeList && listContains(cur, strg))
      return true;  
  }
  return nil;
} 
 
//Topics
t_line : Topic 'straight line/lines' 'line';
t_lines : Topic '# lines' 'lines';
t_circle : Topic 'circle/circles' 'circle';
t_square : Topic 'square/squares' 'square';
t_greatCircle : Topic 'great circle';
t_arc  : Topic 'arc';
t_squareCircle : Topic 'square circle/circles' 'square circle'
 matchNameCommon(origTokens, adjustedTokens)
 {
   if(!listContains(adjustedTokens, 'square')) return t_circle;
   if(!listContains(adjustedTokens, 'circle')) return t_square;
   return self;
 }
;
t_roundSquare : Topic 'round circular square' 'circular square'
 matchNameCommon(origTokens, adjustedTokens)
 {
   if(!listContains(adjustedTokens,'round') &&  
      !listContains(adjustedTokens,'circular')) return t_square;
   return self;
 }
;


t_triangle : Topic 'scalene isoceles equilateral triangle/triangles' 'triangle';
t_hexagon : Topic 'hexagon';
t_point : Topic 'point';
t_geometry : Topic 'geometry';
t_theorems : Topic 'theorem/theorems/proofs' 'theorem';
t_euclidean : Topic 'euclidean space/(geometry)';
t_nonEuclidean: Topic 'non-euclidean space/(geometry)/geometries';
t_sphere : Topic 'sphere';
t_concepts : Topic 'geometrical concepts';
t_geometricalSpace : Topic 'geometrical kinds/space/spaces';
t_angle : Topic 'right acute obtuse reflex angle*angles' 'angle';
t_plane : Topic 'plane/planes' 'plane';
t_centre : Topic 'centre/center' 'centre';
t_circumference : Topic 'circumference/circumfrence';
t_rectangle : Topic 'rectangle';
t_quadrilateral : Topic 'quadrilateral';
t_polygon : Topic 'polygon';


//t_equator : Topic 'equator';
t_surface : Topic 'surface/surfaces' 'surface';

t_crime : Topic 'crime/criminal' 'criminal';
t_criminals : Topic 'criminal criminals/elements' 'criminals' isPlural=true ;
t_contract : Topic 'contract';
t_punishment : Topic 'punishment' abstract=true;
t_justice : Topic 'abstract justice' abstract=true;
t_law : Topic 'law';
t_lawyer : Topic 'lawyer/lawyers' 'lawyer';
t_judge : Topic 'judge/judges' 'judge';
t_prisoner : Topic 'prisoner/prisoners';
t_prison : Topic 'prison';
t_guilt: Topic 'guilt' abstract=true;
t_prevention : Topic 'prevention' abstract=true;
t_trial : Topic 'trial';
t_jury : Topic 'jury';
t_evidence: Topic 'evidence' abstract=true;
t_witness : Topic 'witnesses' isPlural = true;
t_mercy: Topic 'mercy' abstract = true;

t_food : Topic 'food/ration/rations' abstract=true;
t_cannibal : Topic 'cannibal/cannibalism*cannibals' 'cannibal';
t_cell : Topic 'cell';

t_god : Topic 'god' 'God' abstract=true;
t_newEnlightenment : Topic 'new enlightenment/revolution' 'New Enlightenment' aName=(theName);
t_pandra : Topic 'pandra' 'Pandra' abstract=true;
t_tyranny : Topic 'tyranny/tyrrany' 'tyranny' abstract=true;
t_dunderhead : Topic 'prof professor dunderhead' 'Professor Dunderhead' abstract=true;
t_department : Topic 'applied department/rationality/dofar' 'department' aName=(theName);
t_departmentG16 : Topic 'department g16' 'Department G16' abstract=true;
t_government : Topic 'rational government' abstract=true;
t_help : Topic 'help/assistance/favor/favour' 'help' abstract=true;

t_pass : Topic '(visitor) pass/passes';
t_securityCriteria : Topic 'security valid criteria/reasons' abstract=true;
t_shooting : Topic 'shooting/shot/shots/gunfire';
t_reason : Topic 'arid reason/rationality/rationalism' 'reason' abstract=true;
t_religion : Topic 'religion' abstract=true;
t_rulers: Topic 'rulers';
t_hut : Topic '(his) hut/home' 'hut';
t_forest : Topic 'forest';
t_sex : Topic 'sex' abstract=true;
t_society : Topic 'brave new society';
t_love : Topic 'love' abstract=true;
t_manifesto : Topic 'pandra\'s rational principles/government/manifesto/book' 
   'Principles of Rational Government';
t_work : Topic '(his) (her) work/job' 'work' abstract=true;
t_courier : Topic 'courier';
t_joe : Topic 'old farmer joe';
t_signature : Topic 'signature/signing' 'signature';
t_cabal : Topic 'rational denocratic cabal/democrats' 'cabal';
t_wisdom : Topic 'wisdom' abstract = true;
t_plato : Topic 'plato' 'Plato' abstract = true;
t_tradition : Topic 'tradition' abstract = true;
t_idolatry: Topic 'idolatry' abstract = true;
t_clothes: Topic 'clothes/clothing';
t_flood: Topic 'heavy flood/rain/flooding' 'flood';


 /* The endGame function attempts to get round an a bug in versions of TADS
  * prior to 3.0.8 where calling finishGame in certain contexts can cause
  * a crash or bizarre results on attempting to restore. It does no harm to
  * continue the function even with 3.0.8
  */


function endGame(flag)
{
  if(flag == ftVictory)
    gameEnd.finishOptions = [finishOptionUndo,finishOptionFullScore, finishOptionAmusing];
  else
    gameEnd.finishOptions = [finishOptionUndo,finishOptionFullScore];
    
  gameEnd.finishFlag = flag;  
  

  new PromptDaemon(gameEnd, &gameFinish);
  exit;  

}

gameEnd: object
  finishOptions = [finishOptionUndo, finishOptionFullScore]
  finishFlag = ftGameOver
  gameFinish
  {
    finishGameMsg(finishFlag, finishOptions);
  }
;


modify finishOptionAmusing
  doOption()
  {
    amusingMenu.display();

     return true;
  }
;

amusingMenu : MenuItem
  title = 'Other things to try'
;

+ MenuTopicItem
  title = 'Blowing a fuse (or two)'
  menuContents = [
  'Unscrew the silver disc from the round table in the circle room.',
  'Take the disc (and the light cube!) into the cell.',
  'Remove the light bulb.',
  'Insert the disc in its place.',
  'You can now go through the blue steel door with impunity.'
  ]
;

+ MenuLongTopicItem
  title = 'A makeshift projector'
  menuContents = 'Put the light cube under the round glass-topped table and try
   drawing on it (the table, that is). '
;

+ MenuTopicItem
  title = 'Five odd things to do with a pig'
  menuContents = [
  'Try to order him about.',
  'Try talking to him.',
  'Give him a kiss',
  'Offer him a ham sandwich.', 'Attack him. '  
  ]
;

+ MenuTopicItem
  title = 'How did you get past the gate? '
  menuContents = [
    'You can file the iron key with the steel implement, then it\'ll open the lock. ',
    'You can fill the large white box with sand and then stand on it. ',
    'You can fill the small green box with sand and use it to plug the pothole,
     so that you can then push the desk up to the gate and climb over. ',
    'You can unscrew the guard\'s chair (with the steel implement) and then
     carry the chair out to the small yard and stand on it to climb over the gate. ',
    'You can push the trough up to the gate and stand on it to climb back over the
     other way (provided the gate is closed). ' 
  ]
;


+ MenuLongTopicItem
  title = 'Is {the receptionist/she} really that icy?'
  menuContents = 'Try finding out about her love-life! '
;

+ MenuTopicItem
  title = 'Scoring those missing points'
  menuContents = [    
    'If and when you manage to capture the scrap of paper on the hillside,
     make sure you read it. ',
    'Did you show the coveralls to {the hermit/him} after you removed them?',
    'Did you show him the press release?',
    'What did he say? How may times did you try? ',      
    'Destroy the press release. ',
    '(By throwing it either into the fire in the hermit\'s hut or the river
     under the bridge)',
    'Dispose of your prison coveralls in the same way. ',
    'Make sure you read the statement the white-coated man wants you to sign --
      but don\'t sign it!'    
  ]
;

+ MenuTopicItem
  title = 'What was written on that scrap of paper?'
  menuContents = [
    'Did you try showing the scrap to {the hermit/him}? ',
    'Did you then try re-reading it? '
  ]
;

+ MenuTopicItem
  title = 'Magic Words'
  menuContents = [
    'You no doubt tried XYZZY at some point. ',
    'And replied to the rhetorical question you got? ',
    'Did you then try XYZZY a second time? '
  ]
;

 /* The DrawingSurface class and the penHeld precondition are used to allow drawing
  * and writing on various objects in the game.
  */

class DrawingSurface : Thing
  maxDoodles = 10
  maxWritingLength = 1024
  writing = ''
  dobjFor(WriteOn)
  {
    verify() {  }    
    check()
    {
      if(writing.length() + gLiteral.length() > maxWritingLength)
      {
        reportFailure('There isn\'t enough room left on {the dobj/him} to write that. ');
        exit;
      }
    }
    action()
    {
      local txt = gLiteral;
      if(txt.endsWith('"') || txt.endsWith('\'')) txt = txt.substr(1, txt.length()-1);
      if(txt.startsWith('"') || txt.startsWith('\'')) txt = txt.substr(2);
      if(writing.length > 0) writing += ' ';
      writing += txt;
      "{You/he} write{s} <q><<txt>></q> on {the dobj/him}. ";
    }
  }
  dobjFor(DrawOn)
  {
    verify() { logicalRank(150, 'writable'); }    
    check()
    {
      if(doodles != nil && doodles.length >= maxDoodles)
      {
        reportFailure('There\'s no more room to draw on the ' + name);
        exit;
      }       
    }
    action()
    {
      
      local obj = gTopic.getBestMatch;
      if(obj==equator)
        obj = t_greatCircle;
      if(obj != nil && !obj.ofKind(Topic))
         obj = nil;
      local txt = gTopic.getTopicText().toLower(); 
          
      if(txt.find('square') && txt.find('circle')) obj = t_squareCircle;
      if((txt.find('round') || txt.find('circular'))
          && txt.find('square')) obj = t_roundSquare;
      if(obj == nil) 
      {
       if(txt.startsWith('a ')) txt = txt.substr(3);
       if(txt.startsWith('an ')) txt = txt.substr(4);
       if(txt.startsWith('the ')) txt = txt.substr(5);
       obj = new Topic(txt);
      }
      if(obj == t_squareCircle)
      {
        reportFailure(squareCircleMsg);
        exit;
      }
      if(obj == t_roundSquare)
      {
        reportFailure(roundSquareMsg);
        exit;
      }
     initializeVocabWith(txt);
     doodle(obj); 
    }
  }
  doodle(obj)
  {
     if(doodles==nil) doodles = new Vector(10);
     doodles.append(obj);
     "You draw <<obj.aName>> on <<theName>>. ";
  }
  doodles = nil
  squareCircleMsg = ('But you don\'t know how to draw a square circle on ' + theName + '. ')
  roundSquareMsg = ('You can\'t imagine how to draw a round square on ' + theName + '. ')
  
  extras()
  {
    if(!gActor.canTouch(self)) // if we're too far away we can't see what's on me.
      return;
    if(doodles != nil) {
      "\bOn <<theName>> you have drawn ";
      listDoodles(); 
       ". " ;
    }
    if(writing > '') {
      "\bOn <<theName>> you have written <q><<writing>></q>. ";
    }
  }
  
  listDoodles()
  {
   local i = 0;
    foreach(local cur in doodles)
    {
      i++;
      say(cur.aName);      
      if(i == (doodles.length - 1)) " and ";
      else if(i < doodles.length) ", ";
    }
  }
  dobjFor(Examine)
  {
    action()
    {
      inherited;
      extras;
    }
  }
  dobjFor(DrawWhatOn) { verify { } } 
  dobjFor(WriteWhatOn) { verify { } } 
  shouldNotDestroyMsg = 'You see no reason to damage it. '
;

penHeld: PreCondition
     checkPreCondition(obj, allowImplicit)
     {
         /* if the pen is already held, there's nothing we need to do */
         if (pen.isHeldBy(gActor))
             return nil;
         
         /* the pen isn't being held - try an implicit 'take' command */
         if (allowImplicit && pen.tryHolding())
         {
             /* 
              *   we successfully executed the command; check to make sure
              *   it worked, and if not, abort the command without further
              *   comment (if the command failed, presumably the command
              *   showed an explanation as to why) 
              */
             if (!pen.isHeldBy(gActor))
                 exit;
 
             /* tell the caller we executed an implicit command */
             return true;
         }
 
         /* it's not held and we can't take it - fail */
         reportFailure('You must be holding a writing implement before you can '
           + gVerbName + ' anything.<reveal need-pen> ');
         exit;
     }
;

  /*
   *  Modifications and additions to standard library messages
   */
  
modify libMessages
  explainExitsOnOff {
     "<.p><.notification>
     Exits are currently ";
     if(gExitLister.enableRoomDesc || gameMain.showExitsInStatusline)     
     {
         "listed in ";
	     if(gameMain.showExitsInStatusline)
	     {
	       " the status line ";
	       if(gExitLister.enableRoomDesc)
	        " and ";
	       else
	        "only. ";
	     }
	     if(gExitLister.enableRoomDesc)
	     {
	       "room descriptions. ";
	     }
	 }
	 else
	   "not listed in either the status line or room descriptions. ";
       
     "<.p>Type <<aHref('EXITS OFF', 'EXITS OFF','turn off all exit lists' )>>
      to turn off all exit lists, 
     <<aHref('EXITS STATUS', 'EXITS STATUS', 'show exits only in the status line')>> 
     to show exits only in the status line, 
     <<aHref('EXITS LOOK','EXITS LOOK','show exits only in room descriptions')>> 
     to show exits only in room descriptions, 
     and <<aHref('EXITS ON', 'EXITS ON', 'show exits in both the status
      line and room descriptions')>> to show exits in both the status
      line and room descriptions.
     
     
     <./notification> ";
 }
 
  exitsOnOffOkay(stat)
     {
        if(!gExitLister.exitsOnOffExplained)
           explainExitsOnOff;
        else
         "<.parser>The list of exits will <<
         stat ? 'now' : 'no longer'>> be
         displayed in each room description 
         <<stat ? 'and' : 'or'>> in the status line.<./parser> ";

     }

;  
  
  
modify playerActionMessages
  outOfReach(obj)
  { 
    gMessageParams(obj);
    return '{The obj/he} {is} out of reach. ';
  }
  okayPushTravel(obj)
     {
         return '<.p>{You/he} push{es} ' + obj.theNameObj
             + ' into ' + obj.location.destName + '. ';
     }
  doNotDefaceMsg = '{You/he} do{es}n\'t want to deface {it dobj/him}. '
  cannotDrawOnMsg = '{That dobj/he} {is} not really suitable for drawing on. '
  cannotWriteOnMsg = '{That dobj/he} {is} not really suitable for writing on. '
  uselessToScribbleOnMsg = '{You/he} decide{s} that scribbling on {the dobj/him}
    would serve no useful purpose. '
  followUnknownMsg = '{You/he} {is} too far away from {the dobj/him} to
   follow {it dobj/him}. '
  nothingThroughMsg = '{You/he} can\'t see through {the dobj/him}. '
  nothingBehindOpenDoorMsg = 'Opening {the dobj/him} reveals nothing unusual. '
  doorClosesBehindMsg(obj)
     {
         gMessageParams(obj);
//         return '<.p>After {you/he} go{es} through {the obj/him}, {it/he}
//                 close{s} behind {it actor/him}. ';
//		   return '<.p>As {you/he} enter{s} ' + gActor.getOutermostRoom.destName +
//		     ' {the obj/he} close{s} behind {it actor/him}. ';
		   return '<.p>{The obj/he} close{s} behind {it actor/him}. '; 
     }

  moveNoEffectMsg 
  {
    local msg = 'If {you/he} want{s} to move {the dobj/him} somewhere,
     {it actor/he} can just ';
    if(!gDobj.isIn(gActor))
      msg += 'pick {it dobj/him} up and ';
    msg += 'carry {it dobj/him}. ';
    return msg; 
  }
  pushNoEffectMsg { return moveNoEffectMsg; }
  pullNoEffectMsg { return moveNoEffectMsg; }
  cannotPushTravelMsg = 'It would be simpler just to carry {it dobj/him}. '
;


modify Person
  dobjFor(Take) { verify() { illogical(cannotTakeMsg); } }
;

modify OutOfReach
     cannotReachFromOutsideMsg(dest) { return &outOfReach; }
     cannotReachFromInsideMsg(dest) { return  &outOfReach; }
;

modify Room
  enterRoom(traveler) { }
  travelerArriving (traveler, origin, connector, backConnector)  
  {
     inherited (traveler, origin, connector, backConnector);
     enterRoom(traveler);
  }
  private = nil
  travelerLeaving(traveler, dest, connector)
  {
    if((blueCoveralls.isWornBy(traveler) || oldClothes.isWornBy(traveler))
      || traveler != gPlayerChar)
      inherited(traveler, dest, connector);
    else
      {
        "You really <i>don't</i> fancy wandering around naked. ";
        exit;
      }
  }
  up : NoTravelMessage { "Unfortunately, you never learned to fly. " }
  down : NoTravelMessage { "Being neither mole nor rabbit, you can't just burrow into the ground. "}
;

  /*
   *  The following modification to TravelPushable is intended to implement a more
   *  helpful response to PUSH X, where X is a TravelPushable object. If the player
   *  types, for example, PUSH DESK, the response will now be "Where do you want to
   *  push the desk?" and an abbreviated answer such as NORTH or UNDER THE LIGHT will
   *  be correctly recognized (i.e., turned into PUSH DESK NORTH or PUSH DESK UNDER
   *  THE LIGHT).
   */


modify TravelPushable
  dobjFor(Push)
  {
    verify() { }
    action()
    {      
       local dir, newcmd, tokList, tok, lst;
       "Where would you like to push <<theName>>?\b>";
       dir = inputManager.getInputLine(nil, nil);         
       tokList = cmdTokenizer.tokenize(dir);
       tok = getTokVal(tokList[1]);
       lst = cmdDict.findWord(tok);
       if(lst != [] && lst[1].ofKind(predicate))
          executeCommand(gActor, gActor, tokList, true);
       else
       {
         newcmd = 'push ' + disambigName + ' ' + dir;
         tokList = cmdTokenizer.tokenize(newcmd);
         executeCommand(gActor, gActor, tokList, true);
       }
    }
  }
;

modify InConversationState
  
  alwaysSuggestOnEntry = (specialTopics != nil)
  activateState(actor, oldState)
  {
    if (alwaysSuggestOnEntry) "<.topics>";
    previousState = oldState;
  }
  
  /* Michel Nizette's fix for the BYE bug */
  nextState = (getActor().curState == self ? inherited :
       getActor().curState)
 
;

myActorHoldingDescInventoryListerLong: actorHoldingDescInventoryListerLong
  showInventoryCarryingOnly(parent, carrying)
     {
         /* we have only carried items to report */
         "<.p>\^<<parent.nameIs>> holding <<carrying>>. ";
     }
;

 /*
  *  The following modification keeps a SuggestedTopic that's also an EventList
  *  offering itself as a suggestion until all the responses in its eventList
  *  have been seen.
  */
  
modify SuggestedTopic
  timesToSuggest = (eventList != nil ? eventList.length : 1)
;


modify ByeTopic
  matchList = [byeTopicObj, impByeTopicObj]
;

modify ImpByeTopic
  matchScore = 200
;

modify Goal
  closeWhenMoved = nil
  openWhenMoved = nil
  openWhen = (inherited || (openWhenMoved != nil && openWhenMoved.moved))
  closeWhen = (inherited || (closeWhenMoved != nil && closeWhenMoved.moved)) 
;


ftCapture : FinishType 
   finishMsg = 'YOU HAVE BEEN CAPTURED'
;



/* Code for debugging and testing use only */

#ifdef __DEBUG
 
DefineIAction(Change)
  execAction
  {
    if(blueCoveralls.isWornBy(gActor))
    {
       blueCoveralls.moveInto(gActor.location);
       blueCoveralls.makeWornBy(nil);
       oldClothes.moveInto(gActor);
       oldClothes.makeWornBy(gActor);
       "Magically, {your} blue coveralls fall off, to be replaced
        by some old clothes. ";
    }
    else
    {
       oldClothes.moveInto(gActor.location);
       oldClothes.makeWornBy(nil);
       blueCoveralls.moveInto(gActor);
       blueCoveralls.makeWornBy(gActor);
       "Magically, {your} old clothes fall off, to be replaced
        by some blue coveralls. ";
       
    }
  }
;

VerbRule(Change)
  'change'
  : ChangeAction
  verbPhrase = 'change/changing'
;

DefineIAction(Win)
  execAction()
  { 
    endGame(ftVictory);
  }
;

VerbRule(Win)
  'win'
  : WinAction
  verbPhrase = 'win/winning'
;


#endif

 /* Many players like to try the XYZZY command just to see if and how a game has
  * implemented it. The implementation here simply displays a rhetorical question
  * the first time the player types XYZZY, PLUGH or PLOVER, but allows for the 
  * possibility that the player might reply YES or NO. The second (or any even-numbered)
  * time the player types one of these magic words, the player character is teasingly
  * transported (notionally, in thought) to the opening location of "Adventure", the
  * illusion being shattered as soon as movement is attempted.
  */


DefineIAction(Xyzzy)
  execAction()
  {
     if(gActor.isIn(xyzzyRoom))
        "Nothing happens. ";       
     else if(xyzzyRoom.alternate) {
       "Very well -- if you're <i>really</i> that determined to live in a fantasy world...\b";
       inputManager.pauseForMore(true);
       "<TITLE>Adventure</TITLE>";
       if(gActor.posture <> standing) gActor.makePosture(standing);
       xyzzyRoom.oldLoc = gPlayerChar.getOutermostRoom;
       gPlayerChar.moveIntoForTravel(xyzzyRoom);
       xyzzyRoom.statusLineExitsOption = gameMain.showExitsInStatusline;
       gameMain.showExitsInStatusline = nil;  
       xyzzyRoom.oldExitLister = gExitLister;
       gExitLister = nil;     
       gPlayerChar.lookAround(true);    
     }
     else
     {
      "In your imagination declaiming this hoary old incantation causes all prison walls
       everywhere to collapse into piles of dusty rubble, thus bringing you instant fame, fortune
      and freedom. In reality, nothing happens at all; but then you didn't
      seriously expect magic to work in this enlightened age, did you?<.p>";
      XyzzyFuse.fuseID = new Fuse(XyzzyFuse, &fire, 1); 
     }
     xyzzyRoom.alternate = !xyzzyRoom.alternate;
  }
;

XyzzyFuse : object
  fuseID = nil
  fire { fuseID = nil; }
;

VerbRule(Xyzzy)
  'xyzzy' | 'plugh' | 'plover'
  : XyzzyAction
  verbPhrase ='say XYZZY/saying XYZZY'
;

modify YesAction
  execAction()
  {
    if(XyzzyFuse.fuseID != nil)
      "Then you're sadly deluded. Get a grip -- this isn\'t <i>Colossal Cave</i>!";
    else
      inherited;
  }
;

modify NoAction
execAction()
  {
    if(XyzzyFuse.fuseID != nil)
      "No, I should think not! What were you thinking of? No -- don't answer that
       question, just <i>don't</i>!";
    else
      inherited;
  }
;



xyzzyRoom : OutdoorRoom 'At End of Road'
  "You are standing at the end of a road before a small brick building. Around you is
   a forest. A small stream flows out of the building and down a gully. "
  oldLoc = nil
  in: TravelMessage { -> (lexicalParent.oldLoc)
    "As you take your first step your futile daydream shatters...
    <<lexicalParent.returnToReality()>>\b
    ... and you find yourself back in reality. ";
  }
  north asExit(in)
  east asExit(in)
  south asExit(in)
  west asExit(in)
  alternate = nil
  returnToReality()
  {  
    inputManager.pauseForMore(true);
    gameMain.showExitsInStatusline = statusLineExitsOption;
    gExitLister = oldExitLister;
    "<TITLE><<versionInfo.name>></TITLE>";
    foreach(local cur in contents)
      if(!cur.ofKind(NonPortable))
        cur.moveInto(oldLoc);
  }
  cannotGoShowExits(actor) { }
  statusLineExitsOption = true
  oldExitLister = nil
;

+ Fixture 'dirt road' 'road'
  "The road is dirt, not yellow brick. "
  dobjFor(Follow) remapTo(In)
;

+ Fixture 'small brick well building/wellhouse/house' 'small brick building'
  "It's a small brick building. It seems to be a well house. "
  dobjFor(Enter) remapTo(In)
;

+ Fixture 'berry forest/oak/maple/ash/pine/grove/spruce/undergrowth/birch/saplings/leaves/bushes' 'forest'
  "The trees of the forest are large hardwood oak and maple, with an occasional grove
  of pine or spruce. There is quite a bit of undergrowth, largely birch and ash
  saplings plus nondescript bushes of various sorts. This time of year visibility is quite
  restricted by the leaves, but travel is quite easy if you detour around the spruce
  and berry bushes. "
  dobjFor(Enter) remapTo(In)
;

+ Decoration 'stream' 'stream'
  "You see nothing special about the stream. "
;

+ Decoration 'gully' 'gully'
  "That's not something you need to refer to in the course of this game. "
;

+ Unthing 'yellow brick road' 'yellow brick'
  "Yellow brick is what the road is <i>not</i> made of. Got it? Good! "
;

