/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 * Modification History
 *
 * AD551
 * =====
 *
 * 24-Mar-99    djp     Preview version (1.0)
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *                      Changes since version 1.0:
 *                      * Changed phuce to move objects on both sides of
 *                        door (so they will always be visible if we
 *                        come back by a different route)
 *                      * Gave the 'ship' a heredesc
 *                      * Added 'upstream' and 'downstream' travel in many
 *                        rooms.
 *
 * 23-Apr-99    djp     New release (1.10) of Adventure 551.
 *                      Changes since version 1.01:
 *                      * Added code for drinking from the wine fountain.
 *                      * Added a response for attempting to swim in the
 *                        sea.
 *                      * Fixed a coding bug affecting 'throw x at hive.'
 *
 * 30-Apr-99    djp     New release (1.11) of Adventure 551.
 *
 * 17-May-99    djp     New release (1.20)
 *                      Changes since version 1.11
 *                      * Pantry classed as an Indoor location.
 *                      * The pantry and phone booths have been given a
 *                      * notakeallfrom property so that 'take all' from
 *                        the containing room will ignore their contents.
 *                      * The mechanism to make the pantry's contents invisible
 *                        until it has been examined has been reworked.
 *                      * Putting the sack in the pantry is now allowed for
 *                        consistency.
 *                      * Saved more variables after sitting on the throne.
 *                      * Changed the description of an unoccupied booth
 *                        (which might have objects in it, particularly in
 *                        the endgame!)
 *                      * Changed the definition of the decrepit bridge
 *                        location (and associated objects) to always return
 *                        a non-nil value.
 *
 * 20-May-99    djp     New release (1.21)
 *                      Changes since version 1.20:
 *                      * Some adjectives removed due to a change in the
 *                        TADS 2.4.0 parser, which fixes a bug preventing
 *                        the use of an adjective as a noun when another
 *                        object uses the same word as a noun.  This is
 *                        now permitted, making it necessary to exercise more
 *                        caution in the use of adjectives.  For example, the
 *                        crystal bridge defined 'rod' as an adjective; this
 *                        meant that 'wave rod' would refer to the crystal
 *                        bridge if the rod wasn't present.
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this version:
 *                      * Changed the Cloak Pits, so there are now three
 *                        'featureless' pits which can be entered at will
 *                        using ENTER XXX PIT.  They're not really featureless
 *                        any more, because two of the Rings of Protection
 *                        are to be found in them.  The third has been stolen
 *                        by the Wumpus.  A clue to the use of the rings is
 *                        in the Octagonal Room.
 *                      * Added a few extra rooms, reached via the Middle
 *                        crack.  These rooms also provide a new route
 *                        to the Cloakroom when the Tight Crack is blocked.
 *                        (Previously, you could get there only by
 *                        drinking the wine.)  The corroded ladder in the
 *                        Cloak Pits room is a hint to the existence of
 *                        the new route.
 *                      * Added some scenery objects (e.g. a 'crack' object
 *                        seen from within a crack, rocks in the Octagonal
 *                        room, the natural bridge above the Rainbow Room.).
 *                      * Reworked the Limestone Spires puzzle.  Now success
 *                        or failure depends on the objects you carry - apart
 *                        from the sack, any item with the islarge or ishuge
 *                        property will cause you to be skewered.  Of course,
 *                        the cask is now in this category, so you can't get
 *                        the wine unless you enter via Styx.
 *                      * Systematically made 'board' a synonym for 'enter'.
 *                        Cleaned up a few errors in capitalization of
 *                        verDoEnter and doEnter methods.
 *                      * Record whether 'phuce' has been seen (Knoll.seenit)
 *                      * Additionally save Knoll.seenit and
 *                        Rise_Over_Bay.seenit when sitting on the throne.
 *                      * Removed newgame property from classes (where it
 *                        might cause problems in future extensions of the
 *                        game).  Game version properties are now explicitly
 *                        declared for individual items, not inherited from
 *                        classes.
 *                      * Added a few travel methods to reflect the fact that
 *                        'opening' (as a verb) is now a synonym for 'passage'.
 *                      * Changed the description of some Blue Grotto rooms
 *                        to reflect the possibility that they might be
 *                        visited before Grotto_West.
 *                      * Added rubble to E/W corridor.  Added sensible
 *                        responses to 'search rocks' and 'search rubble' in
 *                        various places.
 *                      * Implemented the gate to the Inner Courtyard.
 *                      * Made the Mountain King's sign into a portable object
 *                        (and moved it to ccr_item)
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 *                        Changes in this version
 *                      * Added a missing scenery object for the footsteps
 *                        in the south pit.
 *                      * Added the floatingdecoration class for the Lost
 *                        River.
 *              djp     Bugfix Version 2.02 (not released)
 *                        Changes in this version
 *                      * Changed the coding which warns about leaving on the
 *                        lamp.  It now looks for a special nolampwarn
 *                        property set by lit rooms outside the cave for
 *                        which the warning is to be suppressed.  The
 *                        rooms now include Top_Of_Steps and its associated
 *                        rooms.
 *                      * Changed the ladder room to be NoNPC (to avoid
 *                        unrealistic occurrences like the axe being dropped
 *                        here.)
 *                      * Added a back method for Cloak_Pits, when attempting
 *                        to climb back up the ladder.
 *
 * 17-Feb-00    djp     New release - Version 2.20
 *                        Changes in this version
 *                      * Further generalization of version handling
 *                        Changed CCR_Boothitem to use leaveRoom for
 *                        automatic room-leaving actions.
 * POLYADV
 * =======
 *
 * 24-Aug-99            Pre-release version 0.00
 *              bjs     Added 550-point extensions
 *          djp+bjs     Incorporated ad551 mods up to 2.20
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *                      * Set some doCount methods for scenery objects.
 *                      * Removed redundant location code from
 *                        floatingdecoration objects.
 *                      * Various changes to phone 'dial' code (for future
 *                        extensions)
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version
 *                      * Resurrection code now closes the safe (in the
 *                        551-point game)
 *                      * Added iswavetarget property for cases when the
 *                        rod is waved at a bridge or chasm etc.
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version
 *                      * Changed the code for Sea_Water to look at the
 *                        location, not the game version (needed in 701-point
 *                        game).
 *                      * Added 'up' and 'down' methods for the throne.
 *              
 * 20-Dec-00    djp     Version 2.02: bugfix release
 *                      Changes in this version
 *                      * Fix to north and balcony methods in Sword_Point 
 *                        which should refer to global.game701, not game701
 *                        (not a serious bug because Me.travelTo checks the
 *                        validity of the destination room anyway).
 *                      * Fix to exithints method in Sword_Point to allow
 *                        BACK to return to the balcony  
 *
 * 8-Jun-01     djp     Version 2.05: bugfix release
 *                      Changes in this version
 *                      * Fixed the hive code for putting in objects.
 *                      * Minor fix to ladder_1 sdesc and adesc.
 *
 * 17-Aug-01    djp     Version 2.06: bugfix release with e-mail address update
 *                      Changes in this version
 *                      * Changed room classes to reflect the incorporation 
 *                        of class NotFarIn in class Outside.
 *                      * Added missing 'building' property for In_Forest_3
 *                      * Changed the elfin door to issue the right message
 *                        if we try to unlock it after exiting the
 *                        Cylindrical Room.
 *
 * 22-Aug-01    djp     Version 2.07: bugfix release
 *                      Changes in this version
 *                      * Defined locations individually for Pit_1 etc. to
 *                        circumvent a TADS bug (affecting versions up to 
 *                        2.5.5.)  The bug only affects game files compiled
 *                        without a debugging option, and causes objects
 *                        to go missing from contents lists when (a) the 
 *                        location property is inherited from a class, and 
 *                        (b) the first reference to the contents list is in 
 *                        preinit.
 *
 * 22-Nov-01    djp     Version 2.08: bugfix release
 *                      Changes in this version
 *                      * Defined methods for 'leaving' or 'exiting' certain
 *                        objects.  (Equivalent to typing 'out').
 *                      * Changed the text when the decrepit bridge is
 *                        examined or the message is read.  A warning message
 *                        is omitted when the rod has been used successfully.
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version
 *                      * Added a few scenery objects in the castle garden.
 *                      * Fixed a bug affecting 'search pantry'.
 *                      * Fixed the 'look in' method for phone booths.
 *                      * Added alternative adjectives for the featureless
 *                        pits.
 *                      * Moved functions to ccr-fun.t
 *                      * Updated the pantry and phonebooths to use the new
 *                        Internal_Room class.
 *                      * Updated Multidoor, renamed it CCR_lockableDoorway,
 *                        moved definition to ccr-room.t.  Simplified the
 *                        coding of the elfin door object.
 *                      * Regrow mushrooms if lost at the decrepit bridge.
 *                      * Allow player to find a cake if the steps are
 *                        searched (after trying to eat mushrooms with no
 *                        cakes available).
 *                      * Fixed "Enter pits" in Room with Small Pits
 *                      * Additional rooms not in the 551-point game have
 *                        been moved to ccr-rm11.t
 *                      * Fixed a bug which caused an indefinite loop when
 *                        the player went down from Ledge_Above_Pinnacles
 *
 * 31-May-03    djp     Version 2.10: minor bugfixes and enhancements
 *                      Changes in this version:
 *                      * Changes to the messages when searching the steps
 *                        to find another cake.
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *               
 * 23-Aug-03    djp     Version 2:12 
 *                      * Updated the code for moving the boat to use the
 *                        new moveme method, originally developed for version
 *                        3.   
 *
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *                      * Changed phuce to use the new transmove method.
 *                      * Various minor changes in connection with the
 *                        701+ game mode.
 *
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * In the 701+ point mode, additional game play state
 *                        is saved after a 'throne mishap' (sitting on
 *                        throne with no crown)
 *                      * Phuce now works on the grassy knoll and in the
 *                      * Mountain King's private chamber.  (According to 
 *                        the new logic, it should work where the elves have 
 *                        cast it.)  Also changed the first occurrence of the
 *                        message about an elf shouting "Phuce".
 *                      * Changed 'liminous' to 'luminous' in the Crystal
 *                        Palace description.  (Can't find 'liminous' even in 
 *                        the OED).
 * 7-May-05     djp     Version 3.10 Rev. A
 *                      * Slight adjustment to a response to 'search steps'. 
 *                        It is made clear that fullcount refers to the number
 *                        of hidden cakes which have been found (but not a
 *                        cake which may have been seen to fall when the key
 *                        was left on the shelf).
 *
 * 26-May-05    djp     Version 3.10 Rev. C
 *                      * corrected the featurelessPits class to define a
 *                        location, not a loclist.
 *
 * 9-Aug-05     djp     Version 3.12: Enhancements and bugfixes
 *                      * Save global.verbose in 'throne mishap' code.  Also
 *                        save relevant properties of access shaft grilles.
 *
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Addition of room properties in connection with
 *                        drinking wine.  The property wino_trollstop is set
 *                        to true if the troll would block the route to the
 *                        normal locations (e.g. Cloak_Pits). This is true
 *                        for all the rooms in the Bubble Chamber/ Dark Cove/
 *                        Winery/ Crystal Palace area. (Use of the slippers
 *                        or boat is discounted).
 *                      * Addition of phleece property to outer courtyard. 
 *                        The word works only if the player has seen the 
 *                        elves use the word 'phleece', and if the copper
 *                        bracelet is worn (701+ game).
 *                      * Addition of exit properties from Castle_Room
 *                        (player sent to At End of Road)
 *
/* Additional rooms for 551-point version */

/* Note on compound words.  Compound word constructions were used in the
past to circumvent problems when verbs were used as adjectives; for example,
'actor, west' wouldn't parse correctly if an item defined 'west' as an
adjective.   This fix is no longer needed, and has been backed out from
version 2.00 onwards.

For the benefit of people using TADS 2.4.0 and later, we take care with the
use of nouns as adjectives.  Due to a bugfix in the interpreter, this produced
some odd results - for example, the crystal bridge could be referred to as a
'rod' when the black rod wasn't present.
 */

/* Note the following change from the Fortran version.  For consistency,
'building' from a surface location takes us back to At_End_Of_Road, rather
than Inside_Building. */

/* 141 */
/* Note - in the 701-point game this room combines the 551-point Sword Point 
   room with the 550-point N_Of_Reservoir.  Note that the 701-point game has
   two sword-in-the stone puzzles with different solutions!
 */
Sword_Point: darkroom, NoNPC
    game551 = true
    sdesc = "At Sword Point"
    ldesc = { I();
    "%You% %are% on a narrow promontory at the foot of a waterfall, which
    spurts from an overhead hole in the rock wall and splashes into a
    large reservoir, sending up clouds of mist and spray.
    To the south, the indistinct shape of the opposite shore can be dimly
    seen. ";
    P();
    "Through the thick white mist looms a polished marble slab, to
    which is affixed an enormous rusty iron anvil.  In golden letters
    are written the words: \"Whoso Pulleth Out This Sword of This
    Stone and Anvil, is Rightwise King-Born of All This Mountain.\"
    There is a narrow chimney on the east side of the promontory";
    // In the 701-point game, this room combines the characteristics of
    // the 551-point Sword_Point and the 550-point N_Of_Reservoir.  Players
    // may leave via the chimney, or hit the gong and ride the turtle.
    if (global.game701)
        ", and another passage leads north from here.  Large
        clawed tracks are visible in the damp ground, leading from
        the passage into the water. ";
    else ". ";
    }

    up = Top_Of_Slide  /* 142 */
    east = Top_Of_Slide
    climb = Top_Of_Slide
    chimney = Top_Of_Slide
    slide = Top_Of_Slide
    south = {
        local actor := getActor(&travelActor);
        Reservoir.doCross(actor);
        return nil;
    }
    cross = (self.south)
    across = (self.south)
    myhints = [Swordhint]
    listendesc = "You hear the sound of the waterfall, noisily splashing
    into the water. "
    north = {
        if(global.game701) return Warm_Room;
        else pass north;
    }
    balcony = {
        if(global.game701) return Balcony;
        else pass balcony;
    }
    exithints = {
        if (global.game701) return [Warm_Room, &north, Balcony, &balcony];
        else pass exithints;
    }

;
Anvil: fixeditem
    game551 = true
    sdesc = "rusty anvil"
    ldesc = "It's just a large, rusty iron anvil, fixed to the marble
        slab. "
    noun = 'anvil'
    adjective = 'rusty'
    location = Sword_Point
;
Marbleslab: fixeditem
    game551 = true
    sdesc = "marble slab"
    ldesc = "It's simply a slab of marble, to which a large iron
    anvil is attached. "
    noun = 'slab'
    adjective = 'marble'
    location = Sword_Point
;
/* 142 */
Top_Of_Slide: darkroom, NoNPC
    game551 = true
    sdesc = "At Top of slide"
    ldesc = { I();
    "%You% %are% on a narrow shelf above and east of the top of a very
    steep chimney.  A long smooth granite slide curves down out of sight
    to the east. ";"If %you% go%es% down the slide, %you% may not be able
    to climb back up.";}

    east = In_Misty_Cavern /* 98 */
    down = In_Misty_Cavern
    forward = In_Misty_Cavern
    slide = In_Misty_Cavern

    west = Sword_Point
    chimney = Sword_Point

;
/* 143 */
Throne_Room: darkroom
    game551 = true
    sdesc = "At Entrance to Throne Room"
    ldesc = { I();
    "%You% %are% in the private chamber of the Mountain King.  Hewn into the
    solid rock of the east wall of the chamber is an intricately-wrought
    throne of elvish design.  There is an exit to the west.";
    }

    // In the 701-point game, this room is off the Morion room; otherwise
    // it is reached directly from the Hall of the Mountain King.
    west = {
        if(not global.game701)return In_Hall_Of_Mt_King;
        else return Morion;
    }
    out = In_Hall_Of_Mt_King
    east = Throne_Room_East
    exithints = {
        if(not global.game701)return [In_Hall_Of_Mt_King, &west];
        else return [Morion, &west];
    }
    NPCexit1 = {
        if (not global.game701) return In_Hall_Of_Mt_King;
        else return Morion;
    }
    hasfloor = true
    hasfloordesc = true
    floordesc = "The floor is exceptionally smooth and clean, but isn't
        at all slippery or shiny. ";
;

Throne_Faraway: fixeditem, distantItem
    game551 = true
    sdesc = "throne"
    ldesc = "It's at the far end of the room.  I suggest that %you% get%s%
    closer if %you% want%s% to examine it!"
    noun = 'throne'
    adjective = 'elfin' 'elvish' 'intricate'
    location = Throne_Room
;
/* 144 */
Throne_Room_East: darkroom
    game551 = true
    sdesc = "At East side of Throne Room"
    ldesc = { I();
        // DJP - added small passages for elfin use only!
        "%You% %are% on the east side of the throne room.  Very narrow
        passages go north and south, but are too small to enter.  ";
    }

    west = Throne_Room
    out = In_Hall_Of_Mt_King
    north = {
        "That little elfin passage is much too small for humans to
        enter.";
        return nil;
    }
    south = {return self.north;}
    passage = {
        "The little elfin passages are much too small for humans to
        enter.";
        return nil;
    }
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.smaller; 
        // If the crown etc. hasn't been taken, leave it alone
        if (not crown.moved) crown.isfixed := true;
        if (not thronesign.moved) thronesign.isfixed := true;
        actor.roomMoveTravel(&transmove, Vast_Chamber);
        if (not crown.moved) crown.isfixed := nil;
        if (not thronesign.moved) thronesign.isfixed := nil;
        return nil;
    }
    hasfloor = true
    hasfloordesc = true
    floordesc = "The floor is exceptionally smooth and clean, but isn't
        at all slippery or shiny. ";
;
Vast_Chamber: darkroom, NoNPC
    game551 = true
    sober = true // no drinking here
    sdesc = "In a Vast Chamber"
    ldesc = { I();
        "%You% %are% in a vast chamber - so vast, in fact, that the
        walls and ceiling are too remote for your lamp to show them. 
        All you can see is a slightly rough floor, stretching in all
        directions. ";
    }
    north = {
        "You wander in that direction for a while, but the floor seems to
        stretch endlessly in front of you.  You see no sign of the walls,
        the side passages, or the throne. ";
        if(not self.traveled) {
            P();
            "Maybe the \"Phuce\" spell would be useful somewhere else, but
            it's no use to you here.  I suspect that the elves can control
            its effect, but you don't know how - and it's \(far\) too powerful 
            in here! ";
        }
        room_move(Vast_Chamber,Throne_Room_East);
        self.traveled := true;
        return nil;
    }
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.larger;
        actor.roomMoveTravel(&transmove, Throne_Room_East);
        return nil;
    }
    south = {return self.north;}
    east = {return self.north;}
    west = {return self.north;}
    nw = {return self.north;}
    ne = {return self.north;}
    sw = {return self.north;}
    se = {return self.north;}
    hasfloor = true
    hasfloordesc = true
    floordesc = "The floor is slightly rough, but easy to walk on.  It still
       looks spotlessly clean. ";
;        


Throne: fixeditem, chairitem
    game551 = true
    sdesc = "throne"
    ldesc = {
        "It's rather small, but just large enough for you to sit
        on it.  It is covered in intricately-carved elvish designs. ";
        if(spelunker_today.read) {
            "You recognize it as the throne pictured in the magazines,
            but it's much smaller than you expected it to be! ";
        }
    }
    noun = 'throne'
    adjective = 'elfin' 'elvish' 'intricate'
    onroom = true
    reachable = ([] + self)
    // If we sit on the throne, the result depends on whether we have
    // proof of royal blood.  If not, the game is restarted and the
    // player moved to the Hall of Mists and various objects to the
    // Rainbow Room.  The slippers are moved, and this provides another
    // route to the Gothic Cathedral.  If the crown is worn, the rod is
    // upgraded if carried.  A picture in the magazines gives a clue
    // to this.
    doSiton(actor) = {
        local restartlist := [],newscore, diffscore;
        inherited.doSiton(actor);
        if (not ((crown.location = actor) and crown.isworn)) {
            "A strange cloud of green smoke then
            envelops you, and a strange tune seems to go through your head;
            you then recognize it as 'Somewhere, Over the Rainbow.'
            Coughing and spluttering, you open your eyes and find ...";
            P();
            newscore := global.startscore + global.farinpoints;
            if(global.novicemode) newscore += global.novicepoints;
            if(global.nodwarves) newscore -= 5;
            diffscore := newscore - global.score;
            incscore(diffscore);
            // added vnumber for 701-point game
   /* 1*/   restartlist += global.vnumber;           
   /* 2*/   restartlist += global.novicemode;
   /* 3*/   restartlist += global.turnsofar;
   /* 4*/   restartlist += global.randomized;
   /* 5*/   restartlist += global.nodwarves;
   /* 6*/   restartlist += global.verbose;
            // Certain messages etc. are dependent on the player's
            // memory, so we save the relevant variables
            // Save variables to do with the safe and its combination
   /* 7*/   restartlist += Safe.location;
   /* 8*/   restartlist += SafeCombination.isseen;
   /* 9*/   restartlist += Dial.combination_set;
   /*10*/   restartlist += Safe.hasopened;
            // Note the syntax to add a list as a single element
   /*11*/   restartlist += [Dial.combination];
            // Variables to do with the player's learning experience
            // Player has been to the dead end crack?
   /*12*/   restartlist += Dead_End_Crack.isseen;
            // Player has seen the throne room?
   /*13*/   restartlist += Throne_Room.isseen;
            // Player has found the way to the Oriental room from
            // the Swiss Cheese room?
   /*14*/   restartlist += In_Swiss_Cheese_Room.foundorient;
            // Player knows how to cross the River Styx?
   /*15*/   restartlist += River_Styx_E.isseen;
            // Player has seen inside the pantry?
   /*16*/   restartlist += Pantry.isseen;
   /*17*/   restartlist += Pantry.islookedin;
            // Player has seen the elvish magic words?
   /*18*/   restartlist += Knoll.seenit;
   /*19*/   restartlist += Rise_Over_Bay.seenit;
   /*20*/   restartlist += Outer_Courtyard.seenit;
   /*21*/   restartlist += Blue1.location;
            // Save the Transindection stuff only if we're in the relevant
            // game mode.
            if(global.game701p) {
                // Do we know that the east rocks can be moved?
   /*22*/       restartlist += Rockfalls.movable;
                // Have we seen the Green-level Large Circular Room?
   /*23*/       restartlist += Green_Large_Circular_Room.isseen;
                // Have we completed the Wumpi chase scene?
   /*24*/       restartlist += Green_Tight_Crack_2.isseen;
                // Have we seen the Orange Level Office in the crystal ball?
   /*25*/       restartlist += Orange_Office.seenit;
                // Have we seen the hidden door at the top of the blocks?
   /*26*/       restartlist += Top_Of_Blocks.seendoor;
   /*27*/       restartlist += Access_Shaft_Door.location;
                // Access shafts (re messages about despondency)
   /*28*/       restartlist += Red_Access_East.isseen;
   /*29*/       restartlist += Blue_Access_East.isseen;
                // Do we know how to open the desk?
   /*30*/       restartlist += UnderDesk.isseen;            
                // Have we seen the Control Room in the glass ball?
                // (reveals hidden door at top of blocks)
   /*31*/       restartlist += Control_Room.seenit;
                // Have we read the manual (required before the game will allow
                // the topaz to be put through the hole)
   /*32*/       restartlist += manual.isread;
                // Have we misunderstood the scrawled note?
   /*33*/       restartlist += manual.misunderstood;
                // Have we read 'span' in the meaning of 'spun'?
   /*34*/       restartlist += DisplayBoards.spanseen;
                // Have we found the location of the tower staircase door?
   /*35*/       restartlist += Green_Zarkalon_Doorsite.isseen;
                // Location of key switch (behind Wumpi's bookcase)
   /*36*/       restartlist += Keyswitch.location;
                // Grille properties - have we found out whether they
                // are locked?
   /*37*/       restartlist += Red_Access_Shaft_Grille.isopened;
   /*38*/       restartlist += Blue_Access_Shaft_Grille.isfoundlocked;
   /*39*/       restartlist += Red_Access_Shaft_Grille.isfoundlocked;
            }
            restart(specialrestart,restartlist);
        }
        else {
            P(); I();
            "%Your% crown glows, and %you% really
            feel as if you \(%are%\) the mountain king! ";
            if (not black_rod.isupgraded) {
                if (black_rod.isIn(actor))
                    "Your black rod also glows and issues a strange
                    humming noise.
                    You notice that the star at the end is now shiny,
                    as if it's brand new! ";
                else if(black_rod.isInside(actor))
                    "You hear a strange humming hoise, but you're not
                    quite sure where it's coming from. ";
                if (black_rod.isInside(actor))black_rod.upgrade;
            }
            if (not gray_rod.isupgraded) {
                if (gray_rod.isIn(actor))
                    "Your gray rod also glows and issues a strange
                    whistling noise.
                    You notice that the star at the end is now shiny,
                    as if it's brand new! ";
                else if(gray_rod.isInside(actor))
                    "You hear a strange whistling hoise, but you're not
                    quite sure where it's coming from. ";
                if (gray_rod.isInside(actor))gray_rod.upgrade;
            }
            P();
        }
    }
    location = Throne_Room_East
    up = {
        local actor := getActor(&travelActor);
        self.doUnboard(actor);
        return nil;
    }
    down = {return self.up;}
    ioPutOn(actor,dobj) = {
        if(dobj = thronesign) {
            "You hang up the sign on the throne again, leaving it as you
            found it. ";
            dobj.moveInto(self.location);
            dobj.moved := nil;
        }
        else pass ioPutOn;
    }
    // Search throne now includes the sign.
    doSearch(actor) =
    {
        local fullcont = [];
        if(thronesign.location = self.location) fullcont += thronesign;
        fullcont += self.contents;
  
        if (self.contentsVisible and itemcnt(fullcont,[thronesign]) <> 0)
            "On <<self.thedesc>> %you% 
            see%s% <<listlist(fullcont,[thronesign])>>. ";
        else
            "There's nothing on <<self.thedesc>>. ";
    }


;
/* 145 */
dragged_down: object
    message = {"%You% %are% dragged down, down, into the depths of the
    whirlpool. ";
    "Just as %you% can no longer hold %your% breath, %you% %are% shot out
    over a waterfall into the shallow end of a large reservoir. ";"Gasping
    and sputtering, ";"%you% crawl%s% weakly towards the shore....";P();}

;
/* 146 */
In_Forest_3: room, Outside
    game551 = true
    sdesc = "In Forest"
    ldesc = { I();
        "%You% %are% in dense forest, with a hill to one side.  The trees
        appear
        to thin out towards the north and east.";
    }

    east = At_Hill_In_Road
    up = At_Hill_In_Road
    road = At_Hill_In_Road
    climb = At_Hill_In_Road
    south = {
        if (rand(100) <= 50)
            return In_Forest_2;
        else
            return In_Forest_3;
    }
    forest = {return self.south;}
    west = Knoll
    north = Salt_Marsh_Edge
    building = At_End_Of_Road
    // Exit info. for 'back' command:
    exithints = [
        In_Forest_2, &south,
        In_Forest_3, &south
    ]
;
/* 147 */
Knoll: room, Outside
    game551 = true
    sdesc = "On Grassy Knoll"
    ldesc = { I();
    "%You% %are% at the high point of a wide grassy knoll, partially
    surrounded
    by dense forest.  The land rises to the south and east, and drops off
    sharply to the north and west.  The air smells of sea water.";}

    east = In_Forest_3
    south = In_Forest_3
    north = Salt_Marsh_Edge
    west = Sandy_Beach
    hole = Thunder_Hole
    building = At_End_Of_Road
    thunder = Thunder_Hole

    elfcurse = {
        if (not parserGetMe().isIn(self))return;
        if (rand(100) <= 25) {
            P(); I(); "A tiny elf runs straight at you, shouts
            \"Phuce!\",
            and disappears into the forest.  ";
            if (not self.seenit) 
                "Maybe it was a trick of the light, but you thought you saw
                the elf grow larger as he ran away from you. ";
            self.seenit := true;
        }
    }
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.smaller; 
        // If the clover hasn't been picked, leave it alone
        if (not clover.moved) clover.isfixed := true;
        actor.roomMoveTravel(&transmove, Dense_Jungle);
        if (not clover.moved) clover.isfixed := nil;
        return nil;
    }
;
Dense_Jungle: room, Outside
    game551 = true
    sober = true // no getting out by drinking wine
    sdesc = "In a Dense Jungle"
    ldesc = { I();
        "You're in a dense jungle, surrounded by large leaves which tower above
        your head.  Progress may be possible to the north, east or 
        southwest. ";
    }
    north = {
        "You manage to walk a short distance to the north, only to encounter
        an unfriendly-looking giant spider.  You decide to beat a hasty
        retreat. ";
        return nil;
    }  
    east = {
        "You walk a few yards to the east, only to encounter a giant bird!
        Fearing for your life, you beat a rapid retreat. ";
        return nil;
    }
    sw = {
        "You travel a fair distance to the southwest, only to encounter a
        giant grasshopper!   You retreat to your starting point and ask
        yourself whether this was the right place to use the \"Phuce\" spell.
        Most Elvish magic doesn't work for humans, but this particular
        spell seems to be \(too\) effective if you use it here. ";
        return nil;
    }
/* size of brass key as seen in this room */
    brasskey = large_key
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.larger; 
        actor.roomMoveTravel(&transmove, Knoll);
        return nil;
    }
;
Jungle_Leaves: fixeditem
    game551 = true
    sdesc = "tall leaves"
    ldesc = "They look like giant blades of grass. "
    location = Dense_Jungle
    noun = 'leaves'
    adjective = 'green' 'large' 'tall'
;

/* 148 */
Salt_Marsh_Edge: room, Outside
    game551 = true
    sdesc = "At Edge of Salt Marsh"
    ldesc = { I();
    "%You% %are% at the edge of a trackless salt marsh.  Tall reeds obscure
    the view.";}

    south = {
        if (rand(100) <= 50)
            return Knoll;
        else
            return In_Forest_3;
    }
    east = Salt_Marsh_1
    west = Salt_Marsh_1
    north = Salt_Marsh_1
    building = At_End_Of_Road
    // Exit info. for 'back' command:
    exithints = [
        Knoll, &south,
        In_Forest_3, &south
    ]
;

Saltmud: floatingdecoration
    game551 = true

    sdesc = "mud"
    adesc = "mud"
    ldesc = "It's just salty mud.  No use for anything."
    noun = 'mud'
    adjective = 'salty'
    loclist = [Salt_Marsh_1 Salt_Marsh_2 Salt_Marsh_3 Salt_Dead_End]
;
/* 149 */
Salt_Marsh_1: room, Outside
    game551 = true
    sdesc = "In Salt marsh"
    ldesc = { I();
    "%You're% in a salt marsh.";}

    south = Salt_Marsh_1
    east = Salt_Marsh_3
    west = Salt_Marsh_2
    north = Salt_Dead_End

;
/* 150 */
Salt_Marsh_2: room, Outside
    game551 = true
    sdesc = "In Salty marsh"
    ldesc = { I();
    "%You're% in a salty marsh.";}

    north = Salt_Marsh_3
    south = Salt_Marsh_3
    east = Salt_Marsh_1
    west = Salt_Marsh_1

;
/* 151 */
Salt_Marsh_3: room, Outside
    game551 = true
    sdesc = "In a Salt Marsh"
    ldesc = { I();
    "%You% %are% in a salt marsh.";}

    east = Salt_Marsh_Edge
    west = Salt_Dead_End
    north = Salt_Marsh_2
    south = Salt_Marsh_1

;
/* 152 */
Salt_Dead_End: room, CCR_dead_end_room, Outside
    game551 = true
    sdesc = "At Dead End in Salt Marsh"
    ldesc = { I();
    "%You% %have% reached a dead end in the salt marsh.";}

    east = Salt_Marsh_3
    south = Salt_Marsh_2

;
/* 153 */
Sandy_Beach: room, Outside
    game551 = true
    sdesc = "On Sandy Beach"
    ldesc = { I();
    "%You're% on a sandy beach at the edge of the open sea. The beach
    ends a short distance south and the land rises to a point. To
    the north, the beach ends in cliffs and broken rocks.";}

    north = Broken_Rocks
    south = Rise_Over_Bay
    east = Knoll
    building = At_End_Of_Road
    listendesc = "You hear the sound of the surf pounding against the
    beach and the broken rocks."
;
Sea_Water: roomliquid, floatingdecoration
    game551 = true
    game550 = true // Present in 550-point game as well.
    sdesc = "sea water"
    ldesc = {
        local actor := getActor(&travelActor);
        local toploc := toplocation(actor); 

        if(isclass(toploc,Outside)) {
            "Huge breakers pound against the beach.  Even if you could
            swim, you wouldn't choose to do so today! ";
        }
        else {
            "The calm ocean gently laps against the shore. ";
        }
    }
    noun = 'water' 'sea' 'surf' 'breakers' 'ocean'
    adjective = 'sea' 'salt'
    verDoTake(actor) = {
        "The water here isn't good for anything much. I'd look elsewhere
        if I were you. ";
    }
    verDoDrink(actor) = {
        self.verDoTake(actor);
    }
    verDoPutIn(actor,dobj) = {
        self.verDoTake(actor);
    }

    verDoSwim(actor) = {
        local toploc := toplocation(actor);
        if(isclass(toploc,Outside)) {
            "Surf's up!  Even if you could swim, you wouldn't choose to do
            so today. ";
        }
        else {
            "The calm sea looks very inviting, but unfortunately you
            don't know how to swim. ";
        }
    }
    loclist = [Sandy_Beach Broken_Rocks Thunder_Hole Beach_Shelf Beach]
;
/* 154 */
Broken_Rocks: room, Outside
    game551 = true
    sdesc = "At Broken Rocks"
    ldesc = { I();
    "%You% %are% at a jumble of large broken rocks.  A gentle path leads up
    to the top of the nearby cliffs.  A narrow treacherous path
    disappears among the rocks at the foot of the cliff.";}

    north = {
        if (rand(100) <= 50)
            return Ocean_Vista;
        else
            return Thunder_Hole;
    }
    down = Thunder_Hole
    up = Ocean_Vista
    south = Sandy_Beach
    building = At_End_Of_Road
    listendesc = "You hear the sound of the surf pounding against the
    beach and the broken rocks."

;
/* 155 */
Ocean_Vista: room, Outside
    game551 = true
    sdesc = "At Ocean Vista"
    ldesc = { I();
    "%You% %are% on a high cliff overlooking the sea.  Far below the
    rolling breakers smash into a jumble of large broken rocks.
    The thunder of the surf is deafening.";}

    down = Broken_Rocks
    south = Broken_Rocks
    jump = {return cliff_demise1.death;}
    building = At_End_Of_Road
    listendesc = "You hear the sound of the surf pounding against the
    beach and the broken rocks."

;
/* 156 */
cliff_demise1: object
    ldesc = "%You're% at the bottom of the cliff, smashed to smithereens
    by the pounding surf."
    death = {self.ldesc; die(); return nil;}
;

/* 157 */
Thunder_Hole: room, Outside, Indoor
    game551 = true
    sdesc = "At Thunder Hole"
    ldesc = { I();
    "%You% %are% at Thunder Hole, a funnel shaped cavern opening onto the
    sea.
    The noise of the surf pounding against the outer rocks of the cave is
    amplified by the peculiar shape of the cave, causing a thunder-like
    booming sound to reverberate throughout the cave.  Outside, a narrow
    path leads south towards some large rocks.  The cavern leads in to
    the east.";}

    in = River_Styx_Approach
    east = River_Styx_Approach
    passage = River_Styx_Approach
    out = Broken_Rocks
    south = Broken_Rocks
    up = Broken_Rocks
    building = At_End_Of_Road
    listendesc = "You hear a booming sound, caused by the surf pounding
    against the outer rocks of the cave. "

;
/* 158 */
Top_Of_Steps: room, Outside, Indoor
    game551 = true
    sdesc = "At Top of steps (back of Thunder Hole)"
    ldesc = { I();
    "%You% %are% at the top of some arched steps.  On the east side there
    is
    a blank wall with a tiny door at the base and a shelf overhead.  On
    the other side a westward passage leads to the sea.";}

    west = River_Styx_E
    passage = River_Styx_E
    steps = River_Styx_E
    down = River_Styx_E
    out = Thunder_Hole
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.smaller; 
        actor.roomMoveTravel(&transmove,Ledge_By_Door);
        // also move the objects on the other side of the door
        room_move(Grotto_West,Underground_Sea);
        return nil;
    }
    climb = {"The wall is too smooth to climb."; return nil;}
    up =  {return self.climb;}
    ledge =  {"The shelf is beyond %your% reach."; return nil;}
    in =  {"%You% can't fit through a six-inch door!"; return nil;}
    east =  {return self.in;}
    entrance =  {return self.in;}
    myhints = [Elfindoorhint]
    listendesc = "You hear a booming sound, caused by the surf pounding
    against the outer rocks of the cave. "
    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true
;

Steps: fixeditem, downclimbable, surface
    game551 = true
    sdesc = "steps"
    ldesc = "It's just a normal flight of steps. "
    noun = 'steps'
    adjective = 'arched'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&down);
    }
    location = Top_Of_Steps
    count = 0
    fullcount = 0
    doSearch(actor) = {
        // Check to see if the player has tried to eat the mushroom when 
        // no cakes are available. 
        if(self.cakefind) {
            count++;      // cakes found by actually searching the steps
            fullcount++;  // includes all hidden cakes found on the steps
            "There's nothing on the steps.  ";
            switch(count) {
              case 1:
                "And yet, you have a hunch that you will find a cake if you
                look carefully enough.  So you search each step with great
                care - and find a cake, hidden in a crevice between a step and
                the cave wall!  You take the cake. ";
                break;
              case 2:
                "And yet, you found a cake the last time, so you search the
                steps again.  Once again, your patience is rewarded when you
                notice another cake, hidden in the gap behind a slightly
                loose step!  You take the cake. ";
                break;
              default:
                "And yet, you've spotted a total of <<fullcount>> cakes, 
                all hidden on the steps, so maybe there are more!
                You peer into every nook and cranny, and
                your patience is rewarded again.  Yet another cake, 
                hidden in a gap like the others!  You take the cake. ";
            }
            if (not cakes.knowdrop) {
                cakes.knowdrop := true;
                "Maybe you dropped it when you took the tiny cakes off
                the shelf. ";
            }
            cakes.moveInto(actor);
            self.cakefind := nil;
            P();
            if (itemcnt(self.contents) > 0) pass doSearch;
        }
        else pass doSearch;
    }
;
/* the shelf (when your size is normal */
High_Shelf: fixeditem, distantItem
    game551 = true
    sdesc = "high shelf"
    ldesc = "I can't tell %youm% much about it, because it's so high up."
    noun = 'shelf'
    adjective = 'high'
    location = Top_Of_Steps
    ioThrowAt(actor,dobj) = {
        // large objects can be thrown only if the mushroom has been eaten
        if((dobj.islarge or dobj.ishuge) and not mushroom.is_eaten) {
            caps(); "<<dobj.thedesc>> is too large. ";
        }
        else {
            caps(); "<<dobj.thedesc>> lands on the shelf, out of reach. ";
            dobj.moveInto(Shelf);
        }
    }
;

class Elfindoor: CCR_lockableDoorway
    heredesc = {
        P(); I(); "The door is ";
        if (self.isopen) "open";
        else {
            "closed"; if (self.islocked) " and locked";
            else " but unlocked";
        }
        ". ";
    }
    doorlist = [Small_door_1 Iron_door_1 Small_door_2 Iron_door_2]
;

Small_door_1: Elfindoor
    game551 = true

    noun = 'door' 'lock' 'doorway'
    adjective = 'tiny' 'small'
    sdesc = "tiny door"
    ldesc = {"It's a tiny door, about six inches high.  A cat might be able
    to get through, but %you% can't."; self.heredesc;}


    location = Top_Of_Steps

    doordest = nil
    destination = {"%You% can't fit through a six-inch door!"; return nil;}

    mykey = small_key
    doLookthru (actor) = {
        "%You% peer through the tiny doorway.  %You% see a
        large flooded cavern, lit by a strange bluish glow. ";
        if (Boat.isIn(Grotto_West))
            "There is a small boat on the western shore,
            near the door.  ";
    }
;
Small_door_2: Elfindoor
    game551 = true

    noun = 'door' 'lock' 'doorway'
    adjective = 'tiny' 'small'
    sdesc = "tiny door"

    ldesc = {"It's a tiny door, about six inches high.  A cat might be able
    to get through, but %you% can't.";self.heredesc;}

    location = Grotto_West
    doordest = nil
    destination = {"%You% can't fit through a six-inch door!"; return nil;}

    mykey=small_key
;
Iron_door_1: Elfindoor
    game551 = true

    noun = 'door' 'lock' 'doorway'
    adjective = 'large' 'wrought' 'iron' 'wrought-iron'
    sdesc = "wrought-iron door"

    ldesc = {"It's a large and very substantial door, about eight feet
    high."; self.heredesc;}

    location = Ledge_By_Door
    doordest = Underground_Sea

    mykey=large_key
    doLookthru (actor) = {
        "%You% look through the open doorway.  %You% can see the
        shore of a vast underground sea, lit by a blue glow. ";
        if (Boat.isIn(Grotto_West))
            "A high wooden structure - possibly a ship - extends out
            of the water.  ";
    }
;
Iron_door_2: Elfindoor
    game551 = true

    noun = 'door' 'lock' 'doorway'
    adjective = 'large' 'wrought' 'iron' 'wrought-iron'
    sdesc = "wrought-iron door"

    ldesc = {"It's a large and very substantial door, about eight feet
    high."; self.heredesc;}

    location = Underground_Sea
    doordest = Ledge_By_Door

    mykey=large_key
;
/* 159 */
Cramped_Chamber: room, Outside, Indoor
    game551 = true
    sober = true // no getting out by drinking wine
    sdesc = "In Cramped Chamber"
    ldesc = { I();
    "%You% %are% in a low cramped chamber at the back of a small cave.
    There is a shelf in the rock wall at about the height of ";
    "%your% shoulder.";}

    out = {"%You% %are% now too big to leave the way %you% came in.";
    return nil;}

/* Travel out of here is only achieved by eating the cakes */
    west = {return self.out;}
    steps = {return self.out;}

/* size of brass key as seen in this room */
    brasskey = tiny_key

    phuce = {
        "The shelf appears to rise above your
        head for a few seconds, before returning to the level of your
        shoulder.  It looks as if you'll need to find another way to
        return to your normal size. ";
        return nil;
    }
    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true
;

/* The shelf in the cramped chamber */
Shelf: fixeditem, surface
    game551 = true
    sdesc = "shelf"
    noun = 'shelf'
    adjective = 'rock'
    location = Cramped_Chamber
    ioThrowAt(actor,dobj) = {
        self.ioPutOn(actor,dobj);
    }
;

/* 160 */
Ledge_By_Door: room, Outside, Indoor
    game551 = true
    sober = true // no getting out by drinking
    sdesc = "On Ledge by Wrought-Iron Door"
    ldesc = { I();
    "%You% %are% on a wide ledge, bounded on one side by a rock wall,
    and on the other by a sheer cliff.  The only way past is through
    a large wrought-iron door.";}

    east = Iron_door_1
    in = {return self.east;}
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.larger; 
        actor.roomMoveTravel(&transmove,Top_Of_Steps);
        // also move the objects on the other side of the door
        room_move(Underground_Sea,Grotto_West);
        return nil;
    }

    west =  {"The cliff is unscalable."; return nil;}
    climb =  {return self.west;}
    down =  {return self.west;}
    jump = {return cliff_demise2.death;}
/* size of brass key as seen in this room */
    brasskey = large_key
    // Exit info. for 'back' command:
    exithints = [Underground_Sea, &east]
    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true
;
phuce_messages: object

/* 161 */
    smaller = {
        local actor := getActor(&travelActor);
        local toploc := toplocation(actor);
        "%You% feel dizzy...Everything around %youm% is spinning,
        expanding, growing larger.... ";"Dear me!  ";
        if(toploc.isindoor or not toploc.isoutside)
            "Is the cave bigger ";
        else
            "Has the vegetation grown suddenly, ";
        "or %are% %you% smaller?";
        P();
    }

/* 162 */
    larger = {"%You% %are% again overcome by a sickening vertigo, but
    this time everything around %youm% is shrinking...Shrinking...";P();}

;
eat_messages: object // issued when size-changing foods are eaten
/* 163 */
    smaller =  {
        "%You% %are% closing up like an
        accordian....shrinking..shrinking.  ";"%You% %are% now %your%
        normal size.";P();
    }

/* 164 */
    larger =   {"%You% %are% growing taller, expanding like a telescope!
    Just before %your% head strikes the top of the chamber, the mysterious
    process stops as suddenly as it began.";P();}
;
/* 165 */
cliff_demise2: object
/* This method is used when jumping off the step after using 'phuce'. */

    ldesc = "%You're% at the bottom of the cliff with a broken neck."
    death = {
        self.ldesc; die(); return nil;
    }
;

/* 166 */
Grotto_West: room, NoNPC
    game551 = true
    // no getting out by drinking wine
    sdesc = "At West Wall of Blue Grotto"
    ldesc = {I();
    "%You% %are% at the western tip of the Blue Grotto.  A large lake
    almost
    covers the cavern floor, ";"except for where %you% %are% standing.
    Small holes high in the rock wall to the east admit a dim light.  The
    reflection of the light from the water suffuses the cavern with
    a hazy bluish glow.";}

    in =  {"%You% can't fit through a six-inch door!"; return nil;}
    out = {return self.in;}
    west =  {return self.in;}
/* Directions for use when in the boat */
    boat_east = {if(pole.isIn(Boat))poling_messages.calm;
    return Blue_Grotto_East;}

    boat_cross = {return self.boat_east;}
    boat_over = {return self.boat_east;}
    boat_across = {return self.boat_east;}
    boat_ne = Bubble_Chamber
    boat_south = {if(pole.isIn(Boat))poling_messages.dark;
    return Gravel_Beach;}

    boat_north = {if(pole.isIn(Boat))poling_messages.blue;
    return Dark_Cove;}

    east =  {"%You% can't swim.  %You%'d best go by boat."; return nil;}
    cross =  {return self.east;}
    over =   {return self.east;}
    across =  {return self.east;}
    ne =  {return self.east;}
    south =  {return self.east;}
    north =  {return self.east;}
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.smaller; 
        actor.roomMoveTravel(&transmove,Underground_Sea);
        // also move the objects on the other side of the door
        room_move(Top_Of_Steps,Ledge_By_Door);
        return nil;
    }
    // Exit info. for 'back' command:
    // Note that the exithints must be complete in all rooms which the
    // boat can go to.  This is necessary to ensure that the 'back'
    // command will work correctly while in the boat.
    exithints = [
        Blue_Grotto_East, &east,
        Bubble_Chamber, &ne,
        Gravel_Beach, &south,
        Dark_Cove, &north
    ]
;

/* the lake */

Grotto_Lake: Streamitem, floatingdecoration
    sdesc = "lake"
    ldesc = "It's a large lake, almost covering the floor of the
        chamber.  The reflection of the light from the lake fills the
        room with a bluish glow.  The water is very clear and very deep -
        you won't be able to cross it except by boat."
    noun = 'lake' 'water'
    adjective = 'large' 'clear' 'deep'
    loclist = [Grotto_West Blue_Grotto_East Gravel_Beach Bubble_Chamber
        Dark_Cove]
;
/* The boat */
Boat: fixeditem, vehicle, container
    game551 = true
    noun = 'boat' 'dinghy'
    adjective = 'wooden' 'rowing'
    // statusPrep, outOfPrep now inherited from nestedroom
    isdroploc = nil /* Must put objects in boat explicitly */
    location = Grotto_West
    maxbulk = 10000 /* any objects can be placed in the boat */
    reachable = {
        return self.location.allcontents;
    }
    canReachContents = true
    sdesc = "wooden boat"
    ldesc = {
        "It's a small rowing boat, of traditional construction and
        design. ";
        if(itemcnt(self.contents) > 0) {
            P(); "It contains ";listcont(self);". ";
        }
        else {
            P(); "It's empty. ";
        }
    }
    doSearch(actor) = {
        if(itemcnt (self.contents) > 0) {
            "It contains ";listcont(self);". ";
        }
        else "It's empty. ";
    }

    softfloor = true // allow vase to be put in
    heredesc = {P(); I(); "There is a small wooden boat here.  ";}

    noexit = {"%You% can't go that way by boat."; return nil;}

    verDoTake(actor)={
        if(actor.isIn(Grotto_West)) {
            "The boat is too heavy to carry.  It won't fit through
            the door, and I don't think %you% can use the \"phuce\"
            spell to shrink it. ";
        }
        else {
            "The boat is too heavy to carry, and too bulky for %youm%
            to drag it through the cave passages."; P();
        }
        "If ";"%you% want%s% to cross the water, %you%'ll have to get
        into the boat.";
    }
    verDoMove(actor)={self.verDoTake(actor);}

    verDoPush(actor)={self.verDoTake(actor);}
    verDoPull(actor)={self.verDoTake(actor);}

    genMoveDir = {self.verDoMove(parserGetMe());}

    pole_check = { 
        local actor := getActor(&verbActor);

        if (not actor.isIn(Boat)){
            "%You%'d better get into the boat first!"; return nil;
        }
        if (pole.isIn(self)) return true;
        else {
            "Casting yourself adrift without a paddle is a bad idea.
            The boat's oars were stolen by the dwarves to play
            bing-bong. (That's dwarvish ping-pong -- with rocks!). ";
            "%You%'d better bring something else to propel the boat.";
            return nil;
        }
    }
    poling_check = {
        local actor := getActor(&verbActor);

        if (not pole.isVisible(actor)){
            "I can't see a pole here!"; return nil;
        }
        if (not actor.isIn(Boat)){
            "%You%'d better get into the boat first."; return nil;
        }
        if (pole.isIn(self)) return true;
        else {
            "%You% can't, because %you% %do%n't have a pole with %youm%!";
            return nil;
        }
    }

    boatmove(prop) = {
        local actor := getActor(&travelActor);
        local dest;
        if (self.pole_check){
            dest := self.moveme(prop);
            if (dest) return dest;
            else {
                "%You% can't go that way by boat."; 
                return nil;
            }
        }
        else return nil;
    }
    north = {return self.boatmove(&boat_north);}
    south = {return self.boatmove(&boat_south);}
    east = {return self.boatmove(&boat_east);}
    west = {return self.boatmove(&boat_west);}
    ne = {return self.boatmove(&boat_ne);}
    nw = {return self.boatmove(&boat_nw);}
    se = {return self.boatmove(&boat_se);}
    sw = {return self.boatmove(&boat_sw);}
    cross = {return self.boatmove(&boat_cross);}
    over = {return self.boatmove(&boat_over);}
    across = {return self.boatmove(&boat_across);}

    verDoLaunch(actor) = {}
    verDoRide(actor) = {}
    doLaunch(actor) = {
        if (self.pole_check)
            "Just tell me which direction you want to go in!";
    }
    doRide(actor) = {self.doLaunch(actor);}
    verDoPole(actor) = {}
    doPole(actor) = {
        if (self.poling_check)
            "Just tell me which direction you want to go in!";
    }
    verDoPoleN(actor) = {}
    doPoleN(actor) = {if (self.poling_check){
        actor.travelTo(self,&north);}}
    verDoPoleS(actor) = {}
    doPoleS(actor) = {if (self.poling_check){
        actor.travelTo(self,&south);}}
    verDoPoleE(actor) = {}
    doPoleE(actor) = {if (self.poling_check){
        actor.travelTo(self,&east);}}
    verDoPoleW(actor) = {}
    doPoleW(actor) = {if (self.poling_check){
        actor.travelTo(self,&west);}}
    verDoPoleNW(actor) = {}
    doPoleNW(actor) = {if (self.poling_check){
        actor.travelTo(self,&nw);}}
    verDoPoleNE(actor) = {}
    doPoleNE(actor) = {if (self.poling_check){
        actor.travelTo(self,&ne);}}
    verDoPoleSW(actor) = {}
    doPoleSW(actor) = {if (self.poling_check){
        actor.travelTo(self,&sw);}}
    verDoPoleSE(actor) = {}
    doPoleSE(actor) = {if (self.poling_check){
        actor.travelTo(self,&se);}}

    verDoRow(actor) = {}
    doRow(actor) = {
        actor.travelTo(self,&row);
    }
    // use the exithints list for the containing room
    exithints = {return self.location.exithints;}
;
/* Dummy object */
Oars: item
    sdesc = "oars"
    noun = 'oar' 'oars'
;

/* 167 */

/* The original spoke of a 'wooden structure', but it seems more likely that
the adventurer would recognize it as a large ship. */

Underground_Sea: room, NoNPC
    game551 = true
    sober = true // no getting out by drinking wine
    sdesc = "At Underground Sea"
    ldesc = {
        I();
        "%You% %are% on the shore of an underground sea.  The way west is
        through a wrought-iron door.";
    }
    phuce = {
        local actor := getActor(&travelActor);
        phuce_messages.larger; 
        actor.roomMoveTravel(&transmove,Grotto_West);
        // also move the objects on the other side of the door
        room_move(Ledge_By_Door,Top_Of_Steps);
        return nil;
    }
    west = Iron_door_2
    out = {return self.west;}
    in = {return self.west;}

    east =  {
        if (Boat.isIn(Grotto_West))
            "%You% can't swim, and there's no way into
            the ship, so %you% couldn't possibly cross this sea.";
        else
            "%You% couldn't possibly cross this sea without a large ship.";
        return nil;
    }

    cross =  {return self.east;}
    over =  {return self.east;}
    across =  {return self.east;}
/* size of brass key as seen in this room */
    brasskey = large_key
    // Exit info. for 'back' command:
    exithints = [Ledge_By_Door, &west]

;
ship: fixeditem, upclimbable
    game551 = true
    sdesc = "vast wooden ship"
    noun = 'ship'
    adjective = 'vast' 'wooden' 'huge' 'high'
    ldesc = "Despite its vast scale, it is constructed like an
    old-fashioned wooden rowing boat."
    heredesc = {
        P();I(); "A high wooden ship of vast
        proportions extends out of the water to the east.  There
        doesn't appear to be any way into the ship.";
    }
    locationOK = true
    location = {
        if (Boat.isIn(Grotto_West)) return Underground_Sea;
        else return nil;
    }

    verDoEnter(actor) = {"%You% can't see any way to enter the ship.";}
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoClimb(actor) = {"Even with rock-climbing equipment,
    %you%'d have great difficulty climbing the sheer hull of the ship.";}

;
/* 168 */
Blue_Grotto_East: room
    game551 = true
    sdesc = "At East Side of Blue Grotto"
    ldesc = {I();
    "%You% %are% on the eastern shore of the Blue Grotto.  To the west
    a large lake almost fills the cavern floor, and an ascending
    tunnel disappears into the darkness to the SE.";}

    se = Windy_Tunnel
    up = Windy_Tunnel
    passage = Windy_Tunnel
    boat_north = Bubble_Chamber
    boat_south = Gravel_Beach
    boat_west = Grotto_West
    boat_cross = Grotto_West
    boat_over = Grotto_West
    boat_across = Grotto_West
    west =  {"%You% can't swim.  %You%'d best go by boat."; return nil;}
    north =  {return self.west;}
    south =  {return self.west;}
    // Exit info. for 'back' command:
    exithints = [
        Bubble_Chamber, &north,
        Gravel_Beach, &south,
        Grotto_West, &west,
        Windy_Tunnel, &se
    ]
;
/* 169 */
Bubble_Chamber: room, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    game551 = true
    sdesc = "In Bubble Chamber"
    ldesc = {I();
    "%You% %are% at a high rock on the NE side of a watery chamber at the
    mouth
    of a small brook. An unknown gas bubbles up through the water from
    the chamber floor. ";
    if (not Grotto_West.isseen) {
        "To the southwest lies the Blue Grotto, a large chamber lit
        by a bluish light.  A lake almost completely covers the
        floor. ";
    }
    else {
        "A bluish light can be seen to the southwest. ";
        }
    }
    boat_south = Blue_Grotto_East
    boat_nw = Dark_Cove // Added - DJP
    boat_sw = Grotto_West
    south =  {"%You% can't swim.  %You%'d best go by boat."; return nil;}
    sw = {return self.south;}
    nw = {return self.south;}
    passage = Muddy_Defile
    east = Muddy_Defile
    up = Muddy_Defile
    stream = Muddy_Defile // Added - DJP
    upstream = Muddy_Defile // Added - DJP
    // Exit info. for 'back' command:
    exithints = [
        Blue_Grotto_East, &south,
        Grotto_West, &sw,
        Dark_Cove, &nw,
        Muddy_Defile, &east
    ]
;
/* 170 */
Windy_Tunnel: darkroom
    game551 = true
    sdesc = "In Windy Tunnel"
    ldesc = {I();
    "%You% %are% in a windy E/W tunnel between two large rooms.";}

    east = Bat_Cave
    up = Bat_Cave
    west = Blue_Grotto_East
    down = Blue_Grotto_East

;
/* 171 */
Bat_Cave: darkroom
    game551 = true
    sdesc = "In Bat Cave"
    ldesc = {I();
    "%You% %are% in the Bat Cave.  The walls and ceiling are covered with
    sleeping bats.  ";
    if (Guano.swept) "A mass of dry, foul-smelling guano has been swept
    to one side of the room. ";
    else "The floor is buried by a mass of dry, foul-smelling
    guano.  ";
    "The stench is overpowering. Exits to the NW and east.";}
    hasfloordesc = true
    floordesc = "The floor is covered in dried guano.  The smell is
    indescribable!  Let's get out of here."
    nw = Windy_Tunnel
    down = Windy_Tunnel
    east = Tongue_of_Rock
    up = Tongue_of_Rock

;
Guano: fixeditem
    game551 = true
    sdesc = "guano"
    ldesc = {
        if(self.swept)
            "It's piled up against one side of the room";
        else
            "It covers the floor";
        ", and smells absolutely terrible.  I
        suggest that we make a move out of here as soon as possible.";
    }
    noun = 'guano'
    location = Bat_Cave
    verDoSmell(actor) = {}
    doSmell(actor) = {
        "The stench is overpowering. Let's get out of here.";
    }
    verDoTake(actor) = {
        "I'd rather not, thank you.";
    }
    verDoPutIn(actor,io) = {
        "I'd prefer to leave the stinking guano where it is.";
    }
    verDoLookin(actor) = {
        "There's nothing in the guano, and it stinks like crazy.  Can we
        move on, please?";
    }
    verDoLookunder(actor) = {
        "There's nothing under the guano except the floor of the
        cave.  Please get me out of here - I can't stand the smell.";
    }
    verDoSearch(actor) = {
        "There's nothing hidden in the guano!  Can we go now,
        please - the smell is making me feel unwell.";
    }
    sweepmess = {
        local actor := getActor(&verbActor);

        if (self.swept) {
            if (self.swept2) {
                "If you think there's anything useful to do in this
                room, you're barking up the wrong tree.  So I'll move
                you on! ";
            }
            else {
                "Ignoring my advice, you attempt to sweep the guano once
                again.  However, you are overwhelmed by a feeling of
                nausea, and wisely decide to move on. ";
                self.swept2 := true;
            }
            P();
            if(actor.prevloc = Windy_Tunnel) actor.travelTo(Tongue_of_Rock);
            else actor.travelTo(Windy_Tunnel);
        }
        else
            "In the hope of finding a hidden treasure, you sweep the
            guano to one side of the room, revealing nothing.  Vowing not
            to be beaten, you then sweep it all to the other side - again
            revealing nothing.  We now know for certain that nothing was
            hidden here, and the smell is now absolutely nauseating.  So
            please, PLEASE can we move on.";
            self.swept := true;
    }
    doCleanWith( actor, io ) =
    {
        if(io = whiskbroom) {
            self.sweepmess;
        }
        else pass doCleanWith;
    }
    doSweepWith( actor, io ) =
    {
        if(io = whiskbroom) {
            self.sweepmess;
        }
        else pass doSweepWith;
    }
;
Bats: fixeditem
    game551 = true
    sdesc = "bats"
    ldesc = "The bats are sleeping, and cover the walls and ceiling.  I
        suggest that we leave them alone and move on.  It stinks in
        here! "
    noun = 'bats'
    verDoClean(actor) = {"The bats don't need cleaning.  I suggest that
        you leave them alone.";}
    verDoSweep(actor) = {"Leave the bats alone, please!";}
    verDoCleanWith(actor,io) = {self.verDoClean(actor);}
    verDoSweepWith(actor,io) = {self.verDoSweep(actor);}
    location = Bat_Cave
    verDoWake(actor) = {"Leave the bats alone, please!";}
;

/* 172, 173 */
Tight_Crack: darkroom
    game551 = true
    sdesc = "In Tight N/S Crack"
    ldesc = {
        I(); "%You% %are% in a very tight N/S crack. ";
        if (Cloakroom.caved)
            "The passage south is blocked by a recent cave-in.";
        else
            "The passage seems to widen to the south.";
    }

    north = At_East_End_Of_Long_Hall
    passage = {return self.south;}
    south = {
        if (Cloakroom.caved) {
            "The passage south is blocked by a recent cave-in.";
            return nil;
        }
        else return Cloakroom;
    }

    exithints = [Cloakroom, &south]
    NPCexit1 = {if(Cloakroom.caved)return nil; else return Cloakroom;}
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\" as your entire body collides with a solid wall of 
        concrete.  Fortunately the spell reverses itself, and you find 
        yourself back in the crack. ";
        return nil;
    }

;
InCrack: floatingdecoration
    sdesc = "crack"
    ldesc = "I've already told you all I know about it. "
    loclist = [Tight_Crack Tight_Crack_2 Dead_End_Crack]
    noun = 'crack'
;
crackrocks: floatingdecoration, surface
    game551 = true
    sdesc = "rocks"
    adesc = "rocks"
    ldesc = "The rocks are too large to move or carry, and they are
    now blocking the southern end of the passage.  "
    noun = 'rocks'
    adjective = 'large' 'loose'

    loclist = [Green_Tight_Crack]

    verDoMove(actor) = {"The rocks are far too large to move.";}
    verDoTake(actor) = {"The rocks are too large to carry.";}
    verDoSearch(actor) = {}
    doSearch(actor) = {"You find nothing of interest. ";}
;
/* 174 */
Cloakroom: darkroom
    game551 = true
    sdesc = "In Cloakroom"
    ldesc = {
        I();
        "%You're% in the Cloakroom.  This is where the dreaded Wumpus
        repairs to sleep off heavy meals.  (Adventurers are his favorite
        dinner!)  Two very narrow passages exit NW and NE.  ";
        if (self.caved)" Unfortunately, the NE passage is now blocked by a
        rockslide.  ";
    }

    passage = {
        if (self.caved)
            return self.nw;
        else {
            "There's more than one passage - please tell me which
            direction you want to go. ";
            return nil;
        }
    }
    ne = {if(self.caved){ne_blocked.message; return nil;}
          else return Tight_Crack;}
    crack = {return self.ne;}
    nw = Cloak_Pits
    cave = { // used when cloak is taken
        local actor := getActor(&travelActor), rocklist := crackrocks.loclist;
        P(); I();
        "%You% have jerked the cloak free of the rocks.  However, in doing
        so %you% have caused a small rockslide, blocking the entrance
        and making an unholy din.";
        self.caved := true;
        rocklist += Tight_Crack;
        crackrocks.moveLoclist(rocklist);
        Tight_Crack.isseen := nil; // full description when next seen
        if (Wumpus.isasleep and Wumpus.isIn(self)) 
             {P(); I(); Wumpus.doWake(actor);}
    }

    NPCexit1 = {if(self.caved)return nil; else return Tight_Crack;}
    // Exit info. for 'back' command:
    exithints = [Tight_Crack, &ne]
    myhints = [Cloakhint]
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\".  For a short moment you are surrounded by junk of all
        kinds - old display boards, broken filing cabinets and worn-out
        office machinery.   The pendant then transports you back to 
        Red level. ";
        return nil;
    }
;

cloakrocks: fixeditem, surface
    game551 = true
    sdesc = "rocks"
    adesc = "rocks"
    ldesc = {
        "The rocks are too large to move or carry.  ";
        if(Cloakroom.caved) {
            "Unfortunately, they are now blocking the passage to the
            northeast.  ";
        }
    }

    noun = 'rocks'
    adjective = 'large' 'loose'

    location = Cloakroom

    verDoMove(actor) = {"The rocks are far too large to move.";}
    verDoTake(actor) = {"The rocks are too large to carry.";}
    verDoSearch(actor) = {}
    doSearch(actor) = {"You find nothing of interest. ";}
;
/* 175 */
Cloak_Pits: darkroom
    game551 = true
    sdesc = "In Room with Small Pits"
    ldesc = {I();
    "%You're% in a damp room containing several small climbable pits.
    Passages exit to the east and north.  On the south wall you see the
    remains of an iron ladder which once led upwards, but it is now badly
    corroded and unclimbable. ";
    }
    passage = {
        "There's more than one passage - please tell me which
        direction you want to go. ";
        return nil;
    }
    north = At_High_Hole
    east = Cloakroom
    pitlist = [Cloak_Pit_1, Cloak_Pit_2, Cloak_Pit_3]
    south = Cloak_Pit_1
    ne = Cloak_Pit_2
    nw = Cloak_Pit_3
    pit = {
        local actor := getActor(&travelActor);
        // force the route to be 0 when going down into the pit
        actor.nextRoute := 0;
        // choose one of the three pits at random
        "\n(choosing one of the pits at random)\n";
        return self.pitlist[rand(3)];
    }
    up = {
        "You attempt to climb up, but the remains of the ladder are too
        fragile to bear your weight.  You give up the attempt. ";
        return nil;
    }
    down = {return self.pit;}
    climb = {return self.pit;}
    // Exit info. for 'back' command:
    exithints = [Cloak_Pit_1, &south, Cloak_Pit_2, &ne, Cloak_Pit_3, &nw,
    On_Ladder, &up]
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\" as your body collides with solid concrete.  For a
        brief moment you see that the floor at Blue level has been raised,
        and is almost at the level of your shoulders.  Then the pendant
        returns you to Red level. ";
        return nil;
    }
;
featurelessPits: fixeditem, downclimbable
    sdesc = "featureless pits"
    adesc = {self.sdesc;}
    ldesc = "There are three pits, to the south, northeast and northwest.
    To find out more, I suggest that you enter them. "
    noun = 'pits'
    adjective = 'featureless' 'small'
    location = Cloak_Pits
    verDoClimb(actor) = {
        "You'll have to tell me which pit you want to enter.  For example,
        to enter the south pit say ENTER SOUTH PIT or just SOUTH. ";
    }
    verDoEnter(actor) = {
        self.verDoClimb(actor);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
;

class Featurelesspit: fixeditem, downclimbable
    ldesc = "To find out more, I suggest that you enter <<self.thedesc>>."
    verDoEnter(actor) = {}
    verDoClimb(actor) = {}
    doEnter(actor) = {
        actor.travelTo(self,&mypit);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doClimb(actor) = {self.doEnter(actor);}
    location = Cloak_Pits
;
Pit_1: Featurelesspit
    sdesc = "south pit"
    noun = 'pit'
    adjective = 'south' 'southern' 's'
    mypit = Cloak_Pit_1
;
Pit_2: Featurelesspit
    sdesc = "northeast pit"
    noun =  'pit'
    adjective = 'northeast' 'northeastern' 'ne'
                'north-east' 'north-eastern'
    mypit = Cloak_Pit_2
;
Pit_3: Featurelesspit
    sdesc = "northwest pit"
    noun = 'pit'
    adjective = 'northwest' 'northwestern' 'nw'
                'north-west' 'north-western'

    mypit = Cloak_Pit_3
;
Inpit: floatingdecoration, upclimbable
    ldesc = {
        local actor := getActor(&verbActor);
        local toploc := toplocation(actor); 
        if (toploc) toploc.ldesc;
    }
    noun = 'pit'
    adjective = 'featureless' 'small'
    loclist = [Cloak_Pit_1, Cloak_Pit_2, Cloak_Pit_3]

    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&up);
    }
    verDoEnter(actor) = {
        "You're already in the pit!";
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&up);
    }
;


Ladder_1: fixeditem, upclimbable
    game551 = true
    sdesc = "remains of the ladder"
    adesc = "remains of the ladder"
    ldesc = "Years of corrosion in this damp room have reduced the ladder
    to a few broken pieces of heavily rusted metal.  They certainly
    won't bear your weight. "
    noun = 'ladder' 'remains'
    adjective = 'iron' 'rusted' 'corroded' 'rusty' 'fragile'
    location = Cloak_Pits
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&up);
    }
;

/* 176 */

// From version 2.00 there are three pits.  Two of them contain
// a ring, but the third is empty because the Wumpus has stolen the gold
// ring.  A clue to the significance of the rings is in the Octagonal Room.
Cloak_Pit_1: darkroom, NoNPC
    game551 = true
    sdesc = "In Featureless Pit"
    ldesc = {I();
    "%You% %are% at the bottom of a small pit, which is featureless
    except for large footprints which suggest that
    someone - or something - has been here before you. ";
    }
    // force the route to be always 1 when coming out of the pit, to
    // avoid the backtracking check.
    up = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; return Cloak_Pits;
    }
    climb = {return self.up;}
    out = {return self.up;}
    // Exit info. for 'back' command:
    exithints = [Cloak_Pits, &up]
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\" as your entire body collides with solid concrete.  The
        pendant then returns you to Red level. ";
        return nil;
    }

;
Footprints: fixeditem
    game551 = true
    sdesc = "large footprints"
    ldesc = "They look too large to have been made by a human being. "
    noun = 'footprints'
    adjective = 'large'
    location = Cloak_Pit_1
;
Cloak_Pit_2: darkroom, NoNPC
    game551 = true
    sdesc = "In Very Featureless Pit"
    ldesc = {I();
    "%You% %are% at the bottom of a small, totally featureless pit.";}
    // force the route to be always 1 when coming out of the pit, to
    // avoid the backtracking check.
    up = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; 
        return Cloak_Pits;
    }
    climb = {return self.up;}
    out = {return self.up;}
    // Exit info. for 'back' command:
    exithints = [Cloak_Pits, &up]
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\" as your entire body collides with solid concrete.  The
        pendant then returns you to Red level. ";
        return nil;
    }
;
Cloak_Pit_3: darkroom, NoNPC
    game551 = true
    sdesc = "In Fairly Small Pit"
    ldesc = {I();
    "%You% %are% at the bottom of a fairly small, featureless pit.";}
    // force the route to be always 1 when coming out of the pit, to
    // avoid the backtracking check.
    up = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; 
        return Cloak_Pits;
    }
    climb = {return self.up;}
    out = {return self.up;}
    // Exit info. for 'back' command:
    exithints = [Cloak_Pits, &up]
    ana2 = {
        "You see a brief flash of blue light, then feel a sickening 
        \"crunch\" as your entire body collides with solid concrete.  The
        pendant then returns you to Red level. ";
        return nil;
    }
;
/* 177 */
At_High_Hole: darkroom
    game551 = true
    sdesc = "At High Hole"
    ldesc = {I();
    "%You% %are% at a high hole in a rock wall.";}

    down = At_East_End_Of_Long_Hall
    climb = At_East_End_Of_Long_Hall
    jump = At_East_End_Of_Long_Hall
    south = Cloak_Pits
    ana2 = {
        "You see a brief flash of blue light, and for a short moment you
        seem to be inside a water tank!  More familiar surroundings reappear
        moments later, and you are relieved to find that you and your
        possessions are still dry.  Evidently the pendant is able to protect
        the wearer in situations like this. ";
        return nil;
    }
;

/* 178 */
ne_blocked: object
    message = {"The NE passage is blocked by a recent cave-in.";P();}
;

/* 179 */
Muddy_Defile: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    game551 = true
    sdesc = "In Muddy Defile"
    ldesc = {I();
    "%You% %are% in a sloping muddy defile, next to a tumbling brook.";}

    down = Bubble_Chamber
    downstream = Bubble_Chamber  // Added - DJP
    west = Bubble_Chamber
    up = Fairy_Grotto
    upstream = Fairy_Grotto // Added - DJP
    east = Fairy_Grotto
    stream = Fairy_Grotto

;
/* 180 */
Tongue_of_Rock: darkroom
    game551 = true
    sdesc = "At Tongue of Rock"
    ldesc = {I();
    "%You% %are% in a level E/W passage partially blocked by an overhanging
    tongue of rock.  A steep scramble would take you up over the tongue,
    whence continues an upward crawl.";}

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    west = Bat_Cave
    up = Upper_Passage
    climb = Upper_Passage
    east = Passage_End_at_Hole

;
/* 181 */
dog_message: object
    message = "The dog won't let you pass."
;
/* 182 */
Upper_Passage: darkroom
    game551 = true
    sdesc = "In Upper Passage"
    ldesc = {I();
    "%You%'re in the Upper Passage, a long level E/W tunnel.";}

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    west = Tongue_of_Rock
    east = Star_Chamber

;
/* 183 */
Star_Chamber: darkroom
    game551 = true
    sdesc = "In Star Chamber "
    ldesc = {I();
    "%You% %are% in a star-shaped chamber.  Passages exit north, east,
    south and west.";}

    passage = {
        "There's more than one passage - please tell me which
        direction you want to go. ";
        return nil;
    }

    west = Upper_Passage
    east = {
        if (rand(100) <= 50)
            return Elbow_in_Passage;
        else
            return Tunnel_Intersection;
    }
    south = Dead_End_15
    north = Narrow_E_W_Passage

    NPCexit1 = Elbow_in_Passage
    NPCexit2 = Tunnel_Intersection

    // Exit info. for 'back' command:
    exithints = [
        Elbow_in_Passage, &east,
        Tunnel_Intersection, &east
    ]

;
/* 184 */
Elbow_in_Passage: darkroom
    game551 = true
    sdesc = "At Elbow in Passage"
    ldesc = {I();
    "%You% %are% at an elbow in a winding E/W passage.  %You% can go SW or
    SE.";}

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }

    sw = Star_Chamber
    se = Rotunda

;
/* 185 */
Dead_End_15: room, CCR_dead_end_room
    game551 = true

    north = Star_Chamber

;

/* 186 */
Tunnel_Intersection: darkroom
    game551 = true
    sdesc = "At Tunnel Intersection"
    ldesc = {I();
    "%You're% at the intersection of two long tunnels.  One goes NW,
    the other NE.";}

    passage = {
        "The tunnel goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    nw = Star_Chamber
    ne = Rotunda

;
/* 187 */
Narrow_E_W_Passage: darkroom
    game551 = true
    sdesc = "In Narrow East-West Passage"
    ldesc = {I();
    "%You're% in a long narrow east-west passage which curves out of sight
    at both ends.";}

    passage = {
        "There's more than one passage - please tell me which
        direction you want to go. ";
        return nil;
    }

    west = Star_Chamber
    east = Rotunda

;
/* 188 */
Rotunda: darkroom
    game551 = true
    sdesc = "In Rotunda"
    ldesc = {
        I();
        "%You're% in the Rotunda.  Corridors radiate in all directions.";
        P();I();
        "There is a telephone booth standing against the north wall.";
    }

    hasfloor = true // this seems to be a room in the normal sense rather
                    // than a natural chamber.
    passage = {
        "There's more than one passage - please tell me which
        direction you want to go. ";
        return nil;
    }
    north = Narrow_E_W_Passage
    west = Elbow_in_Passage
    sw = {
        if (rand(100)<= 65) return Devils_Chair;
        else return Tunnel_Intersection;
    }
    down = {return self.sw;}
    in =  {
        local actor := getActor(&travelActor);
        Phone_Booth1.doEnter(actor); return nil;
    }
    leaveRoom (actor) = {
        if (not self.noleaveroom) {
            // leave things as we expect to find them when we re-enter
            // the room.
            // Remove the gnome
            Gnome.moveInto(nil);
            // The booth door will be closed
            Phone_Booth1.mydoor.isopen := nil;
            // If the phone is unbroken, it will be ringing and
            // the gnome may barge in.
            if (not Phone_Booth1.myphone.isbroken) {
                Gnome.isntcoming := nil;
                Phone_Booth1.myphone.isringing := true;
            }
            // If the phone is broken, the gnome won't be coming.
            else Gnome.isntcoming := true;
        }
        self.noleaveRoom := nil;
        pass leaveRoom;
    }


    NPCexit1 = Devils_Chair
    NPCexit2 = Tunnel_Intersection

    // Exit info. for 'back' command:
    exithints = [
        Devils_Chair, &sw,
        Tunnel_Intersection, &sw,
        Phone_Booth1, &in
    ]
;
Gnome: fixeditem
    game551 = true
    // The gnome doesn't need much code because he's always unreachable!
    sdesc = "gnome"
    ldesc = "The large gnome is occupying the phone booth, firmly
            blocking the door.  He is talking excitedly to someone at the
            other end of the phone. "
    noun = 'gnome'
    adjective = 'large'
;

/* 189 */
class CCR_boothitem: Internal_Room, NoNPC
    // telephone booth

    // this is really a kind of nested room, but doesn't have class
    // nestedroom and is treated as a top-level room by the toplocation
    // function.

    // requires the following properties:
    // inldesc  - room description (when seen from inside)
    // outldesc - object description (when seen from outside)
    // mydoor   - the door
    // myphone  - the telephone
    // location - where it is
    istoproom = true // regarded as top-level room by toplocation function
    contentsVisible = true  // assume door is transparent
    reachable = ( [] + self + self.mydoor )
    isopen = (not Gnome.isIn(self)) // closed only when gnome is inside
    isqcontainer = true        // don't report contents when outside
    multisdesc = "phone booth" // Used for actions on multiple objects
    sdesc = "In Phone Booth"
    ldesc = {
        local actor := getActor(&verbActor);
        if(Gnome.isIn(self))
            "At present, it is occupied by a gnome who is talking excitedly
            to someone at the other end of the line.";
        else if(actor.isIn(self)) {
            I(); self.inldesc;
        }
        else
            self.outldesc;
    }
    heredesc = {
        P(); I();
        if(Gnome.isIn(self))
            "The phone booth is occupied by a gnome.  He is talking
            excitedly to someone at the other end.";
        else if (self.myphone.isringing)
            "The telephone booth is unoccupied.  The phone is ringing.";
        else "The telephone booth is unoccupied.";
    }
    islit = {
        if ( self.location ) return( self.location.islit );
        return( nil );
    }
    out = {
        return self.mydoor;

    }
    leaveroom = {
        local actor := getActor(&travelActor);
        if(self.myphone.isoffhook) {
            "\n(Hanging up the phone first)\n";
            self.myphone.dropmethod(actor);
        }
    }
    leaveRoom(actor) = {
        self.leaveroom;
        pass leaveRoom;
    }
    verIoPutIn(actor) = {
        if (Gnome.isIn(self)) {
            "You can't do that while the gnome is blocking the door!";
        }
    }
    ioPutIn( actor, dobj ) = {
        self.gnomecheck; if (Gnome.isIn(self)) return;
        pass ioPutIn;
    }

    verDoOpen(actor) = {
        self.mydoor.verDoOpen(actor);
    }
    doOpen(actor) = {
        self.mydoor.doOpen(actor);
    }
    verDoClose(actor) = {
        self.mydoor.verDoClose(actor);
    }
    doClose(actor) = {
        self.mydoor.doClose(actor);
    }
    verDoLookin(actor) = {}
    verDoSearch(actor) = {}
    doSearch(actor) = {
        if(Gnome.isIn(self)) "All you see is the gnome, chatting
        excitedly on the phone.  To properly examine the interior of
        the booth, you'll need to come back later after the gnome
        has left. ";
        else {
            "%You% see%s% "; listlist(self.contents,[self.myphone]); " inside 
            the booth. ";
        }
    }
    doLookin(actor) = {self.doSearch(actor);}
    verDoEnter(actor) = {
        if(actor.isIn(self)) {
            "%You% are already in ";self.thedesc;". ";
        }
    }
    doEnter(actor) = {
        if(Gnome.isIn(self)) {
            "The gnome firmly blocks the door of the booth.
            %You% can't enter. ";
            return;
        }
        self.gnomecheck;
        if(not Gnome.isIn(self)){
            self.location.noleaveroom := true; // avoid leaveRoom actions
            actor.travelTo(self,&mydoor);
            self.location.noleaveroom := nil;
        }
    }
    verDoUnboard(actor) = {
        if(not actor.isIn(self)) {
            "%You% %are%n't in "; self.thedesc; ". ";
        }
    }
    doUnboard(actor) = {
        actor.travelTo (self.out);
    }
    verDoTouch(actor) = {}
    doTouch(actor) = {
        self.gnomecheck;
        P(); "Touching "; self.thedesc; " doesn't seem to have any
        effect. ";
    }
    verDoRub(actor) = {}
    doRub(actor) = {
        self.gnomecheck;
        P(); "Ok, you just rubbed "; self.thedesc; ".";
    }
    gnomecheck = {} // check for gnome barging in
    verDoMove(actor) = {
        // reference to the 'stationary' (sic) cupboard in
        // frustration.gam
        "I believe you have another game in mind. ";
    }
;

Phone_Booth1: darkroom, CCR_boothitem
    game551 = true
    noun = 'booth'
    adjective = 'phone' 'telephone'
    inldesc = "%You% %are% standing in a telephone booth at the side of a
        large chamber.  Hung on the wall is a banged-up pay telephone of
        ancient design."
    outldesc = "It contains a banged-up pay telephone of ancient design."
    location = Rotunda
    hasfloordesc = true
    floordesc = "You see nothing unusual about the floor of this booth."
    notakeallfrom = true // TAKE ALL from the containing room will ignore
                         // the booth
    mydoor = boothdoor1
    myphone = Phone1
    gnomecount = 0
    gnomecheck = {
       if(Gnome.isIn(self)) {
            "You can't, because the gnome is inside the booth and firmly
            blocks the door.";
            return;
       }
       if(not Gnome.isntcoming and not global.closed and not self.myphone.isbroken and
       (self.gnomecount = 0 or rand(100) <= 55)) {
            self.gnomecount++ ;
            "As you move towards the phone booth, a gnome suddenly streaks
            around the corner, jumps into the booth and rudely slams the
            door in your face.  You can't get in.";
            Gnome.moveInto(self);
            self.myphone.isringing := nil;
            self.mydoor.isopen := nil;
       }
       else Gnome.isntcoming := true;
    }
;
class CCR_boothdoor: doorway // door to a booth
    // The following property should be defined
    // mybooth - the booth to which this door belongs
    // location - same as self.mybooth

    sdesc = "booth door"

    doordest = {
        local actor := getActor(&travelActor);
        if (actor.isIn(self.mybooth))
            return self.mybooth.location;
        else return self.mybooth;
    }

    isopen = nil

    otherside = nil

    doOpen(actor) = {
       self.mybooth.gnomecheck;
       if(Gnome.isIn(self.mybooth))return;
       pass doOpen;
    }
;

boothdoor1: CCR_boothdoor  // door to booth
    game551 = true
    noun = 'door'
    adjective = 'booth'
    mybooth = Phone_Booth1
    location = Rotunda
;

class CCR_phoneitem: fixeditem
    // a telephone.  The following properties are set:
    // enddesc: description of what phone is doing for use in
    // ldesc
    // isringing:  Whether it is ringing.
    // isoffhook:  Whether the receiver has been lifted.
    // isanswering: (When isoffhook is set) we are answering the phone
    // isbroken (when phone is out of order)
    // isdented (when phone has been hit)

    // It is recommended that the following properties be customized
    // as required:
    // mybooth - if non-nil, this is set to the room which the player
    //           must be in to interact with the phone.  By default, it
    //           is set to the location of the phone.
    // ldesc - long description, which may include self.enddesc
    // location - where it is
    // needtopay - true if coins must be inserted before dialing
    // doTake (actor) - all actions when receiver is lifted
    // takemethod (actor) - basic variable settings for when the
    // receiver is lifted
    // dropmethod (actor) - actions when the receiver is replaced.
    // answermethod(actor) - what to do when the phone is answered
    // dialtonemethod(actor) - message to be issued when a dial tone is
    //                         received.
    // brokenmethod - message when the phone is broken
    // dialmethod(actor, value) - what to do when the phone is dialled
    // with value 'value' (value = -1 if no number was specified)

    ldesc = {
        local actor := getActor(&verbActor);
        if (actor.isIn(self.location)) {
            "It's an old battered payphone. ";
            self.enddesc;
        }
        else
            "You can't examine the phone more closely until you
            enter the booth.";
    }
    heredesc = {self.enddesc;}
    enddesc = {
        if(self.isoffhook) {
            P(); "You have lifted the receiver.  ";
        }
        if(self.isringing) {
            P(); "The phone is ringing. ";
        }
        else if(self.isbroken) {
            P(); "The phone is out of order";
            if(self.isdented)
                " and is badly dented";
            ". ";
        }
    }
    mybooth = (self.location)
    isringing = nil
    isoffhook = nil
    isanswering = nil
    isbroken = nil // prevents dialing but not return of coins - set
                   // brokenretain to prevent return of coins.
    brokenretain = nil // phone will fail to return coins when broken
    isdented = nil // phone has been attacked - isbroken will
                   // also be set
    ispaid = nil   // coins have been inserted
    needtopay = true // need to insert coins before dialling
    verDoHangup(actor) = {self.verDoDrop (actor);}
    doHangup (actor) = {self.doDrop (actor);}
    verDoReplace(actor) = {self.verDoDrop (actor);}
    doReplace(actor) = {self.doDrop (actor);}
    verDoTake(actor) = {
        if (self.isoffhook) {
            caps(); self.thedesc; " is already off the hook. ";
        }
    }
    verDoAnswer(actor) = {
        if (not self.isringing) {
            caps(); self.thedesc; " isn't ringing!";
        }
        else self.verDoTake(actor);
    }
    doAnswer(actor) = {
        self.doTake(actor);
    }
    doTake(actor) = {
        self.takemethod(actor);
        if (self.isanswering) self.answermethod(actor);
        else if (self.isbroken) self.brokenmethod(actor);
        else self.dialtonemethod(actor);
    }
    verDoDrop(actor) = {
        if(not self.isoffhook) {
            "The receiver has already been replaced. ";
        }
    }
    doDrop(actor) = {
        self.dropmethod(actor);
        "Done. ";
    }
    verIoDialOn(actor) = {}
    ioDialOn(actor,dobj) = {
        if(dobj <> numObj) {
            "You can't dial that! ";
            return;
        }
        else if(numObj.value < 0) {
            "You can't dial a negative number! ";
            return;
        }
        else if(not self.isoffhook) {
            "\n(lifting the receiver first)\n";
            self.takemethod(actor);"\n";
        }
        else if(self.isanswering) {
            "\n(replacing and lifting the receiver first)\n";
            self.dropmethod(actor);
            self.doTake(actor);"\n";
        }
        if(self.isbroken) self.brokenmethod(actor);
        else self.dialmethod(actor,numObj.value);
    }
    verDoDial(actor) = {
        if(self.isringing) {
            "The phone is ringing. %You%'d better answer it first! ";
        }
    }
    doDial(actor) = {
        local num,prompt,error;
        if(not self.isoffhook) {
            "\n(lifting the receiver first)\n";
            self.takemethod(actor);
        }
        if(self.isbroken) self.brokenmethod(actor);
        else {
            prompt := 'What number do you want to dial?\n>';
            error := 'That\'s not a valid number!';
            num := asknum(prompt,error);
            if(num = nil) return;
            else {
                numObj.value := num;
                self.ioDialOn(actor,numObj);
            }
        }
    }
    verDoAttack(actor) = {}
// Sabotage the phone.  By default, it goes out of order and refuses to
// return your coins.
    doAttack(actor) = {
        if (self.isdented)
            "The telephone is out of order and %your% hand is sore.";
        else {
            "The telephone is now badly dented, and %you% %have% broken
            it beyond all hope.  ";
            self.vandalize;
            self.isbroken := true;
            self.isanswering := nil;
            self.isringing := nil;
            self.brokenretain := true;
        }
    }
    vandalize = { // basic settings when phone is vandalized
            self.isdented := true;
            self.isbroken := true;
            self.isanswering := nil;
            self.isringing := nil;
            self.brokenretain := true;
    }
    // basic actions for lifting receiver (doTake does the complete job)
    takemethod(actor) = {
        self.isoffhook := true;
        if (self.isringing) {
            isanswering := true;
            isringing := nil;
        }
    }
    // method for replacing receiver.  If ispaid is set to an object, it
    // is returned unless the brokenretain property is set.
    dropmethod(actor) = {
        self.isoffhook := nil;
        self.isanswering := nil;
        self.isringing := nil;
        if(self.ispaid and not (self.isbroken and self.brokenretain)) {
            "\nThe phone returns your coins, which you take.\n";
            self.ispaid.moveInto(actor);
        }
        self.ispaid := nil;
    }
    // The following methods should be customized as required
    answermethod(actor) = {
        "You answer the phone, but there is no-one at the other end.";
    }
    dialtonemethod(actor) = {
        "You hear a dial tone.  ";
    }
    brokenmethod(actor) = {
        "The phone appears to be out of order.  There is no dial tone. ";
    }
    // note: coins will be returned unless the phone is broken and
    // brokenretain has been set to true.
    dialmethod(actor,value) = {
        if(value = -1) {
            "Please tell me what number you want to dial, e.g.
            dial <number> on <<self.thedesc>>.";
        }
        else if(self.needtopay and self.ispaid = nil)
            "Nothing happens.  %You% will need to insert coins first.";
        else {
            "You hear a \"number unobtainable\" tone and replace
            the receiver.  ";
            self.dropmethod(actor);
        }
    }
    verIoPutIn(actor) = {}
    // method for putting coins in phone.  By default, if the phone
    // is broken, it will still return coins - but if it has been
    // attacked, it won't.
    ioPutIn(actor,dobj) = {
        if( not isclass(dobj,coinitem) )
            "%You% only want%s% to put coins into the phone. ";
        else if (not self.needtopay)
            "%You% don't need to insert coins to use this phone. ";
        else if (self.isanswering or self.isringing)
            "%You% don't need to insert coins for an incoming call! ";
        else if (self.ispaid)
            "Hold on!  %You% have already put coins into the phone. ";
        else {
            if (not self.isoffhook) {
                "\n(Lifting the receiver)\n";
                self.doTake(actor);
            }
            if (self.isbroken) {
                caps(); dobj.thedesc; " drop into the coinbox with a dull
                'clunk'. There is still no dial tone. ";
            }
            else "Done.  ";
            dobj.moveInto(nil);
            self.ispaid := dobj;
        }
    }
    listendesc = {
        if (self.isringing)
            "The phone is ringing. ";
        else
            pass listendesc;
    }
    // We now use xxxxCheck methods to simplify the implementation,
    // avoiding the need for a 'loccheck' call with every verb:
    dobjCheck(a, v, i, p) = {
        if(self.mybooth = nil) return;
        if(v = gonearVerb or v = countVerb or v = listenVerb) return;
        else if(not a.isIn(self.mybooth)) {
            if(v = inspectVerb) {
                "%You% can't examine the phone properly unless you enter
                the booth. ";
            }
            else {
                "%You% had better enter the booth first. ";
            }
            exit;
        }
    }
    iobjCheck(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb)return;
        self.dobjCheck(a, v, d, p);
    }

;

Phone1: CCR_phoneitem
    game551 = true
    sdesc = "old banged-up payphone"
    adesc = {"an ";self.sdesc;}
    location = Phone_Booth1
    isringing = true
    noun = 'phone' 'telephone' 'payphone' 'receiver' 'handset'
    adjective = 'ancient' 'old' 'banged-up' 'pay'
    answermethod(actor) = {
        "No one replies.  Instead, %you% hear%s% music, which %you% think
        you recognize as \"The Walls of Jericho\".  Maybe it's a crossed
        line.   After a few seconds, the line goes dead with a definite
        \"Click\".  You replace the receiver.  ";
        self.isbroken := true;
        self.dropmethod(actor);
    }

    doAttack(actor) = {
        if (self.isdented)
            "The telephone is out of order and %your% hand is sore.";
        else {
            "A few lead slugs drop from the coinbox.  (Gnomes are
            notoriously cheap....)  But %you% %have% broken the phone
            beyond all hope.  ";
            self.vandalize;
            slugs.moveInto(self.location);
            slugs.moved := nil;
            slugs.heredesc;
        }
    }

;

/* 190 */
Devils_Chair: darkroom
    game551 = true
    sdesc = "At Devil's Chair."
    ldesc = {I();
    "%You're% at the Devil's Chair, a large crystallization shaped like a
    seat, at the edge of a black abyss.  ";"%You% can't see the bottom.
    An upward path leads away from the abyss.";}

    cross = {
        local actor := getActor(&travelActor);
        Decrepit_Bridge.doCross(actor); 
        return nil;
    }
    north =  {return self.cross;}
    across =  {return self.cross;}
    bridge =  {return self.cross;}
    up = Rotunda
    NPCexit1 = {
         if (Decrepit_Bridge.isfallen) return nil;
         else return Dantes_Rest;
    }
    jump = {
            if (not Decrepit_Bridge.isfallen) {
                    "I respectfully suggest you go across the
                    bridge instead of jumping.";

                    return nil;
            }
            else
                    return didnt_make_it.death;
    }
    exithints = [ Devils_Chair, &north ]
    listendesc = {
        if (global.oldgame) pass listendesc;
        global.listenadd := true;
        inherited.listendesc;
        P(); "You hear a distant roar, like the sound of a fast-flowing
             river, from the depths of the abyss. ";
    }
;

Abyss: rfd
    iswavetarget = true // magic can be worked by waving the rod at it ...
    loclist = [Devils_Chair, Dantes_Rest]
    noun = 'abyss' 'chasm'
    adjective = 'dark' 'black' 'deep'
    verDoJump(actor) = {}
    doJump(actor) = {
        actor.travelTo(actor.location,&jump);
    }
    listendesc =  "You hear a distant roar, like the sound of a
    fast-flowing river, from the depths of the abyss. ";
;
Decrepit_Bridge: floatingdecoration
    iswavetarget = true // magic can be worked by waving the rod at it ...
    game551 = true
    sdesc = "bridge"
    ldesc = {
        if(not self.isfallen) {
            "That bridge looks very fragile. ";
            if(self.crosscount >= 0) "I'd take care when crossing
            it. ";
        }
        else
            "The remnants of the bridge can still be seen, but there
            is now no way across the chasm.";
    }
    heredesc = {
        P(); I();
        if(not self.isfallen) {
            "A decrepit natural bridge spans the chasm.  A message
            scrawled into the rock wall reads: \"Bridge out of repair.
            Maximum load: 35 Foonts.\"";
        }
        else {
            "The remnants of a natural bridge partially overhang the
            chasm.";
        }
    }
    noun = 'bridge'
    adjective = 'decrepit' 'natural'
    loclist = [Dantes_Rest Devils_Chair]
    crosscount = 0
    /* Code for crossing the bridge follows.  It can always be crossed
       safely if the total weight of objects carried is 4 or less.  If
       more than 4, the bridge may collapse and we'll end up in the
       Lost River Canyon.  Each time the bridge is crossed, the probability
       of collapse goes up (provided that the allowable weight was
       exceeded).   The upgraded magic wand can temporarily make the
       bridge safer.
    */
    verDoCross(actor) = {}
    doCross(actor) = {
        local newloc := Devils_Chair, safecross := nil, fallpct, i, l, o;
        local olist, wt;
        if (self.isfallen) {
            "There is no longer any way across the chasm.";
            return;
        }
        self.crosscount++;
        if(actor.isIn(Devils_Chair)) newloc := Dantes_Rest;
        wt := addmass(actor.contents);
        if((self.crosscount <= 0) or (wt <= 4))
            safecross := true;
        if(not safecross) {
            fallpct := ((wt + self.crosscount) *
            (wt + self.crosscount))/10;
            if(fallpct < 10) fallpct := 10;
                if(rand(100) <= fallpct) {
                self.isfallen := true;
                "The load is too much for the bridge!  With a roar, the
                entire structure gives way, plunging you headlong into the
                raging river at the bottom of the chasm and scattering all
                your holdings.  As the icy waters close over your head,
                you flail and thrash with all your might, and with your
                last ounce of strength pull yourself onto the south bank
                of the river."; P();
                if(brass_lantern.isIn(actor))
                    brass_lantern.moveInto(Lost_Canyon_E);
                if(axe.isIn(actor))
                    axe.moveInto(Lost_Canyon_S);
                l := length(fresh_batteries.list);
                for (i := 1; i <= l; i++) {
                    o := fresh_batteries.list[i];
                    if(o.isIn(actor))
                        o.moveInto(Lost_Canyon_S);
                }
                l := length(actor.contents); olist := actor.contents;
                for (i := 1; i <= l; i++) {
                    o := olist[i];
                    o.moveInto(nil);
                    if (o = mushrooms or o = mushroom) // Regrow mushrooms
                        notify(o,&regrow,o.growtime);
                }
                actor.travelTo(Lost_Canyon_E);
                return;
            }
            else {
                "The bridge shakes as you cross.  Large hunks of clay and
                rock near the edge break off and hurtle far down into the
                chasm.  Several of the cracks on the bridge surface widen
                perceptibly."; P();
            }
        }
        actor.travelTo(newloc);
    }
;

Bridgemess: floatingdecoration, readable
    game551 = true
    sdesc = "message"
    ldesc = {
        "It reads: \"Bridge out of repair.
        Maximum load: 35 Foonts.\" " ;
        if(Decrepit_Bridge.isfallen) {
            "It's certainly out of repair now.  You should have taken
            more heed of the warning. ";
        }
        else if (Decrepit_Bridge.crosscount >= 0) {
            "The bridge certainly looks very fragile. ";
            P(); "You'll need to keep the weight of your inventory to a bare
            minimum, otherwise it won't be safe to cross. ";
        }
    }
    noun = 'message' 'warning'
    adjective = 'warning'
    loclist = [Dantes_Rest Devils_Chair]

;
/* 191 */
Dead_End_Crack: darkroom, CCR_dead_end_room
    game551 = true
    sdesc = "In Dead End Crack"
    ldesc = {I();
    "%You're% in a dead-end crack.";}

    north = At_East_End_Of_Long_Hall
    ana2 = Blue_Dead_End_Crack
;

/* 192 */
Gravel_Beach: room, NoNPC
    game551 = true
    sober = true // don't allow player to leave by drinking wine
    sdesc = "On Gravel Beach"
    ldesc = {I();
    "%You're% on a small gravel beach at the south wall of the Blue Grotto.
    A gravelly path leads east.";}

    boat_north = Grotto_West
    boat_ne = Blue_Grotto_East
    north =  {"%You% can't swim.  %You%'d best go by boat."; return nil;}
    ne =  {return self.north;}
    east = Vestibule
    // Exit info. for 'back' command:
    exithints = [
        Grotto_West, &north,
        Blue_Grotto_East, &ne,
        Vestibule, &east
    ]
;
/* 193 */
Flower_Room: darkroom, NoNPC
    game551 = true
    sober = true // don't allow player to leave by drinking wine
    sdesc = "In Flower Room"
    ldesc = {I();
    "%You% %are% in the Flower Room.  The walls are covered with colorful,
    intricate, flower-like patterns of crystallized gypsum. A hole leads
    to the west.";}

    west = Vestibule

;

hive: fixeditem, container
    game551 = true
    sdesc = "beehive"
    ldesc = {
        if (not Bees.arefed)
            "Due to the bees, you can't examine the hive closely.";
        else
            "It's a normal-looking beehive, securely fixed to the floor.";
        if (itemcnt(self.contents) > 0) {
            "\nIt contains "; listcont(self);".  ";
        }
    }
    location = Flower_Room

    noun = 'hive' 'beehive'
    // Adjective 'bee' deleted for TADS 2.4.0

    heredesc = {
        P(); I(); if(not Bees.arefed)
            "There is an active beehive nearby.  The bees hum
            protectively around the hive.  ";
        else
            "There is a beehive here, securely fixed to the floor.";
    }

    dobjGen(a, v, i, p) =
    {
        if ((v <> inspectVerb) and (v <> gonearVerb) and (v <> waveVerb)
        and (v <> countVerb) and (v <> askVerb) and (v <> tellVerb) and
        not Bees.arefed)
        {
            "The hum of the bees rises to an angry buzz as you move
                towards the hive.";
                exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb)return;
        self.dobjGen(a, v, d, p);
    }

    verDoLookin(actor) = {if (not Bees.arefed) self.ldesc;}
    verIoThrowAt(actor) = {}
    ioThrowAt(actor,dobj) = {
        if(dobj <> flowers) "That would only enrage the bees!";
        else Bees.doFeedWith(actor,dobj);
    }
    verIoThrowTo(actor) = {}
    ioThrowTo(actor,dobj) = {
        self.ioThrowAt(actor,dobj);
    }

/* Allow objects to be put in the hive provided they are not too bulky, and
   provided that the bees have been fed. */
    ioPutIn ( actor, dobj ) = {
        if(not Bees.arefed) {
            "The bees won't let you anywhere near the hive. ";
        }
        else if(dobj.islong) {
            caps(); dobj.thedesc; " is too long to go into ";
            self.thedesc; ".  ";
        }
        else if(dobj.islarge) {
            caps(); dobj.thedesc; " is too large to go into ";
            self.thedesc; ".  ";
        }
        else if(dobj.ishuge) {
            caps(); dobj.thedesc; " is far too large to go into ";
            self.thedesc; ".  ";
        }
        else pass ioPutIn;
    }
;
/* 194, 195 */

EW_Corridor_E: darkroom
    game551 = true
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "At East End of Short E/W Corridor"
    ldesc = {I();
        if(In_Arched_Hall.jericho) {
            "%You% %are% looking west from the end of a short E/W corridor.
            At %your% feet is a pile of loose rubble. On ";"%your% left is
            a hole into another chamber.";
        }
        else "%You% %are% at the end of a short E/W corridor.";
    }

    hole = {
        if (not In_Arched_Hall.jericho) pass hole;
        else return In_Arched_Hall;
    }
    south = {
        if (not In_Arched_Hall.jericho) pass south;
        else return In_Arched_Hall;
    }
    wall = {return self.hole;}
    cross = {return self.hole;}
    left = {return self.hole;}
    west = Crypt

    // Avoid getting NPC's trapped in here.
    NPCexit1 = In_Arched_Hall
    // Exit info. for 'back' command:
    exithints = [In_Arched_Hall, &hole]
    listendesc = {
        if (self.jericho or global.oldgame)
            pass listendesc;
        else
            "You pace around the room, listening carefully to your
            footsteps.  You have a strong feeling that the wall to your
            south is hollow. ";
    }

;
Corridor_Rubble: fixeditem
    game551 = true
    sdesc = "loose rubble"
    ldesc = "It's just loose rubble. "
    noun = 'rubble'
    adjective = 'loose'
    verDoSearch(actor) = {}
    doSearch(actor) = {"You sift through the rubble, but find nothing
    of interest. ";}
    verDoMove(actor) = {}
    doMove(actor) = {"You move the rubble to one side, but find nothing
    of interest. ";}
    verDoTake(actor) = {"You've come here to find treasures, not to cart
    useless rubble around the cave.  I suggest that you leave it where
    it is. ";}
    verDoPutIn(actor,io) = {self.verDoTake(actor);}

;
/* 196 incorporated with In_Arched_Hall */
/* 197 */
Vestibule: darkroom, NoNPC
    game551 = true
    sober = true // don't allow the player to leave by drinking wine
    sdesc = "In the Vestibule"
    ldesc = {I();
    "%You're% in the Vestibule, a short east-west passage between two
    rooms.";}

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    east = Flower_Room
    west = Gravel_Beach

;
/* 198 */
Fairy_Grotto: room
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    game551 = true
    sdesc = "In the Fairy Grotto"
    ldesc = {I();
    "%You% %are% in the Fairy Grotto.  All around %youm% innumerable
    stalactites,
    arranged in immense colonnades, form elegant arches.  On every side
    you hear the dripping of water, like the footsteps of a thousand
    fairies.  A small stream runs from the SW corner.  A bright glow
    emanates from the south side of the grotto, and a steep passage
    descends to the east.";}

    sw = Muddy_Defile
    stream = Muddy_Defile
    downstream = Muddy_Defile
    passage = Muddy_Defile
    down = Cold_Passage
    east = Cold_Passage
    south = {
        local actor := getActor(&travelActor);
        if (brass_lantern.isIn(actor) and brass_lantern.islit) {
            "You go a short way down the bright passage, but the light
            grows to blinding intensity.  You can't continue.";
            return nil;
        }
        else return Crystal_Palace;
    }
    up = {return self.south;}
    NPCexit1 = Crystal_Palace
    // Exit info. for 'back' command:
    exithints = [Crystal_Palace, &south]
;
/* 199 */
too_cold: object
    message = {"%You% %have% approached the lower end of a steep passage,
    but it is just too cold here to hang around, and ";"%you% aren't
    properly equipped to continue.  With teeth chattering, ";"%you% climb
    back up....";P();}
;
/* 200 */
Crystal_Palace: room
    // This is the reverse of a darkroom.  When the lamp is on, the light is
    // blinding.
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In the Crystal Palace."
    ldesc = {I();
    if (self.blinding) {
        "The glare from the walls is absolutely
        blinding.  If you were to proceed you would almost certainly fall
        into a pit.";
    }
    else {
        "%You% %are% in the Crystal Palace.  An overhead vein of
        phosphorescent quartz casts a luminous glow which is reflected by
        countless chips of mica embedded in both walls, which consist of
        some sort of highly reflective glass, apparently of volcanic
        origin.  A winding path of yellow sandstone leads west and rises
        steeply to the east.";}
    }
// DJP - can't find "liminous" even in the OED.  Changed it to 'luminous'.

    west = {
        if (self.blinding) {
            "The glare from the walls is absolutely blinding.
            If you tried to proceed you would almost certainly fall into
            a pit.";
            return nil;
        }
        else return Fairy_Grotto;
    }

    east = {
        if (self.blinding) {
            "The glare from the walls is absolutely blinding.
            If you tried to proceed you would almost certainly fall into
            a pit.";
            return nil;
        }
        else return Yellow_Path;
    }
    up = {return self.east;}

    NPCexit1 = Fairy_Grotto
    NPCexit2 = Yellow_Path

    // Exit info. for 'back' command:
    exithints = [
        Fairy_Grotto, &west,
        Yellow_Path, &east
    ]
    // This property indicates that the light level in the room is too
    // high ...
    blinding = (brass_lantern.islit and brass_lantern.isIn(self))
    replace lookAround( verbosity ) =
    {
        // DJP: Added check for blinding room.
        if ( not self.blinding ) pass lookAround;
        else {
            "\(Blinding Room\)"; P(); "The glare from the walls is absolutely
            blinding.  If you tried to proceed you would almost certainly fall
            into a pit.";
        }
    }
    // The roomAction and roomCheck methods similarly look for a blinding
    // level of illumination instead of darkness.
    roomAction( actor, v, dobj, prep, io ) =
    {
        // DJP - vet the 'put', 'change' and 'replace' verbs to allow
        // batteries to be changed, but no other actions.
        if (v = putVerb and self.blinding) {
            if (isclass(dobj,fresh_batteries) and prep = inPrep and
            io = brass_lantern and brass_lantern.isReachable(actor) and
            dobj.isReachable(actor)) {
                pass roomAction;
            }
            else {
                "The glare from the walls is so bright that you can't see
                a thing.\n";
                exit;
            }
        }
        if ((v = changeVerb or v=replaceVerb) and self.blinding) {
            if (isclass(dobj,fresh_batteries) and
            brass_lantern.isReachable(actor) and dobj.isReachable(actor)) {
                pass roomAction;
            }
            else {
                "The glare from the walls is so bright that you can't see
                a thing.\n";
                exit;
            }
        }
        // allow dropping in the blinding light but not 'drop on' or 'drop in'
        if ((v = dropVerb) and self.blinding and (io <> nil)) {
            "The glare from the walls is so bright that you can't see
            a thing.\n";
            exit;
        }
        if ( self.blinding and not v.isDarkVerb)
        {
            "%You% can't see a thing. ";
            exit;
        }
        else pass roomAction;
    }
    roomCheck( v ) =
    {
        if ( (not self.blinding) or v.issysverb or v.isTravelVerb or
        v.isDarkVerb )
        return true;
        else
        {
            "The glare from the walls is so bright that you can't see
            a thing.\n";
            return nil;
        }
    }
;
/* 201 */
Yellow_Path: room
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "On the Yellow Path"

/* Mention of the north passage has been added, on the grounds of fairness
to Adventurers.  */

    ldesc = {I();
    "%You% %are% following an east-west yellow sandstone path.  There is a
    glow to the west.   A dark passage branches off to the north. ";}
    west = {
        local actor := getActor(&travelActor);
        if (brass_lantern.isIn(actor) and brass_lantern.islit) {
            "You go a short way down the bright passage, but the light
            grows to blinding intensity.  You can't continue.";
            return nil;
        }
        else return Crystal_Palace;
    }
    down = {return self.west;}
    east = Rainbow_Room
    passage = Ledge_Above_Pinnacles
    north = Ledge_Above_Pinnacles
    NPCexit1 = Crystal_Palace
    // Exit info. for 'back' command:
    exithints = [Crystal_Palace, &west]
;
/* 202 */
Rainbow_Room: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In the Rainbow Room."
    ldesc = {I();
    "%You% %are% in a very tall chamber whose walls are comprised of many
    different rock strata.  Layers of red and yellow sandstone
    intertwine with bright bands of calcareous limestone in a rainbow-
    like profusion of color.  The rainbow effect is so real, you
    are almost tempted to look for a pot of gold!  Poised far over ";
    "%your% head, a gigantic slab, wedged tightly between the north and
    south walls, forms a natural bridge across the roof of the chamber.
    A trail leads east and west.";}

    west = Yellow_Path
    click = Over_Rainbow
    east = Green_Lake_Room
    myhints = [Bridgehint]
;
RainbowSlab: rfd
    sdesc = "natural bridge"
    loclist = [Rainbow_Room Over_Rainbow]
    noun = 'slab' 'bridge'
    adjective = 'natural' 'gigantic'
    // The natural bridge is a distant item if seen from below.
    dobjGen(a, v, i, p) =
    {
        if (a.isIn(Rainbow_Room) and
        (v <> inspectVerb) and (v <> gonearVerb) and (v <> countVerb)) {
            "It's too far away. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb)return;
        self.dobjGen(a, v, d, p);
    }
;
/* 203 */
Cold_Passage: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Cold Passage"
    ldesc = {I();
    "%You're% in a steeply sloping passage.  It is very cold here.";}

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    down = {
        local actor := getActor(&travelActor);
        if (cloak.location = actor and cloak.isworn)
            return Hall_Of_Ice;
        else {
            too_cold.message;
            return nil;
        }
    }
    east = {return self.down;}
    ice = {return self.down;}
    up = Fairy_Grotto
    west = Fairy_Grotto
    // No NPCexits - hall of ice is off limits to dwarves
    // Exit info. for 'back' command:
    exithints = [Hall_Of_Ice, &down]
;

/* 204 */
Hall_Of_Ice: darkroom, NoNPC //too cold for NPC's
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Hall of Ice"
    ldesc = {I();
    "%You% %are% in the Hall of Ice, in the deepest part of the caverns.
    During winter, frigid outside air settles here, making this room
    extremely cold all year round.  The walls and ceiling are covered
    with a thick coating of ice.  An upward passage exits to the west.";}

    up = Cold_Passage
    passage = Cold_Passage
    west = Cold_Passage
    out = Cold_Passage

;

Ice: fixeditem
    game551 = true
    sdesc = "ice"
    adesc = "ice"
    ldesc = "It covers the walls and ceiling. "
    noun = 'ice'
    location = Hall_Of_Ice
;

/* 205 */
Over_Rainbow: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "Over the Rainbow (Room)"
    ldesc = {I();
    "%You% %are% standing on a natural bridge far above the floor of a
    circular
    chamber whose walls are a rainbow of multi-colored rock.  The bridge
    was formed eons ago by a huge slab which fell from the ceiling and
    is now jammed between the north and south walls of the chamber.";}

    north = Gothic_Cathedral
    click = Rainbow_Room
    jump = {return rainbow_demise.death;}

;
/* 206 */
Green_Lake_Room: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Green Lake Room"
    ldesc = {I();
    "%You% %are% in a low, wide room below another chamber.  A small green
    pond fills the center of the room.  The lake is apparently spring-
    fed.  A small stream exits through a narrow passage to the north.
    A larger passage continues west.";}

    passage = {
        "There's more than one passage - please tell me which
        direction you want to go. ";
        return nil;
    }

    stream = Red_Rock_Crawl
    downstream = Red_Rock_Crawl // Added - DJP
    north = Red_Rock_Crawl
    crawl = Red_Rock_Crawl
    west = Rainbow_Room
    up =  {"The hole is too far up for %youm% to reach. "; return nil;}
    hole =  {return self.up;}
    climb =  {return self.up;}

;
Green_Lake: Streamitem
    game551 = true
    sdesc = "green pond"
    ldesc = "It would be better described as a small pond rather than a
            lake.  It fills the center of the room and has a definite
            green tint, probably caused by minerals dissolved from the
            floor of the chamber. "
    noun = 'pond' 'lake' 'water'
    adjective = 'green'
    location = Green_Lake_Room
;
/* 207 */
Red_Rock_Crawl: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Red Rock Crawl"
    ldesc = {I();
    "%You% %are% in a tight north/south crawl through a stratum of red
    colored rock.  The air is damp with mist.";}

    north = Lost_Canyon_S
    south = Green_Lake_Room
    upstream = Green_Lake_Room //added - DJP
    downstream = Lost_Canyon_S //added - DJP
;

/* 208 */
Lost_Canyon_S: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "On South Side of Lost River Canyon."
    ldesc = {I();
    "%You% %are% in a tall canyon on the south side of a swift, wide river.
    Written in the mud in crude letters are the words: \"You Have Found
    Lost River.\"  A wide path leads east and west along the bank.  A tight
    crawlway would take you south out of the canyon.";}

    east = Lost_Canyon_E
    upstream = Lost_Canyon_E // Added - DJP
    south = Red_Rock_Crawl
    crawl = Red_Rock_Crawl
    west = Lost_Canyon_End
    downstream = Lost_Canyon_End // Added - DJP
    cross =  {"The river is too wide and deep to cross.";return nil;}
    across = {return self.cross;}
    north =  {return self.cross;}
    listendesc = "You hear the roar of the wide, fast-flowing river. "
;

Lost_Canyon_Mud: fixeditem, readable
    sdesc = "mud"
    ldesc = {
        "It's just ordinary mud, on which a previous adventurer has
        written the following:"; P();I();
        self.readdesc;
    }
    readdesc = "You have found Lost River."
    noun = 'mud' 'lettering' 'note'
    adjective = 'crude'
    location = Lost_Canyon_S
;


Lost: Streamitem, floatingdecoration
    sdesc = "river"
    ldesc = "The river is wide, deep and fast-flowing.  There's no way
        to cross it."
    noun = 'river' 'water' 'stream'
    adjective = 'wide' 'deep' 'swift' 'rapid' 'fast' 'flowing'
        'fast-flowing' 'lost'

    loclist = [Lost_Canyon_S Lost_Canyon_End Lost_Canyon_E
        Niche_Above_River Narrow_Ledge]

    verDoCross(actor) = {"The river is too wide and deep to cross.";}

    // The river is a distant item in some locations.
    dobjGen(a, v, i, p) = {
        if (self.isIn(Lost_Canyon_S) or self.isIn(Lost_Canyon_E)) return;
        if ((v <> inspectVerb) and (v <> gonearVerb) and
        (v <> countVerb)) {
            "It's too far away. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb)return;
        self.dobjGen(a, v, d, p);
    }
    listendesc = "You hear the roar of the wide, fast-flowing river. "
;


/* 209 */
Lost_Canyon_End: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At End of Lost River Canyon."
    ldesc = {I();
    "%You% %are% standing on a large flat rock table at the western end of
    Lost River Canyon.  Beneath %your% feet, the river disappears amidst
    foam and spray into a large sinkhole.  A gentle path leads east
    along the river's south shore.  Another leads sharply upward along
    the river's north side.";}

    downstream = {"Don't be ridiculous! "; return nil;} // Added - DJP
    jump = {return self.downstream;} // Added - DJP
    east = Lost_Canyon_S
    upstream = Lost_Canyon_S // Added - DJP
    up = Niche_Above_River
    climb = Niche_Above_River
    listendesc = "You hear the roar of the wide, fast-flowing river. "
;
/* 210 */
Niche_Above_River: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Niche in Ledge above Lost River."
    ldesc = {I();
    "%You% %are% at a niche in the canyon wall, far above a raging river.
    The air is filled with mist and spray, making it difficult to see
    ahead.  A downward sloping ledge narrows to the east. The path
    to the west is easier.";}

    down = Narrow_Ledge
    east = Narrow_Ledge
    west = Lost_Canyon_End
    listendesc = "You hear the roar of the wide, fast-flowing river, far
    below you. "

;
/* 211 */
Narrow_Ledge: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Narrow Ledge"
    ldesc = {I();
    "The ledge is growing very narrow and treacherous, and falls off almost
    vertically.  ";"%You% could go down, but %you% won't be able to climb
    back.";}

    up = Niche_Above_River
    west = Niche_Above_River
    down = On_Ne_Side_Of_Chasm
    east = On_Ne_Side_Of_Chasm
    // Let NPC's get out of here
    NPCexit1 = On_Sw_Side_Of_Chasm
    listendesc = "You hear the roar of the wide, fast-flowing river, far
    below you. "

;
/* 212 */
Phone_Booth_2: room, NoNPC
    game551 = true
    sdesc = "In Phone Booth"
    ldesc = {I();
    "%You% %are% standing in a telephone booth at the side of the
    Repository.";}

    out = At_Ne_End
    south = At_Ne_End

;
/* 213 */
Passage_End_at_Hole: darkroom
    game551 = true
    sdesc = "At East End of Level Passage"
    ldesc = {I();
    "%You're% at the east end of a level passage at a hole in the floor.";}

    down = Green_Lake_Room
    hole = Green_Lake_Room
    passage = Tongue_of_Rock
    west = Tongue_of_Rock

;
/* 214 */
Dark_Cove: room, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    game551 = true
    sdesc = "In Dark Cove"
    ldesc = {I();
    "%You're% at the north edge of a dark cove.  To your south lies
    the Blue Grotto.  A large lake almost fills the cavern floor. ";}

    boat_south = Grotto_West
    boat_se = Bubble_Chamber
    south =  {"%You% can't swim.  %You%'d best go by boat."; return nil;}
    se =  {return self.south;}
    ne = Dry_Basin
    // Exit info. for 'back' command:
    exithints = [
        Grotto_West, &south,
        Bubble_Chamber, &se,
        Dry_Basin, &ne
    ]
;
/* 215 */
Dry_Basin: darkroom, NoNPC
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Dry Basin"
    ldesc = {I();
    "%You% %are% in a dry granite basin, worn smooth eons ago by water
    swirling down from a now-dry spillway.";}

    sw = Dark_Cove
    up = Old_Spillway

;
/* 216 */
Old_Spillway: darkroom, NoNPC
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Old Spillway"
    ldesc = {I();
    "%You're% in a dry spillway east of and above a smooth rock basin.";}

    down = Dry_Basin
    east = Winery
    up = Winery

;
/* 217 */
Winery: darkroom, NoNPC
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Winery"
    ldesc = {I();
    "%You% %are% in the Winery, a cool dark room which extends some
    distance off to the east.";}

    west = Old_Spillway
    down = Old_Spillway
    out = Old_Spillway
    east = Limestone_Pinnacles
    pit = Limestone_Pinnacles

;

Wine: fixeditem,roomliquid
    game551 = true

    liquid = "wine"
    /* wine prefers to go into the cask by default */
    contlist = [cask bottle]

    sdesc = "fountain of wine"
    ldesc = "It appears to be expensive vintage wine!  It would be
        very valuable if you could find a suitable cask to carry
        it in.  "
    heredesc = {P(); I(); "There is a fountain of sparkling vintage wine
        here!";}

    noun = 'wine' 'fountain'
    adjective = 'wine' 'fountain'

    location = Winery

    verDoDrink(actor) = {
        if (Wumpus.ischasing) {
            "You'd better do something about the Wumpus
            first - this isn't going to help! ";
        }
        else if (Dwarves.numberhere(actor) = 1) {
            "You'd better do something about the dwarf first -
            this isn't going to help! ";
        }
        else if (Dwarves.numberhere(actor) > 1) {
            "You'd better do something about the dwarves
            first - this isn't going to help! ";
        }
    }
    doDrink(actor) = {cask.winocode;}
;


/* 218 */
Limestone_Pinnacles: darkroom, NoNPC
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Limestone Pinnacles"
    ldesc = {I();
    "%You% %are% to the east of the Winery, where the room ends in a
    thicket
    of high, sharp, pointed, climable limestone pinnacles.  There is a
    narrow ledge just above the top of the spires.  If you go up, it
    might be difficult to get back down.";}

    west = Winery
    up = Ledge_Above_Pinnacles
    climb = Ledge_Above_Pinnacles

;
/* 219 */
Gothic_Cathedral: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "In Gothic Cathedral"
    ldesc = {I();
    "%You% %are% in a high-vaulted cavern whose roof rises over fifty
    meters to culminate in a series of pointed arches directly over ";
    "%your% head.  There are also two low arches to either side, forming
    side portals.  The whole effect is that of a gothic cathedral. ";
    "%You% can proceed north, south, east, or west.";}

    south = Over_Rainbow
    bridge = Over_Rainbow
    north = Altar_Foot
    altar = Altar_Foot
    east = East_Portal
    west = West_Portal
    pray = Inside_Building

;
/* 220 */
East_Portal: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "At East Portal of Gothic Cathedral"
    ldesc = {I();
    "%You're% at the east portal of the Gothic Cathedral. The path
    leads east and west.";}

    west = Gothic_Cathedral
    east = Gothic_Chapel

;
/* 221 */
West_Portal: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "At West Portal of Gothic Cathedral"
    ldesc = {I();
    "%You're% at the west portal of the Gothic Cathedral.";}

    east = Gothic_Cathedral

;
/* 222 */
Altar_Foot: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "At Foot of Altar"
    ldesc = {I();
    "%You% %are% at the foot of the Altar, an immense, broad stalagmite.
     An opening leads south.";}

    passage = Gothic_Cathedral
    south = Gothic_Cathedral
    up = Top_of_Stalagmite
    climb = Top_of_Stalagmite
    pray = Inside_Building

;
Altar: fixeditem, upclimbable
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "immense broad stalagmite"
    ldesc = "It does have an uncanny resemblance to a large cathedral
    altar. "
    noun = 'altar' 'stalagmite'
    adjective = 'immense' 'enormous' 'broad'
    location = Altar_Foot
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&climb);
    }
;
/* 223 */
Top_of_Stalagmite: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "On Top of Stalagmite"
    ldesc = {I();
    "%You're% on top of an enormous, broad stalagmite.  There is a hole
    in the ceiling overhead.";}

    down = Altar_Foot
    climb = Altar_Foot
    up = Crypt
    hole = Crypt
;
Stalagmite: fixeditem, downclimbable
    game551 = true
    sdesc = "enormous broad stalagmite"
    ldesc = "If you want to find out more, I suggest that you climb down
    it. "
    noun = 'altar' 'stalagmite'
    location = Top_of_Stalagmite
    adjective = 'immense' 'enormous' 'broad'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&down);
    }
;

/* 224 */
Crypt: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "In the Crypt"
    ldesc = {I();
    "%You% %are% in a room the size and shape of a small crypt.  A narrow
    cut exits east.  There is a hole in the floor.";}

    down = Top_of_Stalagmite
    hole = Top_of_Stalagmite
    east = EW_Corridor_E

;
/* 225 */
Gothic_Chapel: darkroom
    game551 = true
    // Don't allow drinking unless the Walls of Jericho are down.
    sober = {return not In_Arched_Hall.jericho;}
    sdesc = "In Gothic Chapel"
    ldesc = {I();
    "%You% %are% in the Gothic Chapel, a small chamber adjoining the Gothic
    Cathedral. A path leads west.";}

    west = East_Portal

;
/* 226 */
rainbow_demise: object
    ldesc = "%You% %are% on the floor of the Rainbow Room.  In fact,
    %you% %are% spread *ALL OVER* the floor of the Rainbow Room."
    death = {self.ldesc; die(); return nil;}
;
/* 227 */
River_Styx_Approach: room, Outside, Indoor
    game551 = true
    sdesc = "At Approach to River Styx"
    ldesc = {I();
    "%You% %are% in a dimly lit E/W passage behind Thunder Hole.
    Etched into
    the rock wall are the ominous words: \n
    *\ \ \ \"You are approaching the River Styx.\ \ \ \ \ *\n
    *\ \ \ Lasciate Ogni Speranza Voi Ch'Entrate.\"\ \ *";}

    west = Thunder_Hole
    out = Thunder_Hole
    up = Thunder_Hole

    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }

    east = {
        if (Dog.isasleep or not Dog.isIn(self)) return River_Styx;
        else {dog_message.message; return nil;}
    }

    in = {return self.east;}
    down = {return self.east;}
    // Exit info. for 'back' command:
    exithints = [River_Styx, &east]
    myhints = [Doghint]
    listendesc = {
        "You hear the sound of the river, and in the background there
        is a booming sound, caused by the surf pounding
        against the outer rocks of the cave. ";
        if(not Dog.isasleep) {
            P(); "You also hear the dog growling at you!  I wouldn't
            go too close if I were you. ";
        }
    }
;

/* 228 */
River_Styx: room, Outside, Indoor
    game551 = true
    sdesc = "At River Styx"
    ldesc = {I();
    "%You% %are% at the River Styx, a narrow little stream cutting directly
    across the passageway.  The edge of the stream is littered with sticks
    and other debris washed in by a recent rainfall.  On the far side
    of the river, the passage continues east.";}

    north = {"The stream flows out of one very small crack and into
    another."; return nil;}

    south =  {return self.north;}
    upstream =  {return self.north;}
    downstream =  {return self.north;}
    crack =  {return self.north;}
    up = River_Styx_Approach
    west = River_Styx_Approach
    out = River_Styx_Approach
    jump = River_Styx_E
    // Modified to allow us across once we've worked out how to
    // cross the stream.
    east =  {
        if (River_Styx_E.isseen) {
            "\n(jumping the river)\n";
            return self.jump;
        }
        else {
            "How do you propose to cross the river?";
            return nil;
        }
    }
    in =  {return self.east;}
    across =  {return self.east;}
    cross =  {return self.east;}
    over =  {return self.east;}
    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    listendesc = "You hear the sound of the river, and in the background
    there is a booming sound, caused by the surf pounding against the
    outer rocks of the cave. "
;

Styx: Streamitem, floatingdecoration
    sdesc = "River Styx"
    ldesc = {
        "It's a narrow but very deep little stream cutting across the
        passageway.  It flows out of one very small crack and into
        another, so you can't follow it upstream or downstream. ";
        if(River_Styx_E.isseen) {
            "However, you can jump across! ";
        }
        else {
            "You can't wade across, but there might be another way. ";
        }
    }
    loclist = [River_Styx River_Styx_E]
    noun = 'river' 'styx' 'stream' 'water'
    adjective = 'river' 'deep' 'narrow'
    verDoCross(actor) = {}
    doCross(actor) = {
        actor.travelTo(actor.location,&cross);
    }
    verDoJump(actor) = {}
    doJump(actor) = {
        actor.travelTo(actor.location,&jump);
    }
    listendesc = "You hear the sound of rushing water. "
    verIoSpanOver(actor) = {}
    ioSpanOver(actor,dobj) = {
        if(dobj.islong)
            "You can span <<dobj.thedesc>> over the river, but it's not
            much use as a bridge.  I'm sure there's a simpler and
            safer way to get across. ";
        else
            "Good try, but <<dobj.thedesc>> is too short to span 
            over <<self.thedesc>>.";
    }
;

Sticks: floatingdecoration
    sdesc = "sticks at Styx"
    ldesc = "The sticks and branches litter the edge of the stream."
    noun = 'sticks' 'debris' 'branches'
    loclist = [River_Styx River_Styx_E]
    verDoTake(actor) = {
        "The sticks and branches are all in a tangle and are stuck in the
        mud.  You'd need a shovel to dig them out. ";
    }
;
/* 229 */
River_Styx_E: room, Outside, Indoor
    game551 = true
    sdesc = "On East Side of River Styx"
    ldesc = {I();"%You're% on the east side of the river's sticks.";}

    north =  {"The stream flows out of one small crack and into
    another."; return nil;}

    south =  {return self.north;}
    upstream =  {return self.north;}
    downstream =  {return self.north;}
    crack =  {return self.north;}
    jump = River_Styx
    west =  {
        "\n(jumping the river)\n";
        return self.jump;
    }
    out =  {return self.west;}
    across =  {return self.west;}
    cross =  {return self.west;}
    over =  {return self.west;}
    east = Top_Of_Steps
    in = Top_Of_Steps
    steps = Top_Of_Steps
    passage = {
        "The passage goes in two directions.  Please tell me which way
        you want to go ";
        return nil;
    }
    listendesc = "You hear the sound of the river, and in the background
    there is a booming sound, caused by the surf pounding against the
    outer rocks of the cave. "

;
/* 230 */
Ledge_Above_Pinnacles: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "On ledge above limestone pinnacles."
    ldesc = {I();
    "%You% %are% on a ledge at the northern end of a long N/S crawl.  The
    ledge is above a large number of sharp vertical limestone spires.
    An attempt to climb down with bulky items could be dangerous, if
    you get my *point*!";}

/* The effect of going down here has been reworked.  Now you can get down
   safely unless you are carrying any bulky items. */

    down = {
        local actor := getActor(&travelActor);
        local str, stat, tx := nil, skewered := nil,
        o := firstobj(item);
        while (o) {
            if ((o.ishuge or o.islarge) and not (o = sack) and not o.isfixed) {
                if (o.isInside(actor)) {
                    skewered := true;
                    if (tx = nil) {
                        stat := outcapture(true);
                        o.sdesc;
                        str := outcapture(stat);
                        tx := 'the bulky '+str;
                    }
                    else tx := 'the bulky items';
                }
            }
            o := nextobj(o,item);
        }
        if (skewered) return pinnacle_demise.bulkdeath(tx);
        else return Limestone_Pinnacles;
    }
    // Jumping down will always get you skewered.
    jump = {return pinnacle_demise.jumpdeath;}
    south = Yellow_Path
    crawl = Yellow_Path
    // Exit info. for 'back' command:
    exithints = [Limestone_Pinnacles, &down]
    myhints = [Caskhint]
;
/* 231 */
pinnacle_demise: object
    jump = "That wasn't exactly your most brilliant move!
    You are very neatly skewered on the point of a sharp
    rock. "
    bulk(tx) = {
        "Your attempt to climb down is hampered by
        <<tx>> you are carrying.  You lose your grip and fall to
        your death! You are now very neatly skewered on the point
        of a sharp rock. ";
    }

    // Note that we don't drop the player's possessions at
    // Limestone Spires.  This would be realistic, but it would allow the
    // player to obtain the wine after reincarnation!  The main
    // objection to this is that the player might be misled into seeking
    // the wrong type of solution (i.e. getting the cask down safely
    // instead of finding the way round.)

    jumpdeath = {self.jump; die(); return nil;}
    bulkdeath(tx) = {self.bulk(tx); die(); return nil;}
;

/* 232-234 */
poling_messages: object
    calm={"%You% %have% poled %your% boat across the calm water.";P();}
    dark={"%You% %have% poled %your% boat across the dark water.";P();}
    blue={"%You% %have% poled %your% boat across the Blue Grotto.";P();}

;
/* 235 */
Dantes_Rest: darkroom
    game551 = true
    sdesc = "At Dante's Rest"
    ldesc = {I();
    "%You're% at Dante's Rest, on the north side of a yawning dark chasm.
    A passage continues west along the chasm's edge.";}

    cross = {
        local actor := getActor(&travelActor);
        Decrepit_Bridge.doCross(actor); 
        return nil;
    }
    south =  {return self.cross;}
    across =  {return self.cross;}
    bridge =  {return self.cross;}
    passage = In_Misty_Cavern
    west = In_Misty_Cavern
    NPCexit1 = {
         if (Decrepit_Bridge.isfallen) return nil;
         else return Devils_Chair;
    }
    jump = {
        if (not Decrepit_Bridge.isfallen) {
            "I respectfully suggest you go across the
            bridge instead of jumping.";

            return nil;
        }
        else
            return didnt_make_it.death;
    }
    exithints = [ Devils_Chair, &south ]
    listendesc = {
        if (global.oldgame) pass listendesc;
        global.listenadd := true;
        inherited.listendesc;
        P(); "You hear a distant roar, like the sound of a fast-flowing
             river, from the depths of the chasm. ";
    }
;
/* 236 */
Lost_Canyon_E: darkroom
    game551 = true
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At East End of Lost River Canyon"
    ldesc = {I();
    "%You% %are% at the east end of a riverbank path in Lost River
    Canyon.";}

    west = Lost_Canyon_S
    downstream = Lost_Canyon_S // Added - DJP
    upstream = {"The path ends here and you can't go any further
        upstream. "; return nil;} // Added - DJP
;

/* 238 */
Pantry: room, Internal_Room, Outside, Indoor
    game551 = true
    // this really a nested room - from the outside it looks like a
    // container, and from the inside it looks like a room.  Objects
    // are visible only after the room has been entered or looked into.
    multisdesc = "caretaker's pantry"
    sdesc = "In the Caretaker's Pantry"
    inldesc = {I(); "%You're% in the Caretaker's Pantry.";}
    outldesc = "It's a small room on the north side of the building. "
    noun = 'pantry' 'doorway'
    adjective = 'small' 'caretaker\'s'

    location = Inside_Building

    isopen = (self.isseen or self.islookedin)
    // non-large objects can be placed inside the pantry, but the vase
    // breaks unless the pillow has been placed there.
    verIoPutIn(actor) = {}
    ioPutIn( actor, dobj ) = {
        if (dobj.islong) {
            caps(); dobj.thedesc; " is too long to go into that little
            pantry. ";
            return;
        }
        else if ((dobj.ishuge or dobj.islarge) and not (dobj = sack)) {
            caps(); dobj.thedesc; " is too large to go into that little
            pantry. ";
            return;
        }
        else inherited.ioPutIn(actor,dobj);
        if(itemcnt(self.contents) > 0 and not self.contentsVisible) {
            P(); "%You% notice%s% that the pantry now contains ";
            listcont(self); ". "; P();
        }
            self.islookedin := true;
    }
    verDoEnter(actor) = {
        if(actor.isIn(self)) {
            "%You% are already in ";self.thedesc;". ";
        }
    }
    // Special method for entering the pantry.  We exclude most large/long 
    // items, mainly to make the player drop the chest in the building, not in
    // here.  (The pillow and vase can be dropped here and will get the
    // full score if you do so, because the pantry is located in the
    // building.)
    enter(actor) = {
        local i,o,l;
        if(actor.isIn(self)) {
            "%You're% already in the pantry. ";
            return nil;
        }
        l := length(actor.contents);


        for (i := 1; i <= l; i++) {
            o := actor.contents[i];
            // bar most large/long objects from the pantry
            if (o.islong) {
                caps(); o.thedesc; " is too long to go with %you% into
                the pantry.  %You%'ll have to drop it first. ";
                return nil;
            }
            else if ((o.islarge or o.ishuge) and not (o = sack)) {
                caps(); o.thedesc; " is too large to go with %you% into
                the pantry.  %You%'ll have to drop it first. ";
                return nil;
            }
        }
        return self;
    }
    doEnter(actor) = {
        actor.travelTo(self.enter(actor));
    }
    doSynonym('Enter') = 'Gothrough'

    verDoOpen(actor) = {
        "It's already open. ";
    }

    verDoClose(actor) = {
        "The pantry is entered through an open doorway, and there's no
        way to close it. ";
    }

    out = Inside_Building
    entrance = Inside_Building
    south = Inside_Building

    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true
    // Methods for looking into and searching the pantry.
    verDoLookin(actor) = {}
    doSearch(actor) = {
        self.islookedin := true;
        pass doSearch;
    }

;
/* 239 */
Rise_Over_Bay: room, Outside
    game551 = true
    sdesc = "On a Small Rise Over the Bay"
    ldesc = {I();
    "%You% %are% on a small rise overlooking a beautiful bay. In the center
    of the bay is the castle of the elves.";}

    north = Sandy_Beach
    ne = Sandy_Beach
    smichel = Castle_Pinnacle
    building = At_End_Of_Road
    myhints = [Castlehint]
;
Castle: fixeditem, distantItem
    game551 = true
    sdesc = "The Castle of the Elves"
    ldesc = "It looks like a fairy-tale \"castle in the air.\"  It's in
            the middle of the bay, and I can see no obvious way to get to
            it!"
    noun = 'castle'
    verDoEnter = {"You'll have to tell me how to do that.";}
    verDoBoard(actor) = {self.verDoEnter(actor);}
    location = Rise_Over_Bay
;

/* 240 */
Castle_Pinnacle: room, Outside
    game551 = true
    sdesc = "On Castle Pinnacle"
    ldesc = {I();
    "%You% %are% on the highest pinnacle of the castle in the bay.
    Steps lead down into the garden.";}

    ne = Rise_Over_Bay
    across = Rise_Over_Bay
    cross = Rise_Over_Bay
    smichel = Rise_Over_Bay
    down = Outer_Courtyard
;

CastleSteps: floatingdecoration, surface
    game551 = true
    isThem = true
    sdesc = "steps"
    adesc = "steps"
    ldesc = "Stone steps lead from the top of the tower to the outer 
        courtyard of the garden. "
    loclist = [Castle_Pinnacle, Outer_Courtyard]
    noun = 'steps'
    adjective = 'stone'
;

Pinnacle_Garden: fixeditem, distantItem
    sdesc = "garden" // as seen from above
    ldesc = "Below you, to the west of an outer courtyard, lies a 
    multicolored maze of hedges.  The colors appear to be produced by leaves,
    flowers and fruits, but look unnaturally brilliant and seem to dance
    around as you look at them.  To the far west you see an inner courtyard, 
    in the center of which grows a strange tree with silvery bark and shiny 
    metallic leaves. "

    noun = 'garden' 'hedges' 'maze' 'courtyard' 'courtyards'
    location=Castle_Pinnacle
;


/* 241 */
Outer_Courtyard: room, Outside
    game551 = true
    sdesc = "In Outer Courtyard"
    ldesc = {I();
    "%You% %are% in the outer courtyard of the garden of the elves.
    Steps lead up to the tower, and to the west, separating you
    from the inner courtyard, is a maze of hedges, living things,
    but almost crystalline in their multicolored splendor. ";}

    up = Castle_Pinnacle
    west = Living_Maze_1
    phleece = {
        // only if bracelet is worn.  The bracelet is
        // available only in the 701+ point version.
        local actor := getActor(&travelActor);
        actor.travelTo(Castle_Room); 
        P(); I();
        "You hear a bustle of activity from the rooms around you - meals being
        prepared, numerous conversations in Elvish, footsteps on stairs ...
        then an abrupt silence, as if a switch has been thrown.  The lighting
        appears to dim slightly at the same time. ";
    }
;

CastleTower: fixeditem
    sdesc = "tower"
    ldesc = "An octagonal stone tower, the top of which is the highest 
        pinnacle of the castle. "
    noun = 'tower' 'pinnacle'
    adjective = 'stone' 'octagonal'
    location = Outer_Courtyard
;

Courtyard: fixeditem
    sdesc = "courtyards and garden"
    ldesc = "The courtyards and garden form an octagonal area, bounded by the
    stone walls of the castle. "
    location = Outer_Courtyard
    noun = 'courtyard' 'garden'
    adjective = 'outer' 'inner'
;

/* 242 */
Living_Maze_1: room, Outside
    game551 = true
    sdesc = "In Living Maze (red berries)"
    ldesc = {I();
    "From the inside the maze looks like a kaleidoscope, with
    swatches of color dancing as you move. In this part the colors
    are produced by shining red berries on the branches.";}

    // actor.kaleid implements the 'kaleidoscope code' in the original
    // version.  Here the implementation is simple: global.kaleid is
    // set to true whenever you enter the first room, but false if
    // you make a wrong move.
    enterRoom(actor) = {
        actor.kaleid := true;
        pass enterRoom;
    }
    east = Outer_Courtyard
    sw = Living_Maze_2
    west = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; return Living_Maze_3;
    }
    nw = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; return Living_Maze_5; 
    }

;
Redberries: fixeditem
    game551 = true
    sdesc = "red berries"
    adesc = "red berries"
    ldesc = "They look very attractive, but they're probably deadly
        poisonous.  I'd leave them alone if I were you."
    noun = 'berries'
    adjective = 'red'
    location = Living_Maze_1
    leavealone = "I think these berries are poisonous, so I suggest that
        you leave them alone."
    verDoPick(actor) = {self.leavealone;}
    verDoTake(actor) = {self.leavealone;}
    verDoEat(actor) = {self.leavealone;}
    verDoPutIn(actor,io) = {self.leavealone;}
;
/* 243 */
Living_Maze_2: room, Outside
    game551 = true
    sdesc = "In Living Maze (orange flowers)"
    ldesc = {I();
    "%You% %are% surrounded by a tall hedge with sharp iridescent leaves
    and metallic orange flowers.";}
    ne = Living_Maze_1
    north = Living_Maze_3
    nw = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_6;
    }
;
/* 244 */
Living_Maze_3: room, Outside
    game551 = true
    sdesc = "In Living Maze (yellow leaves)"
    ldesc = {I();
    "%You% %are% in the center of the living maze. The plants here are
    dormant this season, but still carry brilliant yellow leaves.";}

    east = Living_Maze_1
    south = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_2;
    }
    west = Living_Maze_4
    north = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_5;
    }

;
/* 245 */
Living_Maze_4: room, Outside
    game551 = true
    sdesc = "In Living Maze (green leaves)"
    ldesc = {I();
    "Unlike the other areas of the hedge system, this area seems to
    have no metallic gleam; nevertheless it is still breathtaking.
    The trees and bushes are all variegated shades of green, the
    evergreens being a rich dark shade while the seasonal bushes
    are a lighter yellowish green, making a startling contrast.";}

    south = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil;  
        return Living_Maze_2;
    }
    east = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_3;
    }
    north = Living_Maze_5
    west = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_6;
    }
;
/* 246 */
Living_Maze_5: room, Outside
    game551 = true
    sdesc = "Near Edge of Maze (blueberries)"
    ldesc = {I();
    "%You% %are% near the edge of the maze. There are delicious-looking
    blueberries on the bushes.  %You're% tempted to sample them!";}

    se = Living_Maze_1
    south = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_4;
    }
    sw = Living_Maze_6

;
Blue2:  Blueberries
    game551 = true
    location = Living_Maze_5
;
/* 247 */
Living_Maze_6: room, Outside
    game551 = true
    sdesc = "Western Edge of Maze (violets)"
    ldesc = {I();
    "%You% %are% at the western end of the living maze. Beside the
    shrubs forming the walls are tastefully planted beds of
    violets and brilliant purple pansies.
    To the west, through a small gate, is the inner garden.";}

    se = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_2;
    }
    east = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_4;
    }
    ne = {
        local actor := getActor(&travelActor);
        actor.kaleid := nil; 
        return Living_Maze_5;
    }
    west =  {
        local destno;
        local actor := getActor(&travelActor);
        if (actor.kaleid) return Courtyard_Gate;
        else {
            // Open the gate for consistency
            if(not Courtyard_Gate.isopen) {
                "\n(Opening << Courtyard_Gate.thedesc >>)\n)";
                Courtyard_Gate.isopen := true;
                if (Courtyard_Gate.otherside)
                    Courtyard_Gate.otherside.isopen := true;
            }
            // Move the actor randomly to one of the first five rooms
            // of the maze
            "You get a tingling feeling as you walk through the gate,
            and ..."; P();
            destno:= rand(5);
            switch(destno) {
                case 1:
                    return Living_Maze_1;
                case 2:
                    return Living_Maze_2;
                case 3:
                    return Living_Maze_3;
                case 4:
                    return Living_Maze_4;
                case 5:
                    return Living_Maze_5;
                default:
                    "\nInternal error, destno = ";
                    say(destno);"\n";
                    return nil;
            }
        }
    }
    gate =  {return self.west;}
    in =  {return self.west;}
;

Courtyard_Gate: doorway
    sdesc = "gate"
    noun = 'gate'
    adjective = 'small'
    doordest = Inner_Courtyard
    otherside = Inner_Gate
    location = Living_Maze_6
;

/* 248 */
Inner_Courtyard: room, Outside
    game551 = true
    sdesc = "In Inner Courtyard"
    ldesc = {I();
    "%You% %are% in the inner garden of the elves. In the center is
    a living tree, with shimmering silvery bark, glistening metallic
    green leaves, and flowers ripe with nectar. As the nectar falls
    to the ground it forms droplets of silver. Around the tree is
    a hedge of briars which cannot be crossed. Unfortunately for
    adventurers such as you, most of the nectar falls inside the hedge.
    The exit is to the east.";}

    east = Inner_Gate
    gate = Inner_Gate
    out = Inner_Gate
    cross = {"You can't cross the hedge of briars."; return nil;}
    in = {return self.cross;}
;

Inner_Gate: doorway
    sdesc = "gate"
    noun = 'gate'
    adjective = 'small'
    doordest = Living_Maze_6
    otherside = Courtyard_Gate
    location = Inner_Courtyard
;

Briars: fixeditem
    sdesc = "hedge of briars"
    ldesc = "The shoulder-high hedge is about three foot wide and is
        full of razor-sharp thorns.   You'd be torn to shreds if you
        tried to force your way through."
    noun = 'hedge' 'briars' 'thorns'
    adjective = 'hedge'
    verDoCross(actor) = {
        "The thorns are razor sharp.  You can't cross the hedge of
        briars."; return nil;
    }
    location = Inner_Courtyard
    verDoJump(actor) = {
        "A top Olympic athlete might be able to get over the hedge that
        way, but it's unlikely that you could.  You'd be torn to
        shreds in the attempt.";
    }
    verDoKick(actor) = {
        "Those thorns look very nasty.  Your legs would be torn to
         shreds.";
    }
    verDoCutWith(actor,io) = {
        if(isclass(io, weapon)) {
            "You slash <<self.thedesc>> vigorously with <<io.thedesc>>.
            This has no effect whatsoever on <<self.thedesc>>, but you
            have noticeably blunted the edge of your weapon.  I'd
            give up trying to cross the hedge if I were you. ";
        }
        else
            "I don't see how you could cut a hedge with <<io.thedesc>>.";
    }
    verDoAttackWith(actor,io) = {self.verDoCutWith(actor,io);}
;
Living_Tree: fixeditem, distantItem, upclimbable
    sdesc = "living tree"
    ldesc = "It's the most remarkable tree you've ever seen.
        Unfortunately there is no way to cross the hedge to get a closer
        look."
    noun = 'tree'
    adjective = 'living'
    location = Inner_Courtyard
    verDoClimb(actor) = {"You might be able to climb this tree, but
        you can't cross the hedge to get to it.";}
;
Nectar: fixeditem, distantItem
    sdesc = "nectar"
    ldesc = "I've already told you all I know about it."
    noun = 'nectar'
    location = Inner_Courtyard
;
Castle_Walls: rfd
    sdesc = "walls"
    noun = 'wall' 'walls'
    ldesc = {
        local actor := getActor(&travelActor);
        "They enclose an octagonal area, and look like normal
        stone-built walls.  ";
        if(actor.isIn(Outer_Courtyard))
            "There is something odd about them, though - I can't
            see a door anywhere! ";
    }
    adjective = 'castle'
    loclist = [
        Castle_Pinnacle Outer_Courtyard
    ]
    doCount(actor) = {"The walls form an octagonal shape - so there are
        eight of them. ";}
;

Hedges: rfd
    isThem = true
    sdesc = "hedges and their leaves and flowers"
    ldesc = "Sections of vivid color are produced by leaves, flowers and
    fruits. "
    noun = 'hedge' 'hedges' 'trees' 'flowers' 'violets' 'pansies' 'leaves'
           'maze' 'garden'
    adjective = 'multicolored'
    loclist = [
        Outer_Courtyard Living_Maze_1 Living_Maze_2
        Living_Maze_3 Living_Maze_4 Living_Maze_5 Living_Maze_6
    ]
    doCount(actor) = {"You quickly give up the attempt to count the
        multicolored hedges. ";}
;

Castle_Room: room, Outside, Indoor
    game551 = true
    sdesc = "Octagonal Castle Room"
    ldesc = { 
        I();
        "You're in an a large octagonal room with shiny white marble walls.
        Doorways, each marked with a sign in Elvish, lead out in all compass 
        directions.  ";
    }
    north = {
        local actor := getActor(&travelActor);
        "As you approach the doorway, a loud buzzer sounds and your bracelet
        glows red.   (I guess the Elves' security systems are better than
        we thought).  You then get a strange tinging
        sensation, and you find yourself... ";
        P();
        // move any objects which the player dropped
        room_move(self,At_End_Of_Road);
        actor.travelTo(At_End_Of_Road);
        return nil;
    }
    south = {return self.north;}
    east = {return self.north;}
    west = {return self.north;}
    nw = {return self.north;}
    sw = {return self.north;}
    ne = {return self.north;}
    se = {return self.north;}
    phleece = Outer_Courtyard

    listendesc = "There's nothing but a strange silence.  Something isn't
        quite right here ... ";
;
