/* Copyright (c) 1998 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  attach.t - attachable object class
Function
  Defines an object that can be connected to other objects.

  To connect two objects:

     connect/attach x to y   [synonyms in adv.t]
     fasten x to y
     put x on y

  To disconnect two objects that are connected:

     disconnect/detach x from y   [synonyms in adv.t]
     disconnect/detach x          [synonyms in adv.t]
     unfasten x from y
     unfasten x
     take x off of y/remove x from y  [synonyms in adv.t]

  Attachment is symmetrical: x and y in the above commands are generally
  interchangeable.
Notes
  
Modified
  09/01/98 MJRoberts  - Creation
*/

class attachableItem: item
    /*
     *   Determine if this object is attached to the given object.
     *   Returns true if so, nil if not.  
     */
    isAttachedTo(obj) = { return (find(self.attachmentList, obj) != nil); }

    /*
     *   Display a sentence listing an objects attachments, or say nothing
     *   at all if it's not attached to anything. 
     */
    attachmentDesc =
    {
        local cnt;
        
        cnt := length(self.attachmentList);
        if (cnt != 0)
        {
            local i;

            "Attached to <<self.itobjdesc>> <<cnt = 1 ? "is" : "are">> ";
            for (i := 1 ; i <= cnt ; ++i)
            {
                self.attachmentList[i].adesc;

                if (cnt > 1 && i + 1 <= cnt)
                {
                    if (i + 1 < cnt)
                        ", ";
                    else if (cnt = 2)
                        " and ";
                    else
                        ", and ";
                }
            }
            ". ";
        }
    }
    
    /* 
     *   List of attached objects.  If the object is initially attached to
     *   another object, initialize the list to include the attached
     *   object, and initialize the attached object's list to include this
     *   object. 
     */
    attachmentList = []

    /*
     *   To be overridden: verify that the given object can be attached to
     *   this object.  If the attachment is allowed, this should simply
     *   return true.  If not, this should display an appropriate error
     *   message and return nil.  
     */
    verifyAttachTo(actor, otherObj) =
    {
        /* make sure I'm not already attached to this object */
        if (self.isAttachedTo(otherObj))
        {
            "\^<<self.thedesc>> is already attached
            to <<otherObj.thedesc>>. ";
            return nil;
        }

        /* we have no objections */
        return true;
    }

    /*
     *   To be overridden: process attaching this object to another
     *   object, updating this object's state accordingly.  This should
     *   not generally need to update the other object's state, since this
     *   same method will be called in the other object either before or
     *   after it's called in this object (depending on how the user
     *   phrased the command).
     *   
     *   isFirst is true for the first object of the pair that this method
     *   is called on, nil for the second.
     *   
     *   This method must minimally be overridden to provide a message for
     *   the attachment.  The overrider must coordinate message generation
     *   so that only one message is generated, since this method will be
     *   called in both affected objects.  Generally, the object that has
     *   more specialized behavior on attachment should display the
     *   message (in other words, if you have one object that can be
     *   connected to a number of other objects with varying results, the
     *   various other objects should display the messages).  When the
     *   behavior is entirely symmetrical so there is no clear preference,
     *   the isFirst argument can be used - simply display the message
     *   when isFirst is true, and suppress the message when isFirst is
     *   nil.  
     */
    processAttachTo(actor, otherObj, isFirst) =
    {
        /* add this object to my list of attached objects */
        self.attachmentList += otherObj;
    }

    /*
     *   To be overridden: verify that the given object can be detached
     *   from this object.  If the detachment is allowed, this should
     *   simply return true.  If not, this should display an appropriate
     *   error message and return nil. 
     */
    verifyDetachFrom(actor, otherObj) =
    {
        /* if I'm not attached to the given object, we can't detach */
        if (!self.isAttachedTo(otherObj))
        {
            "\^<<self.thedesc>> isn't attached to <<otherObj.thedesc>>. ";
            return nil;
        }

        /* looks okay */
        return true;
    }

    /*
     *   To be overridden: process detachment from the given object,
     *   updating my state appropriately.  This should only update this
     *   object, not the other object, since this same method will be
     *   called in the other object either before or after it's called in
     *   this object.
     *   
     *   This routine should be overridden to provide a message in one
     *   (but not both) of the involved objects.  The isFirst argument can
     *   be used the same way as in processAttachTo to determine where the
     *   message is generated, if necessary.  
     */
    processDetachFrom(actor, otherObj, isFirst) =
    {
        /* remove this object from my attachment list */
        self.attachmentList -= otherObj;
    }

    verIoAttachTo(actor) = { }
    verDoAttachTo(actor, iobj) =
    {
        /* verify that I can connect to the other object */
        if (!self.verifyAttachTo(actor, iobj))
            return;

        /* verify that the other object can connect to me */
        iobj.verifyAttachTo(actor, self);
    }

    ioAttachTo(actor, dobj) =
    {
        /* process attachment from my perspective */
        self.processAttachTo(actor, dobj, true);

        /* process attachment from the other object's perspective */
        dobj.processAttachTo(actor, self, nil);
    }

    verIoDetachFrom(actor) = { }
    verDoDetachFrom(actor, iobj) =
    {
        /* verify that I can be detached from the other object */
        if (!self.verifyDetachFrom(actor, iobj))
            return;

        /* verify that the other object can detach from me */
        iobj.verifyDetachFrom(actor, self);
    }

    ioDetachFrom(actor, dobj) =
    {
        /* process detachment from my perspective */
        self.processDetachFrom(actor, dobj, true);

        /* process detachment from the other object's perspective */
        dobj.processDetachFrom(actor, self, nil);
    }

    verDoDetach(actor) =
    {
        switch(length(self.attachmentList))
        {
        case 0:
            "It's not attached to anything. ";
            break;

        case 1:
            /* 
             *   Treat this the same as "detach from list[1]".  This
             *   assumes that the attached object is valid as an object in
             *   a command, since we skip the normal validation the parser
             *   does; we have to count on the verifyDetachFrom method to
             *   enforce appropriate validity rules if an object can be
             *   accessible at the same time an object attached to it is
             *   inaccessible.  This should be a rare situation in any
             *   case.  
             */
            self.verDoDetachFrom(actor, self.attachmentList[1]);
            break;

        default:
            "You will have to be more specific about what you
            want to detach <<self.thedesc>> from. ";
            break;
        }
    }

    doDetach(actor) =
    {
        /* 
         *   because of the verification tests, we should never get here
         *   unless we have exactly one item in the attachment list - act
         *   as though they explicitly asked us to detach it from that
         *   object 
         */
        self.ioDetachFrom(actor, self.attachmentList[1]);
    }

    /* "put x on y" is a synonym for "connect x to y" */
    ioSynonym('AttachTo') = 'PutOn'
    doSynonym('AttachTo') = 'PutOn'

    /* "take x off of y" is a synonym for "detach x from y" */
    ioSynonym('DetachFrom') = 'TakeOff' 'TakeOut'
    doSynonym('DetachFrom') = 'TakeOff' 'TakeOut'

    /* "unfasten x" is a synonym for "detach x" */
    doSynonym('Detach') = 'Unfasten'
;

