
// This interface implements the sound manager functionality.

#ifndef __ILTSOUNDMGR_H__
#define __ILTSOUNDMGR_H__

	// Base sound mgr interface.  (shared between client and server)
	class ILTSoundMgr
	{
	public:

		// Play a sound with full control
		// Arguments:
		//		pPlaySoundInfo - Sound control structure.
		//		hResult - (return) Newly created sound handle.
		// Returns:
		//		LT_OK - Successful.
		//		LT_ERROR - Unable to play sound.
		//		LT_INVALIDPARAMS - pPlaySoundInfo is invalid
		//		LT_MISSINGFILE - Sound file was not found
		// Notes:
		// Specifying the PLAYSOUND_GETHANDLE flag in pPlaySoundInfo will fill m_hSound
		// with the handle to the client sound, and prevent the sound from being automatically
		// deleted when the sound is done playing.  If it is specified, you should therefore
		// call KillSound.
		virtual LTRESULT	PlaySound( PlaySoundInfo *pPlaySoundInfo, HLTSOUND &hResult ) = 0;

		// Get total length in seconds of sound.
		// Arguments:
		//		hSound - Handle to sound.
		//		fDuration - (return) Duration of sound.
		// Returns:
		//		LT_OK - Successful.
		//		LT_INVALIDPARAMS - hSound not available.
		virtual LTRESULT	GetSoundDuration( HLTSOUND hSound, LTFLOAT &fDuration ) = 0;

		// Check if sound finished playing or if object it was attached to was removed.
		// Arguments:
		//		hSound - Handle to client only sound
		//		bDone - (return) Finished state of sound
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - hSound is invalid
		virtual LTRESULT	IsSoundDone( HLTSOUND hSound, LTBOOL &bDone ) = 0;

		// Kill a sound.
		// Arguments:
		//		hSound - Handle to client only sound
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - hSound is invalid or not playing
		virtual LTRESULT	KillSound( HLTSOUND hSound ) = 0;

		// Kills a looping sound.  Sound will continue to play until it reaches
		// the end, then remove itself.
		// Arguments:
		//		hSound - Handle to client side sound.
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - hSound is invalid or not playing
		virtual LTRESULT	KillSoundLoop( HLTSOUND hSound ) = 0;

	protected:
		ClientServerType	m_ClientServerType; // Tells if this is on the client or server.
	};


	struct InitSoundInfo
	{
		// Name of 3d provider to use
		char						m_sz3DProvider[_MAX_PATH+1];

		// Number of sw voices
		uint8						m_nNumSWVoices;

		// Number of 3D voices
		uint8						m_nNum3DVoices;

		// Output sound format.  Sample rate (8000, 11025, 22050 or
		// 44100 kHz), Bits per sample (8 or 16)...
		unsigned short				m_nSampleRate;
		unsigned short				m_nBitsPerSample;

		// Use INITSOUNDINFOFLAG_xxx flags
		unsigned long				m_dwFlags;

		// Engine fills in this parameter with INITSOUNDINFORESULTS_xxx after InitSound is called.
		unsigned long				m_dwResults;

		// Initial volume (0-100)
		unsigned short				m_nVolume;

		// Distance factor in meters/game unit
		float						m_fDistanceFactor;

		// Initialize the InitSoundInfo structure to the default values...
		inline void Init()
			{
				m_sz3DProvider[0]	= 0;									
				m_nNumSWVoices		= 32;									
				m_nNum3DVoices		= 0;									
				m_nSampleRate		= 22050;								
				m_nBitsPerSample	= 16;									
				m_dwFlags			= 0;									
				m_nVolume			= 100;									
				m_fDistanceFactor	= 1.0f;
			}
	};

	// Client-specific sound mgr interface.
	class ILTClientSoundMgr : public ILTSoundMgr
	{
	public:
		// Gets a list of the 3d sound providers available on system, to be
		// used in the InitSoundInfo structure.
		// Arguments:
		//		pSound3DProviderList - (return) The list of available 3D sound providers
		//		bVerify - Tell the engine not to report providers that aren't 
		//			completely supported on the system.  This takes longer and causes 
		//			speaker popping.  
		// Returns:
		//		LT_OK - Successful
		// Notes:
		//		Games should only need  to do this when a different provider 
		//		is chosen.  EAX support can only be checked when the provider
		//		is opened, so without the bVerify, the SOUND3DPROVIDER_CAPS_REVERB 
		//		cannot be set.  InitSound will report this info in the 
		//		m_dwResults flags.  !!IMPORTANT!! Be sure to release the list 
		//		with ReleaseSound3DProviderList.
		virtual LTRESULT	GetSound3DProviderLists( Sound3DProvider *&pSound3DProviderList, LTBOOL bVerify ) = 0;
		// Releases a list of 3D sound providers generated by GetSound3DProviderList
		// Arguments:
		//		pSound3DProviderList - The 3D sound provider list to be released
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - An invalid provider list was provided
		virtual LTRESULT	ReleaseSound3DProviderList( Sound3DProvider *pSound3DProviderList ) = 0;

		// Initializes the sound driver.
		// Arguments:
		//		pSoundInfo - The initialization parameters
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - pSoundInfo is invalid
		//		LT_ERROR - Unable to initialize the sound driver
		virtual LTRESULT	InitSound( InitSoundInfo *pSoundInfo ) = 0;

		// Retrieve the global volume 
		// Arguments:
		//		nVolume - (return) Current volume level (0 - 100)
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	GetVolume( uint16 &nVolume) = 0;
		// Set the global volume
		// Arguments:
		//		nVolume - Desired volume level (0 - 100)
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	SetVolume( uint16 nVolume ) = 0;

		// Controls the reverb properties.
		// Arguments:
		//		pReverbProperties - The desired reverb properties
		// Returns:
		//		LT_OK - Successful
		virtual	LTRESULT	SetReverbProperties( ReverbProperties *pReverbProperties ) = 0;
		// Retrieves the current reverb properties
		// Arguments:
		//		pReverbProperties - (return) Current reverb properties
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	GetReverbProperties( ReverbProperties *pReverbProperties ) = 0;

		// Controls the sound occlusion level
		// Arguments:
		//		hSound - Handle to the client only sound.
		//		fLevel - Level of occlusion
		// Returns:
		//		LT_OK - Successful
		virtual	LTRESULT	SetSoundOcclusion( HLTSOUND hSound, LTFLOAT fLevel ) = 0;
		// Retrieves the occlusion level of a sound
		// Arguments:
		//		hSound - Handle to the client only sound.
		//		pLevel - Variable to return the level of occlusion into
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	GetSoundOcclusion( HLTSOUND hSound, LTFLOAT *pLevel ) = 0;

		// Controls the sound obstruction level
		// Arguments:
		//		hSound - Handle to the client only sound.
		//		fLevel - Level of obstruction
		// Returns:
		//		LT_OK - Successful
		virtual	LTRESULT	SetSoundObstruction( HLTSOUND hSound, LTFLOAT fLevel ) = 0;
		// Retrieves the occlusion level of a sound
		// Arguments:
		//		hSound - Handle to the client only sound.
		//		pLevel - Variable to return the level of obstruction into
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	GetSoundObstruction( HLTSOUND hSound, LTFLOAT *pLevel ) = 0;

		// Sets a sound's filter (Applies only to 2D non-streaming sounds)
		// Arguments:
		//		hSound - Handle to the sound
		//		pFilter - Name of the filter
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - hSound is invalid or does not support filters
		//		LT_NOTFOUND - Unknown filter requested
		virtual LTRESULT	SetSoundFilter( HLTSOUND hSound, const char *pFilter ) = 0;

		// Sets a parameter of a filter
		// Arguments:
		//		hSound - Handle to the sound
		//		pParam - Name of the parameter
		//		fValue - Value of the parameter
		// Returns:
		//		LT_OK - Successful
		//		LT_ERROR - hSound is invalid or does not support filters
		//		LT_NOTFOUND - Unknown parameter requested
		virtual LTRESULT	SetSoundFilterParam( HLTSOUND hSound, const char *pParam, float fValue ) = 0;

		// Gets a parameter of a filter
		// Arguments:
		//		hSound - Handle to the sound
		//		pParam - Name of the parameter
		//		pValue - Value of the parameter (return)
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - pValue is NULL.
		//		LT_ERROR - hSound is invalid or does not support filters
		//		LT_NOTFOUND - Unknown parameter requested
		virtual LTRESULT	GetSoundFilterParam( HLTSOUND hSound, const char *pParam, float *pValue ) = 0;

		// Get a filter name
		// Arguments:
		//		nIndex - Index of the filter
		//		pFilter - Name of the filter (return)
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - pFilter is NULL
		//		LT_NOTFOUND - Unknown index requested
		virtual LTRESULT	GetFilterName( uint32 nIndex, const char **pFilter ) = 0;

		// Get a filter parameter name
		// Arguments:
		//		nIndex - Index of the filter
		//		pFilter - Name of the filter
		//		pParam - Name of the parameter (return)
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - pFilter or pParam is NULL
		//		LT_NOTFOUND - Unknown index or filter requested
		virtual LTRESULT	GetFilterParamName( uint32 nIndex, const char *pFilter, const char **pParam ) = 0;

		// Update position and orientation of a client only sound.
		// Arguments:
		//		hSound - Handle to client only sound.
		//		pPos - New position of sound. Can be NULL.
		// Returns:
		//		LT_OK - Successful.
		//		LT_INVALIDPARAMS - Invalid parameters.
		//		LT_ERROR - Unable to find hSound
		virtual LTRESULT	SetSoundPosition( HLTSOUND hSound, LTVector *pPos ) = 0;

		// Get current position and orientation of a client only sound.
		// Arguments:
		//		hSound - Handle to client only sound.
		//		pPos - Destination of position. Can be NULL.
		// Returns:
		//		LT_OK - Successful.
		//		LT_INVALIDPARAMS - Invalid parameters.
		//		LT_ERROR - Unable to find hSound.
		virtual LTRESULT	GetSoundPosition( HLTSOUND hSound, LTVector *pPos ) = 0;

		// Pause/resume sounds.
		// Returns:
		//		LT_OK - Successful
		//		LT_ERROR - Unable to pause/resume
		virtual LTRESULT	PauseSounds( ) = 0;
		virtual LTRESULT	ResumeSounds( ) = 0;

		// Set the listener status, position and orientation.  
		// Arguments:
		//		bListenerInClient - If TRUE, then pPos and pRot are ignored and 
		//			can be set to NULL.
		//		pPos - Position of the listener
		//		pRot - Rotation of the listener
		//		bTeleport - Consider the movement a teleportation (don't apply velocity)
		// Returns:
		//		LT_OK - Successful
		virtual LTRESULT	SetListener( LTBOOL bListenerInClient, LTVector *pPos, LTRotation *pRot, LTBOOL bTeleport ) = 0;

		// Notifies the sound manager that the client is no longer going to use the handle
		// and the sound may therefore be freed whenever it's done playing
		// Arguments:
		//		hSound - Handle to client side sound.
		// Returns:
		//		LT_OK - Successful
		//		LT_INVALIDPARAMS - hSound is invalid or not playing
		//		LT_ERROR - hSound doesn't have the GETHANDLE flag set
		virtual LTRESULT	ReleaseSoundHandle( HLTSOUND hSound ) = 0;
	};


#endif  // __ILTSOUNDMGR_H__
