/**
  ******************************************************************************
  * @file    BSP/Src/eeprom.c 
  * @author  MCD Application Team
  * @version V1.5.0
  * @date    14-April-2017
  * @brief   This example code shows how to manage I2C EEPROM memory
  *          =================================================================== 
  *          Notes:
  *           - This driver is intended for STM32F107xC families devices only. 
  *           - The I2C EEPROM memory (M24CXX) is available is available directly on
  *             STM3210C-EVAL RevC board.
  *          ===================================================================
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup STM32F1xx_HAL_Examples
  * @{
  */

/** @addtogroup BSP
  * @{
  */ 

/* Private typedef -----------------------------------------------------------*/
typedef enum {FAILED = 0, PASSED = !FAILED} TestStatus; 
/* Private define ------------------------------------------------------------*/
#define EEPROM_FEATURES_NUM     2
#define EEPROM_TEST_NB          1
#define BUFFER_SIZE1            (countof(Tx1Buffer)-1 + 17)
#define EEPROM_WRITE_ADDRESS1   0x50
#define EEPROM_READ_ADDRESS1    0x50
/* Private macro -------------------------------------------------------------*/
#define countof(a) (sizeof(a) / sizeof(*(a)))
/* Private variables ---------------------------------------------------------*/
static uint8_t EepromFeature = 0;
static uint8_t EepromSelected[EEPROM_TEST_NB] = {BSP_EEPROM_M24C64_32};
static uint8_t EepromTestNb = 0;
uint8_t EepromDetected = 1;
/* Private function prototypes -----------------------------------------------*/
static void EEPROM_SetHint(void);
static void EEPROM_Show_Feature(uint8_t feature);
static TestStatus Buffercmp(uint8_t* pBuffer1, uint8_t* pBuffer2, uint16_t BufferLength);

extern __IO uint8_t NbLoop;
/* Private functions ---------------------------------------------------------*/

/**
  * @brief  EEPROM Demo
  * @param  None
  * @retval None
  */
void EEPROM_demo (void)
{ 
  EEPROM_SetHint();
  EepromFeature = 0;
  EepromTestNb = 0;

  /* Select EEPROM Component */
  BSP_EEPROM_SelectDevice(EepromSelected[EepromTestNb++]);
  EEPROM_Show_Feature(EepromFeature++);

  while (1)
  {
    if(CheckForUserInput() > 0)
    {
      if(EepromFeature < EEPROM_FEATURES_NUM)
      {
       EEPROM_Show_Feature(EepromFeature++); 
      }
      else
      {
        return;
      }
      
      if(EepromTestNb < EEPROM_TEST_NB)
      {
        /* Select another EEPROM */
        BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
        BSP_LCD_DisplayStringAt(0, 205, (uint8_t *)"Press Key push-button", CENTER_MODE); 
        BSP_LCD_DisplayStringAt(0, 220, (uint8_t *)"to select other EEPROM", CENTER_MODE); 
        BSP_EEPROM_SelectDevice(EepromSelected[EepromTestNb++]);
        
        /* Restart example feature */
        EepromFeature = 0;
      }
    }
    HAL_Delay(100);
  }
}

/**
  * @brief  Display EEPROM Demo Hint
  * @param  None
  * @retval None
  */
static void EEPROM_SetHint(void)
{
  /* Clear the LCD */ 
  BSP_LCD_Clear(LCD_COLOR_WHITE);
  
  /* Set LCD Demo description */
  BSP_LCD_SetTextColor(LCD_COLOR_BLUE);
  BSP_LCD_FillRect(0, 0, BSP_LCD_GetXSize(), 80);
  BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
  BSP_LCD_SetBackColor(LCD_COLOR_BLUE); 
  BSP_LCD_SetFont(&Font24);
  BSP_LCD_DisplayStringAt(0, 0, (uint8_t*)"EEPROM", CENTER_MODE);
  BSP_LCD_SetFont(&Font12);
  BSP_LCD_DisplayStringAt(0, 30, (uint8_t*)"This example shows the different", CENTER_MODE);
  BSP_LCD_DisplayStringAt(0, 45, (uint8_t*)"EEPROM Features, use Key push-button", CENTER_MODE); 
  BSP_LCD_DisplayStringAt(0, 60, (uint8_t*)"to start EEPROM data transfer", CENTER_MODE); 

   /* Set the LCD Text Color */
  BSP_LCD_SetTextColor(LCD_COLOR_BLUE);  
  BSP_LCD_DrawRect(10, 90, BSP_LCD_GetXSize() - 20, BSP_LCD_GetYSize()- 100);
  BSP_LCD_DrawRect(11, 91, BSP_LCD_GetXSize() - 22, BSP_LCD_GetYSize()- 102);
 }

/**
  * @brief  Show EEPROM Features
  * @param  feature : feature index
  * @retval None
  */
static void EEPROM_Show_Feature(uint8_t feature)
{
  uint8_t Tx1Buffer[] = "STM32F107xC EEPROM";
  uint8_t EEPROM_I2C[] = "I2C"; 
  uint8_t *Bus = NULL;
  
  uint8_t Rx1Buffer[BUFFER_SIZE1] = {0}; 
  uint8_t Tx2Buffer[BUFFER_SIZE1] = {0}; 
  __IO TestStatus TransferStatus1 = FAILED;
  __IO uint32_t NumDataRead = 0; 

  BSP_LCD_SetBackColor(LCD_COLOR_WHITE);  
  BSP_LCD_SetTextColor(LCD_COLOR_WHITE);    
  BSP_LCD_FillRect(12, 92, BSP_LCD_GetXSize()- 24, BSP_LCD_GetYSize() - 104);
  BSP_LCD_SetTextColor(LCD_COLOR_BLACK);

  /* Initialize the EEPROM driver --------------------------------------------*/
  if (BSP_EEPROM_Init() != EEPROM_OK)
  {
    BSP_LCD_SetTextColor(LCD_COLOR_RED);    
    BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Initialization problem", CENTER_MODE); 
    BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"Check if HW connected or", CENTER_MODE); 
    BSP_LCD_DisplayStringAt(0, 145, (uint8_t*)"HW version not supported", CENTER_MODE); 
    return;
  }
  EepromDetected = 1;
  
  switch (feature)
  {
  case 0:
    /* Read old parameter in EEPROM */
    if (EepromDetected == 1)
    {
      /* Set the Number of data to be read */
      NumDataRead = (uint32_t)BUFFER_SIZE1;
      
      /* Read from EEPROM from EEPROM_READ_ADDRESS1 */
      if (BSP_EEPROM_ReadBuffer(Rx1Buffer, EEPROM_READ_ADDRESS1, (uint32_t *)(&NumDataRead)) != EEPROM_OK)
      {
        BSP_LCD_SetTextColor(LCD_COLOR_RED);    
        BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Init issue at read old data", CENTER_MODE); 
        BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
        BSP_LCD_DisplayStringAt(0, 145, (uint8_t*)"Press again Key push-button", CENTER_MODE); 
        BSP_LCD_DisplayStringAt(0, 160, (uint8_t*)"To write new data", CENTER_MODE); 
        return;
      }
      BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"String read", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"in the current EEPROM selected:", CENTER_MODE); 
      BSP_LCD_SetTextColor(LCD_COLOR_BLUE);    
      BSP_LCD_DisplayStringAt(0, 160, Rx1Buffer, CENTER_MODE); 
      BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
      BSP_LCD_DisplayStringAt(0, 190, (uint8_t*)"Press Key push-button", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 205, (uint8_t*)"To write new data", CENTER_MODE); 
    }
    else
    {
      BSP_LCD_SetTextColor(LCD_COLOR_RED);    
      BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Problem to communicate", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"with EEPROM", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 145, (uint8_t*)"Press again Key push-button", CENTER_MODE); 
    }
    
    break;
    
  case 1:
     /* Write new parameter in EEPROM */
    if (EepromDetected == 1)
    {
      Bus = EEPROM_I2C;

      snprintf((char*)Tx2Buffer, BUFFER_SIZE1, "%s %s Ex. test %d", Tx1Buffer, Bus, NbLoop);

      /* First write in the memory followed by a read of the written data ----*/
      /* Write on EEPROM to EEPROM_WRITE_ADDRESS1 */
      if (BSP_EEPROM_WriteBuffer(Tx2Buffer, EEPROM_WRITE_ADDRESS1, BUFFER_SIZE1) != EEPROM_OK)
      {
        BSP_LCD_SetTextColor(LCD_COLOR_RED);    
        BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Init issue at write", CENTER_MODE); 
        return;
      }
      
      /* Set the Number of data to be read */
      NumDataRead = (uint32_t)BUFFER_SIZE1;
      
      /* Read from I2C EEPROM from EEPROM_READ_ADDRESS1 */
      if (BSP_EEPROM_ReadBuffer(Rx1Buffer, EEPROM_READ_ADDRESS1, (uint32_t *)(&NumDataRead)) != EEPROM_OK)
      {
        BSP_LCD_SetTextColor(LCD_COLOR_RED);    
        BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Init issue at read", CENTER_MODE); 
        return;
      }
      
      /* Check if the data written to the memory is read correctly */
      TransferStatus1 = Buffercmp(Tx2Buffer, Rx1Buffer, BUFFER_SIZE1);
      if(TransferStatus1 != FAILED)
      {
        BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"String writes", CENTER_MODE); 
        BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"in the current EEPROM selected:", CENTER_MODE); 
        BSP_LCD_SetTextColor(LCD_COLOR_BLUE);    
        BSP_LCD_DisplayStringAt(0, 160, Tx2Buffer, CENTER_MODE); 
      }
      else 
      {
        BSP_LCD_SetTextColor(LCD_COLOR_RED);    
        BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"FAILED to write!", CENTER_MODE); 
        BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"Press Key push-button", CENTER_MODE);
        BSP_LCD_DisplayStringAt(0, 145, (uint8_t*)"to end test.", CENTER_MODE);
      }
    }
    else
    {
      BSP_LCD_SetTextColor(LCD_COLOR_RED);    
      BSP_LCD_DisplayStringAt(0, 115, (uint8_t*)"Problem to communicate", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 130, (uint8_t*)"again with EEPROM", CENTER_MODE); 
      BSP_LCD_DisplayStringAt(0, 145, (uint8_t*)"Press Key push-button to end test", CENTER_MODE); 
    }
    break;
  }
}

/**
  * @brief  Basic management of the timeout situation.
  * @param  None.
  * @retval 0.
  */
void BSP_EEPROM_TIMEOUT_UserCallback(void)
{
  EepromDetected = 0;
}

/**
  * @brief  Compares two buffers.
  * @param  pBuffer1, pBuffer2: buffers to be compared.
  * @param  BufferLength: buffer's length
  * @retval PASSED: pBuffer1 identical to pBuffer2
  *         FAILED: pBuffer1 differs from pBuffer2
  */
static TestStatus Buffercmp(uint8_t* pBuffer1, uint8_t* pBuffer2, uint16_t BufferLength)
{
  while(BufferLength--)
  {
    if(*pBuffer1 != *pBuffer2)
    {
      return FAILED;
    }
    
    pBuffer1++;
    pBuffer2++;
  }

  return PASSED;  
}
/**
  * @}
  */ 
/**
  * @}
  */ 
/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
