/******************************************************************************/
/*                                  FILELIST.CPP                              */
/*                                                                            */
/*                      Simple file list management routines                  */
/*                                                                            */
/*  Sean Wilson, 2/Feb/1993                                                   */
/******************************************************************************/
#include <string.h>
#include <dir.h>

#include "filelist.h"
#include "slist.hpp"

/******************************************************************************/
/* The DirEntry class is the basic element of a linked list of files.  It is  */
/* derived from the Link class for constructing singly linked ordered Lists   */
/******************************************************************************/

class DirEntry: public Link {
    friend class DirList;               // Allow DirList to access my members

private:
    CLAUSHORT       attr;               // File attribute in Clarion format
    CLATIME         time;               // File time in Clarion format
    CLADATE         date;               // File date in Clarion format
    CLAULONG        size;               // File size in Clarion format
    char            name[13];           // File name

    int Compare(DirEntry *tryLink);     // Compare tryLink against this Link
    void MakeClaDate(unsigned DosDate); // Create a Clarion format DATE
    void MakeClaTime(unsigned DosTime); // Create a Clarion format TIME

public:
    int             order;              // Determines list order

    DirEntry(struct ffblk &FileInfo, int Order);
   ~DirEntry();

    FILESPEC EntryInfo();               // Return info as a FILESPEC

    int operator<(const void *tryLink); // Is tryLink < this Link ?
};


/** COMPARE *******************************************************************
 *
 *  Synopsis:
 *      int Compare(tryLink);
 *
 *          DirEntry   *tryLink;        Link to compare against this Link
 *
 *  Description:
 *      Compare() compares tryLink against this link using order to determine
 *      the method of comparison.
 */

int DirEntry::Compare(DirEntry *tryLink)
{   int Cmp;

    switch (order) {
    default:
    case NAMEORDER:
        Cmp = strcmp(name, tryLink->name);
        break;
    case DATEORDER:
        if (date.s.usYear > tryLink->date.s.usYear)
            Cmp = 1;
        if (date.s.usYear < tryLink->date.s.usYear)
            Cmp = -1;
        else if (date.s.ucMonth > tryLink->date.s.ucMonth)
            Cmp = 1;
        else if (date.s.ucMonth < tryLink->date.s.ucMonth)
            Cmp = -1;
        else if (date.s.ucDay > tryLink->date.s.ucDay)
            Cmp = 1;
        else if (date.s.ucDay < tryLink->date.s.ucDay)
            Cmp = -1;
        else
            Cmp = 0;
        break;
    case SIZEORDER:
        Cmp = size > tryLink->size ? 1: -1;
        break;
    }

    return Cmp;
}

/** MAKECLADATE ***************************************************************
 *
 *  Synopsis:
 *      CLADATE MakeClaDate(DosDate);
 *
 *          unsigned    DosDate;        Date in DOS format
 *
 *  Description:
 *      MakeClaDate() takes a dos format date in DosDate and returns a
 *      Clarion format date.
 */

void DirEntry::MakeClaDate(unsigned DosDate)
{   union {
        unsigned    u;
        struct {
        unsigned    day     : 5;
        unsigned    month   : 4;
        unsigned    year    : 7;
        }           s;
    }   d;

    d.u = DosDate;

    date.s.usYear  = d.s.year + 1980;
    date.s.ucMonth = d.s.month;
    date.s.ucDay   = d.s.day;
}

/** MAKECLATIME ***************************************************************
 *
 *  Synopsis:
 *      CLADATE MakeClaTime(DosTime);
 *
 *          unsigned    DosTime;        Time in DOS format
 *
 *  Description:
 *      MakeClaTime() takes a dos format time in DosTime and returns a
 *      Clarion format time.
 */

void DirEntry::MakeClaTime(unsigned DosTime)
{   union {
        unsigned    u;
        struct {
        unsigned    seconds : 5;
        unsigned    minutes : 6;
        unsigned    hours   : 5;
        }           s;
    }   t;

    t.u = DosTime;

    time.s.ucHour      = t.s.hours;
    time.s.ucMinute    = t.s.minutes;
    time.s.ucSecond    = t.s.seconds * 2;
    time.s.ucHund      = 0;
}

/** DIRENTRY ******************************************************************
 *
 *  Synopsis:
 *      DirEntry(FileInfo, Order);
 *
 *          struct ffblk &FileInfo;     File information block
 *          int           Order;        Determine file ordering
 *
 *  Description:
 *      DirEntry() creates a directory entry that will be entered into an
 *      ordered directory List.
 */

DirEntry::DirEntry(struct ffblk &FileInfo, int Order)
{   order   = Order;
    attr    = FileInfo.ff_attrib;
    size    = FileInfo.ff_fsize;

    MakeClaTime(FileInfo.ff_ftime);
    MakeClaDate(FileInfo.ff_fdate);

    strncpy(name, FileInfo.ff_name, 13);
}

/** ENTRYINFO *****************************************************************
 *
 *  Synopsis:
 *      EntryInfo();
 *
 *  Description:
 *      EntryInfo() returns the information from the current entry as a
 *      FILESPEC.
 */

FILESPEC DirEntry::EntryInfo()
{   FILESPEC Info;

    Info.attr = attr;
    Info.time = time;
    Info.date = date;
    Info.size = size;
    memmove(Info.name, name, sizeof(Info.name));

    return Info;
}

/** OPERATOR< *****************************************************************
 *
 *  Synopsis:
 *      int operator<(tryLink);
 *
 *          const void  *tryLink;        Link to compare against this Link
 *
 *  Description:
 *      operator < determines whether tryLink is less than this link using
 *      order to determine the method of comparison.
 */

int DirEntry::operator<(const void *tryLink)
{   return Compare((DirEntry *)tryLink) < 0;
}


/******************************************************************************/
/* The DirList class implements an ordered directory list.  Entries may be    */
/* ordered by name, size or date.                                             */
/******************************************************************************/

class DirList: public List {
public:

    DirList(char *Path, CLAUSHORT Attr, CLAUSHORT Order);
    DirList();

    void ReOrder(int Order);
};

CLAUSHORT FileCount;                    // How many files are in the list?


/** DIRLIST *******************************************************************
 *
 *  Synopsis:
 *      DirList(Path, Attr, Order)
 *
 *          char       *Path;           File specification
 *          CLAUSHORT   Attr;           File attributes
 *          CLAUSHORT   Order;          Determine file ordering
 *
 *  Description:
 *      DirList() creates a directory listing of files whose path matches
 *      Path and attributes are Attr.  The list is ordered according to the
 *      value of Order.
 */

DirList::DirList(char *Path, CLAUSHORT Attr, CLAUSHORT Order)
{   struct ffblk FileInfo;

    FileCount = 0;

    if (findfirst(Path, &FileInfo, Attr) != -1)
    {   do
        {   (*this) += new DirEntry(FileInfo, Order);
            FileCount++;

        }   while (findnext(&FileInfo) != -1);
    }
}


/** DIRLIST *******************************************************************
 *
 *  Synopsis:
 *      DirList()
 *
 *  Description:
 *      DirList() creates a new empty list.
 */

DirList::DirList()
{   FileCount = 0;
}

/** REORDER *******************************************************************
 *
 *  Synopsis:
 *      ReOrder(Order)
 *
 *          int     Order;              Specifies new ordering
 *
 *  Description:
 *      ReOrder() imposes a new order on the the list.  It does this by
 *      removing the elements one by one from the old list and inserting
 *      them in the new list.
 */

void DirList::ReOrder(int Order)
{   DirList     NewLst;
    DirEntry   *Entry;

    while ((Entry = (DirEntry *)(*this)--) != NULL)
    {   Entry->order = Order;
        NewLst += Entry;
    }

    head        = NewLst.head;          // Point to the new list
    NewLst.head = (Link *)0;            // Ensure the list doesn't get deleted
}

/******************************************************************************/
/* The following code does nothing more than provide entrypoints for the      */
/* Clarion code to access the functionality of the class library.  NOTE that  */
/* since Clarion performs no name-mangling and cannot access classes or their */
/* members, the API is necessarily fairly simple.                             */
/******************************************************************************/

DirList *FileList = NULL;

CLASHORT MakeFileList(char *Path, CLAUSHORT Attr, CLAUSHORT Order)
{   CLASHORT    Ret = 0;

    if (FileList != NULL)
        ReleaseFileList();

    FileList = new DirList(Path, Attr, Order);

    return Ret;
}


CLASHORT GetFile(FILESPEC *File)
{   DirEntry   *CurrFile    = (DirEntry *)(*FileList)++;
    CLASHORT    Ret         = 1;

    if (CurrFile)
    {   *File = CurrFile->EntryInfo();

        Ret = 0;
    }

    return Ret;
}


CLASHORT ResetFileList(void)
{   CLASHORT Ret = -1;

    if (FileList)
    {   FileList->restart();
        Ret = 0;
    }

    return Ret;
}


CLASHORT GetNFile(FILESPEC *File, CLAUSHORT N)
{   CLASHORT Ret = -1;

    if (FileList)
    {   DirEntry *CurrFile;

        FileList->restart();

        while (N--)
            CurrFile = (DirEntry *)(*FileList)++;

        *File = CurrFile->EntryInfo();

        Ret = 0;
    }

    return Ret;
}

CLASHORT OrderFileList(CLAUSHORT Order)
{
    if (FileList)
        FileList->ReOrder(Order);

    return FileList ? 0: -1;
}


void ReleaseFileList()
{   delete FileList;
    FileList = NULL;
}

