/******************************************************************************/
/*                                  FILELIST.H                                */
/*                                                                            */
/*                      Simple file list management routines                  */
/*                                                                            */
/*  Sean Wilson, 2/Sep/1992                                                   */
/******************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <dos.h>

#include "filelist.h"

typedef struct node_t {
    CLAUSHORT       attr;
    CLATIME         time;
    CLADATE         date;
    CLAULONG        size;
    char            name[13];
    struct node_t  *next;
    } NODE;

static struct ffblk  FileInfo;

static NODE         *FileList   = NULL;
static NODE         *CurrFile   = NULL;
CLAUSHORT            FileCount  = 0;


/** MAKECLADATE ***************************************************************
 *
 *  Synopsis:
 *      static CLADATE MakeClaDate(DosDate);
 *
 *          unsigned    DosDate;        Date in DOS format
 *
 *  Description:
 *      MakeClaDate() takes a dos format date in DosDate and returns a
 *      Clarion format date.
 */

static CLADATE MakeClaDate(unsigned DosDate)
{   CLADATE ClaDate;
    union {
        unsigned    u;
        struct {
        unsigned    day     : 5;
        unsigned    month   : 4;
        unsigned    year    : 7;
        }           s;
    }   d;

    d.u = DosDate;

    ClaDate.s.usYear  = d.s.year + 1980;
    ClaDate.s.ucMonth = d.s.month;
    ClaDate.s.ucDay   = d.s.day;

    return ClaDate;
}

/** MAKECLATIME ***************************************************************
 *
 *  Synopsis:
 *      static CLADATE MakeClaTime(DosTime);
 *
 *          unsigned    DosTime;        Time in DOS format
 *
 *  Description:
 *      MakeClaTime() takes a dos format time in DosTime and returns a
 *      Clarion format time.
 */

static CLATIME MakeClaTime(unsigned DosTime)
{   CLATIME ClaTime;
    union {
        unsigned    u;
        struct {
        unsigned    seconds : 5;
        unsigned    minutes : 6;
        unsigned    hours   : 5;
        }           s;
    }   t;

    t.u = DosTime;

    ClaTime.s.ucHour      = t.s.hours;
    ClaTime.s.ucMinute    = t.s.minutes;
    ClaTime.s.ucSecond    = t.s.seconds * 2;
    ClaTime.s.ucHund      = 0;

    return ClaTime;
}

/** COMPARE *******************************************************************
 *
 *  Synopsis:
 *      static int Compare(N1, N2, Order);
 *
 *          NODE    *N1;                First node to compare
 *          NODE    *N2;                Second node in comparison
 *          int      Order;             Ordering flags
 *
 *  Description:
 *      Compare() compares N1 and N2 for ordering according to Order.
 *
 *  Returns:
 *      -1 (N1 < N2), 0 (N1 == N2), 1 (N1 > N2)
 */

static int Compare(NODE *N1, NODE *N2, int Order)
{   int Cmp;

    switch (Order) {
    default:
    case NAMEORDER:
        Cmp = strcmp(N1->name, N2->name);
        break;
    case DATEORDER:
        if (N1->date.s.usYear > N2->date.s.usYear)
            Cmp = 1;
        if (N1->date.s.usYear < N2->date.s.usYear)
            Cmp = -1;
        else if (N1->date.s.ucMonth > N2->date.s.ucMonth)
            Cmp = 1;
        else if (N1->date.s.ucMonth < N2->date.s.ucMonth)
            Cmp = -1;
        else if (N1->date.s.ucDay > N2->date.s.ucDay)
            Cmp = 1;
        else if (N1->date.s.ucDay < N2->date.s.ucDay)
            Cmp = -1;
        else
            Cmp = 0;
        break;
    case SIZEORDER:
        Cmp = N1->size > N2->size ? 1: -1;
        break;
    }

    return Cmp;
}


CLASHORT MakeFileList(char *Path, CLAUSHORT Attr, CLAUSHORT Order)
{   CLASHORT    Ret = 0;
    NODE       *Tmp = NULL;

    if (FileList != NULL)
        ReleaseFileList();

    FileCount = 0;

    if (findfirst(Path, &FileInfo, Attr) == -1)
    {   Ret = errno;
    }
    else
    {   do  /*** Insert file into list and find next one                    ***/
        {   if ((Tmp = (NODE *)malloc(sizeof(NODE))) == NULL)
            {   Ret = errno;
            }
            else
            {   Tmp->attr = FileInfo.ff_attrib;
                Tmp->time = MakeClaTime(FileInfo.ff_ftime);
                Tmp->date = MakeClaDate(FileInfo.ff_fdate);
                Tmp->size = FileInfo.ff_fsize;

                strncpy(Tmp->name, FileInfo.ff_name, 13);

                Tmp->next = FileList;
                FileList  = Tmp;

                FileCount++;

                if (findnext(&FileInfo) == -1)
                    break;
            }
        }   while (Ret == 0);

        OrderFileList(Order);
    }

    return Ret;
}


CLASHORT GetFile(FILESPEC *File)
{   CLASHORT    Ret = 1;

    if (CurrFile)
    {   memmove(File, CurrFile, sizeof(FILESPEC));
        CurrFile = CurrFile->next;
        Ret = 0;
    }

    return Ret;
}


CLASHORT ResetFileList(void)
{   CLASHORT Ret = -1;

    if (FileList)
    {   CurrFile = FileList;
        Ret = 0;
    }

    return Ret;
}


CLASHORT GetNFile(FILESPEC *File, CLAUSHORT N)
{   NODE       *Tmp = FileList;
    CLASHORT    Ret = -1;

    if (FileList)
    {   Ret = 0;

        while (N--)
            Tmp = FileList->next;

        memmove(File, Tmp, sizeof(FILESPEC));
    }

    return Ret;
}

CLASHORT OrderFileList(CLAUSHORT Order)
{   NODE       *NewLst  = NULL;         /* New ordered list                   */

    /*** Initialise the new ordered list                                    ***/
    NewLst          = FileList;         /* NewLst is the start of FileList    */
    FileList        = FileList->next;   /* Increment FileList                 */
    NewLst->next    = NULL;             /* NewLst is a singleton              */

    /*** Iterate through FileList inserting each item in order into NewLst  ***/
    while (FileList)
    {   NODE       *Lst     = NULL;     /* Iterates through NewLst            */
        NODE       *Prv     = NULL;     /* Previous item in NewLst            */
        NODE       *Tmp     = NULL;     /* Current item to insert             */
        int         Cmp;

        Tmp         = FileList;         /* Tmp is the next item in FileListq  */
        FileList    = FileList->next;   /* Increment FileList                 */
        Tmp->next   = NULL;             /* Tmp is  a singleton                */

        Lst         = NewLst;           /* Lst is the start of NewLst         */
        Prv         = NULL;             /* No previous item yet               */

        /*** Iterate through NewLst to find the insertion point             ***/
        do
        {   Cmp = Compare(Tmp, Lst, Order);

            if ((Cmp < 0) || (Lst == NULL))
            {   if (Prv)
                {   Tmp->next = Lst;
                    Prv->next = Tmp;
                }
                else
                {   Tmp->next = NewLst;
                    NewLst    = Tmp;
                }

                break;
            }
            else
            {   Prv = Lst;

                if ((Lst = Lst->next) == NULL)
                {   Prv->next = Tmp;
                }
            }
        }   while (Lst);
    }

    FileList = NewLst;                  /* FileList is the NewLst             */
    CurrFile = FileList;                /* CurrFile is start of FileList      */

    return FileList ? 0: -1;
}


void ReleaseFileList()
{   while (FileList)
    {   NODE *Tmp = FileList;

        FileList = FileList->next;
        free(Tmp);
    }

    FileCount = 0;
    FileList = NULL;
    CurrFile = NULL;
}


void ProgParam(unsigned uArg, unsigned uLen, char *pszParam)
{   if (uArg >= 0 && uArg <= _argc)
    {   strncpy(pszParam, _argv[uArg], uLen);
    }
    else
    {   *pszParam = '\0';
    }
}


unsigned ParamCount(void)
{   return _argc;
}


