/******************************************************************************/
/*                                  FILELIST.H                                */
/*                                                                            */
/*              Interface to simple file list management routines             */
/*                                                                            */
/*  Sean Wilson, 2/Feb/1993                                                   */
/******************************************************************************/

#ifndef FILELIST_H
#define FILELIST_H

#include "cla.h"

typedef struct filespec_t {             /* Structure for file information    */
    CLAUSHORT       attr;
    CLATIME         time;
    CLADATE         date;
    CLAULONG        size;
    char            name[13];
    } FILESPEC;

/*** File ordering flags, only one may be specified                        ***/
#define     NAMEORDER   0               /* Ascending name order              */
#define     DATEORDER   1               /* Ascending date order              */
#define     SIZEORDER   2               /* Ascending size order              */

/*** File attributes, may be combined                                       ***/
#define     NORMAL      0x00            /* normal file                        */
#define     RDONLY      0x01            /* Read only attribute                */
#define     HIDDEN      0x02            /* Hidden file                        */
#define     SYSTEM      0x04            /* System file                        */
#define     LABEL       0x08            /* Volume label                       */
#define     DIRECTRY    0x10            /* Directory                          */
#define     ARCHIVE     0x20            /* Archive                            */

#if defined(__cplusplus)
extern "C" {
#endif

extern CLAUSHORT FileCount;             /* How many files are in the list?    */


/** MAKEFILELIST **************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      CLASHORT MakeFileList(Path, Attr, Order);
 *
 *          char       *Path;           File spec, may include wild cards
 *          CLAUSHORT   Attr;           File attribute flags
 *          CLAUSHORT   Order;          Ordering to apply
 *
 *  Description:
 *      MakeFileList() builds an internal list of all the files that conform
 *      to the specification in Path, with attributes Attr.  The list will
 *      be Ordered according to the flags in Order.  Information can be
 *      retreived from the list using GetFile().
 *
 *  Returns:
 *      0 on success, DOS error code otherwise.
 */

CLASHORT MakeFileList(char *Path, CLAUSHORT Attr, CLAUSHORT Order);


/** GETFILE *******************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      CLASHORT GetFile(File);
 *
 *          FILESPEC    *File;          File information buffer
 *
 *  Description:
 *      GetFile() places the file information from the first element of the
 *      internal list into the File buffer, then updates the list to the
 *      next item. i.e. GetFile() iterates through the list.
 *
 *  Returns:
 *      0 on success, -1 if the end of the list is reached.
 */

CLASHORT GetFile(FILESPEC *File);


/** GETNFILE ******************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      CLASHORT GetNFile(File, N);
 *
 *          FILESPEC    *File;          File information buffer
 *          CLAUSHORT    N;             File index
 *
 *  Description:
 *      GetNFile() places the file information from the Nth element of the
 *      internal list into the File buffer.
 *
 *  Returns:
 *      0 on success, -1 if N is out of range.
 */

CLASHORT GetNFile(FILESPEC *File, CLAUSHORT N);


/** ORDERFILELIST ************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      CLASHORT OrderFileList(Order);
 *
 *          CLASHORT    Order;          New ordering flags
 *
 *  Description:
 *      OrderFileList() reorders the list according to the value in the
 *      Order parameter.  Files may be ordered by name, size or date.
 *
 *  Returns:
 *      0 on success, -1 if MakeFileList() hasn't been called.
 */

CLASHORT OrderFileList(CLAUSHORT Order);


/** RELEASEFILELIST **********************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      void ReleaseFileList();
 *
 *  Description:
 *      ReleaseFileList() should be called when you have finshed using the
 *      list to release memory.
 *
 *  Returns:
 *      Void.
 */

void ReleaseFileList(void);


/** PROGPARAM *****************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      void ProgParam(uArg, uLen, pszParam);
 *
 *          CLAUSHORT   uArg;
 *          CLAUSHORT   uLen;
 *          char       *pszParam;
 *
 *  Description:
 *      ProgParam() returns the uArg'th string from the program's command
 *      line in pszParam, which is a CSTRING of uLen bytes.
 *
 *  Returns:
 *      Void.
 */

void ProgParam(unsigned uArg, unsigned uLen, char *pszParam);


/** PARAMCOUNT ****************************************************************
 *
 *  Synopsis:
 *      #include "filelist.h"
 *
 *      unsigned ParamCount();
 *
 *  Description:
 *      ParamCount() determines the number of strings passed on the program's
 *      command line.
 *
 *  Returns:
 *      The number of program parameters
 */

unsigned ParamCount(void);

#if defined(__cplusplus)
}
#endif

#endif /* FILELIST_H */

