(******************************************************************************)
(*                                  FILELIST.MOD                              *)
(*                                                                            *)
(*                      Simple file list management routines                  *)
(*                                                                            *)
(*  Sean Wilson, 13/Jan/1993                                                  *)
(******************************************************************************)
IMPLEMENTATION MODULE FileList;

  IMPORT  Lib, FIO, Str;
  FROM    Storage IMPORT ALLOCATE, DEALLOCATE;

  TYPE
    NODEP = POINTER TO NODE;
    NODE  = RECORD
      CASE : BOOLEAN OF
      | TRUE:
        attr  : Cla.USHORT;
        time  : Cla.TIME;
        date  : Cla.DATE;
        size  : Cla.LONG;
        name  : ARRAY [0..12] OF CHAR;
        next  : NODEP;
      ELSE
        fspec : FILESPEC;
      END
    END;

  VAR
    FileInfo  : FIO.DirEntry;
    FilesList : NODEP;
    CurrFile  : NODEP;

(** MAKECLADATE ***************************************************************
 *
 *  Synopsis:
 *      PROCEDURE MakeClaDate(DosDate: CARDINAL): Cla.DATE;
 *
 *        DosDate                       Date in DOS format
 *
 *  Description:
 *      MakeClaDate() takes a dos format date in DosDate and returns a
 *      Clarion format date.
 *)

PROCEDURE MakeClaDate(DosDate: CARDINAL): Cla.DATE;
  VAR
    ClaDate : Cla.DATE;
BEGIN
  ClaDate.usYear    := Cla.USHORT(BITSET(DosDate >> 9H) * BITSET(7FH)) + 1980;
  ClaDate.ucMonth   := Cla.BYTE(BITSET(DosDate >> 5)  * BITSET(0FH));
  ClaDate.ucDay     := Cla.BYTE(BITSET(DosDate) * BITSET(1FH));

  RETURN ClaDate;
END MakeClaDate;

(** MAKECLATIME ***************************************************************
 *
 *  Synopsis:
 *      PROCEDURE MakeClaTime(DosTime: CARDINAL): Cla.TIME;
 *
 *        DosTime                       Time in DOS format
 *
 *  Description:
 *      MakeClaTime() takes a dos format time in DosTime and returns a
 *      Clarion format time.
 *)

PROCEDURE MakeClaTime(DosTime: CARDINAL): Cla.TIME;
  VAR
    ClaTime : Cla.TIME;
BEGIN
  ClaTime.ucHour    := Cla.BYTE(BITSET(DosTime >> 11) * BITSET(1FH));
  ClaTime.ucMinute  := Cla.BYTE(BITSET(DosTime >> 5) * BITSET(3FH));
  ClaTime.ucSecond  := Cla.BYTE(CARDINAL(BITSET(DosTime) * BITSET(1FH)) << 1);
  ClaTime.ucHund    := 0;

  RETURN ClaTime;
END MakeClaTime;

(** COMPARE *******************************************************************
 *
 *  Synopsis:
 *      PROCEDURE Compare(VAR N1, N2: NODE; Order: INTEGER): INTEGER;
 *
 *        N1                            First node to compare
 *        N2                            Second node in comparison
 *        Order;                        Ordering flags
 *
 *  Description:
 *      Compare() compares N1 and N2 for ordering according to Order.
 *
 *  Returns:
 *      -1 (N1 < N2), 0 (N1 == N2), 1 (N1 > N2)
 *)

PROCEDURE Compare(VAR N1, N2: NODE; Order: INTEGER): INTEGER;
  VAR
    Cmp : INTEGER;
BEGIN
  CASE Order OF
  | DATEORDER:
    IF (N1.date.usYear > N2.date.usYear) THEN
      Cmp := 1;
    ELSIF (N1.date.usYear < N2.date.usYear) THEN
      Cmp := -1;
    ELSIF (N1.date.ucMonth > N2.date.ucMonth) THEN
      Cmp := 1;
    ELSIF (N1.date.ucMonth < N2.date.ucMonth) THEN
      Cmp := -1;
    ELSIF (N1.date.ucDay > N2.date.ucDay) THEN
      Cmp := 1;
    ELSIF (N1.date.ucDay < N2.date.ucDay) THEN
      Cmp := -1;
    ELSE
      Cmp := 0;
    END;

  | SIZEORDER:
    IF (N1.size > N2.size) THEN
      Cmp := 1;
    ELSE
      Cmp := -1;
    END;

  ELSE  (* NAMEORDER *)
    Cmp := Str.Compare(N1.name, N2.name);
  END;

  RETURN Cmp;
END Compare;


PROCEDURE MakeFileList(Path: ARRAY OF CHAR; Attr, Order: Cla.USHORT): Cla.SHORT;
  VAR
    Spec: ARRAY [0..128] OF CHAR;
    Ret : Cla.USHORT;
    Tmp : NODEP;
BEGIN
  Lib.FastMove(ADR(Path), ADR(Spec), SIZE(Spec));
  Ret   := 0;
  Tmp   := NIL;

  IF (FilesList # NIL) THEN
    ReleaseFileList()
  END;

  FileCount := 0;

  IF FIO.ReadFirstEntry(Spec, FIO.FileAttr(Attr), FileInfo) THEN
    LOOP    (*** Insert file into list and find next one                    ***)
      NEW(Tmp);

      Tmp^.attr := Cla.USHORT(FileInfo.attr);
      Tmp^.time := MakeClaTime(FileInfo.time);
      Tmp^.date := MakeClaDate(FileInfo.date);
      Tmp^.size := FileInfo.size;

      Str.Copy(Tmp^.name, FileInfo.Name);

      Tmp^.next := FilesList;
      FilesList := Tmp;

      INC(FileCount);

      IF NOT FIO.ReadNextEntry(FileInfo) THEN
        EXIT;
      END;

      Ret := OrderFileList(Order);
    END;
  ELSE
    Ret := Lib.SysErrno();
  END;

  RETURN Ret;
END MakeFileList;


PROCEDURE GetFile(VAR File: FILESPEC): Cla.SHORT;
  VAR
    Ret : Cla.SHORT;
BEGIN
  Ret := -1;

  IF CurrFile # NIL THEN
    File      := CurrFile^.fspec;
    CurrFile  := CurrFile^.next;
    Ret       := 0;
  END;

  RETURN Ret;
END GetFile;


PROCEDURE ResetFileList(): Cla.SHORT;
  VAR
    Ret : Cla.SHORT;
BEGIN
  Ret := -1;

  IF FilesList # NIL THEN
    CurrFile  := FilesList;
    Ret       := 0;
  END;

  RETURN Ret;
END ResetFileList;


PROCEDURE GetNFile(VAR File: FILESPEC; N: Cla.USHORT): Cla.SHORT;
  VAR
    Tmp : NODEP;
    Ret : Cla.SHORT;
BEGIN

  IF FilesList # NIL THEN
    WHILE (N > 0) DO
      Tmp := FilesList^.next;
      DEC(N);
    END;

    File  := Tmp^.fspec;
    Ret   := 0;
  END;

  RETURN Ret;
END GetNFile;


PROCEDURE OrderFileList(Order: Cla.USHORT): Cla.SHORT;
  VAR
    NewLst  : NODEP;                    (* New ordered list                   *)
    Lst     : NODEP;                    (* Iterates through NewLst            *)
    Prv     : NODEP;                    (* Previous item in NewLst            *)
    Tmp     : NODEP;                    (* Current item to insert             *)
    Cmp     : INTEGER;
BEGIN
  Lst     := NIL;
  Prv     := NIL;
  Tmp     := NIL;

  (*** Initialise the new ordered list                                      ***)
  NewLst        := FilesList;           (* NewLst is the start of FilesList   *)
  FilesList     := FilesList^.next;     (* Increment FilesList                *)
  NewLst^.next  := NIL;                 (* NewLst is a singleton              *)

  (*** Iterate through FilesList inserting each item in order into NewLst   ***)
  WHILE FilesList # NIL DO
    Tmp         := FilesList;           (* Tmp is the next item in FilesList  *)
    FilesList   := FilesList^.next;     (* Increment FilesList                *)
    Tmp^.next   := NIL;                 (* Tmp is  a singleton                *)

    Lst         := NewLst;              (* Lst is the start of NewLst         *)
    Prv         := NIL;                 (* No previous item yet               *)

    (*** Iterate through NewLst to find the insertion point                 ***)
    LOOP
      Cmp := Compare(Tmp^, Lst^, Order);

      IF (Cmp < 0) OR (Lst = NIL) THEN
        IF Prv # NIL THEN
          Tmp^.next := Lst;
          Prv^.next := Tmp;
        ELSE
          Tmp^.next := NewLst;
          NewLst    := Tmp;
        END;

        EXIT;
      ELSE
        Prv := Lst;
        Lst := Lst^.next;

        IF Lst = NIL THEN
          Prv^.next := Tmp;

          EXIT;
        END
      END
    END
  END;

  FilesList := NewLst;                 (* FilesList is the NewLst             *)
  CurrFile  := FilesList;              (* CurrFile is start of FilesList      *)

  IF FilesList = NIL THEN
    RETURN -1;
  ELSE
    RETURN 0;
  END
END OrderFileList;


PROCEDURE ReleaseFileList();
  VAR
    Tmp : NODEP;
BEGIN
  WHILE FilesList # NIL DO
    Tmp := FilesList;

    FilesList := FilesList^.next;
    DISPOSE(Tmp);
  END;

  FileCount := 0;
  FilesList := NIL;
  CurrFile  := NIL;
END ReleaseFileList;

BEGIN
  FileCount := 0;
  FilesList := NIL;
  CurrFile  := NIL;
END FileList.

