(******************************************************************************)
(*                                  FILELIST.PAS                              *)
(*                                                                            *)
(*                      Simple file list management routines                  *)
(*                                                                            *)
(*  Sean Wilson, 14/May/1993                                                  *)
(******************************************************************************)
IMPLEMENTATION UNIT FileList;

  IMPORT  PasLib, CDOS;

  TYPE
    NODEP = ^NODE;
    NODE  = RECORD
      CASE BOOLEAN OF
      TRUE :
       (attr : Cla.USHORT;
        time : Cla.TIME;
        date : Cla.DATE;
        size : Cla.LONG;
        name : ARRAY[0..12] OF CHAR;
        next : NODEP);
      FALSE:
       (fspec: FILESPEC);
    END;

  VAR
    FileInfo  : CDOS.ffblk;
    FilesList : NODEP;
    CurrFile  : NODEP;

(** MAKECLADATE ***************************************************************
 *
 *  Synopsis:
 *      MakeClaDate(DosDate: WORD): Cla.DATE;
 *
 *        DosDate                       Date in DOS format
 *
 *  Description:
 *      MakeClaDate() takes a dos format date in DosDate and returns a
 *      Clarion format date.
 *)

FUNCTION MakeClaDate(DosDate: WORD): Cla.DATE;
  VAR
    ClaDate : Cla.DATE;
BEGIN
  ClaDate.usYear    := ((DosDate >> 9H) AND (7FH)) + 1980;
  ClaDate.ucMonth   := ((DosDate >> 5) AND (0FH));
  ClaDate.ucDay     := ((DosDate) AND (1FH));

  MakeClaDate := ClaDate;
END;

(** MAKECLATIME ***************************************************************
 *
 *  Synopsis:
 *      FUNCTION MakeClaTime(DosTime: WORD): Cla.TIME;
 *
 *        DosTime                       Time in DOS format
 *
 *  Description:
 *      MakeClaTime() takes a dos format time in DosTime and returns a
 *      Clarion format time.
 *)

FUNCTION MakeClaTime(DosTime: WORD): Cla.TIME;
  VAR
    ClaTime : Cla.TIME;
BEGIN
  ClaTime.ucHour    := ((DosTime >> 11) AND (1FH));
  ClaTime.ucMinute  := ((DosTime >> 5) AND (3FH));
  ClaTime.ucSecond  := ((DosTime) AND (1FH)) << 1;
  ClaTime.ucHund    := 0;

  MakeClaTime := ClaTime;
END;

(** COMPARE *******************************************************************
 *
 *  Synopsis:
 *      PROCEDURE Compare(VAR N1, N2: NODE; Order: INTEGER): INTEGER;
 *
 *        N1                            First node to compare
 *        N2                            Second node in comparison
 *        Order;                        Ordering flags
 *
 *  Description:
 *      Compare() compares N1 and N2 for ordering according to Order.
 *
 *  Returns:
 *      -1 (N1 < N2), 0 (N1 == N2), 1 (N1 > N2)
 *)

FUNCTION Compare(VAR N1, N2: NODE; Order: INT16): INT16;
  VAR
    Cmp : INT16;
BEGIN
  CASE Order OF
  DATEORDER:
    IF (N1.date.usYear > N2.date.usYear) THEN
      Cmp := 1
    ELSE IF (N1.date.usYear < N2.date.usYear) THEN
      Cmp := -1
    ELSE IF (N1.date.ucMonth > N2.date.ucMonth) THEN
      Cmp := 1
    ELSE IF (N1.date.ucMonth < N2.date.ucMonth) THEN
      Cmp := -1
    ELSE IF (N1.date.ucDay > N2.date.ucDay) THEN
      Cmp := 1
    ELSE IF (N1.date.ucDay < N2.date.ucDay) THEN
      Cmp := -1
    ELSE
      Cmp := 0;

  SIZEORDER:
    IF (N1.size > N2.size) THEN
      Cmp := 1
    ELSE
      Cmp := -1;

  ELSE  (* NAMEORDER *)
    Cmp := PasLib.Compare(N1.name, N2.name, Size(N1.name));
  END;

  Compare := Cmp;
END;


FUNCTION MakeFileList{(VAR Path; Attr, Order: Cla.USHORT): Cla.SHORT};
  VAR
    Ret   : Cla.USHORT;
    Tmp   : NODEP;
BEGIN
  Ret   := 0;
  Tmp   := NIL;

  IF (FilesList <> NIL) THEN
    ReleaseFileList;

  FileCount := 0;

  Ret := CDOS.FindFirst(Path, FileInfo, Attr);

  IF Ret = 0 THEN
    REPEAT        (*** Insert file into list and find next one              ***)
      NEW(Tmp);

      Tmp^.attr := FileInfo.ff_attrib;
      Tmp^.time := MakeClaTime(FileInfo.ff_ftime);
      Tmp^.date := MakeClaDate(FileInfo.ff_fdate);
      Tmp^.size := FileInfo.ff_fsize;
      Tmp^.name := FileInfo.ff_name;

      Tmp^.next := FilesList;
      FilesList := Tmp;

      FileCount := FileCount + 1;

    UNTIL CDOS.FindNext(FileInfo) <> 0;

    Ret := OrderFileList(Order);

  MakeFileList := Ret;
END;


FUNCTION GetFile{(VAR File: FILESPEC): Cla.SHORT};
  VAR
    Ret : Cla.SHORT;
BEGIN
  Ret := -1;

  IF CurrFile <> NIL THEN
  BEGIN
    FileSpc   := CurrFile^.fspec;
    CurrFile  := CurrFile^.next;
    Ret       := 0;
  END;

  GetFile := Ret;
END;


FUNCTION ResetFileList: Cla.SHORT;
  VAR
    Ret : Cla.SHORT;
BEGIN
  Ret := -1;

  IF FilesList <> NIL THEN
  BEGIN
    CurrFile  := FilesList;
    Ret       := 0;
  END;

  ResetFileList := Ret;
END;


FUNCTION GetNFile{(VAR File: FILESPEC; N: Cla.USHORT): Cla.SHORT};
  VAR
    Tmp : NODEP;
    Ret : Cla.SHORT;
BEGIN

  IF FilesList <> NIL THEN
  BEGIN
    WHILE (N > 0) DO
    BEGIN
      Tmp := FilesList^.next;
      N   := N - 1;
    END;

    FileSpc := Tmp^.fspec;
    Ret     := 0;
  END;

  GetNFile := Ret;
END;


FUNCTION OrderFileList{(Order: Cla.USHORT): Cla.SHORT};
  VAR
    NewLst  : NODEP;                    (* New ordered list                   *)
    Lst     : NODEP;                    (* Iterates through NewLst            *)
    Prv     : NODEP;                    (* Previous item in NewLst            *)
    Tmp     : NODEP;                    (* Current item to insert             *)
    Cmp     : INT16;
    Found   : BOOLEAN;
BEGIN
  Lst     := NIL;
  Prv     := NIL;
  Tmp     := NIL;

  (*** Initialise the new ordered list                                      ***)
  NewLst        := FilesList;           (* NewLst is the start of FilesList   *)
  FilesList     := FilesList^.next;     (* Increment FilesList                *)
  NewLst^.next  := NIL;                 (* NewLst is a singleton              *)

  (*** Iterate through FilesList inserting each item in order into NewLst   ***)
  WHILE FilesList <> NIL DO
  BEGIN
    Tmp         := FilesList;           (* Tmp is the next item in FilesList  *)
    FilesList   := FilesList^.next;     (* Increment FilesList                *)
    Tmp^.next   := NIL;                 (* Tmp is  a singleton                *)

    Lst         := NewLst;              (* Lst is the start of NewLst         *)
    Prv         := NIL;                 (* No previous item yet               *)

    (*** Iterate through NewLst to find the insertion point                 ***)

    Found := FALSE;

    REPEAT
      Cmp := Compare(Tmp^, Lst^, Order);

      IF (Cmp < 0) OR (Lst = NIL) THEN
      BEGIN
        IF Prv <> NIL THEN
        BEGIN
          Tmp^.next := Lst;
          Prv^.next := Tmp;
        END ELSE
        BEGIN
          Tmp^.next := NewLst;
          NewLst    := Tmp;
        END;

        Found := TRUE;
      END ELSE
        Prv := Lst;
        Lst := Lst^.next;

        IF Lst = NIL THEN
        BEGIN
          Prv^.next := Tmp;
          Found     := TRUE;
        END
    UNTIL Found
  END;

  FilesList := NewLst;                 (* FilesList is the NewLst             *)
  CurrFile  := FilesList;              (* CurrFile is start of FilesList      *)

  IF FilesList = NIL THEN
    OrderFileList := -1
  ELSE
    OrderFileList := 0;
END;


PROCEDURE ReleaseFileList;
  VAR
    Tmp : NODEP;
BEGIN
  WHILE FilesList <> NIL DO
  BEGIN
    Tmp := FilesList;

    FilesList := FilesList^.next;
    DISPOSE(Tmp);
  END;

  FileCount := 0;
  FilesList := NIL;
  CurrFile  := NIL;
END;

BEGIN
  FileCount := 0;
  FilesList := NIL;
  CurrFile  := NIL;
END.

