#include <adv3.h>
#include <en_us.h>


/*
 *   -------------------------------------------------------------------------
 *   Modifications for en_us.t
 */
/* modifications: change the default language for the GameInfo metadata class */
modify GameInfoModuleID
    languageCode = 'en-GB'
;

/* modifications: use "and" in spelt-out numbers when necessary */
modify Thing
    countNameFrom(count, singularStr, pluralStr) {
        /* if the count is one, use 'one' plus the singular name */
        if (count == 1)
            return 'one ' + singularStr;

        /*
         *   Get the number followed by a space - spell out numbers below
         *   100, but use numerals to denote larger numbers.  Append the
         *   plural name to the number and return the result.
         */
        return spellIntBelowExt(count, 100, SpellIntAndTens | SpellIntCommas, 
                                DigitFormatGroupSep) + ' ' + pluralStr;
    }
;

/* modifications: use "and" in spelt-out numbers when necessary */
modify spellInt(val) {
    return spellIntExt(val, SpellIntAndTens | SpellIntCommas);
}

/* modifications: use "and" in spelt-out numbers when necessary */
modify spellIntBelow(val, threshold) {
    return spellIntBelowExt(val, threshold, SpellIntAndTens | SpellIntCommas, 0);
}

/* 
 *   modifications: remove "billion" from powers list, since a UK billion is 
 *   too large for ordinary integers to handle
 */
modify spellIntExt(val, flags) {
    local str;
    local trailingSpace;
    local needAnd;
    local powers = [1000000,       ' million ',
                    1000,          ' thousand ',
                    100,           ' hundred '];

    /* start with an empty string */
    str = '';
    trailingSpace = nil;
    needAnd = nil;

    /* if it's zero, it's a special case */
    if (val == 0)
        return 'zero';

    /*
     *   if the number is negative, note it in the string, and use the
     *   absolute value
     */
    if (val < 0)
    {
        str = 'negative ';
        val = -val;
    }

    /* do each named power of ten */
    for (local i = 1 ; val >= 100 && i <= powers.length() ; i += 2)
    {
        /*
         *   if we're in teen-hundreds mode, do the teen-hundreds - this
         *   only works for values from 1,100 to 9,999, since a number like
         *   12,000 doesn't work this way - 'one hundred twenty hundred' is
         *   no good 
         */
        if ((flags & SpellIntTeenHundreds) != 0
            && val >= 1100 && val < 10000)
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';

            /* spell it out as a number of hundreds */
            str += spellIntExt(val / 100, flags) + ' hundred ';

            /* take off the hundreds */
            val %= 100;

            /* note the trailing space */
            trailingSpace = true;

            /* we have something to put an 'and' after, if desired */
            needAnd = true;

            /*
             *   whatever's left is below 100 now, so there's no need to
             *   keep scanning the big powers of ten
             */
            break;
        }

        /* if we have something in this power range, apply it */
        if (val >= powers[i])
        {
            /* if desired, add a comma if there was a prior power group */
            if (needAnd && (flags & SpellIntCommas) != 0)
                str = str.substr(1, str.length() - 1) + ', ';

            /* add the number of multiples of this power and the power name */
            str += spellIntExt(val / powers[i], flags) + powers[i+1];

            /* take it out of the remaining value */
            val %= powers[i];

            /*
             *   note that we have a trailing space in the string (all of
             *   the power-of-ten names have a trailing space, to make it
             *   easy to tack on the remainder of the value)
             */
            trailingSpace = true;

            /* we have something to put an 'and' after, if one is desired */
            needAnd = true;
        }
    }

    /*
     *   if we have anything left, and we have written something so far,
     *   and the caller wanted an 'and' before the tens part, add the
     *   'and'
     */
    if ((flags & SpellIntAndTens | SpellIntCommas) != 0
        && needAnd
        && val != 0)
    {
        /* add the 'and' */
        str += 'and ';
        trailingSpace = true;
    }

    /* do the tens */
    if (val >= 20)
    {
        /* anything above the teens is nice and regular */
        str += ['twenty', 'thirty', 'forty', 'fifty', 'sixty',
                'seventy', 'eighty', 'ninety'][val/10 - 1];
        val %= 10;

        /* if it's non-zero, we'll add the units, so add a hyphen */
        if (val != 0)
            str += '-';

        /* we no longer have a trailing space in the string */
        trailingSpace = nil;
    }
    else if (val >= 10)
    {
        /* we have a teen */
        str += ['ten', 'eleven', 'twelve', 'thirteen', 'fourteen',
                'fifteen', 'sixteen', 'seventeen', 'eighteen',
                'nineteen'][val - 9];

        /* we've finished with the number */
        val = 0;

        /* there's no trailing space */
        trailingSpace = nil;
    }

    /* if we have a units value, add it */
    if (val != 0)
    {
        /* add the units name */
        str += ['one', 'two', 'three', 'four', 'five',
                'six', 'seven', 'eight', 'nine'][val];

        /* we have no trailing space now */
        trailingSpace = nil;
    }

    /* if there's a trailing space, remove it */
    if (trailingSpace)
        str = str.substr(1, str.length() - 1);

    /* return the string */
    return str;
}

/* modifications: use "and" in spelt-out numbers where necessary */
modify spellIntOrdinal(n) {
    return spellIntOrdinalExt(n, SpellIntAndTens | SpellIntCommas);
}

/* modifications: "off of" -> "off" */
modify Surface
    actorOutOfPrep = 'off'
;

/* modifications: "off of" -> "off" */
modify BasicChair
    actorOutOfPrep = 'off'
;

/* modifications: "off of" -> "off" */
modify BasicPlatform
    actorOutOfPrep = 'off'
;

/* modifications: "off of" -> "off" */
modify Floor
    actorOutOfPrep = 'off'
;

/* modifications: "off of" -> "off" */
modify VerbRule(GetOffOf)
    :
    verbPhrase = 'get/getting (off what)'
;


/*
 *   -------------------------------------------------------------------------
 *   Modifications for msg_neu.t
 */
modify libMessages
    /* modifications: placement of punctuation relative to quotation marks */
    againCannotChangeActor() {
        "<.parser>To repeat a command like <q>turtle, go north</q>,
        just say <q>again</q>, not <q>turtle, again</q>.<./parser> ";
    }

    /* modifications: spelling of "cancelled" */
    saveCanceled() { "<.parser>Cancelled.<./parser> "; }

    /* modifications: spelling of "cancelled" */
    restoreCanceled() { "<.parser>Cancelled.<./parser> "; }
    
    /* modifications: spelling of "cancelled" */
    scriptingCanceled = "<.parser>Cancelled.<./parser> "
    
    /* modifications: spelling of "cancelled" */
    recordingCanceled = "<.parser>Cancelled.<./parser> "
    
    /* modifications: spelling of "cancelled" */
    replayCanceled = "<.parser>Cancelled.<./parser> "
;

modify playerActionMessages
    /* modifications: spelling of "burnt" */
    candleOutOfFuelMsg =
        '{The dobj/he} {is} too burnt down; {it/he} {cannot} be lit. '
    
    /* modifications: new message (avoiding "can't seem to cut" construction) */
    cutNoEffectMsg = '{The iobj/he} {is} unable to cut {the dobj/him}. '
    
    /* modifications: "off of" -> "off" */
    cannotGetOffOfMsg = '{You/he} {can\'t} get off {that dobj/him}. '
;


/*
 *   -------------------------------------------------------------------------
 *   Modifications for instruct.t
 */
modify InstructionsAction
    /* 
     *   modifications: placement of punctuation relative to quotation marks; 
     *   spelling of "apologise", "memorise"
     */
    conversationInstructions =
        "You can talk to other characters by asking or
        telling them about things in the story.  For example, you might
        ASK WIZARD ABOUT WAND or TELL GUARD ABOUT ALARM.  You should
        always use the ASK ABOUT or TELL ABOUT phrasing; the story
        won&rsquo;t be able to understand other formats, so you don&rsquo;t
        have to worry about thinking up complicated questions like <q>ask
        guard how to open the window</q>.
        In most cases, you&rsquo;ll get the best results by asking
        about specific objects or other characters you&rsquo;ve encountered
        in the story, rather than about abstract topics such as
        MEANING OF LIFE; however, if something in the story leads you
        to believe you <i>should</i> ask about some particular abstract
        topic, it can&rsquo;t hurt to try.

        \bIf you&rsquo;re asking or telling the same person about several
        topics in succession, you can save some typing by abbreviating
        ASK ABOUT to A, and TELL ABOUT to T.  For example, once you&rsquo;re
        talking to the wizard, you can abbreviate ASK WIZARD ABOUT AMULET
        to simply A AMULET.  This addresses the question to the same
        character as in the last ASK or TELL.

        <<firstObj(InConversationState, ObjInstances) != nil ?
          "\bTo greet another character, type TALK TO (Person).  This
          tries to get the other character&rsquo;s attention and start a
          conversation.  TALK TO is always optional, since you can start
          in with ASK or TELL directly if you prefer." : "">>

        <<firstObj(SpecialTopic, ObjInstances) != nil ?
          "\bThe story might occasionally suggest some special conversation
          commands, like this:

          \b\t(You could apologise, or explain about the aliens.)

          \bIf you like, you can use one of the suggestions just by
          typing in the special phrasing shown.  You can usually
          abbreviate these to the first few words when they&rsquo;re long.

          \b\t&gt;APOLOGISE
          \n\t&gt;EXPLAIN ABOUT ALIENS

          \bSpecial suggestions like this only work right at the moment
          they&rsquo;re offered, so you don&rsquo;t have to worry about
          memorising them, or trying them at other random times in the story.
          They&rsquo;re not new commands for you to learn; they&rsquo;re just
          extra options you have at specific times, and the story will always
          let you know when they&rsquo;re available.  When the story offers
          suggestions like this, they don&rsquo;t limit what you can do; you
          can still type any ordinary command instead of one of the
          suggestions." : "">>

        <<firstObj(SuggestedTopic, ObjInstances) != nil ?
          "\bIf you&rsquo;re not sure what to discuss, you can type TOPICS any
          time you&rsquo;re talking to someone.  This will show you a list of
          things that your character might be interested in discussing
          with the other person.  The TOPICS command usually won&rsquo;t list
          everything that you can discuss, so feel free to explore other
          topics even if they&rsquo;re not listed." : "">>

        \bYou can also interact with other characters using physical
        objects.  For example, you might be able to give something to
        another character, as in GIVE MONEY TO CLERK, or show an object
        to someone, as in SHOW IDOL TO PROFESSOR.  You might also be
        able to fight other characters, as in ATTACK TROLL WITH
        SWORD or THROW AXE AT DWARF.

        \bIn some cases, you can tell a character to do
        something for you.  You do this by typing the character&rsquo;s name,
        then a comma, then the command you want the character to perform,
        using the same wording you&rsquo;d use for a command to your own
        character.  For example:

        \b\t&gt;ROBOT, GO NORTH

        \bKeep in mind, though, that there&rsquo;s no guarantee that other
        characters will always obey your orders.  Most characters have
        minds of their own and won&rsquo;t automatically do whatever you
        ask. "
    
    /* modifications: spelling of "cancelled" */
    #ifndef INSTRUCTIONS_MENU
    showInstructions() {
        local startedScript;

        /* presume we won't start a new script file */
        startedScript = nil;
        
        /* show the introductory message */
        "The story is about to show a full set of instructions,
        designed especially for people who aren&rsquo;t already familiar
        with interactive fiction.  The instructions are lengthy";

        /*
         *   Check to see if we're already scripting.  If we aren't, offer
         *   to save the instructions to a file. 
         */
        if (scriptStatus.scriptFile == nil)
        {
            local str;
            
            /* 
             *   they're not already logging; ask if they'd like to start
             *   doing so 
             */
            ", so you might want to capture them to a file (so that
            you can print them out, for example).  Would you like to
            proceed?
            \n(<<aHref('yes')>>Y</a> is affirmative, or type
            <<aHref('script')>>SCRIPT</a> to capture to
            a file) &gt; ";

            /* ask for input */
            str = inputManager.getInputLine(nil, nil);

            /* if they want to capture them to a file, set up scripting */
            if (rexMatch('<nocase><space>*s(c(r(i(pt?)?)?)?)?<space>*', str)
                == str.length())
            {
                /* try setting up a scripting file */
                ScriptAction.setUpScripting(nil);

                /* if that failed, don't proceed */
                if (scriptStatus.scriptFile == nil)
                    return;
                
                /* note that we've started a script file */
                startedScript = true;
            }
            else if (rexMatch('<nocase><space>*y.*', str) != str.length())
            {
                "Cancelled. ";
                return;
            }
        }
        else
        {
            /* 
             *   they're already logging; just confirm that they want to
             *   see the instructions 
             */
            "; would you like to proceed?
            \n(Y is affirmative) &gt; ";

            /* stop if they don't want to proceed */
            if (!yesOrNo())
            {
                "Cancelled. ";
                return;
            }
        }

        /* make sure we have something for the next "\b" to skip from */
        "\ ";

        /* show each chapter in turn */
        showCommandsChapter();
        showAbbrevChapter();
        showTravelChapter();
        showObjectsChapter();
        showConversationChapter();
        showTimeChapter();
        showSaveRestoreChapter();
        showSpecialCmdChapter();
        showUnknownWordsChapter();
        showAmbiguousCmdChapter();
        showAdvancedCmdChapter();
        showTipsChapter();

        /* if we started a script file, close it */
        if (startedScript)
            ScriptOffAction.turnOffScripting(nil);
    }
    #endif /* INSTRUCTIONS_MENU */
    
    /* 
     *   modifications: placement of punctuation in relation to quotation 
     *   marks 
     */
    showCommandsChapter()
    {
        "\b<b>Entering Commands</b>\b
        You&rsquo;ve probably already noticed that you interact with the story
        by typing a command whenever you see the <q>prompt</q>, which
        usually looks like this:
        \b";

        gLibMessages.mainCommandPrompt(rmcCommand);

        "\bKnowing this much, you&rsquo;re probably thinking one of two things:
        <q>Great, I can type absolutely anything I want, in plain English,
        and the story will do my bidding,</q> or <q>Great, now I have to
        figure out yet another heinously complex command language for
        a computer program; I think I&rsquo;ll go play Minefield.</q>  Well,
        neither extreme is quite true.

        \bIn actual play, you&rsquo;ll only need a fairly small set of
        commands, and the commands are mostly in ordinary English, so
        there&rsquo;s not very much you&rsquo;ll have to learn or remember.
        Even though that command prompt can look intimidating, don&rsquo;t
        let it scare you off &mdash; there are just a few simple things you
        have to know.

        \bFirst, you&rsquo;ll almost never have to refer to anything that
        isn&rsquo;t directly mentioned in the story; this is a story, after
        all, not a guessing game where you have to think of everything that
        goes together with some random object.  For example, if
        you&rsquo;re wearing a jacket, you might assume that the jacket has
        pockets, or buttons, or a zip &mdash; but if the story never
        mentions those things, you shouldn&rsquo;t have to worry about them.

        \bSecond, you won&rsquo;t have to think of every conceivable action
        you could perform.  The point of the game isn&rsquo;t to make you
        guess at verbs.  Instead, you&rsquo;ll only have to use a relatively
        small number of simple, ordinary actions.  To give you an idea
        of what we mean, here are some of the commands you can use:";
        
        "\b
        \n\t LOOK AROUND
        \n\t INVENTORY
        \n\t GO NORTH (or EAST, SOUTHWEST, and so on, or UP, DOWN, IN, OUT)
        \n\t WAIT
        \n\t TAKE THE BOX
        \n\t DROP THE DISK
        \n\t LOOK AT THE DISK
        \n\t READ THE BOOK
        \n\t OPEN BOX
        \n\t CLOSE BOX
        \n\t LOOK IN THE BOX
        \n\t LOOK THROUGH WINDOW
        \n\t PUT FLOPPY DISK INTO BOX
        \n\t PUT BOX ON TABLE
        \n\t WEAR THE CONICAL HAT
        \n\t TAKE OFF HAT
        \n\t TURN ON LANTERN
        \n\t LIGHT MATCH
        \n\t LIGHT CANDLE WITH MATCH
        \n\t PUSH BUTTON
        \n\t PULL LEVER
        \n\t TURN KNOB
        \n\t TURN DIAL TO 11
        \n\t EAT COOKIE
        \n\t DRINK MILK
        \n\t THROW PIE AT CLOWN
        \n\t ATTACK TROLL WITH SWORD
        \n\t UNLOCK DOOR WITH KEY
        \n\t LOCK DOOR WITH KEY
        \n\t CLIMB THE LADDER
        \n\t GET IN THE CAR
        \n\t SIT ON THE CHAIR
        \n\t STAND ON THE TABLE
        \n\t STAND IN FLOWER BED
        \n\t LIE ON THE BED
        \n\t TYPE HELLO ON COMPUTER
        \n\t LOOK UP BOB IN PHONE BOOK";

        /* show the conversation-related verbs */
        foreach (local cur in conversationVerbs)
            "\n\t <<cur>>";

        /* show the custom verbs */
        foreach (local cur in customVerbs)
            "\n\t <<cur>>";

        /* 
         *   if the list is exhaustive, say so; otherwise, mention that
         *   there might be some other verbs to find 
         */
        if (allRequiredVerbsDisclosed)
            "\bThat&rsquo;s it &mdash; every verb and every command phrasing
            you need to complete the story is shown above.
            If you ever get stuck and feel like the story is expecting
            you to think of something completely out of the blue, remember
            this: whatever it is you have to do, you can do it with one
            or more of the commands shown above. 
            Don&rsquo;t ever worry that you have to start trying lots of
            random commands to hit upon the magic phrasing, because
            everything you need is shown plainly above. ";
        else
            "\bMost of the verbs you&rsquo;ll need to complete the story are
            shown above; there might be a few additional commands you&rsquo;ll
            need from time to time as well, but they&rsquo;ll follow the same
            simple format as the commands above.";

        "\bA few of these commands deserve some more explanation.
        LOOK AROUND (which you abbreviate to LOOK, or even just L)
        shows the description of the current location.  You can use
        this if you want to refresh your memory of your character&rsquo;s
        surroundings.  INVENTORY (or just I) shows a list of everything
        your character is carrying. WAIT (or Z) just lets a little
        time pass in the story.";
    }
    
    /* modifications: placement of punctuation relative to quotation marks */
    showTravelChapter() {
        "\b<b>Travel</b>
        \bAt any given time in the story, your character is in a
        <q>location</q>.  The story describes the location when your
        character first enters, and again any time you type LOOK.
        Each location usually has a short name that&rsquo;s displayed just
        before its full description; the name is useful when drawing a map,
        and the short name can help jog your memory as you&rsquo;re finding
        your way around.

        \bEach location is a separate room, or a large outdoor area,
        or the like.  (Sometimes a single physical room is so large that
        it comprises several locations in the game, but that&rsquo;s fairly
        rare.)  For the most part, going to a location is as specific
        as you have to get about travel; once your character is in a
        location, the character can usually see and reach everything
        within the location, so you don&rsquo;t have to worry about where
        exactly your character is standing within the room.  Once in
        a while, you might find that something is out of reach, perhaps
        because it&rsquo;s on a high shelf or on the other side of a moat; in
        these cases, it&rsquo;s sometimes useful to be more specific about
        your character&rsquo;s location, such as by standing on something
        (STAND ON TABLE, for example).

        \bTraveling from one location to another is usually done using
        a direction command: GO NORTH, GO NORTHEAST, GO UP, and so on.
        (You can abbreviate the cardinal and vertical directions to one
        letter each &mdash; N, S, E, W, U, D &mdash; and the diagonal
        directions to two: NE, NW, SE, SW.)  The story should always
        tell you the directions that you can go when it describes a
        location, so you should never have to try all the unmentioned
        directions to see if they go anywhere.

        \bIn most cases, backtracking (by reversing the direction you
        took from one location to another) will take you back to where you
        started, although some passages might have turns.

        \bMost of the time, when the story describes a door or passageway,
        you won&rsquo;t need to open the door to go through the passage, as
        the story will do this for you.  Only when the story specifically
        states that a door is blocking your way will you usually have to
        deal with the door explicitly.";
    }
    
    /* modifications: placement of punctuation relative to quotation marks */
    showTimeChapter() {
        "\b<b>Time</b>";

        if (isRealTime)
        {
            "\bTime passes in the story in response to the commands
            you type.  In addition, some parts of the story
            might unfold in <q>real time</q>, which means that things
            can happen even while you&rsquo;re thinking about your next move.

            \bIf you want to stop the clock while you&rsquo;re away from your
            computer for a moment (or you just need some time to think),
            you can type PAUSE.";
        }
        else
        {
            "\bTime passes in the story only in response to commands
            you type.  This means that nothing happens while the story
            is waiting for you to type something.  Each command takes
            about the same amount of time in the story.  If you
            specifically want to let some extra time pass within the
            story, because you think something is about to happen,
            you can type WAIT (or just Z). ";
        }
    }
    
    /* modifications: typo correction */
    showSaveRestoreChapter() {
        "\b<b>Saving and Restoring</b>
        \bYou can save a snapshot of your current position in the story
        at any time, so that you can later restore the story to the
        same position.  The snapshot will be saved to a file on your
        computer&rsquo;s disk, and you can save as many different snapshots
        as you&rsquo;d like (to the extent you have space on your disk,
        anyway).\b";

        switch (crueltyLevel)
        {
        case 0:
            "In this story, your character will never be killed, and
            you&rsquo;ll never find yourself in a situation where
            it&rsquo;s impossible to complete the story.  Whatever happens
            to your character, you&rsquo;ll always be able to find a way
            to complete the story. So, unlike in many text games, you
            don&rsquo;t have to worry about saving positions to protect
            yourself against getting stuck in impossible situations.  Of
            course, you can still save as often as you&rsquo;d like, to
            suspend your session and return to it later, or to save
            positions that you think you might want to revisit.";
            break;

        case 1:
        case 2:
            "It might be possible for your character to be killed in
            the story, or for you to find your character in an impossible
            situation where you won&rsquo;t be able to complete the story.
            So, you should be sure to save your position
            <<crueltyLevel == 1 ? 'from time to time' : 'frequently'>>
            so that you won&rsquo;t have to go back too far if this should
            happen. ";

            if (crueltyLevel == 2)
                "(You should make a point of keeping all of your old saved
                positions, too, because you might not always realize
                immediately when a situation has become impossible.
                You might find at times that you need to go back further
                than just the last position that you <i>thought</i>
                was safe.)";

            break;
        }

        "\bTo save your position, type SAVE at the command prompt.
        The story will ask you for the name of a disk file to use to store
        the snapshot.  You&rsquo;ll have to specify a filename suitable for
        your computer system, and the disk will need enough free space
        to store the file; you&rsquo;ll be told if there&rsquo;s any problem
        saving the file.  You should use a filename that doesn&rsquo;t already
        exist on your machine, because the new file will overwrite any
        existing file with the same name.

        \bYou can restore a previously saved position by typing RESTORE
        at any prompt.  The story will ask you for the name of the file
        to restore.  After the computer reads the file, everything in
        the story will be exactly as it was when you saved that file.";

        "\b<b>Undo</b>
        \bEven if you haven&rsquo;t saved your position recently, you can
        usually take back your last few commands with the UNDO command.
        Each time you type UNDO, the story reverses the effect of one command,
        restoring the story as it was before you typed that command.  UNDO
        is limited to taking back the last few commands, so this isn&rsquo;t
        a substitute for SAVE/RESTORE, but it&rsquo;s very handy if you
        find your character unexpectedly in a dangerous situation, or you
        make a mistake you want to take back.";
    }
    
    /* modifications: spelling of "recognise" */
    showSpecialCmdChapter() {
        "\b<b>Some Other Special Commands</b>
        \bThe story understands a few other special commands that
        you might find useful.

        \bAGAIN (or just G): Repeats the last command.  (If your last input
        line had several commands, only the last single command on the line
        is repeated.)
        \bINVENTORY (or just I): Shows what your character is carrying.
        \bLOOK (or just L): Shows the full description of your
        character&rsquo;s current location.";

        /* if the exit lister module is active, mention the EXITS command */
        if (gExitLister != nil)
            "\bEXITS: Shows the list of obvious exits from the current
            location.
            \bEXITS ON/OFF/STATUS/LOOK: Controls the way the game
            displays exit lists.  EXITS ON puts a list of
            exits in the status line and also lists exits in each room
            description.  EXITS OFF turns off both of these listings.
            EXITS STATUS turns on just the status line exit list, and
            EXITS ROOM turns on only the room description lists.";
        
        "\bOOPS: Corrects a single misspelled word in a command, without
        retyping the entire command.  You can only use OOPS immediately
        after the story tells you it didn&rsquo;t recognise a word in your
        previous command.  Type OOPS followed by the corrected word.
        \bQUIT (or just Q): Terminates the story.
        \bRESTART: Starts the story over from the beginning.
        \bRESTORE: Restores a position previously saved with SAVE.
        \bSAVE: Saves the current position in a disk file.
        \bSCRIPT: Starts making a transcript of your story session,
        saving all of the displayed text to a disk file, so that you
        can peruse it later or print it out.
        \bSCRIPT OFF: Ends a transcript that you started with SCRIPT. 
        \bUNDO: Takes back the last command.
        \bSAVE DEFAULTS: Saves your current settings for things like
        NOTIFY, EXITS, and FOOTNOTES as defaults.  This means that your
        settings will be restored automatically the next time you start
        a new game, or RESTART this one.
        \bRESTORE DEFAULTS: Explicitly restores the option settings
        you previously saved with SAVE DEFAULTS. ";
    }
    
    /* modifications: spelling of "recognise" */
    showUnknownWordsChapter() {
        "\b<b>Unknown Words</b>
        \bThe story doesn&rsquo;t pretend to know every word in the English
        language.  The story might even occasionally use words in its
        own descriptions that it doesn&rsquo;t understand in commands.  If
        you type a command with a word the story doesn&rsquo;t know, the
        story will tell you which word it didn&rsquo;t recognise.  If the
        story doesn&rsquo;t know a word for something it described, and
        it doesn&rsquo;t know any synonyms for that thing, you can probably
        assume that the object is just there as a detail of the setting,
        and isn&rsquo;t important to the story. ";
    }
    
    /* modifications: "full stop" rather than "period" */
    showAdvancedCmdChapter() {
        "\b<b>Advanced Command Formats</b>
        \bOnce you get comfortable with entering commands, you might
        be interested to know about some more complex command formats
        that the story will accept.  These advanced commands are all
        completely optional, because you can always do the same things
        with the simpler formats we&rsquo;ve talked about so far, but
        experienced players often like the advanced formats because
        they can save you a little typing.

        \b<b>Using Multiple Objects at Once</b>
        \bIn most commands, you can operate on multiple objects in
        a single command.  Use the word AND, or a comma, to separate
        one object from another:

        \b
        \n\t TAKE THE BOX, THE FLOPPY DISK, AND THE ROPE
        \n\t PUT DISK AND ROPE IN BOX
        \n\t DROP BOX AND ROPE
        
        \bYou can use the words ALL and EVERYTHING to refer to everything
        applicable to your command, and you can use EXCEPT or BUT
        (right after the word ALL) to exclude specific objects:

        \b
        \n\t TAKE ALL
        \n\t PUT ALL BUT DISK AND ROPE INTO BOX
        \n\t TAKE EVERYTHING OUT OF THE BOX
        \n\t TAKE ALL OFF SHELF

        \bALL refers to everything that makes sense for your command,
        excluding things inside other objects matching the ALL.  For example,
        if you&rsquo;re carrying a box and a rope, and the box contains
        a floppy disk, typing DROP ALL will drop the box and the rope,
        and the floppy will remain in the box.

        \b<b><q>It</q> and <q>Them</q></b>
        \bYou can use IT and THEM to refer to the last object or objects
        that you used in a command:

        \b
        \n\t TAKE THE BOX
        \n\t OPEN IT
        \n\t TAKE THE DISK AND THE ROPE
        \n\t PUT THEM IN THE BOX

        \b<b>Multiple Commands At Once</b>
        \bYou can put multiple commands on a single input line by
        separating the commands with full stops or the word THEN, or
        with a comma or AND.  For example:

        \b
        \n\t TAKE THE DISK AND PUT IT IN THE BOX
        \n\t TAKE BOX. OPEN IT.
        \n\t UNLOCK THE DOOR WITH THE KEY. OPEN IT, THEN GO NORTH.

        \b If the story doesn&rsquo;t understand one of the commands, it will
        tell you what it couldn&rsquo;t understand, and ignore everything after
        that on the line.";
    }
    
    /* modifications: spelling of "recognise" */
    showTipsChapter() {
        "\b<b>A Few Tips</b>
        \bNow that you know the technical details of entering commands,
        you might be interested in some strategy pointers.  Here are
        a few techniques that experienced interactive fiction players use
        when approaching a story.

        \bEXAMINE everything, especially when you enter a new location
        for the first time.  Looking at objects will often reveal details
        that aren&rsquo;t mentioned in the broader description of the location.
        If examining an object mentions detailed parts of the object,
        examine them as well.

        \bMake a map, if the story has more than a few locations.  When
        you encounter a new location for the first time, note all of its
        exits; this will make it easy to see at a glance if there are any
        exits you haven&rsquo;t explored yet.  If you get stuck, you can scan
        your map for any unexplored areas, where you might find what
        you&rsquo;re looking for.

        \bIf the story doesn&rsquo;t recognise a word or any of its synonyms,
        the object you&rsquo;re trying to manipulate probably isn&rsquo;t
        important to the story.  If you try manipulating something, and the
        story responds with something like <q>that isn&rsquo;t important,</q>
        you can probably just ignore the object; it&rsquo;s most likely just
        there as scenery, to make the setting more detailed.

        \bIf you&rsquo;re trying to accomplish something, and nothing you do
        seems to work, pay attention to what&rsquo;s going wrong.  If
        everything you try is met with utter dismissal (<q>nothing
        happens</q> or <q>that&rsquo;s not something you can open</q>), you
        might simply be on the wrong track; step back and think about other
        ways of approaching the problem.  If the response is something more
        specific, it might be a clue. <q>The guard says <q>you can&rsquo;t
        open that here!</q>\ and snatches the box from your hands</q> &mdash;
        this might indicate that you have to get the guard to leave, or
        that you should take the box somewhere else before you open it,
        for example.

        \bIf you get completely stuck, you might try setting aside the
        current puzzle and working on a different problem for a while.
        You might even want to save your position and take a break from
        the game.  The solution to the problem that&rsquo;s been
        thwarting your progress might come to you in a flash of insight
        when you least expect it, and those moments of insight can be
        incredibly rewarding.  Some stories are best appreciated when
        played over a period of days, weeks, or even months; if you&rsquo;re
        enjoying the story, why rush through it?

        \bFinally, if all else fails, you can try asking for help.  A
        great place to go for hints is the Usenet newsgroup
        <a href='news:rec.games.int-fiction'>rec.games.int-fiction</a>. ";

        "\n";
    }
;
