#include <adv3.h>
#include <en_us.h>

#include <storyguide.h>


/* 
 *   For convenience, a couple of macros to let me easily check whether the game
 *   is running in A) an interpreter that supports HTML and B) an interpreter
 *   and environment that supports banner windows.
 */
#define gHtmlTerp (systemInfo(SysInfoInterpClass) == SysInfoIClassHTML || gameMain.usingWebUI)
#define gBannerTerp (systemInfo(SysInfoBanners) && !gameMain.usingWebUI)


/*
 *   Our game credits and version information.  This object isn't required 
 *   by the system, but our GameInfo initialization needs this for some of 
 *   its information.
 */
versionInfo: GameID
    IFID = '35eba4cb-d2a3-6097-2240-f088c8aede85'
    name = 'It'
    byline = 'by Emily Boegheim'
    htmlByline = 'by <a href="mailto:emily.boegheim@gmail.com">Emily 
        Boegheim</a>'
    version = '2.2'
    firstPublished = '2011'
    authorEmail = 'Emily Boegheim <emily.boegheim@gmail.com>'
    desc = 'The rules of the game are easy. I\'m It, so I go and hide. You 
        and the others count to 50, then you have to look for me. The house and
        the shed are out of bounds. If you find me, you have to get into the 
        hiding spot with me. If you\'re the last person still looking for me, 
        then you lose. Got it?'
    genre = 'Slice of Life'
    licenseType = 'Freeware'
    copyingRules = 'No Restrictions'
    
    
    /* Show the ABOUT text. */
    showAbout()
    {
        "<i>It</i> is a small slice of life game. It isn't very
        long; I expect that playing through to an ending will 
        take five or ten minutes. However, it has many different 
        endings to discover, and I encourage you to play several 
        times to see some of the variations available.<.p>
        In general, the verbs you need to use are all standard
        or made clear in the game text. If you're new to 
        interactive fiction, you can visit the <<gHtmlTerp ? 
          aHref('help') + 'help</a>' : 'HELP'>> section to 
        learn the basic commands you need to know. For everyone
        else, probably the only thing you need to know is that
        <i>It</i> uses ASK/TELL for conversation. If you're
        struggling to get a good ending, or you want some ideas
        about finding some different endings, I've also got some
        general <<gHtmlTerp ? aHref('hints') + 'hints</a>' : 
            'HINTS'>> for you.<.p>
        Finally, a lot of other people helped make this 
        story possible. Please read the <<gHtmlTerp ? aHref('credits') + 
          'credits</a>' : 'CREDITS'>> and 
        appreciate their hard work. If you want to send me bug 
        reports, praise, advice, criticism, transcripts (I 
        <i>love</i> transcripts) or
        anything else, you can email me at <a 
        href='mailto:emily.boegheim@gmail.com'>emily.boegheim@gmail.com</a>.
        <.p>
        Thanks for playing. I hope you enjoy <i>It</i>! ";
    }
    
    
    showCredit()
    {
        /* title */
        "<center>
        <h2><b><<versionInfo.name>></b></h2><.p><br>";
        
        /* main by-line */
        showCreditsBlock('Written and Programmed By', ['Emily Boegheim']);
        "<br>";
        
        /* TADS 3 credit */
        showCreditsBlock('Programmed Using', tadsCredits, ', by ');
        "<br>";
        
        /* extensions credits */
        showCreditsBlock('Extensions', extCredits, ', by ');
        "<br>";
        
        /* cover art credits */
        showCreditsBlock('Cover Art', coverCredits, ' by ');
        "<br>";
        
        /* testing credits */
        showCreditsBlock('Testers', testCredits);
        
        /* post-release feedback credits */
        showCreditsBlock('Post-Release Feedback', feedbackCredits);
        
        /* stop centering the text */
        "</center>";
    }
    
    /* 
     *   Show a block of credits differently depending on whether the 
     *   interpreter is HTML-enabled.
     *
     *   "heading" is the heading of the block (e.g. "Extensions") and 
     *   "credits" is either a list of names or a list of lists of role/name 
     *   pairs. The final optional arguments, "sep" is a string to be used 
     *   in non-HTML interpreters to separate role/name pairs.
     */
    showCreditsBlock(heading, credits, ...)
    {
        /* grab the separator, if there is one */
        local sep = getArg(argcount);
        
        /* Can we use HTML? */
        local html = gHtmlTerp;
        
        /* heading */
        "<.p><b><<heading>></b><.p>";
        
        /* start the table */
        if (html)
            "<table cellspacing='6'>";
        
        foreach (local cred in credits)
        {
            /* start the row */
            if (html)
                "<tr>";
            
            if (cred.ofKind(String))
            {
                /* start the column, centre-aligned */
                if (html)
                    "<td align='center'>";
                
                /* print the name */
                say(cred);
            }
            else
            {
                /* start the column, right-aligned */
                if (html)
                    "<td align='right' width='50%'>";
            
                /* say the role */
                say(cred[1]);
                
                /* show the separator */
                if (!html && sep)
                    say(sep);
                else if (html)
                    "</td><td align='left' width='50%'>";
                
                /* say the name */
                say(cred[2]);
            }
            
            /* end the row */
            if (html)
                "</td></tr>";
            else
                "<br>";
        }
        
        /* end the table */
        if (html)
            "</table>";
        else
            "<.p>";
    }
    
    /* Various credits blocks */
    tadsCredits = [
        ['TADS 3', 'Mike Roberts']
    ]
    
    extCredits = [
        ['SayTopic & QueryTopic', 'Eric Eve'],
        ['CQuotes', 'Stephen Granade'],
        ['Custom Messages', 'Mark Engelberg, Eric Eve and Emily Boegheim'],
        ['Real NC Debug Actions', 'Nikos Chantziaras'],
        ['Spelling Corrector', 'Steve Breslin'],
        ['Story Guide', 'Kevin Forchione']
    ]
    
    coverCredits = [
        ['Artwork', 'Lisa Boegheim'],
        ['Digital Editing', 'Emily Boegheim']
    ]
    
    testCredits = [
        'Jonathan Blask',
        'Lisa Boegheim',
        'A. J. Crowley',
        'Kevin Jackson-Mead',
        'Paul Laroquod',
        'C. E. J. Pacian',
        'Savaric',
        'Ryan Veeder'
    ]
    
    feedbackCredits = [
        'Clash of the Type-Ins',
        'Victor Gijsbers',
        'Paul Lee',
        'Ralph Merridew',
        'Ron Newcomb',
        'Melvin Rangasamy',
        'Hannes Schller',
        'Andrew Schultz',
        'Mike Sousa',
        'NR Turner',
        'Matt Weiner',
        'and all the IFComp reviewers'
    ]
;

/*
 *   The "gameMain" object lets us set the initial player character and
 *   control the game's startup procedure.  Every game must define this
 *   object.  For convenience, we inherit from the library's GameMainDef
 *   class, which defines suitable defaults for most of this object's
 *   required methods and properties.
 */
gameMain: GameMainDef
    initialPlayerChar = me
    
    /* The status line is not immediately visible. */
    showStatus = nil
    statusHasBeenActive = nil
    
    /* Is the game running under WebUI? */
#ifdef TADS_INCLUDE_NET
    usingWebUI = true
#else
    usingWebUI = nil
#endif
    
    /* 
     *   The game begins abruptly, with Emma (but it isn't immediately clear 
     *   who is speaking) explaining the rules and demanding to know whether 
     *   the player understands. Jump straight into a yes/no ConvNode.
     */
    showIntro() 
    {
        /* Clear the screen. */
        cls();
        
        /* Hide the status line. */
        MenuItem.removeStatusLine();
        
        /* 
         *   The description in the game metadata is the same as the intro. Just
         *   print that to save duplicating the text.
         */
        "<.p><<versionInfo.desc>><.p>";
        
        /* Enter the first ConvNode. */
        emma.setConvNode('understand-rules');
    }
    
    /* 
     *   A method to show the title screen. This should be handled 
     *   differently depending on whether it's being shown in an 
     *   HTML-enabled interpreter.
     */
    showTitles() 
    {
        /* 
         *   Having to press any key every time I restart is getting really
         *   annoying. Stop that.
         */
        if (!titlesInfo.titlesSeen)
        {
            /* Flush the output of the main window. */
            flushOutput();
            
            /* Pause. */
            inputManager.pauseForMore(true);
        }
        
        /* 
         *   If running under WebUI, try showing the cover art; otherwise just
         *   use the text titles, appropriately formatted.
         */
        local titles = '<center>\b\b\b
        <h1><<versionInfo.name>></h1>\b
        <<versionInfo.byline>>
        </center>\b\b\b';
        
        if (gameMain.usingWebUI)
        {
            "<center>
            <img src=\"coverart/cover.jpg\" width=500 alt=\"It by Emily
            Boegheim\">
            </center><.p>";
            
            if (!titlesInfo.titlesSeen)
            {
                inputManager.pauseForMore(true);
            }
        }
        else
        {
            /* 
             *   Print the info in the main window - in bold if not an
             *   HTML-enabled terp, or formatted with heading tags if
             *   HTML-enabled.
             */
            if (!gHtmlTerp)
            {
                "<b>";
            }
            
            say(titles);
            
            if (!gHtmlTerp)
            {
                "</b>";
            }
        }
        
        /* 
         *   If the player has seen the titles before, it's not necessary to go
         *   any further. Otherwise, note that the titles have now been seen and
         *   keep going.
         */
        if (titlesInfo.titlesSeen)
        {
            return;
        }
        else
        {
            titlesInfo.titlesSeen = true;
        }
        
        /* 
         *   Suggest that the player read the ABOUT text to learn more about the
         *   game.
         */
        "<.notification>If this is the first time you've played <i>It</i>, please type
        ABOUT to learn more about the game.<./notification><.p>";


        
        /* 
         *   Is the game running in an HTML-enabled interpreter that also
         *   supports banner windows and images? (Since the creation of the
         *   WebUI, not all HTML-enabled interpreters support banner windows,
         *   hence the double-check.)
         */
        local supportedTerp = (gHtmlTerp && gBannerTerp
                               && systemInfo(SysInfoPrefImages));
        
        /* 
         *   If using an HTML-enabled, banner-enabled, image-enabled
         *   interpreter, try to create and write to a banner window taking up
         *   the entire screen. This makes the cover art appear alone in the
         *   window.
         *
         *   Wait for a key-press, then remove the banner, returning the player
         *   to the main game window.
         *
         *   I do it this rather hacky way so that I can have the cover art
         *   appear alone in the window without having to clear the screen
         *   (which would mean losing scrollback).
         */
        if (supportedTerp)
        {
            local win = bannerCreate(nil, BannerFirst, nil, BannerTypeText,
                         BannerAlignTop, 100, BannerSizePercent,
                         BannerStyleMoreMode);
            if (win) 
            {
                bannerSay(win, '<center>\b\b\b<img src="coverart/cover.jpg" 
                    width=500 alt="It by Emily Boegheim">
                    </center>');
                inputManager.pauseForMore(true);
                bannerDelete(win);
            }
        }
    }
;



/* 
 *   Keep track of whether the player has seen the title screen in this play
 *   session. Since a normal play session will involve at least one RESTART,
 *   this info needs to be stored in a transient object, so it persists across
 *   RESTART and UNDO.
 */
transient titlesInfo: object
    titlesSeen = nil
;




/* Help/instructions for playing It. */
VerbRule(Help)
    'help'
    : InstructionsAction
    verbPhrase = 'request/requesting help'
;

modify InstructionsAction
    showInstructions()
    {
        "<i>It</i> is a work of interactive fiction, a story you can participate in.
        The computer will describe the setting and tell you what is happening, while
        you guide the main character by typing in simple English phrases.<.p>
        For instance, to have a look at what's around your character, just type LOOK
        and press Enter. If you see something interesting, like a potential hiding spot, 
        you can LOOK AT it or SEARCH it. You can move around the landscape using 
        compass directions: GO SOUTHWEST (or just SOUTHWEST). When you find Emma, you 
        might need to HIDE IN her hiding spot.<.p>
        The other characters in the story have their own agendas. You can try speaking
        to them, though they might not be inclined to chat. If they ask you a question,
        you can SAY YES or NO. You can ASK or TELL them about things: for example, 
        ASK THE GIRL ABOUT THE GAME. Occasionally, they may also react to strange or 
        annoying things that your character does.<.p>
        You'll need to use other commands as well, sometimes. Pay attention to the 
        descriptions of places, things, and events; they may include hints about
        what you can do. For instance, at the beginning of the game you'll be told
        to CLOSE YOUR EYES and then COUNT. Some things may have obvious uses, like
        CLIMBing a ladder. Or sometimes you might just want to WAIT around for a
        while and see what happens.<.p>
        Lastly, if you make a mistake or don't like the results of an action, you can 
        always UNDO the action or RESTART right from the beginning. And if you 
        want to stop playing and come back later, you can SAVE your game, QUIT 
        for now, and then RESTORE the game when you're ready. ";
    }
;



/* Some general hints for playing the game. */
modify HintAction
    execSystemAction()
    {
        "If you're having trouble finding a good ending, or want to see some
        different endings, here are a few things you can try:<.p> ";
        
        /* 
         *   If running in a HTML interpreter, use a bulleted list. 
         *   Otherwise, create the bullet points in text.
         */
        if (gHtmlTerp)
        {
            "<ul>";
            foreach (local hint in hintsList)
            {
                "<li><<hint>>";
            }
            "</ul>";
        }
        else
        {
            foreach (local hint in hintsList)
            {
                "- <<hint>><br>";
            }
            "<.p>";
        }
    }
    
    hintsList = [
        'Explore. ',
        'Behave badly. ',
        'Bend the rules. Break the rules. Change the game. ',
        'Pay attention to the other characters\' actions and motivations. ',
        'Don\'t jump to conclusions about the other characters. ',
        'See if you can distract the other characters from the game, or 
        tempt them away. '
    ]
;



/* Define the player character. */
me: Person
    vocabWords = 'self'
    pcDesc = "You look just like normal. Well, maybe a bit more neat and tidy
        than normal. Mum was all fussy this morning. "
    location = lawnWest
    
    
    /* The game starts in the middle of a conversation with Emma. */
    lastInterlocutor = emma
    
    
    /* The player may or may not count. */
    hasCounted = nil
    
    
    /* 
     *   Did the player go east before Yvonne could? This can only happen if 
     *   the PC cheated and caught Yvonne cheating, but the player might 
     *   do that and still not go east before Yvonne, so this must be 
     *   tracked separately.
     */
    wentEastBeforeYvonne = nil
    
    
    /* 
     *   The PC's path to Emma's hiding spot will vary wildly, but always 
     *   begins at the west end of the lawn.
     */
    pathToHidingSpot = [lawnWest]
    
    
    /* 
     *   Yvonne is scary sometimes. Mostly 0 scary, but sometimes 1 
     *   scary, and sometimes even *2* scary. That's pretty scary.
     */
    isScaredOfYvonne = 0
    
    
    /* Does the PC know that Tiana prefers Hide & Seek to Sardines? */
    knowsTianaPrefersHnS = nil
    
    
    actorAction() 
    {
        /* 
         *   During the prologue, very few actions are allowed. Check with 
         *   the prologue object to see whether the current action is okay.
         */
        if (gCurrStoryPoint == prologue && !prologue.allowAction(gAction))
        {
            emma.impatientList.doScript();
            exit;
        }
        
        /* 
         *   Instead of looking when the player's eyes are closed, try 
         *   opening the eyes.
         */
        if (!eyes.isOpen && gActionIs(Look))
        {
            /* 
             *   There's a bit more going on here than your average LOOK, so
             *   we'll make this one take a turn, even though LOOK doesn't
             *   usually.
             */
            gAction.actionTime = 1;
            replaceAction(Open, eyes);
        }
    }
    
    
    /* Choosing roles for Hide and Seek. */
    choosesHnSRole(role)
    {
        gPlayerChar.roleInHnS = role;
        
        "<q>I'll <<role == roleCounting ? 'count' : 'hide'>>,</q> you say, 
        <q>and you can <<role == roleCounting ? 'hide' : 'count'>>, 
        okay?</q><.p>";
        
        local players = Person.peoplePlayingHnS - gPlayerChar;
        gMessageParams(players);
        
        "{The/list players} nod{s}. ";
    }
    
    
    /* 
     *   Don't look around after restoring the game if the prologue is still
     *   happening. (Have to test for SaveAction rather than RestoreAction
     *   because after restoring, gAction gets reset to the action that was
     *   happening when you saved the game!)
     */
    lookAround(verbose)
    {
        if (gActionIs(Save) && gCurrStoryPoint == prologue)
        {
            if (eyes.isOpen)
            {
                "<.p>(Emma is waiting for you to close your eyes.) ";
            }
            return;
        }
        else
        {
            inherited(verbose);
        }
    }
;


/* 
 *   If the PC is too scared of Yvonne, she will refuse to follow Yvonne 
 *   across the lawn.
 */
+ BeforeAction
    actor = gPlayerChar
    action = TravelViaAction
    condition = (gPlayerChar.isScaredOfYvonne == 2 &&
                 gPlayerChar.isIn(lawnWest) &&
                 (gDobj is in (lawnEast, cubbyArea)))
    stopAction = true
    
    react()
    {
        "No way. Getting beaten up by Yvonne would hurt. You'd better find 
        some other place to search for now. ";
    }
;


/* If the PC leaves the lawn, reset her scared-of-Yvonne-ness. */
+ AfterAction
    actor = gPlayerChar
    action = TravelViaAction
    condition = (gPlayerChar.getOutermostRoom() != lawnWest)
    isOneOff = true
    
    react()
    {
        gPlayerChar.isScaredOfYvonne = 0;
    }
;



/* Silently note the fact that the PC went east before Yvonne could. */
+ AfterAction
    actor = gPlayerChar
    action = TravelViaAction
    dobj = lawnEast
    condition = (yvonne.isIn(lawnWest))
    
    react()
    {
        gPlayerChar.wentEastBeforeYvonne = true;
    }
;


/* Silently note which rooms the PC has visited. */
+ AfterAction
    actor = gPlayerChar
    action = TravelViaAction
    
    react()
    {
        if (gDobj != noTravel)
        {
            gPlayerChar.pathToHidingSpot += gDobj;
        }
    }
;


/* 
 *   If all the characters are cheating, and Yvonne is cheating because the 
 *   PC told her to, the game is completely broken.
 */
+ AfterAction
    condition = (yvonne.isCheating && tiana.isCheating && 
                 gPlayerChar.isCheating && 
                 !yvonne.isCheatingOfHerOwnAccord)
    
    react()
    {
        "<.p>You, {the tiana/she} and Yvonne all head towards the bush 
        Emma is hiding in. The three of you try to avoid catching each 
        other's eyes.<.p>
        Emma looks at you suspiciously. <q>You were all really quick,</q>
        she points out. <q>You're not supposed to help each other search,
        you know, it's like a race.</q><.p>
        None of you say anything. {The tiana/she} wriggles and looks at 
        her feet.<.p>
        Emma stares at you all. <q>Why are you being so weird?</q> she 
        demands. <q>I don't want to play this game any more if---</q>
        She stops suddenly and dashes off towards the house, leaving 
        the rest of you to follow in silence. ";
        
        endGame('You have broken the game');
    }
;


/* 
 *   Instead of saying HELLO or TALKing TO any of the NPCs, suggest the 
 *   player ASK or TELL them about something specific.
 */
+ BeforeAction
    actor = gPlayerChar
    action = [TalkToAction]
    condition = (gDobj.ofKind(Person) && gDobj != gPlayerChar)
    
    stopAction = true
    
    react()
    {
        "Try ASKing or TELLing {the dobj/her} about something specific 
        instead. ";
        
        /* Treat this as if it were a system action - it takes no time. */
        gAction.actionTime = 0;
    }
;


/* 
 *   The player has eyes that can be open or closed. At one point in the 
 *   story (before counting) they must be closed.
 */
+ eyes: Openable, Component
    'eye/eyes'
    'your eyes'
    "You can't see them without a mirror. "
    isQualifiedName = true
    isPlural = true
    initiallyOpen = true
    openStatusReportable = nil
    
    haveBeenOpen = nil
    
    /* 
     *   Closing your eyes is necessary during the prologue (except for the 
     *   early endings), possible when playing Hide and Seek instead of 
     *   Sardines, and pointless at any other time.
     */
    dobjFor(Close) 
    {
        
        /* Don't need to be able to touch your eyes to close them. */
        preCond = []

        check() 
        {
            inherited();
            
            if (gCurrStoryPoint not in (prologue, counting, playingHnS))
            {
                failCheck('There isn\'t any point in closing your eyes now. ');
            }
            
            /* 
             *   If the Hide and Seek players have decided that the PC will 
             *   hide, there's no point in her closing her eyes.
             */
            if (gPlayerChar.roleInHnS == roleHiding)
            {
                failCheck('You\'re supposed to hide this time, not count. ');
            }
        }
        
        action()
        {
            if (gCurrStoryPoint == prologue)
            {
                inherited();
                
                /* 
                 *   This suppresses the default "Closed." report and any 
                 *   paragraph breaks that would follow. Reporting is handled by 
                 *   the scene change mechanism.
                 */
                "<.P0>";
            }
            else if (gCurrStoryPoint == counting)
            {
                /* This is the PC deciding not to cheat after all. */
                gPlayerChar.isCheating = nil;
                
                /* Move the characters back into darkness. */
                [me, yvonne, tiana].forEach({x: x.moveIntoForTravel(dark)});
                
                /* Put Yvonne and Tiana back into their counting states.*/
                tiana.setCurState(tianaSCounting);
                yvonne.setCurState(yvonneCounting);
                
                /* Leave the status line as is. */
                
                /* Close the eyes. */
                makeOpen(nil);
                
                /* Report the action. */
                "You close your eyes again. Yvonne and {the tiana/she} continue
                to count:<.p>";
            }
            else /* gCurrStoryPoint == playingHnS */
            {
                /* 
                 *   If the players haven't decided on roles yet, treat this 
                 *   action as an answer to that question.
                 */
                if (gPlayerChar.roleInHnS == nil)
                {
                    gPlayerChar.choosesHnSRole(roleCounting);
                }
                
                /* 
                 *   It's clear that this is intended to be an action, not 
                 *   just an answer to Tiana's question, so go ahead and 
                 *   count as well (which ends the game).
                 */
                replaceAction(Count);
            }
        }
    }
    
    /* Opening your eyes during "counting" is cheating. */
    dobjFor(Open)
    {
        /* Don't need to be able to touch your eyes to open them. */
        preCond = []
        
        action()
        {
            /* Note that the player is cheating. */
            gPlayerChar.isCheating = true;
            
            /* Do standard eye-opening stuff. */
            counting.openEyes();
        }
    }
    
    /* Kids, don't go sticking things into your eyes. */
    iobjFor(PutIn) 
    {
        verify() {}
        
        check() 
        {
            failCheck(dontTouchMsg);
        }
    }
    
    /* 
     *   Allow attempts to manhandle the eyes. (They'll be caught by the 
     *   beforeAction handler.)
     */
    dobjFor(Take) { verify() {} }
    dobjFor(Push) { verify() {} }
    dobjFor(Pull) { verify() {} }
    dobjFor(Move) { verify() {} }
    
    dontTouchMsg = 'That might hurt. '
;


/* 
 *   A general catch-all for actions that require you to touch your eyes. 
 *   Check the action's preconditions for touchObj - if found, reject the 
 *   action.
 */
++ BeforeAction
    actor = gPlayerChar
    
    condition()
    {
        if (gDobj == eyes && 
            eyes.(gAction.preCondDobjProp).valWhich({x: x == touchObj}))
        {
            return true;
        }
        
        if (gIobj == eyes && 
            eyes.(gAction.preCondIobjProp).valWhich({x: x == touchObj}))
        {
            return true;
        }
        
        return nil;
    }
    
    stopAction = true
    
    react() {
        say(location.dontTouchMsg);
    }
;
