#include <adv3.h>
#include <en_us.h>

#include <storyguide.h>

/* 
 *   A macro for removing the vocabulary for verbs (so that although the 
 *   verbs still exist, the player can't access them).
 */
#define removeVerb(name) modify VerbRule(name) ' ' :


/* It's distracting to have so many useless verbs, so remove them. */
removeVerb(Throw);
removeVerb(ThrowTo);
removeVerb(ThrowToType2);
removeVerb(ThrowAt);
removeVerb(ThrowDir);
removeVerb(ThrowDirDown);

removeVerb(Port);
removeVerb(Starboard);

removeVerb(DigWith);

removeVerb(JumpOver);
removeVerb(JumpOff);
removeVerb(JumpOffI);

removeVerb(MoveTo);
removeVerb(MoveWith);

removeVerb(TurnTo);
removeVerb(TurnWith);

removeVerb(Set);
removeVerb(SetTo);

removeVerb(TypeOn);
removeVerb(TypeLiteralOn);
removeVerb(TypeLiteralOnWhat);
removeVerb(EnterOn);
removeVerb(EnterOnWhat);

removeVerb(Consult);
removeVerb(ConsultAbout);
removeVerb(ConsultWhatAbout);

removeVerb(Switch);
removeVerb(Flip);
removeVerb(TurnOn);
removeVerb(TurnOff);

removeVerb(Light);
removeVerb(Burn);
removeVerb(BurnWith);
removeVerb(Extinguish);

removeVerb(CutWithWhat);
removeVerb(CutWith);

removeVerb(Pour);
removeVerb(PourInto);
removeVerb(PourOnto);

removeVerb(Clean);
removeVerb(CleanWith);

removeVerb(AttachTo);
removeVerb(AttachToWhat);
removeVerb(DetachFrom);
removeVerb(Detach);

removeVerb(Sleep);

removeVerb(Fasten);
removeVerb(FastenTo);
removeVerb(Unfasten);
removeVerb(UnfastenFrom);

removeVerb(PlugInto);
removeVerb(PlugIntoWhat);
removeVerb(PlugIn);
removeVerb(UnplugFrom);
removeVerb(Unplug);

removeVerb(Screw);
removeVerb(ScrewWith);
removeVerb(Unscrew);
removeVerb(UnscrewWith);

removeVerb(PushTravelDir);
removeVerb(PushTravelThrough);
removeVerb(PushTravelEnter);
removeVerb(PushTravelGetOutOf);
removeVerb(PushTravelClimbUp);
removeVerb(PushTravelClimbDown);

removeVerb(HintsOff);

removeVerb(QueryImplicit);


/* "Y" can also mean yes. And so can a bunch of other synonyms. */
modify VerbRule(Yes)
    (('say' | 'nod' | ) ('yes' | 'y' | 'affirmative' | 'sure' | 'yep' | 'yeah' | 
                'okay' | 'ok' | 'got' 'it' | 'yup' | 'definitely' | 'fine'
                | 'whatever' | 'right')
     | 'nod')
    :
;

/* synonyms for "no" */
modify VerbRule(No)
    ('say' | ) ('no' | 'negative' | 'no' 'way' | 'nuh-uh' | 'nuh' 'uh' |
                'definitely' 'not' | 'not' 'happening')
    :
;

/* 
 *   Can't make "N" always mean "no", since it's a common abbreviation for 
 *   north. But I can fudge it.
 */
modify TravelAction
    beforeAction() 
    {
        if (gAction.getEnteredVerbPhrase == 'n' && nCanMeanNo())
            replaceAction(No);
        inherited();
    }
    
    nCanMeanNo() 
    {
        if (prologue.active_)
            return true;
        return nil;
    }
;



/* 
 *   Make LOOK and EXAMINE actions take no time (unless EXAMINE also triggers a
 *   SEARCH action, but deal with that elsewhere).
 */
modify LookAction
    actionTime = 0
;

modify ExamineAction
    actionTime()
    {
        if (gCurrStoryPoint == counting)
            return 1;
        else
            return 0;
    }
;

/* 
 *   Travel actions should only take time if they succeed - i.e. if they make it
 *   as far as becoming a TravelViaAction.
 */
modify BasicLocation
    cannotGoThatWay()
    {
        gAction.actionTime = 0;
        inherited();
    }
;


/* Counting actions. We have to handle both COUNT and COUNT [TO] #. */
DefineIAction(Count)
    /* Only necessary during the counting or playing Hide & Seek scenes */
    checkAction()
    {
        if (gCurrStoryPoint not in (counting, playingHnS))
        {
            if (counting.completed_)
            {
                reportFailure('You\'ve already counted. Now you just need to 
                    find Emma as quickly as possible. ');
                exit;
            }
            else
            {
                reportFailure('You have to close your eyes before you count. 
                    ');
                exit;
            }
        }
    }
    
    execAction()
    {
        /* 
         *   If Tiana is the one doing the counting, it must be because the 
         *   PC told her to.
         */
        if (gActor == tiana && tiana.curConvNode == tianaCountOrHide)
        {
            tellTianaYoullHide.topicResponse();
            exit;
        }
        
        
        /* If in the counting scene, count to 50 and end the scene. */
        if (gCurrStoryPoint == counting)
        {
            "{You/he} <<counting.countedTo == 0 ? 'begin to count' : 'join in 
                with the others'>>: ";
            gPlayerChar.hasCounted = true;
            
            /* if the count is up to 50, skip the continue counting bit */
            if (counting.countedTo < 50)
            {
                "<q>";
                
                /* begin/continue counting */
                counting.continueCount();
                
                /* a bit of a break */
                "</q><.p>...<.p>";
            }
            
            /* end counting */
            counting.endCount();
        }
        else /* gCurrStoryPoint == playingHnS */
        {
            /* 
             *   If this is in response to Tiana's question and the roles 
             *   haven't been decided yet, treat it as speaking only - don't 
             *   go ahead with the action. But I need to make it clear to 
             *   the player that I'm doing it this way, so s/he doesn't 
             *   think it's a bug. Print a command clarification to let the 
             *   player knows what's going on.
             */
            if (tiana.curConvNode == tianaCountOrHide &&
                !gPlayerChar.roleInHnS)
            {
                /* remap to SayToAction */
                remapToSayTo(tiana, tCounting);
                
                /* 
                 *   make it clear to the player that s/he has only *agreed* 
                 *   to count and still needs to *actually* count
                 */
                "<.p>{The tiana/she} fidgets, waiting for you to actually
                start counting. ";
                tiana.lastAction = WaitAction.createActionInstance();
                
                /* stop the action */
                exit;
            }
            
            /* If the PC's role is hiding, then remind the player and exit. */
            if (gPlayerChar.roleInHnS == roleHiding)
            {
                reportFailure('{The tiana/she} is counting. You\'re supposed 
                    to hide this time. ');
                exit;
            }
            
            /* Counting once the roles are decided ends the game. */
            local players = Person.peoplePlayingHnS - gPlayerChar;
            gMessageParams(players);
            
            "Closing your eyes, you begin to count: <q>One... two...
            three...</q> Footsteps patter across the grass and fade into
            the distance as {list/the players} run{s} off to hide.<.p>";
            
            if (yvonne.isPlayingHnS) /* Emma isn't, Tiana is */
            {
                "When you start to search for Yvonne and {the tiana/her},
                Emma is nowhere to be seen. ";
                
                if (!emma.knowsEveryoneIsPlayingHnS)
                {
                    "She doesn't turn up until after you've found both
                    of them. ";
                }
                else
                {
                    ". ";
                }
                
                playingHnS.playingWithYvonne();
            }
            else /* Only Tiana is playing with you */
            {
                "It's not until you finish counting and start searching
                for {the tiana/her} that Emma and Yvonne turn up.<.p>";
                
                playingHnS.playingWithTiana();
            }
            
            /* 
             *   It's also possible to get all three playing with you, but 
             *   the game finishes as soon as you persuade Emma to join, so 
             *   she doesn't need to be accounted for here.
             */
        }
    }
;

VerbRule(Count)
    'count'
    : CountAction
    verbPhrase = 'count/counting'
;


/* 
 *   COUNT TO #. This is a bit trickier, as there's no way to tell TADS that 
 *   we need a number in the command. So we'll have to accept a literal 
 *   phrase and then parse it to get a number, rejecting it if it is not a 
 *   number.
 */
DefineIAction(CountTo)
    execAction
    {
        /* get the number the player entered */
        local num = numMatch.getval();
        
        /* if num is nil, tell the player she must enter a number */
        if (num == nil)
        {
            reportFailure('{You/he} can only count to a number. ');
            exit;
        }
        
        /* 
         *   check with the main CountAction to see if it's the right point 
         *   in the story for pointing
         */
        CountAction.checkAction();
        
        /* 
         *   If this is the start of the game, require the PC to count to 50 
         *   exactly.
         */
        if (!gPlayerChar.isPlayingHnS)
        {
            /* 
             *   check whether the player is counting to 50; if not, object and 
             *   stop the action
             */
            if (num < 50)
            {
                reportFailure('{You/he} have to count to 50. It would be cheating 
                    if {you/he} only counted to ' + num + '. ');
                exit;
            }
            else if (num > 50)
            {
                reportFailure('{You/he} only have to count to 50. If {you/he} count 
                    more than that, the others will get ahead of {you/him}. ');
                exit;
            }
        }
        
        /* replace the action with the main CountAction */
        replaceAction(Count);
    }
;

VerbRule(CountTo)
    ('count' ('to' | ) | 'say') singleNumber
    : CountToAction
    verbPhrase = 'count/counting (to what)'
;


/* Peeking. */
VerbRule(Peek)
    'peek'
    : LookAction
    verbPhrase = 'peek/peeking'
;



/* Searching. */
VerbRule(LookBetween)
    ('look' | 'l' | 'peek') ('between' | 'among' | 'around' | 'within') dobjList
    : SearchAction
    verbPhrase = 'search/searching (what)'
;

VerbRule(LookUnderneath)
    ('look' | 'l' | 'peek') ('under' | 'underneath' | 'beneath' | 'below') dobjList
    : LookUnderAction
    verbPhrase = 'look/looking under (what)'
;

modify VerbRule(LookBehind)
    ('look' | 'l' | 'peek') 'behind' dobjList
    :
    verbPhrase = 'look/looking (behind what)'
;

modify Thing    
    /* 
     *   Looking in, under or behind something is the same as searching it. 
     *   The library redirects "search" to "look in", but I prefer it the 
     *   other way round, so I'll do the bit of extra work to Do It My Way. 
     *   It makes my source code cleaner and more consistent.
     */
    dobjFor(LookIn) asDobjFor(Search)
    dobjFor(LookUnder) asDobjFor(Search)
    dobjFor(LookBehind) asDobjFor(Search)
    
    dobjFor(Search)
    {
        /* Get rid of the library's default remap to LookIn. */
        remap() {}
        preCond = [objVisible, touchObj]
        verify() {}
        
        /* 
         *   By default, searching something just points out that it isn't a 
         *   possible hiding place.
         */
        check() 
        {
            failCheck(&notAHidingSpotMsg);
        }
    }
    
    notAHidingSpotMsg = 'You can tell just by looking at {the dobj/him} that 
        Emma isn\'t hiding there. '
;


/* 
 *   Containers override Thing's handling of SEARCH, so we have to do a bit 
 *   of extra handling here.
 */
modify Container
    dobjFor(LookIn) asDobjFor(Search)
    
    dobjFor(Search)
    {
        remap = inherited Thing()
        preCond = inherited Thing()
        verify = inherited Thing()
        check = inherited Thing()
    }
;


/* 
 *   Attempting to search an entire room should prompt the player to search 
 *   individual objects instead.
 */
modify Room
    dobjFor(Search)
    {
        check()
        {
            /* 
             *   get a list of all nearby hiding spots, included those 
             *   already searched
             */
            local hsList = gActor.getNearbyHidingSpots(nil);
            
            /* prompt the player */
            "<.parser>You need to be more specific. Try searching some of the 
            possible hiding spots around here";
            
            /* 
             *   print a list of suggested hiding spots, if there are any 
             *   available
             */
            if (hsList.length() > 0)
            {
                gMessageParams(hsList);
                ", like {the hsList/list}";
            }
            
            ".<./parser> ";
            
            /* stop the action */
            exit;
        }
    }
;


/* SEARCHing FOR a character */
DefineTAction(SearchFor);

VerbRule(SearchFor)
    (('search' | 'look' | 'hunt') 'for' | 'find') dobjList
    : SearchForAction
    verbPhrase = 'searching/searching (for whom)'
;

modify Thing
    dobjFor(SearchFor)
    {
        action()
        {
            "You can already see {the dobj/her}. ";
            
            /* Is the object is in the same place as the PC? */
            if (gDobj.location == gPlayerChar.location)
            {
                "{It's/she's dobj} right here. ";
            }
            /* 
             *   Some items, e.g. room parts like the grass, don't technically
             *   have a location. I don't think it's worth giving them special
             *   behaviour, so I'll just restrict this code to objects that do
             *   have a location.
             */
            else if (gDobj.location)
            {
                /* If not, tell the player where it is. */
                local loc = gDobj.location;
                gMessageParams(loc);
                "{It's/she's dobj} <<gDobj.posture == standing ? '' : 
                  gDobj.posture.participle>> {in loc}";
                
                /* 
                 *   If the object is not directly in a room and isn't in the
                 *   same room as the PC, mention the outer location as well.
                 */
                if (!loc.ofKind(Room) && 
                    gDobj.getOutermostRoom != gPlayerChar.getOutermostRoom)
                {
                    local outerLoc = gDobj.getOutermostRoom();
                    gMessageParams(outerLoc);
                    " {in outerLoc}";
                }
                
                /* Either way, end the sentence. */
                ".";
            }
        }
    }
;

modify Room
    dobjFor(SearchFor)
    {
        action()
        {
            /* Is the PC in the room in question? */
            if (gPlayerChar.isIn(gDobj))
            {
                "You're already {in dobj}. ";
            }
            else
            {
                /* 
                 *   TODO: I should really put in some pathfinding instead to
                 *   help people who are feeling a bit lost.
                 */
                "{The dobj/he} {is} just over there. ";
            }
        }
    }
;



/* 
 *   Since I've changed all occurrences of the word "Exits" to "Directions", 
 *   allow DIRECTIONS as a synonym for the EXITS command.
 */
VerbRule(Directions)
    'directions'
    : ExitsAction
    verbPhrase = 'show/showing directions'
;

VerbRule(DirectionsMode)
    'directions' ('on'->on_ | 'all'->on_
             | 'off'->off_ | 'none'->off_
             | ('status' ('line' | ) | 'statusline') 'look'->on_
             | 'look'->on_ ('status' ('line' | ) | 'statusline')
             | 'status'->stat_ ('line' | ) | 'statusline'->stat_
             | 'look'->look_)
    : ExitsModeAction
    verbPhrase = 'turn/turning off directions display'
;



/* Allow L THING, LOOK THING and WATCH THING as synonyms for LOOK AT THING. */
VerbRule(LTransitive)
    ('l' | 'look' | 'watch') dobjList
    : ExamineAction
    verbPhrase = 'look/looking at (what)'
;



/* Easter egg actions. */

/* SWING and SWING ON. */
DefineIAction(Swing)
    execAction()
    {
        askForDobj(SwingOn);
    }
;

VerbRule(Swing)
    'swing'
    : SwingAction
    verbPhrase = 'swing/swinging'
;

DefineTAction(SwingOn);

VerbRule(SwingOn)
    'swing' 'on' singleDobj
    : SwingOnAction
    verbPhrase = 'swing/swinging (on what)'
;

modify Thing
    dobjFor(SwingOn)
    {
        verify()
        {
            illogical('{The dobj/he} {is}n\'t really something you can 
                swing on. ');
        }
        
        preCond = [actorInObj]
    }
;

/* 
 *   The actor must be in or on a thing in order to swing on it, which means 
 *   creaing a new PreCondition.
 */
actorInObj: PreCondition
    verifyPreCondition(obj)
    {
        if (!gActor.isIn(obj))
            logicalRank(80, 'actor isn\'t in');
    }
    
    checkPreCondition(obj, allowImplicit)
    {
        /* if the actor is already in the object, no worries */
        if (gActor.isIn(obj))
            return nil;
        
        /* 
         *   If implicit actions are allowed, try to meet the condition. The 
         *   only swingable thing in the game is the swing, so the implicit 
         *   action can just be SitOn. Simplicity!
         */
        if (allowImplicit && tryImplicitAction(SitOn, obj))
        {
            if (gActor.isIn(obj))
                /* Success! We're done! */
                return true;
            else
                /* Failure, give up. */
                exit;
        }
        
        /* 
         *   If we've got this far, then we can't implicitly sit on it. Oh 
         *   dear.
         */
        reportFailure('You\'d have to be sitting on it to swing on it. ');
        
        /* make it the pronoun */
        gActor.setPronounObj(obj);

        /* abort the command */
        exit;
    }
;



modify WaitAction
    execAction()
    {
        /* 
         *   Waiting when the PC is in a hiding spot at any other time 
         *   triggers the "misunderstanding" ending.
         */
        if (gActor.location.ofKind(HidingSpot))
        {
            gAction.setMessageParam('loc', gActor.location);
            "Hiding {in loc}, you settle down to wait. And wait. And wait.
            <.p>
            Ages later, you hear the others calling your name. When you 
            emerge from {the loc/him}, they stare at you, puzzled.<.p>
            <q>What were you doing there?</q> Emma demands. <q>We waited
            ages for you to find us!</q><.p>
            You say nothing.<.p>
            Emma laughs suddenly. <q>You thought we were playing Hide and
            Seek, didn't you? That's not how the game goes! When I'm It,
            you have to come find me, not the other way around!</q><.p>
            Yvonne and {the tiana/she} giggle and whisper. Never mind, it's 
            almost time to go home. ";
            endGame('You misunderstood the game');
        }
        
        
        /* 
         *   Waiting when the PC is in the tree triggers an ending. A good 
         *   ending! Too few of those in this game.
         */
        if (gActor.isIn(upTree))
        {
            if (gActor.isPlayingHnS)
            {
                /* treat it as hiding */
                replaceAction(HideIn, topOfTree);
            }
            else
            {
                "You sit very still, very quiet, and wait. The baby bird cheeps
                and flaps its tiny wings and stretches up in the hope of being fed.
                You wait. Your nose itches. You wait. Your knees start to ache
                from gripping the branch, but you wait.<.p>
                Finally the mother bird returns with an insect of some sort and 
                drops it into the baby's open beak. The baby gulps it down and 
                chirps cheerfully. The mother bird hardly even glances at you as she
                settles back down into the nest.<.p>
                When you climb back down the tree, the others complain about you
                not playing the game properly, but you don't pay much attention. ";
                endGame('You have found something better to do');
            }
        }
        
        
        /* 
         *   If the method got this far, then there's nothing special about 
         *   the PC's situation. Just do the ordinary "time passes" thing.
         */
        inherited();
    }
;



/* 
 *   Hiding. Originally this was just going to be a synonym for ENTER, but 
 *   since I decided to make hiding a posture, it needs to be a bit more 
 *   complicated than that. 
 */
DefineTAction(HideIn);

VerbRule(HideIn)
    ('hide' | 'go' | 'get')
    ('in' | 'inside' | 'within' | 'between' | 'among' | 'behind' | 'under' | 
     'underneath' | 'beneath') singleDobj
    : HideInAction
    verbPhrase = 'hide/hiding (in what)'
;

modify Thing
    dobjFor(HideIn)
    {
        verify()
        {
            illogical('{The dobj/he} wouldn\'t make a very good hiding spot. ');
        }
        
        preCond = [touchObj]
    }
;

DefineIAction(Hide)
    execAction()
    {
        /* 
         *   If Tiana is the one taking this action, it must be because the 
         *   PC told her to.
         */
        if (gActor == tiana && tiana.curConvNode == tianaCountOrHide)
        {
            tellTianaYoullCount.topicResponse;
            exit;
        }
        
        /* 
         *   If Tiana just asked the PC which Hide and Seek role she wants 
         *   to take, treat this as an answer rather than an action.
         */
        if (tiana.curConvNode == tianaCountOrHide &&
            !gPlayerChar.roleInHnS)
        {
            /* remap to SayToAction */
            remapToSayTo(tiana, tHiding);
            
            /* stop the action */
            exit;
        }
        
        
        /* Otherwise, just look for an appropriate direct object. */
        askForDobj(HideIn);
    }
;

VerbRule(Hide)
    'hide'
    : HideAction
    verbPhrase = 'hide/hiding'
;


DefineTAction(HideWith)
    /* 
     *   The PC can't hide in more than one place. If the direct objects 
     *   aren't in the same place, protest and stop the action.
     */
    beforeActionMain()
    {
        if (dobjList_.length > 1)
        {
            foreach (local i in dobjList_)
            {
                foreach (local j in dobjList_)
                {
                    if (!i.obj_ || !j.obj_ ||
                        i.obj_.location != j.obj_.location)
                    {
                        local msg = 'You can\'t hide with ' + 
                            (dobjList_.length == 2 ? 'both' : 'all') + ' of 
                                them. They aren\'t in the same place. ';
                        failCheck(msg);
                    }
                }
            }
            
            /* 
             *   If we get this far safely, the NPCs are all in the same 
             *   place. Trim the direct object list back to a single NPC.
             */
            dobjList_ = dobjList_.sublist(1, 1);
        }
    }
;

VerbRule(HideWith)
    'hide' 'with' dobjList
    : HideWithAction
    verbPhrase = 'hide/hiding (with whom)'
;

modify Thing
    dobjFor(HideWith)
    {
        verify()
        {
            illogical('You can\'t hide with {the dobj/her} --- {it dobj/she}
                {is}n\'t even hiding! ');
        }
    }
;

modify Person
    dobjFor(HideWith)
    {
        verify()
        {
            if (!isIn(bushNearCubby))
                illogicalNow('{The dobj/she} {is}n\'t hiding. ');
        }
        
        action()
        {
            replaceAction(HideIn, location);
        }
    }
;


/* 
 *   modifications to SIT, which in most cases should behave similarly to HIDE
 *   IN
 */
modify VerbRule(SitOn)
    'sit' (('down' | ) 'on' | 'in' | 'inside' | 'within' | 'between' | 'among' | 
           'behind' | 'under' | 'underneath' | 'beneath')
        singleDobj
    : 
;

modify HidingSpot
    dobjFor(SitOn)
    {
        verify()
        {
            if (gActor.isIn(self) && gActor.posture == hiding)
                illogicalNow('{You/she} {is} already hiding {in dobj}. ');
            
            inherited();
            

        }
        
        action()
        {
            performEntry(hiding);
        }
    }
;



/* Climbing. Remap CLIMB UP to CLIMB. */
modify Thing
    dobjFor(ClimbUp)
    {
        remap = [ClimbAction, self]
    }
;



/* Extra vocabulary for SAVE/RESTORE */
VerbRule(SaveGame)
    'save' ('your' | 'my' | 'the' | ) 'game'
    : SaveAction
    verbPhrase = 'save/saving the game'
;

VerbRule(RestoreGame)
    'restore' ('your' | 'my' | 'the' | ) 'game'
    : RestoreAction
    verbPhrase = 'restore/restoring the game'
;



/* 
 *   Allow jumping while in the tree. (So it can be caught with a 
 *   BeforeAction. Yes, this is a dumb way to implement this. See if I care.)
 */
modify JumpAction
    preCond()
    {
        if (gActor.isIn(upTree))
            return [];
        else
            return inherited();
    }
    
    /* Jumping while in the cubby is also a pretty dumb idea. */
    execAction()
    {
        if (gActor.isIn(cubbyHouse))
        {
            "You hit your head on the ceiling of the cubby. ";
        }
        else
        {
            "You jump up and down a bit. ";
            
            if (gCurrStoryPoint != counting)
            {
                /* The NPCs looks at you strangely. */
                local npcList = new Vector(2);
                foreach (local npc in [yvonne, tiana])
                {
                    if (!npc.isIn(bushNearCubby) && npc.canSee(gActor))
                        npcList.append(npc);
                }
                
                if (npcList.length() > 0)
                {
                    gMessageParams(npcList);
                    "{The/list npcList} stare{s} at you in confusion. ";
                }
            }
        }
    }
;



/* Sliding down the slide */
DefineTAction(SlideDown);

VerbRule(SlideDown)
    ('slide' | 'ride') ('down' | 'on' | ) singleDobj
    : SlideDownAction
    verbPhrase = 'slide/sliding (down what)'
;

modify Thing
    dobjFor(SlideDown)
    {
        verify()
        {
            illogical('You can\'t slide down {the dobj/him}. ');
        }
    }
;


DefineIAction(Slide)
    execAction()
    {
        askForDobj(SlideDown);
    }
;

VerbRule(Slide)
    ('slide' | 'ride') ('down' | 'on' | )
    : SlideAction
    verbPhrase = 'slide/sliding'
;



/* Digging */
modify Thing
    dobjFor(Dig)
    {
        check()
        {
            failCheck('You can\'t make any impression on {the dobj/him}
                with your bare hands. ');
        }
    }
;



/* Playing */
DefineTAction(PlayOn);

VerbRule(PlayOn)
    'play' ('on' | 'in' | 'with' | ) singleDobj
    : PlayOnAction
    verbPhrase = 'play/playing (on what)'
;

modify Thing
    dobjFor(PlayOn)
    {
        verify()
        {
            illogical('You\'ll have to be more specific about
                what you want to do. ');
        }
    }
;

modify Person
    dobjFor(PlayOn)
    {
        verify()
        {
            illogicalSelf('You are not a plaything. ');
            
            illogical('{The dobj/she} probably wouldn\'t like that. At
                all. ');
        }
    }
;


DefineIAction(PlayVague)
    checkAction()
    {
        failCheck('You\'re already playing ' + 
                  (gPlayerChar.isPlayingHnS ? 'Hide and Seek' : 
                  'Emma\'s game') + '. If you want to  play something 
                      different, you\'ll have to be more specific. ');
    }
;

VerbRule(PlayVague)
    'play'
    : PlayVagueAction
    verbPhrase = 'play/playing'
;


DefineTopicAction(Play)
    checkAction()
    {
        /* what game does the player want to play? */
        local topic = gAction.getTopic.getBestMatch();
        
        /* if the topic is actually a Thing, try PLAYing ON it instead */
        if (topic && topic.ofKind(Thing) && gActor.canSee(topic))
        {
            replaceAction(PlayOn, topic);
        }
        
        /* If it's Sardines, IS THE PLAYER CRAZY? */
        if (topic == tGame)
        {
            if (!gPlayerChar.isPlayingHnS)
            {
                failCheck('You\'re already playing Emma\'s game. ');
            }
            else
            {
                failCheck('You only just managed to get out of playing
                    Emma\'s game. ');
            }
        }
        
        /* 
         *   If it's Hide and Seek, give the player a nudge in the right 
         *   direction.
         */
        if (topic == tHnS)
        {
            if (gPlayerChar.isPlayingHnS)
            {
                failCheck('You\'re already playing Hide and Seek. Try ' +
                          (gPlayerChar.roleInHnS == counting ? 'counting' :
                           'hiding') + ', if you don\'t know what to do next. ');
            }
            else
            {
                failCheck('That would be more fun, but you can\'t play it on
                    your own. You need to play it with someone else. ');
            }
        }
        
        /* 
         *   You can't play anything else. Well, you can, but not using this 
         *   verb (PLAY SWING etc. will be handled by PlayOnAction).
         */
        failCheck('You\'re not sure how to play that game. ');
    }
;

VerbRule(Play)
    'play' singleTopic
    : PlayAction
    verbPhrase = 'play/playing (what)'
;


DefineTopicTAction(PlayWith, DirectObject);

VerbRule(PlayWith)
    'play' singleTopic 'with' dobjList
    : PlayWithAction
    verbPhrase = 'play/playing (what) (with whom)'
;

modify Thing
    dobjFor(PlayWith)
    {
        verify()
        {
            illogical('{The iobj/he} wouldn\'t be a very good playmate. ');
        }
    }
;

modify Person
    dobjFor(PlayWith)
    {
        verify() 
        {
            if (self == gActor)
                illogicalSelf('Unfortunately, there\'s only one of you. ');
        }
        
        check()
        {
            /* get the conversation topic */
            local topic = gTopic.getBestMatch();
            
            /* if it's Sardines, the PC isn't interested */
            if (topic == tGame)
            {
                if (!gActor.isPlayingHnS && !self.isPlayingHnS)
                {
                    failCheck('You\'re both already playing Emma\'s game. ');
                }
                else
                {
                    failCheck('You only just got out of playing Emma\'s 
                        game. ');
                }
            }
            
            /* if it's Hide and Seek, is the PC playing already? */
            if (topic == tHnS)
            {
                if (gActor.isPlayingHnS && self.isPlayingHnS)
                {
                    failCheck('You\'re both playing Hide and Seek already. ');
                }
                else
                {
                    /* let the action go ahead */
                    return;
                }
            }
            
            /* if it's anything else, the PC doesn't know that game */
            failCheck('You\'re not sure how to play that game. ');
        }
        
        action()
        {
            nestedAction(AskAbout, self, gTopic);
        }
    }
;


/* PLAY ON [thing] WITH [NPC] */
DefineTIAction(PlayOnWith)
    objInScope(obj)
    {
        if (obj is in (swing, cubbyHouse, cubbySlide, sandpit))
            return true;
        else
            return inherited(obj);
    }
    
    cacheScopeList()
    {        
        local vec = new Vector(100);
        forEachInstance(Thing, new function(x) {
            if (objInScope(x))
                vec.append(x);                             
        });
        scope_ = vec.toList();
    }
;

VerbRule(PlayOnWith)
    'play' (('on' | 'in') singleDobj 'with' iobjList |
            'with' iobjList ('on' | 'in') singleDobj)
    : PlayOnWithAction
    verbPhrase = 'play (on what) (with whom)'
;

modify Thing
    dobjFor(PlayOnWith)
    {
        verify()
        {
            illogical('You\'ll have to be more specific about what
                you want to do with {the dobj/him}. ');
        }
    }
    
    iobjFor(PlayOnWith)
    {
        verify()
        {
            illogical('{The iobj/she} wouldn\'t make a very good playmate. ');
        }
    }
;

modify Person
    iobjFor(PlayOnWith)
    {
        verify() 
        { 
            if (self == gActor)
                illogicalSelf('Unfortunately, there\'s only one of you. ');
        }
        
        check()
        {
            if (gActor.isPlayingHnS)
            {
                failCheck('But you\'re already playing Hide and Seek. ');
            }
        }
        
        action()
        {
            local topic = ResolvedTopic.wrapActionObject(DirectObject);
            replaceAction(AskAbout, self, topic);
        }
    }
;


DefineTAction(PlayWithVague);

VerbRule(PlayWithVague)
    'play' 'with' dobjList
    : PlayWithVagueAction
;

modify Thing
    dobjFor(PlayWithVague) remapTo(PlayOn, self)
;

modify Person
    dobjFor(PlayWithVague)
    {
        remap = nil
        
        check()
        {
            if (gActor == self)
                failCheck('Unfortunately, there\'s only one of you. ');
            
            if (isPlayingHnS && gActor.isPlayingHnS)
                failCheck('{The dobj/she} is already playing Hide and Seek
                    with you. ');
            
            if (!isPlayingHnS && !gActor.isPlayingHnS)
                failCheck('{The dobj/she} is already playing Emma\'s game
                    with you. ');
        }
        
        action()
        {
            replaceAction(AskAbout, self, tHnS);
        }
    }
;


/* 
 *   Remap commands like TIANA, PLAY HIDE AND SEEK WITH ME to PLAY HIDE AND 
 *   SEEK WITH TIANA.
 */
playWithRemap: GlobalRemapping
    getRemapping(issuingActor, targetActor, action)
    {
        /* grab a match for the direct object, if the action has one */
        local dobj = ((action.dobjList_ && action.dobjList_.length > 0) ? 
                      action.dobjList_[1] : nil);
        dobj = (dobj ? dobj.obj_ : nil);
        
        /* get a match for the indirect object, if the action has one */
        local iobj = ((action.iobjList_ && action.iobjList_.length > 0) ? 
                  action.iobjList_[1] : nil);
        iobj = (iobj ? iobj.obj_ : nil);
        
        /* remap a PlayAction */
        if (action.actionOfKind(PlayAction) && 
            targetActor != gPlayerChar)
        {
            /* create a new action */
            local newAction = PlayWithAction.createActionFrom(action);
            
            /* set the referrent for the pronoun "you" */
            newAction.setPronounOverride(PronounYou, targetActor);
            
            /* set the direct object */
            dobj = new PreResolvedProd(targetActor);
            newAction.setObjectMatches(dobj, action.topicMatch);
            
            /* return the new action with the correct target actor */
            return [issuingActor, newAction];
        }
        
        /* remap a PlayWithAction */
        if (action.actionOfKind(PlayWithAction) && 
            targetActor != gPlayerChar && 
            dobj && dobj == gPlayerChar)
        {
            /* set the referrent for the pronoun "you" */
            action.setPronounOverride(PronounYou, targetActor);
            
            /* set the direct object */
            dobj = new PreResolvedProd(targetActor);
            action.setObjectMatches(dobj, action.topicMatch);
            
            /* return the new action with the correct target actor */
            return [issuingActor, action];
        }
        
        /* remap a PlayWithVagueAction */
        if (action.actionOfKind(PlayWithVagueAction) && 
            targetActor != gPlayerChar && 
            dobj && dobj == gPlayerChar)
        {
            /* set the referrent for the pronoun "you" */
            action.setPronounOverride(PronounYou, targetActor);
            
            /* set the direct object */
            dobj = new PreResolvedProd(targetActor);
            action.setObjectMatches(dobj);
            
            /* return the new action with the correct target actor */
            return [issuingActor, action];
        }
        
        /* remap a PlayOnWithAction */
        if (action.actionOfKind(PlayOnWithAction) && 
            targetActor != gPlayerChar && 
            iobj && iobj == gPlayerChar)
        {
            /* set the referrent for the pronoun "you" */
            action.setPronounOverride(PronounYou, targetActor);
            
            /* get the direct object */
            dobj = new PreResolvedProd(dobj);
            
            /* get the indirect object */
            iobj = new PreResolvedProd(targetActor);
            
            /* set the objects */
            action.setObjectMatches(dobj, iobj);
            
            /* return the new action with the correct target actor */
            return [issuingActor, action];
        }
        
        /* doesn't need remapping */
        return nil;
    }
;



/* PATting the bird */
VerbRule(Pat)
    ('pat' | 'pet' | 'stroke') dobjList
    : FeelAction
    verbPhrase = 'pat/patting (what)'
;



/* LICKing stuff */
VerbRule(Lick)
    'lick' dobjList
    : TasteAction
    verbPhrase = 'lick/licking (what)'
;



/* 
 *   TIPping stuff. Absolutely not a reference to any other hide and seek game
 *   that might or might not bear a passing resemblence to It.
 */
DefineTAction(Tip);

VerbRule(Tip)
    'tip' dobjList
    : TipAction
    verbPhrase = 'tip/tipping (what)'
;

modify Thing
    dobjFor(Tip) asDobjFor(Push)
;

modify Actor
    dobjFor(Tip)
    {
        verify() { }
        check()
        {
            failCheck('You don\'t need to tip the other kids in this game. You
                must be thinking of some other kind of hide and seek game. ');
        }
    }
;



/* 
 *   HUGging people. I told Wade that you had to be able to HUG AYLA in Six and
 *   then never implemented hugging Tiana. Whoops.
 */
DefineTAction(Hug);

VerbRule(Hug)
    ('hug' | 'put' ('your' | 'my' | ) 'arms' ('around' | 'round')) dobjList
    : HugAction
    verbPhrase = 'hug/hugging (whom)'
;

modify Thing
    dobjFor(Hug)
    {
        verify()
        {
            illogical('{The dobj/she} {does}n\'t share your affection for 
                {it dobj/her}. ');
        }
    }
;

modify Actor
    dobjFor(Hug)
    {
        verify() { }
        
        /* 
         *   The actual results of hugging a person are implemented on an
         *   individual basis.
         */
    }
;


/* MURDER, for Jenni Polodna. */
VerbRule(Murder)
    'murder' dobjList
    : AttackAction
    verbPhrase = 'attack/attacking (whom)'
;
