#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

/* Provide some standard NoTravel messages */

modify OutdoorRoom
    down: NoTravelMessage { "There are no visible entrances to caves in the vicinity. " }
    up: NoTravelMessage { "Unfortunately, you\'re not equipped with wings. " }    
;

modify Room
    out: NoTravelMessage { "Which way do you want to go? 
        <<me.getOutermostRoom.cannotGoShowExits(me)>> " }
;


/***************************************************************************
 *   Things that are visible from other locations, or present in multiple 
 *   locations, in the outdoor area.
 ***************************************************************************
 */

Occluder, DistanceConnector [endOfDriveway, sidewalkByFrontYard, driveway, 
     driveByHouse]
;
Occluder, DistanceConnector [endOfDriveway, sidewalkByFrontYard, driveway,
    frontYard]
;

sidewalkGround: defaultGround 'sidewalk/pavement/walk' 'pavement'
    "The sidewalk runs east and west along the edge of the street. It\'s
        a bit cracked and chipped, but not so buckled or warped you
        couldn\'t skateboard on it. "
        dobjFor(JumpOff)
        {
            verify() {}
            check()
            {
                failCheck('You\'d better not --- you might be hit by a passing
                    car. ');
            }
        }
        
        beforeAction()
        {
            if(gActionIn(JumpOffI) && gActor.canTouch(self))
                replaceAction(JumpOff, self);
        }
;

viewOfStreet: MultiLoc, Decoration 
    'busy passing street/road/traffic/vehicles/cars' 
    'street'
    "Traffic is flowing up and down the street, making it much too dangerous to
    cross here. "
    remoteDesc(pov)
    {
        "The street still looks pretty busy; there's no end to the traffic. ";
    }
    locationList = [onTheSidewalk, endOfDriveway]
    sightSize = large
    soundSize = large
    dobjFor(Enter) remapTo(South)    
    dobjFor(ListenTo) { verify() { inherited Thing; } }
;

+ Noise 'passing din/sound/noise/(vehicles)' 'din of passing vehicles'
    sourceDesc = "The din of passing vehicles is nearly constant. "
    descWithSource = "It's a constant din. "
    hereWithSource = (sourceDesc)    
    isAmbient = true
;


streetBarrier: FakeConnector
    travelDesc = "There\'s too much traffic for you to consider crossing the
        street.<.reveal off-map> "
;

/* 
 *   This object is slightly misnamed, as it's also the fence by the driveway
 *   and on the west side of the front yard. The back yard has its own 
 *   fence.
 */
sidewalkFence: MultiLoc, Decoration
    'white gray water-stained stained solid vertical tall painted fence/boards' 
    'fence'
    "The fence is tall, and made of solid vertical boards. Once they were
    painted white, but by now they\'re gray and a bit water-stained. "
    locationList = [onTheSidewalk, driveway, driveByHouse, frontYard]
    canMatchThem = true
    dobjFor(Climb) {
        verify() {}
        check() {
            failCheck ('On any other day, you might consider it. Today you
                have more important things on your mind. ');
        }
    }
    dobjFor(JumpOver) { verify() { illogical(cannotJumpOverMsg); } }
    cannotJumpOverMsg = 'It\'s a bit too tall for that. '
;

fenceBarrier: NoTravelMessage
    travelDesc = "The fence looks to be quite solid, and there\'s no gate. "
;

enterGrounds: EventList
    [
        {: "You take a cautious step onto Mrs. Pepper's property. When nothing
            terrible happens, you take another, your <<gRevealed('skate-find')
              ? 'desire to retrieve your skateboard' : 'curiosity'>> overcoming
            your apprehension of the batty old woman. "  }
    ]    
    showWhenEnteringFrontYard = true    
    
    doScriptEvent(evt)
    {
        if(showWhenEnteringFrontYard || destination == driveway)
            inherited(evt);            
    }
    
    destination = nil
    
    execute(conn)
    {
        destination = conn.destination;
        doScript();
    }
;

//-------------------------------------------------------------------------------
// The initial location:

onTheSidewalk: OutdoorRoom 'On the Sidewalk' 'the sidewalk'
    "This stretch of sidewalk runs more or less east-west between the busy
    street to the south and a tall, water-stained fence to the north. Just a
    few yards further west the fence ends where Mrs. Pepper's driveway thrusts
    into the roadway. <<endOfDriveway.visited ? '' : 'Unfortunately there\'s no
        option but to cross that driveway if you want to get home. '>>"
    west = endOfDriveway
    south = streetBarrier
    east: FakeConnector { "You\'ve just finished school for the day --- you
        don\'t want to go back there now!" }
    north = fenceBarrier
    
    inRoomName(pov) { return 'on the sidewalk'; }
    roomParts = inherited - defaultGround + sidewalkGround
;

+ Unthing 'mrs. pepper\'s driveway' 'driveway'
    'You can\'t quite see it from here; it\'s just beyond the corner of the
    fence. '
;

+ Decoration 'section/corner/(fence)' 'corner of the fence'
    "The corner is where this section of fence comes to an end; beyond that is
    Mrs. Pepper\'s driveway. "
;

//------------------------------------------------------------------------------
// At the end of Mrs. Pepper's driveway.

endOfDriveway: TravelWithMessage, OutdoorRoom, EventList 
    'At the End of the Driveway' 'the end of the driveway'
    "You\'re at the outer end of Mrs.\ Pepper\'s driveway, which extends to
    the north alongside her house. <<(oldCar.location == driveByHouse) ? 'Her
        old gray car is parked beside the house, and before' : 'Before'>>  the
    house is a square of sadly neglected lawn. The sidewalk runs east and west
    from here, and the street is to the south. "
    vocabWords = 'outer end/(driveway)'
    east = onTheSidewalk
    west = sidewalkByFrontYard
    south = streetBarrier
    north: TravelMessage { ->driveway "<<enterGrounds.execute(self)>>" }
    northwest: TravelMessage { ->frontYard "<<enterGrounds.execute(self)>>" }
    
    eventList = [ 'You step gingerly off the sidewalk into the mouth of Mrs.\
        Pepper\'s driveway. There\'s no way of avoiding it, but that
        doesn\'t make it any less hazardous. More than once the nasty old
        woman has backed her car out quite rapidly just as you were crossing,
        as if bent on running you over, so this step across Mrs.\
        Pepper\'s driveway is the part of your daily walk home that you\'ve
        most come to dread....\b
        But today you\'re in luck. Her car isn\'t moving, and she\'s not in it. '
        ]
    
    inRoomName(pov) { return 'at the end of the driveway'; }
    roomParts = [sidewalkGround, defaultSky]
    
    /* 
     *   Mrs Pepper is mentioned several times when the player char enters 
     *   this room, so the PC is likely to think of Mrs Pepper as 'her' at 
     *   this point.
     */
    enteringRoom(traveler)
    {
        me.setHer(mrsPepper);
    }
;

+ viewOfDriveway: Decoration 'drive/driveway' 'driveway'
    "The driveway continues northward along the side of the
    house<<(oldCar.location == driveByHouse) ? ', where an old gray car is parked. ' :
      '. '>>"
    owner = mrsPepper
;

//------------------------------------------------------------------------------
// The sidewalk by the front yard.

sidewalkByFrontYard: OutdoorRoom 'On the Sidewalk by the Front Yard'
    'the sidewalk by the front yard'
    "To the north, across a sadly dried-up and decrepit yard, stands Mrs.\
    Pepper\'s house. You can also go east or
    west along the sidewalk. The busy street is to the south. A telephone pole
    plastered with the scraps of old posters stands here. "
    east = endOfDriveway
    west: FakeConnector { 
        travelDesc {
            if (orangeTree.grewAnOrange) {
                if ((!skateboard.isIn(me)) && (me.location != skateboard)) {
                    "Wait --- you\'d better go back and get your skateboard! ";
                    return;
                }
                // And in this branch, the orange tree has grown an orange,
                // and you've got your skateboard, so the game is over:
                "After taking a hurried detour to put Mrs. Pepper\'s
                possessions back where you found them, lock her house<<
                  garageDoorOutside.isLocked ? '' : ' and garage'>>, and toss
                the keys under the back porch, you head home, happy as can be:
                Not only did you meet an elf and save him from certain death,
                but you got your skateboard back too. ";
                finishGameMsg(ftVictory, [finishOptionUndo,finishOptionFullScore]);
            }
            "If you go on home now, you\'ll spend the rest of the afternoon
            wondering ";
            if (frontYard.heardVoice) {
                if (!frontBedroom.seen) {
                    "about that weak little voice you heard coming from Mrs.\
                    Pepper\'s house, asking for help. Maybe there\'s
                    nothing you can do, but still you ought to try. ";
                    return;
                }
                // In this branch, the bedroom has been seen.
                "whether you could have helped {the elf/him} escape from Mrs.\
                Pepper\'s clutches --- probably the rest of the year, in fact.
                There is definitely something you need to accomplish before you
                go home. ";
                return;
            }    
            "whether you might have been able to <<skateboard.discovered ?
              'retrieve' : 'find'>> your skateboard if you had been bold enough
            to poke around a little more. <.reveal skate-find>";
        }
    }
    south = streetBarrier
    north: TravelMessage { ->frontYard "<<enterGrounds.execute(self)>>" }
    northeast: TravelMessage { ->driveway "<<enterGrounds.execute(self)>>" }
    roomParts = inherited - defaultGround + sidewalkGround
;

+ Surface, CustomFixture 'telephone pole' 'telephone pole'
    "The telephone pole looks much like every other telephone pole you\'ve ever
    seen, though this one has had more posters stapled to it over the years
    than some. <<flyer.discover>><<flyer.moved ? '' : 'Tacked to the pole is an
        advertising flyer that looks crisp and new. '>>"
    contentsListedInExamine = nil
    notifyInsert (obj, newCont) {
        "To do that, you\'d need staples or thumbtacks, neither of which you
        have. ";
        exit;
    }
    sightSize = medium
    cannotClimbMsg = 'You might be able to, but you really don\'t fancy the
        idea; there\'s nothing at the top you want, and if you fell off into the
        road you could end up getting badly hurt. '
    cannotTakeMsg = 'The maintenance department obviously do a good job around
        here; straining with all your might, you can\'t make the pole budge by
        so much as a fraction of an inch. '
    cannotEatMsg = 'That might be biting off more than you can chew. '
    cannotCleanMsg = dobjMsg('You\'re not sure you\'ve ever seen anyone trying
        to clean a telephone pole, but it\'s certainly not your job. ')
    cannotJumpOverMsg = &needJetPackMsg
;

++ flyer: Hidden, Readable
    'crisp new colorful advertising advert ad (mall) (wig) (store) flyer/poster'
    'advertising flyer'
    "The flyer is crisp, new, and colorful. In big bright
    letters it advertises a special sale at Flogg & Grabby\'s Stufftown, a
    shopping mall that lies just half a mile down the road. If the flyer is to
    be believed, the sale ends this very afternoon. <.q>INCREDIBLY LOW PRICES
    ON GORGEOUS LADIES\' WIGS!!!<./q> it promises.\b
    
    Mrs.\ Pepper is not your idea of a gorgeous lady (if anything, just the
    opposite). But she certainly is fond of wigs. Every time you see her, it
    seems, her wig is a different color and style, each more outlandish and
    ill-fitting than the last. What they all have in common is that they look
    as if she bought them in a thrift shop. "
    isPaper = true
    notifyMoveInto(newCont) {
        if (newCont == me) achievement.awardPointsOnce();
    }
    achievement: Achievement { +1 "taking the flyer from the telephone pole" }
    bulk = 2
    dobjFor(Take) {
        action() {
            gReveal( 'pickup-something');
            inherited;
        }
    }
;

+ Decoration 'faded torn old scraps/posters' 'posters'
    "Many are faded, many are torn, and many are both. The ones that remain
    legible advertise everything from last year's flower show to the merits of
    the Independent Candidate in some long-forgotten election.
    <<flyer.discover()>><<flyer.moved ? '' : 'Amongst this mass of posters well
        past their read-by date is an advertising flyer that looks crisp and
        new.' >>"
    isPlural = true
    notImportantMsg = 'These posters were mostly useless the day they were
        printed; they are utterly useless now. '
;

//------------------------------------------------------------------------------
// The driveway. Not much going on here, it's just a room that needs to exist.

driveway: OutdoorRoom 'Driveway' 
    "The driveway continues northward along the side of the house,
    <<(oldCar.location == driveByHouse) ? 'where an old gray car is parked, ' :
      ''>>and leads south out to sidewalk and beyond it the street. On the east
    the driveway is bordered by a high fence, and Mrs.\ Pepper\'s front yard
    lies to the west. "
    vocabWords = 'drive/driveway'
    remoteDesc(pov)
    {
        if(pov.getOutermostRoom is in (endOfDriveway, sidewalkByFrontYard))
            "The driveway runs northward along the right side of Mrs.\ Pepper\'s
            house. <<(oldCar.location == driveByHouse) ? 'Standing in the drive
                next to the house is Mrs.\ Pepper\'s old gray car. ' : ''>>";
        
        if(pov.isIn(frontYard))
            "The driveway runs along the east edge of the yard, and disappears
            around the corner of the house. ";
        
        if(pov.isIn(driveByHouse))
            "The driveway runs southward from here toward the street.
            <<(oldCar.location == driveByHouse) ? 'Standing in the driveway is
                Mrs.\ Pepper\'s old gray car. ' : ''>>";
    }
    
    enteringRoom(traveler)
    {
        me.setHer(mrsPepper);
    }
    
    west = frontYard
    south = endOfDriveway
    southwest = sidewalkByFrontYard
    north = driveByHouse
    east = fenceBarrier
;

//------------------------------------------------------------------------------
// The front yard, which contains an orange tree and an important clue.

frontLawn: defaultGround
    'bare dirt patchy sadly neglected burned-out burned out brown lawn/grass'
    'lawn'
    "The lawn is patchy and mostly dead, mottled with patches of bare dirt. "
    sightSize = large
    cannotDigMsg = 'It\'s an intriguing thought that Mrs. Pepper might keep
        something interesting buried in her front yard, but you don\'t have
        time to time to dig up the entire lawn to find out. Besides, she\'d be
        really angry. Considering how much trouble she causes you when she\'s
        <i>not</i> angry, do you really want to take the chance? '
;

frontYard: TravelWithMessage, OutdoorRoom, EventList 'Front Yard'
    "The front yard is mostly bare dirt and burned-out brown lawn, except for a
    few hardy weeds. A paved walkway crosses it, leading from the sidewalk on
    the south up to the front porch on the north. The driveway runs past the
    yard on the east side, and a tall fence guards what must be the
    neighbors\' yard on the west. Growing next to the walkway is a scraggly,
    half-dead tree. <<hearTheVoice>>"
    
    distantDesc = "The front yard is mostly bare dirt and burned-out brown
        lawn, except for a few hardy weeds. A paved walkway crosses it, leading
        from the sidewalk up to the front porch. Growing next to the walkway is
        a scraggly, half-dead tree. "
    
    vocabWords = 'sadly decrepit dried-up front yard'
    
    roomParts = static inherited - defaultGround + frontLawn
    
    enteringRoom(traveler)
    {
        me.setHer(mrsPepper);
    }
    
    north = porchStepsUp
    south = sidewalkByFrontYard
    east = driveway
    west = fenceBarrier
    southeast = endOfDriveway
    
    heardVoice = nil
    hearTheVoice {
        if (heardVoice) return;
        heardVoice = true;
        "<.p>As you\'re crossing the front yard, a momentary lull in the
        traffic noise allows you to hear something very odd. Weakly, somewhere
        in the near distance, a voice is crying, <.q>Help me! Oh, please, help
        me!<./q> The voice is high-pitched and very hoarse, as if whoever is
        calling for help can barely speak at all.\b
        
        You stop dead in your tracks, looking around to see where the voice
        might be coming from. It seems to be coming from the upstairs window in
        Mrs.\ Pepper\'s house. You stare hard at the window, and for a moment
        it seems something might be moving there --- but you can\'t make out
        what it might be. After a moment the movement stops, and the voice
        falls silent.\b
        
        Could it have been your imagination? No, you\'re certain you heard
        something. And it was coming from inside the house. <.reveal elf-cry>";
    }
;

+ porchStepsUp: StairwayUp -> porchStepsDown
    'short wooden flight/stairs/steps' 'short flight of wooden steps'
    "A short flight of steps leads up from the yard to the porch. "
    sightSize = large
    travelBarrier = [ 
        new PushSkateBarrier('You can\'t push {the obj/him} up the steps. '),
        new RideSkateBarrier('You can hardly ride {the obj/him} up the steps. ')
    ]
;

+ Decoration '(few) parched withered hardy weeds' 'weeds'
    "The weeds look parched and withered. "
    notImportantMsg = 'You\'ve never had any time for weeds. '
    isPlural = true
    sightSize = medium
;

+ Platform, CustomFixture 'paved walkway/path*paths' 'paved walkway'
    "The walkway crosses the lawn, running from the sidewalk on the south up to
    the front porch on the north. "
    bulkCapacity = 1000
    maxSingleBulk = 200
    sightSize = large
    notFollowableMsg = 'Which way do you want to go: north or south? '
;

+ Fixture 'narrow front small covered peaked mrs mrs. pepper\'s 
    house/building/structure/home/porch/roof/corner' 
    'Mrs.\ Pepper\'s house'
    "Mrs.\ Pepper\'s house is two stories tall, and narrow. It hasn\'t
    been painted in a long time. A small covered porch is attached to the
    front, and the driveway runs along the right side. <<porchStepsUp.desc()>>"
    sightSize = large
    isProperName = true
    
    /* 
     *   This face of the house is not visible from the driveway by the side 
     *   of the house
     */
    isOccludedBy(occluder, sense, pov)
    {
        return pov.isIn(driveByHouse);
    }
    
    dobjFor(Enter)
    {
        verify() {}
        action()
        {
            "You walk up the steps into the porch.<.p>";
            replaceAction(TravelVia, porchStepsUp);
        }
    }
    dobjFor(GoTo) asDobjFor(Enter)
    cannotJumpOverMsg = &needJetPackMsg
    vocabLikelihood = (me.isIn(frontYard) ? 10 : -10)
;

+ orangeTree: CustomFixture 'scraggly half-dead half dead sickly moribund lone 
    orange tree*trees' 'orange tree'
    "A few green leaves cling to the crooked branches of the tree, but really
    it looks more dead than alive. Mrs.\ Pepper calls it her orange tree --- in
    fact, one of your more memorable run-ins with her occurred when she accused
    you of stealing oranges from the tree. <<!grewAnOrange ? 'But in truth
        you\'ve never seen an orange growing on the tree. The idea that it\'s
        an orange tree is purely a theory, as far as you\'re concerned. ' :
      ''>><<(orange.location == orangeTree) ? 'Growing from a low-hanging
          branch is a single golden orange. ' : ''>>"
    
    distantDesc = "A few green leaves cling to the crooked branches of the
        tree, but really it looks more dead than alive. Mrs.\ Pepper calls it
        her orange tree --- in fact, one of your more memorable run-ins with
        her occurred when she accused you of stealing oranges from the tree.
        <<!grewAnOrange ? 'But in truth you\'ve never seen an orange growing on
            the tree. The idea that it\'s an orange tree is purely a theory, as
            far as you\'re concerned. ' : ''>>"
    sightSize = large
    
    dobjFor(Shake) {
        check() {}
        action() {
            "You grasp the trunk and give the tree a shake. The branches rattle
            and rustle a bit, but nothing else happens. ";
        }
    }
    /* 
     *   Note: The player may want to try to climb the tree to look in the 
     *   upstairs window, so we're not going to claim that there's no 
     *   point in climbing it.
     */
    cannotClimbMsg = 'The tree is too spindly to support your weight. '
    cannotEatMsg = 'If you eat that much wood you won\'t have any room left for
        your mom\'s pumkin pie; besides, it\'d probably give you splinters in
        your tummy. '
    cannotTakeMsg = 'It doesn\'t look too healthy, but it\'s probably best left
        where it is. '
    uselessToAttackMsg = dobjMsg('Pick on something your own size! ')
    shouldNotBreakMsg = 'The poor thing looks sickly enough as it is, without
        you adding to its troubles! '
    cannotJumpOverMsg = &needJetPackMsg
    grewAnOrange = nil
;

++ Decoration '(few) green leaf/leaves' 'leaves' 
    "A few green leaves cling to the crooked branches<<(orange.location ==
        orangeTree) ? '' : ', the only sign of life on the otherwise moribund
            tree'>>. "

    isPlural = true
    notImportantMsg = 'The leaves hardly look worth messing with. '
;

++ Component 'crooked twisted dry limb/limbs/branch/branches' 'branches'
    "The branches look twisted and dry, with a mere handful of leaves sprouting
    pathetically from them. "
    cannotTakeComponentMsg(loc)
    {
        return 'As sickly as the orange tree looks, its branches are still
            firmly attached to it. ';
    }
    isPlural = true
    cannotClimbMsg = (location.cannotClimbMsg)
;

+ Openable, Distant 'narrow upstairs window/curtains' 'upstairs window'
    "The narrow window is lodged beneath the peaked roof of Mrs.\ Pepper\'s
    house. <<extraDesc()>>"
    
    extraDesc()
    {
        if(me.hasSeen(fbWindow))
        {
            if(fbWindow.isOpen == fbCurtains.isOpen)
                "The window and its curtains are both <<fbWindow.openDesc>>. ";
            else
                "The window is <<openDesc>> while its curtains are
                <<fbCurtains.openDesc>>. ";
        }
        else
            "The curtains are open, but the window itself is closed, so the
            movement you glimpsed there could hardly have been the wind
            stirring the curtains.";
    }
    
    dobjFor(Board) asDobjFor(Open)
    dobjFor(Enter) asDobjFor(Open)
    dobjFor(Open) 
    {
        verify() 
        {
            if(isOpen)
                illogicalAlready('It\'s already open. ');
        }
        check() 
        {
            failCheck ('It\'s too high up for you to reach. ');
        }
    }
    dobjFor(Break) 
    {
        verify() {}
        check() 
        {
            failCheck ('That would be vandalism! (Besides, your mother might
                find out.) ');
        }
    }
    
    iobjFor(ThrowAt)
    {
        check() { checkDobjBreak(); }
    }
    canMatchThem = true
    openStatusReportable = nil
    masterObject = fbWindow
;

++ SimpleNoise 'high-pitched (high) pitched crying hoarse voice/cry' 'voice'
    "You strain to hear the voice from the window again, but either it has
    fallen silent altogether, or else it can no longer make itself heard above
    the traffic. "
    isEmanating = (gRevealed('elf-cry') && !me.hasSeen(elf))
;

orange: Food 'single round firm plump golden ripe juicy orange/peel/skin' 'orange'
    "The orange is round and <<isPeeled ? 'juicy' : 'firm, with a pebbly skin'>>. "
    tasteDesc {
        if (isPeeled) "It\'s deliciously juicy and sweet. ";
        else "Until you\'ve peeled it, there\'s no way to do that. ";
    }
    smellDesc = "The orange smells tart and sweet. " 
    feelDesc = "The skin feels firm and pebbly. "   
    initSpecialDesc = "A single orange is growing from a branch of the tree. "
    bulk = 6
    eatCount = 0
    isListed = nil
    isPeeled = nil
    isListedInContents = (location != orangeTree)
    dobjFor(Peel) 
    {
        preCond = [objHeld]
        verify() 
        {
            if (isPeeled) illogicalAlready ('You\'ve already peeled the orange. ');
        }
        action() 
        {
            isPeeled = true;
            "You peel the orange. It smells delicious! ";
        }
    }    
    dobjFor(Eat) 
    {
        action() 
        {
            eatCount++;
            if (eatCount == 1) 
            {
                if (!isPeeled)
                    "You peel the orange carefully, remove a section, and ";
                else
                    "You remove a section and ";
                "bite into it. Quite possibly it\'s the best-tasting orange
                you\'ve ever had in your life! ";
                isPeeled = true;
                return;
            }
            if (eatCount < 5) 
            {
                "You separate another section of the orange and eat it avidly. ";
                return;
            }
            // Okay, now we've finished the orange:
            "You finish off the last of the orange, lick your fingers, and toss
            the peel away. Yum! ";
            moveInto(nil);
        }
    }
    dobjFor(FruitPick) asDobjFor(Take)
    
    dobjFor(Take)
    {
        action()
        {
            if (location == orangeTree) "You pick the orange. ";
            inherited;
            
            /* 
             *   If we didn't award the points for filling the glass from the
             *   birdbath (since we watered the elf by getting it to rain on
             *   him) award the equivalent number of points for picking the 
             *   orange instead, to keep the total number of points the same.
             */
            if(birdbath.achievement.scoreCount == 0)
                achievement.addToScoreOnce(birdbath.achievement.points);
        }
    }
    
    achievement: Achievement { "picking the orange" }
;

//------------------------------------------------------------------------------
// The front porch.

frontPorch: Room 'Front Porch'
    "The front door lies immediately to the north; the front yard is to the
    south. There\'s a doorbell button just by the front door. "
    roomParts = static inherited - defaultSouthWall
    north = frontDoorOutside
    south = porchStepsDown
    in asExit(north)
    out asExit(south)
    vocabWords = 'front porch'
    iobjFor(PutOn)
    {
        verify() {}
        action()
        {
            replaceAction(Drop, gDobj);
        }        
    }
    
    iobjFor(PutIn)
    {
        verify() {}
        action()
        {
            replaceAction(Drop, gDobj);
        }        
    }
    owner = mrsPepper
;

+ porchStepsDown: StairwayDown 'short three wooded flight/steps/stairs' 
    'short flight of wooden steps'
    "A short flight of three wooden steps leads back down to the front yard. "
    canMatchThem = true
    travelBarrier = [ 
        new PushSkateBarrier('It wouldn\'t be safe to push {the
        obj/him} down the steps into the front yard. '),
        new RideSkateBarrier('It\'s probably not safe to skateboard down the
            steps. ') 
    ]
;

+ frontDoorOutside: LockableWithKey, Door 'front door*doors' 'front door'
    "The door is of solid wood, except for a tiny peephole covered with a
    filigree of wrought iron. "
    keyList = [keyBunch]
    knownKeyList = [keyBunch]
    knockNoEffectMsg 
    {
        if (!isLocked) 
            return 'You\'ve unlocked it --- why not just walk on in? ';
        if (mrsPepper.isGone) 
            return '{You/he} knock{s} on {the dobj/him}, but nobody answers. ';
        return 'You tap softly on the door, and then glance uncertainly at the
            doorbell button. If you\'re serious about wanting to talk to Mrs.\
            Pepper, ringing the doorbell might work better. ';
    }
    iobjFor(PutUnder) 
    {
        verify() {}
        check() {
            if (!isOpen) 
                failCheck('The door is tightly fitted to its frame. There\'s
                    not even a tiny gap at the bottom. ');
            else 
                failCheck ('The door is open --- why not just take
                    {the dobj/him} inside? ');
        }
    }
    
    iobjFor(PutIn) maybeRemapTo(gAction.getEnteredVerbPhrase is in ('put (dobj)
        through (iobj)', 'put (dobj) thru (iobj)'), PutUnder, DirectObject,
                                self)
    
    /* Customization so as to trigger an Extra Hint: */
    dobjFor(Open) {
        check() {
            gReveal('door-tried');
            inherited;
        }
    }
    owner = mrsPepper
;

++ Component 'tiny wrought filigree/hole/peephole' 'peephole'
    "The peephole would be used by someone on the inside to see who\'s at
    the door. At the moment, it\'s closed. " 
    dobjFor(LookIn) asDobjFor(LookThrough)
    nothingThroughMsg = 'You can\'t see through the peephole, because it\'s
        closed on the other side. '
    
    dobjFor(Open) 
    {
        verify() {}
        check() 
        {
            failCheck ('You can\'t do that from this side of the door. ');
        }
    }
;

+ doorbell: Button, CustomFixture 
    '(door) ivory ivory-colored colored doorbell bell/button' 
    'doorbell'
    "The button is small, round, and vaguely ivory-colored. "
    dobjFor(Ring) asDobjFor(Push)
    dobjFor(Push) 
    {
        verify() {}
        check() 
        {
            if ((trashcan.getOutermostRoom == frontPorch) && !mrsPepper.isGone)
                failCheck ('Do you really want Mrs. Pepper to come to the door
                    and see that you\'ve been playing with her trash can? That
                    might not be so smart. ');
        }
        action() 
        {
            if (!dingDongDaemonID) 
            {
                dingDongCountDown = 2;
                dingDongDaemonID = new Daemon (self, &dingDongDaemon, 1);
            }
            "You thumb the doorbell. Within the house a rather discordant chime
            sounds. ";
        }
    }
    dingDongDaemonID = nil
    dingDongCountDown = 2
    dingDongDaemon 
    {
        dingDongCountDown--;
        if (dingDongCountDown > 0) return;
        if (dingDongDaemonID) 
        {
            dingDongDaemonID.removeEvent;
            dingDongDaemonID = nil;
        }
        if (mrsPepper.isGone) return;
        if (me.location == frontPorch) pepperFromPorch.doScript;
        else 
        {
            if (flyer.location == frontPorch) 
            {
                "After a brief pause, the front door opens. Mrs.\ Pepper
                ventures out on the porch, looks around to make sure no one is
                close enough to sneak past her into the house, and then bends
                to retrieve the flyer. For long seconds, she peers at it
                closely, holding it in both hands. Maybe it\'s only your
                imagination, but it looks as if her hands are actually
                trembling with desire. Then she turns on her heel and dives
                back into the house.\b
                A moment later she bursts out again, a large purse over her
                arm. After carefully locking the door, she strides off in the
                direction of the driveway. You hear her car coughing to life,
                and in a squeal of tires it backs down the driveway, careens
                into the street, reverses direction, and roars away in a cloud
                of blue exhaust vapors. ";
                mrsPepper.isGone = true;
                flyer.moveInto(nil);
                oldCar.moveInto(nil);
                if (trashcan.location == oldCar)
                {
                    trashcan.moveInto (driveByHouse);
                    if ((trashcanLid.isIn(trashcan.subSurface)) ||
                        (trashcanLid.location == oldCar))
                        trashcanLid.moveInto(driveByHouse);
                    
                    "Meanwhile, the trashcan has flown off the top of her car
                    and landed on the driveway with a mighty clatter, of which
                    Mrs. Pepper seems blissfully unaware. ";
                }
                if (trashcanLid.location == oldCar)
                {
                    trashcanLid.moveInto (driveByHouse);
                    "Meanwhile, the trashcan lid has flown off the top of the
                    car into the driveway and landed with a clang. ";
                }
                achievement.awardPointsOnce();
            }
            else pepperFromLawn.doScript;
        }
    }
    achievement: Achievement { +4 "getting Mrs.\ Pepper to leave the house" }
    pepperFromPorch: StopEventList 
    {[
        'The door flies open, and Mrs.\ Pepper glares down at you. <.q>You
        again!<./q> she says with a sneer. <.q>Why don\'t you go play in the
        street, good-for-nothing brat?<./q> Mrs.\ Pepper is wearing one of her
        outlandish wigs, as usual. This one is fire-engine red, and the hair
        is crimped from scalp to tip in tiny waves, like a very bad imitation
        of a waterfall. Also, it\'s on a bit crooked.\b
        You try to ask her if maybe you can have your skateboard back, but
        before you can get more than two words out, she laughs nastily and
        slams the door in your face. <.reveal door-slam>',
        
        'After a somewhat longer delay than the first time, the door flies
        open. Mrs.\ Pepper snarls at you. <.q>Why are you bothering me? Do you
        want me to call the police?<./q> The door slams again, leaving you on
        the outside and Mrs.\ Pepper on the inside. ',
        
        'The little peephole in the top part of the door slides open with a
        scraping noise. You hear a guttural <.q>harrumph<./q> from behind the
        door, as if she\'s clearing her throat in preparation for spitting at
        you. But then the peephole snaps shut. '
    ]}
    
    pepperFromLawn: StopEventList 
    {[
        'As you watch from the far corner of the lawn, the door opens and Mrs.\
        Pepper emerges. Her head darts left and right as if she\'s hunting for
        the miscreant who rang her doorbell and then ran. Of course it doesn\'t
        take her long to spot you. <.q>Go away!<./q> she shrieks. <.q>Get off
        my property!<./q> Her grotesque bright red wig catches the afternoon
        sun like a stop sign caught in the high beams. A moment later she
        stomps inside and slams the door. ',
        
        'Again you retreat to a far corner of the lawn (wishing the trunk of
        the orange tree were thick enough to hide behind). Mrs.\ Pepper stomps
        out on the porch, shakes her fist at you, and stomps back inside. You
        find yourself worrying that her patience may wear thin. Might she
        actually attack you if you keep on ringing the doorbell to no
        purpose? ',
        
        'There is an ominous silence from the direction of the porch. Maybe
        she\'s opening the peephole and looking out --- you\'re too far away to
        tell. But it doesn\'t look as if she\'s going to venture outdoors again
        ... unless maybe you drop something on the porch that will attract her
        attention. '
    ]}
;

+ Unthing '(letter) (mail) box/slot/mailbox/letterbox' 'mail slot'
    'Mrs. Pepper\'s front door doesn\'t seem to have a slot for her mail;
    perhaps people just leave it on her porch. '
;


//------------------------------------------------------------------------------
// The driveway along the side of the house.

driveByHouse: OutdoorRoom 'Driveway by House' 
    'the section of driveway beside the house'
    "Mrs.\ Pepper\'s house lies directly to the west; the drive continues to
    the south, and a gravel path leads around the house to the northwest.
    Standing directly to the north is a detached garage. "    
    
    south = driveway
    northwest = gravelPath
    east = fenceBarrier
    west = kitchenWindow
    in asExit(west)
    north = garageSideDoor
    inRoomName(pov) { return 'beside the house'; }
;

+ gravelPath : PathPassage 'gravel path' 'gravel path' 
    "The path leads around the corner of the house to the northwest. "  
    destination = backYard  
; 

+ oldCar: Heavy, Platform
    'gray grey (door) car handle slate-gray slate battered boxy old 
    car/Oldsmobile/Olds/model' 'car'
    "The car is an Oldsmobile, a boxy model that\'s at least 20 years old.
    It\'s slate-gray, and the rear fender droops at one side, as if perhaps
    Mrs.\ Pepper has succeeded in hitting something --- or someone --- with
    it. "
    initSpecialDesc = "A battered old car is parked in the drive, just under an
        open window. "
    distantDesc = "The car is an Oldsmobile, a boxy model that\'s at least 20
        years old. It\'s slate-gray, and the rear fender droops at one side, as
        if perhaps Mrs.\ Pepper has succeeded in hitting something --- or
        someone --- with it. "
    
    /* 
     *   I want to handle this description manually, within the paragraph of 
     *   other locations' descs, so I'm overriding it.
     */
    distantInitSpecialDesc {
        //if ((me.location == driveway) || (me.location == endOfDriveway)) 
            return;
        //"A battered old car is parked beside the house. ";
    }
    sightSize = large
    bulkCapacity = 400
    maxSingleBulk = 400
    down asExit(out)
    
    dobjFor(Board)
    {
        remap = ([gAction.isGetOn ? StandOnAction : EnterAction, self])
    }
    dobjFor(Enter) 
    {
        verify() {}
        check() 
        {
            failCheck ('You rattle the car\'s door handle, but the door is
                securely locked. ');
        }
    }
    dobjFor(Unlock) 
    {
        verify() {}
        check() 
        {
            failCheck ('You don\'t have the key. ');
        }
    }
    
    canReachFromInside(obj, dest)
    {
        /* Allow touching here, because this is checked elsewhere */
        if(dest == kitchenWindow)
            return true;
        return nil;
    }
            
    cannotCleanMsg = dobjMsg('It could certainly do with a clean, but you
        can\'t think of a single reason why you might want to clean Mrs.
        Pepper\'s car for her. ')
    cannotOpenMsg = 'You can\'t; it\'s locked. '
    cannotCloseMsg = 'It\'s not open. '
    cannotUnboardMsg = 'You\'re not in it. '
    cannotGetOffOfMsg = 'You\'re not on it. '
    cannotJumpOverMsg = 'You don\'t think you can. '
    lookInDesc = "There's just the usual stuff you'd expect to see inside a car.
        Even if you were skillful enough to break into it
        (which you aren\'t) it doesn't look like there'd be anything worth
        stealing. "
    nothingUnderMsg = 'There\'s nothing interesting under the car, not even
        those odd patches of liquid your parents worry might mean there\'s
        something wrong with theirs. '
    nothingBehindMsg = 'So far as you can tell, there\'s nothing behing the car
        but the house. '
    cannotLockMsg = dobjMsg('It\'s already locked. ')
    dobjFor(Climb) asDobjFor(StandOn)
    dobjFor(ClimbUp) asDobjFor(StandOn)
    owner = mrsPepper
;

++ Component 'drooping rear fender/bumper' 'rear fender'
    "It droops to one side, as if Mrs. Pepper has managed to hit something ---
    or someone --- with it. "
    sightSize = large
;

+ garageSide: Fixture
    'detached single-car weathered avocado green garage/planks' 'garage'
    "The single-car garage is built of weathered planks, which are painted an
    unlikely shade of avocado green. "
    cannotEnterMsg = 'There\'s no entrance on this side of the garage, other
        than the wide door through which a car would be driven in and out, and
        that door, on close inspection, seems to have been shut for so long
        that the most recent layer of paint has sealed it up. '
    cannotJumpOverMsg = 'Not even the champion high-jumper at your school could
        manage that, and though your English teacher is fond of telling you
        that you\'re <.q>in for the high jump<./q>, you\'ve never been much
        good at it. '
    cannotClimbMsg = 'It doesn\'t look a particularly easy climb, and there\'s
        nothing up there you want. '
    cannotStandOnMsg = (cannotClimbMsg)
    cannotSitOnMsg = (cannotClimbMsg)    
    getFacets = [garageFront]
    dobjFor(Open) remapTo (Open, garageSideDoor)
;

++ garageSideDoor: IndirectLockable, Door, Component
    'wide (garage) (green) door' 'garage door'
    "It's wide enough for a car to be driven through, but
    \v<<cannotOpenLockedMsg>>"
    cannotOpenLockedMsg = 'It seems to have been shut for so long that the most
        recent layer of paint has sealed it up. '
    cannotUnlockMsg = (cannotOpenLockedMsg)   
    openStatusReportable = nil
    isConnectorListed = nil
;

+++ Decoration 'most recent green darker shade/layer/paint' 'layer of paint'
    "It's green, like the rest of the garage, but a darker shade. "
;

+ houseSide: Fixture 'small box-shaped (box) (shaped) corner/house' 'house'   
    "The house is two stories high, but small and box-shaped. There\'s only
    one window on this side, set six or seven feet up from the ground. "
    
    /* 
     *   This may seem counter-intuitive, but we want this view of the house 
     *   to be visible only from this location.
     */
    sightSize = small
    dobjFor(Enter) asDobjFor(Board)
    dobjFor(Board) {
        preCond = [] 
        verify() {}
        check() {
            if ((oldCar.location == driveByHouse) && (me.location == oldCar))
                failCheck ('You can\'t quite reach the window from here. ');
            else
            if (!kitchenWindow.canObjReachContents(me))
            failCheck ('The window is too high up for you to reach. Perhaps if
                you were standing on something.... ');
        }
        action () {
            replaceAction (Enter, kitchenWindow);
        }
    }
    cannotJumpOverMsg = &needJetPackMsg
;

++ kitchenWindow: OutOfReach, TravelWithMessage, Door 
    'kitchen open closed window' 'window'
    "The window is <<isOpen ? 'standing open' : 'closed'>>, and doesn\'t appear
    to have a screen. It\'s six or seven feet up the wall from the paved
    surface of the driveway. "
    initiallyOpen = true
    
    canObjReachContents(obj) 
    { 
        return obj.isIn(trashcan.subSurface) 
            && obj.posture == standing && trashcan.nominalLocation == self; 
    }

    cannotReachFromOutsideMsg(dest) 
    { 
        if (me.location == oldCar)
            return 'From atop the car, the window is almost close enough
                for you to reach --- but not quite. 
                <.reveal window-reach>';
        return 'You can<./s>t reach the window from where you<./s>re '
            + me.posture.participle + '. <.reveal window-reach>'; 
    }
    iobjFor(PutIn) 
    {
        verify() {}
        check() 
        {
            local str = cannotReachFromOutsideMsg(self);
            if (trashcan.nominalLocation != self)
                failCheck(str);
            
            if (me.location != trashcan.subSurface)
                failCheck(str);
            
            if (!isOpen) 
                failCheck ('The window is closed. ');
            
            failCheck ('There probably isn\'t much reason to do that. ');
        }
    }
        
    connectorStagingLocation = (trashcan.nominalLocation == self ?
                                trashcan.subSurface : nil )
    dobjFor(Board) asDobjFor(Enter)
    dobjFor(LookIn) asDobjFor(LookThrough)
    iobjFor(MoveTo)
    {
        verify() {}
        check()
        {
            if(oldCar.isIn(driveByHouse))
                failCheck('There isn\'t enough space between the car and the
                    side of the house for you to do that. <.reveal car-block>');
        }        
    }
    
    iobjFor(PutUnder) remapTo(MoveTo, DirectObject, self)
    
    dobjFor(Close)
    {
        check()
        {
            failCheck('You find you can\'t close the window from the
                outside. ');
        }
    }
    
    dobjFor(Open)
    {
        check()
        {
            failCheck('You\'re unable to open the window from the outside. ');
        }
    }
    
    travelDesc = "You manage to clamber through the window --- but as you do so,
        you accidentally knock the lid off the can. <<scoreIt>>"
    scoreIt 
    { 
        trashcanLid.moveInto(trashcan.roomLocation());
        achievement.awardPointsOnce(); 
    }
    achievement: Achievement { +3 "climbing through the window" }
    /* 
     *   The window's description already states whether it's open or 
     *   closed, so we don't want to repeat the information.
     */
    descContentsLister: openableContentsLister
    {
        showListEmpty(pov, parent) {}
    }  
    
    allStates = [openState, closedState]
    getState = (isOpen ? openState : closedState)
;

++ openState: ThingState
    stateTokens = ['open']
;

++ closedState: ThingState
    stateTokens = ['closed']
;

+trashcan: ComplexContainer 'gray waste trash dust garbage
    can/trashcan/dustbin/bin/wastebin/container/receptacle' 'trash can'
    "The old, dented trash can is made of gray metal<<(trashcanLid.location ==
        subSurface) ? ', and has a sturdy lid' : ''>>. <<(nominalLocation ==
            kitchenWindow) ? 'It\'s sitting directly under the window. ' : ''>>
    <<(nominalLocation == plankShelf) ? 'It\'s sitting directly below the
        shelf. ' : ''>>"
    sightSize = small
    bulk = 300
    initSpecialDesc = "<<(me.location == subSurface) ? 'You\'re standing on a
        trash can' : 'A trash can is standing'>> in the driveway. "
    actorInPrep = 'on'
    smellDesc = "Just a whiff of the trash can is more than enough to satisfy
        your curiosity. It smells quite foul. "
    dontThrowMsg = 'That would make a lot of noise, and Mrs.\
        Pepper would probably hear it and come to investigate. Better not. '
    
    dobjFor(GetOffOf) remapTo(GetOffOf, subSurface)
    
    dobjFor(Throw) 
    {
        verify() {}
        check() 
        {
            if (!mrsPepper.isGone) 
                failCheck ( dontThrowMsg );
        }
    }
    dobjFor(ThrowAt) 
    {
        check() 
        {
            if (!mrsPepper.isGone) 
                failCheck ( dontThrowMsg );
        }
    }
    dobjFor(Take)
    {
        verify()
        {
           if(gActor.isIn(subSurface))
                illogicalNow('You can\'t take {the dobj/him} while you\'re ' +
                             gActor.posture.participle + ' on it. ');
            inherited;
        }
        action() {
            gReveal( 'pickup-something');
            inherited;
        }
    }

    subSurface: ComplexComponent, RestrictedSurface, Platform 
    {
        validContents = [trashcanLid]
        contentsListedInExamine { return nil; }
        actorInPrep = 'on'
        down asExit(out)
        notifyInsert (obj, newCont) 
        {                        
            /* 
             *   Anything else in the game will fit neatly inside the can, 
             *   with the exception of the rake and the bambooStick:
             */
            if (rake.location == trashcan.subContainer) 
            {
                "In order to put the lid on the can, you\'ll need to take the
                long-handled rake out of it. ";
                exit;
            }
            if (bambooStick.location == trashcan.subContainer) 
            {
                "In order to put the lid on the can, you\'ll need to take the
                long bamboo stick out of it. ";
                exit;
            }
            
            if(obj == trashcanLid && newCont == self)
                "You put the lid back on the trash can. ";
        }
        
        allowedPostures = [standing, sitting]
        
        dobjFor(StandOn)
        {                       
            verify() 
            {
                inherited;
                
                if (trashcanLid.location != self) 
                    illogicalNow ('You can\'t stand on the can after removing
                        the lid. ');
            }
            check()
            {
                if(lexicalParent.location != lexicalParent.getOutermostRoom)
                    failCheck('It doesn\'t look very stable. ');
            }
        }
        
        dobjFor(SitOn)
        {
            verify() 
            {
                inherited;
                
                if (trashcanLid.location != self) 
                    illogicalNow ('You can\'t sit on the can after removing
                        the lid. ');
            }
            check() {  checkDobjStandOn();  }
        }

        stagingLocations = [lexicalParent.location]
        
        /*  
         *   We can touch both the window and the side of the house if we're 
         *   standing on the trash can while it's under the window. We can touch
         *   the jar if we're on the can while it's under the shelf.
         */             
        canReachFromInside(obj, dest) 
        { 
            if(dest == houseSide 
               && lexicalParent.nominalLocation == kitchenWindow)
                return true;
            
            if (dest.isOrIsIn(plankShelf)
                && lexicalParent.nominalLocation == plankShelf)
                return true;

            return (dest == kitchenWindow);
        }
    }
    subContainer: ComplexComponent, Container 
    {
        isOpen = (!trashcanLid.isIn(lexicalParent.subSurface))
        bulkCapacity = 150
        maxSingleBulk = 150
        notifyInsert(obj, newCont) 
        {
            if (obj == trashcanLid) 
                failCheck('The lid won<./s>t fit inside the can. ');
        }
        iobjFor(PutIn) 
        {
            verify () 
            {
                if (!lexicalParent.isOpen) 
                    illogicalNow ('In order to put anything in the trash can,
                        you\'ll need to take off the lid. ');
                if (gDobj && (gDobj == trashcan)) 
                    illogicalSelf ('That makes no sense. ');
            }
        }
        
        dobjFor(Open) 
        {
            verify() {}
            check() 
            {
                if (trashcanLid.location != lexicalParent.subSurface)                    
                    failCheck ('The trash can is already open. ');
                
                if (me.isIn(trashcan.subSurface))
                    failCheck ('You can\'t open the can while you\'re standing
                        on the lid! ');
            }
            action() 
            {
                trashcanLid.moveInto(me);
                inherited;
                "You remove the lid from the trash can. ";
            }
        }
        
        dobjFor(LookIn) { preCond = [objOpen, touchObj] }
    }
    
    dobjFor(Close) maybeRemapTo(isOpen, PutOn, trashcanLid, subSurface)
    
    dobjFor(Attack) 
    {
        verify() {}
        check() 
        {
            if (me.isIn(trashcan.subSurface))
                failCheck ('You can\'t kick the can while you\'re on it. ');
            
            if (location != getOutermostRoom)
                failCheck ('It would probably fall over. ');
        }
        action () 
        {
            "You jab the trash can with your toe. The noise is satisfying,
            but you don\'t seem to have accomplished anything. ";
        }
    }
    dobjFor(AttackWith) 
    {
        verify() {}
        check() 
        {
            if (me.isIn(trashcan.subSurface))
                failCheck ('You can\'t hit the can with anything while you\'re
                    on it. ');
        }
        action () 
        {
            "You hit the trash can with {the iobj/him}. The noise is satisfying,
            but you don\'t seem to have accomplished anything. ";
        }
    }
    dobjFor(Board)
    {
        remap = (gAction.isGetOn ? [StandOnAction, subSurface] : [EnterAction, self])
    }
    dobjFor(Enter) 
    {
        verify() {}
        check() 
        {
            failCheck ('Tossing yourself into the wastebin at such a young age?
                You contemplate the idea for a moment, and reject it with a
                shudder. ');
        }
    }

    dobjFor(Climb) remapTo(StandOn, subSurface)    
    dobjFor(StandOn) remapTo(StandOn, subSurface)
    dobjFor(SitOn) remapTo(SitOn, subSurface)

    dobjFor(Move) asDobjFor(Push)
    
    dobjFor(Push) 
    {
        verify() {}
        check() 
        {
            if (location == me) 
                failCheck ('You can\'t push the trash can while you\'re holding
                    it. ');
            
            if (location != getOutermostRoom)
                failCheck ('It would probably fall over. ');
            
            if (me.isIn(trashcan.subSurface))
                failCheck ('You can\'t push the can while you\'re on it! ');
        }
        action() 
        {
            "You push the trash can around aimlessly. ";
        }
    }
    dobjFor(MoveTo)
    {
        verify() 
        {
            if(gIobj && gIobj == nominalLocation)
                illogicalAlready('It\'s already there. ');
        }
        check()
        {
            if((gIobj != kitchenWindow) && (gIobj != plankShelf))
                failCheck('There\'s not much point moving it there. ');
        }
        action()
        {
            if (getOutermostRoom == driveByHouse) 
            {
                if (location == me) 
                {
                    moveInto(driveByHouse);
                    "You set down the trash can directly below the window. ";
                }
                else
                "You push {the dobj/him} under {the iobj/him}. ";
            }
            if (getOutermostRoom == cellar) 
            {
                if (location == me) 
                {
                    moveInto(cellar);
                    "You set down the trash can directly below the plank shelf. ";
                }
                else
                "You push {the dobj/him} under {the iobj/him}. ";
            }
            nominalLocation = gIobj;
            achievement.awardPointsOnce();
        }
    }
    achievement: Achievement { +2 "moving the trash can underneath the window" }
    cannotPushTravelMsg = 'You don\'t need to move it that far. '
    
    nominalLocation = nil
    
    moveInto(newLoc)
    {
        inherited(newLoc);
        
        /* If the trashcan is moved, it loses its nominalLocation */
        nominalLocation = nil;
    }
    
    dobjFor(JumpOver)
    {
        verify() 
        {
            if(gRevealed('grazed-knee'))
                illogicalAlready('The last time you tried that you grazed your
                    knee and very nearly twisted your ankle. You\'re in no
                    hurry to repeat the experience. ');
        }
        action()
        {
            "You jump over {the dobj/him}";
            if(trashcanLid.isIn(subSurface))
            {
                ", but your foot catches the lid, knocking it noisily to the
                ground, while you land awkwardly next to it, grazing your knee
                and nearly spraining your ankle. You're not in any hurry to
                try that again! <.reveal grazed-knee>";
                trashcanLid.moveInto(getOutermostRoom);
            }
            else
                " and land, somewhat joltingly, on the other side. ";
        }
    }
   cannotCleanMsg = dobjMsg('You\'ve always believed trashcans are <i>meant</i>
       to be dirty.' )
;

trashcanLid: Surface '(trash) (can) (trashcan) (dustbin) (bin) (gray) sturdy 
    lid/top' 'trash can lid'
    "The lid is made of gray metal. "
    location = trashcan.subSurface
    bulk = 60
    dobjFor(StandOn) 
    {
        verify() 
        {
            if (isIn(gActor)) 
                illogicalNow ('You can\'t stand on the lid while holding
                    it! ');
        }
        check () 
        {
            if (location != trashcan.subSurface) 
                failCheck ('There\'s probably not much point in standing on the
                    lid. It\'s not a magic carpet or a flying saucer, that\'s
                    for sure. ');
        }
        remap()
        {
            if(location == trashcan.subSurface)
                return [StandOnAction, location];
            
            return nil;
        }  
    }

    dobjFor(SitOn) 
    {
        verify() 
        {
            if (isIn(gActor)) 
                illogicalNow ('You can\'t sit on the lid while holding
                    it! ');
        }
        check () 
        {
            if (location != trashcan.subSurface) 
                failCheck ('There\'s probably not much point in sitting on the
                    lid. It\'s not a magic carpet or a flying saucer, that\'s
                    for sure. ');
        }
        remap()
        {
            if(location == trashcan.subSurface)
                return [SitOnAction, location];
            
            return nil;
        }  
    }

    dobjFor(Doff) 
    {
        verify() {}
        check() 
        {
            if (location != trashcan.subSurface)
                failCheck('You\'ve already removed the lid from the trash can. ');
        }
        action() {   replaceAction(Take, self);  }
    }
    dobjFor(Take)
    {
        verify()
        {
            if(gActor.isIn(trashcan.subSurface))
                illogicalNow('{You/he} can\'t take the lid while {you\'re/he\'s}
                    ' + gActor.posture.participle + ' on it. ');
            
            inherited;
        }
        action() 
        {
            gReveal( 'pickup-something');
            inherited;
        }
    }
    
    dobjFor(JumpOver) maybeRemapTo(isIn(trashcan.subSurface), JumpOver,
                                   trashcan)
    
    stagingLocations = (isIn(trashcan.subSurface) ? [trashcan.location] :
                        [location])
;

//------------------------------------------------------------------------------
// The back yard -- an interesting place!

viewOfRamp: MultiLoc, Fixture 'ramp' 'ramp'
    "The ramp ascends smoothly from the ground to the porch. "
    locationList = [backYard, backPorch]
    dobjFor(Climb) asDobjFor(ClimbUp)
    dobjFor(ClimbUp)
    {
        verify ()
        {
            if (me.location == backPorch) 
                illogicalNow ('But you\'re already at the top of the ramp. ');
        }
        action ()  {  replaceAction(TravelVia, backPorch);  }
    }
    dobjFor(ClimbDown)
    {
        verify ()
        {
            if (me.location == backYard) 
                illogicalNow ('But you\'re already at the foot of the ramp. ');
        }
        action ()  {   replaceAction(TravelVia, backYard);  }
    }
;

birdbathWater: Fixture 'clean nice fresh rain water/rainwater' 'fresh rainwater'
    "The water in the birdbath looks to be clean and fresh. "
    
    dobjFor(PutIn) 
    {
        preCond = [touchObj]
        verify() {}
        check() 
        {
            if (gIobj != chippedGlass) 
                failCheck ('You can\'t put the fresh rainwater in that. ');
        }
        action()  {   replaceAction(FillWith, gDobj, birdbath);  }
    }
    
    iobjFor(PutIn) remapTo(PutIn, DirectObject, birdbath)
    
    iobjFor(FillWith)
    {
        verify()
        {
            if(!birdbath.containsWater)
                illogicalNow('Since the birdbath is empty, you can\'t very well
                    fill anything from it. ');
        }
        check() {
            if (gDobj != chippedGlass)
                failCheck ('There doesn\'t seem to be any way to do that with
                    {the dobj/him}. ');
        }
        action()
        {
            "{You/he} fill{s} {the dobj/him} with fresh rainwater from the
            birdbath. ";
            gDobj.getState = glassFull;
            gDobj.waterType = freshWater;
            birdbath.achievement.awardPointsOnce();
        }
    }
    
    dobjFor(Take) 
    {
        verify() {}
        check() 
        {
            failCheck('Scooping up the water in your bare hands wouldn\'t be
                too useful. Maybe if you had a container to carry it in, you
                could fill the container with it. ');
        }
    }
    
    dobjFor(Taste) asDobjFor(Drink)
    
    dobjFor(Drink) 
    {
        preCond = []
        verify() {}
        action() 
        {
            "You bend over the birdbath and take a sip of the water. It\'s
            deliciously fresh and cool. ";
        }
    }
    
    dobjFor(MoveWith) 
    {
        verify() {}
        check () 
        {
            if (gIobj != chippedGlass)
                failCheck ('There doesn\'t seem to be any way to do that with
                    {the iobj/him}. ');
        }
    }
;

backYard: OutdoorRoom 'Back Yard'
    "The back yard is in a dreadful state. Except for a few sorry weeds,
    nothing grows here. Someone has dug holes of various sizes, leaving mounds
    of dirt and gravel everywhere. A concrete birdbath stands in the center of
    the yard. The back porch of the house lies directly to the south.<.p>
    
    The driveway is off to the southeast, past the corner of the garage. The
    garage itself stands directly to the east. <<cueTheToad>>"
    vocabWords = 'back yard'
    toadcounter = 11
    toadDaemonID = nil
    toadDaemon 
    {
        if (me.location != backYard) 
            return;
        
        toadcounter--;
        
        if (toadcounter <= 0) 
        {
            if (toadDaemonID) toadDaemonID.removeEvent;
            toadDaemonID = nil;
            "A fat toad that you hadn\'t noticed before hops away under the
            back porch. ";
        }
    }
    toadCued = nil
    cueTheToad 
    {
        if (toadCued) 
            return;
        
        toadCued = true;
        
        if (!toadDaemonID) 
        {
            toadDaemonID = new Daemon(backYard, &toadDaemon, 1);
        }
    }
    southeast = driveByHouse
    in asExit (east)
    east = garageDoorOutside
    south = backPorch
    north = fenceBarrier
    west: NoTravelMessage { "<<backyardFence.cannotClimbMsg>>" }
    
    yellCheck()
    {
        if(!neighbour.isIn(nextDoorGarden))
            failCheck('{The neighbour/she} has gone back inside her house, and
                probably won\'t hear your shout. Even if she does you have no
                reason to make her come outside again, so there\'s no need to
                annoy her. ');
    }
    yellDesc()
    {
        "You call over the fence to attract {the neighbour/she}\'s
        attention.<.p>";
        replaceAction(TalkTo, neighbour);
    }
;

+toad: Unthing 'fat frog/toad/amphibian' 'toad'
    notHereMsg 
    {
        /* 
         *   if the toad has been cued and the daemon already shut off, then 
         *   the message about the toad has been printed.
         */
        if (backYard.toadCued && !backYard.toadDaemonID)
            return 'The toad is gone --- it hopped away under the porch. ';
        
        return 'You haven\'t seen a toad since you left biology class this
            morning. ';
    }
    isKnown = true
    
    dobjFor(Follow) 
    {
        verify() {}
        check () 
        {
            if (!backYard.toadCued || backYard.toadDaemonID)
                failCheck ('Toad --- what toad? ');
        }
        action() {
            replaceAction (LookUnder, porchFromBackYard);
        }
    }
;

+ backyardFence: Fixture 'tall solid lower north west fence' 'fence'
    "The fence to the north is tall and solid. The fence to the west is
    somewhat lower, allowing you to look over it into the adjoining yard, where
    a neighbor is at work in her garden. "
    cannotClimbMsg 
    {
        if (neighbour.location == nextDoorGarden)
            return 'Climbing the fence while the neighbor is working in her
                garden would be impolite. ';
        
        return 'You peer over the fence, but it doesn\'t seem there\'s anything
            else in the garden next door that you\'d have a use for. ';
    }
    sightSize = large
    cannotJumpOverMsg = (cannotClimbMsg)
;

+ birdbath: Container, Fixture 'thick ugly gray grimy concrete (bird) empty full
    basin/bath/birdbath*birdbaths' 'birdbath'
    "The gray concrete birdbath is thick and ugly<<containsWater ? '. Its
        basin is brimming with fresh rain water. ' : ', and looks as if it
    hasn\'t held water for a long time. The bottom of the basin is coated
    with a thin layer of grime. '>>"
    bulkCapacity = 60
    maxSingleBulk = 60
    containsWater = nil
    allStates = [emptyState, fullState]
    getState = (containsWater ? fullState : emptyState)
    achievement: Achievement { +2 "filling the glass from the birdbath" }
    sightSize = medium
    cannotJumpOverMsg = 'You\'d rather not risk it. '
    cannotBoardMsg = 'Just because one of your teachers called you a bird-brain
        today, there\'s no need to prove him right. Besides, the birdbath is
        too small for you, and there\'s a much nicer bath at home. '
    cannotEnterMsg = (cannotBoardMsg)
    dobjFor(Push) 
    {
        verify() {}
        check() 
        {
            failCheck ('You give the birdbath a shove, but it doesn\'t budge. ');
        }
    }
    
    cannotTakeMsg = 'It surely weighs more than a hundred pounds! '
    notifyInsert(obj, newCont)
    {
        if(getState == fullState && newCont == self)
            obj.makeDamp(self);
    }
    
    iobjFor(FillWith) 
    {
        verify()
        {
            if(getState == fullState)
                logicalRank(120, 'water source');
        }
        check() 
        {
            if (getState != fullState)
                failCheck('There\'s no water in the birdbath. ');
            
            if (gDobj != chippedGlass)
                failCheck('You can\'t fill {the dobj/him} from the birdbath. ');
        }
        action() 
        {
            chippedGlass.getState = glassFull;
            chippedGlass.waterType = freshWater;
            achievement.awardPointsOnce();
            "You dip the glass in the birdbath and fill it with fresh rainwater. ";
        }
    }
    
    iobjFor(PourInto) 
    {
        verify() {}
        check() 
        {
            if (containsWater) 
                failCheck ('It\'s already full of water. ');
            else if (gDobj != chippedGlass) 
                failCheck ('It\'s not clear what you\'re trying to do. ');
            else if (chippedGlass.getState == glassEmpty) 
                failCheck ('There\'s nothing in the glass. ');
            else
                /* 
                 *   We'll put this in check even though it's technically an 
                 *   action because that will stop the glass from emptying 
                 *   itself. We don't want that to happen, because the 
                 *   birdbath state wouldn't change -- and couldn't, without 
                 *   a lot of messy recoding.
                 */
                failCheck ('You splash a few drops of water into the birdbath. ');
        }
    }
;

/* 
 *   We put these ThingStates here not because they're actually 'in' the 
 *   birdbath, but (a) to keep them out of the way of the rest of the 
 *   containment hierarchy while (b) showing their close association with 
 *   the birdbath object.
 */

++ emptyState: ThingState
    stateTokens = ['empty']
;

++ fullState: ThingState
    stateTokens = ['full']
;

++basinGrime: Decoration 'layer/grime' 'grime'
    "The grime is --- well, it\'s grimy. No surprise there. "
;

+ Unthing 'bird/birds' 'birds'
    'The birdbath has failed to attract a single bird. In fact, it seems
    nothing in Mrs. Pepper\'s back yard is attractive to birds; there\'s
    not a single bird in sight. '
    isPlural = true
;

+ garageFront: Enterable ->garageDoorOutside 'green (wooden) (wood) shed/garage' 
    'garage'
    "The garage is pretty much just a wooden shed. It\'s painted green, and
    there\'s a plain wooden door on this side. "    
    cannotJumpOverMsg = (garageSide.cannotJumpOverMsg)
    cannotClimbMsg = (garageSide.cannotClimbMsg)
    cannotStandOnMsg = (cannotClimbMsg)
    cannotSitOnMsg = (cannotClimbMsg)    
    getFacets = [garageSide]
    dobjFor(Unlock) remapTo (Unlock, garageDoorOutside)
    dobjFor(Lock) remapTo (Lock, garageDoorOutside)
;

++ garageDoorOutside: LockableWithKey, Door '(garage) plain wooden wood door*doors' 
    'garage door'
    "It\'s a plain wooden door. "
    keyList = [keyBunch]
    knownKeyList = [keyBunch]

    dobjFor(Open) 
    {
        check() 
        {
            gReveal('door-tried');
            inherited;
        }
    }
;

+ porchFromBackYard: Underside, Enterable ->backPorch 'rear back porch' 'back porch'
    "The back porch is a plain wooden structure, with no railing or decoration.
    It looks as if it was tacked onto the house as an afterthought. It\'s
    several feet above the ground, with a ramp leading up to one side. The
    crawl space beneath it is deep in shadow. "
    contentsListed = nil
    contentsListedInExamine = nil
    
    dobjFor(GoUnder)
    {
        verify() 
        {
            illogical ('There\'s not enough space for you to wedge yourself
                beneath the porch. ');
        }
    }
    dobjFor(Search) asDobjFor(LookUnder)
    
    dobjFor(LookUnder)
    {
        action()
        {
            if (!skateboard.moved) 
            {
                "Deep in the shadows beneath the porch, you spy your
                skateboard!<<skateboard.discover>> ";
                achievement.awardPointsOnce();
                exit;
            }
            else inherited;
        }
    }
    
    achievement: Achievement { +2 "locating your skateboard" }
    
    notifyInsert (obj, newCont)
    {
        "The space under the porch is not a very good hiding place. For one
        thing, whatever you put under there might be hard to get back. ";
        exit;
    }
    cannotJumpOverMsg = &needJetPackMsg
    
    /* 
     *   We want the parser to choose this object or the back porch 
     *   depending on the location of the player char.
     */
    
    filterResolveList(lst, action, whichObj, np, requiredNum)
    {
        local obj = (me.isIn(backPorch) ? self : backPorch);
        local idx = lst.indexWhich({x:x.obj_== obj});
        if(idx && lst.length() > 1)
            lst = lst.removeElementAt(idx);
        
        return lst;
            
    }
    getFacets = [backPorch]
;

++ crawlSpace: Container, Component 'crawl deep space/shadow/crawlspace' 
    'crawl space'
    "You can\'t see much under the porch except spider
    webs<<skateboard.discover>><<skateboard.moved ? '. ' : ' --- but way back,
        near the wall of the house, there\'s your missing skateboard! '>>"
    dobjFor(Enter) 
    {
        verify() 
        { 
            illogical ('It\'s a very small crawl space --- there isn\'t even
                enough room for you to crawl into it. ');
        }
    }
    dobjFor(PutIn) remapTo (PutUnder, DirectObject, porchFromBackYard)
    bulkCapacity = 200
    maxSingleBulk = 200
;

++ skateboard: Platform, Vehicle, Hidden 'flame decals/skateboard/board'
    'your skateboard'
    "The flame decals are starting to peel off, but that\'s because
    you\'ve spent so many happy hours zipping around town on your
    skateboard. "
    bulk = 35
    bulkCapacity = 220
    maxSingleBulk = 220
    owner = me
    isQualifiedName = true
    isKnown = true
    dobjFor(Take)
    {
        check()
        {
            #ifdef __DEBUG
            /* 
             *   Don't put any restrictions here on using the SNARF verb when
             *   debugging.
             */
            if(gActionIs(Snarf))
                return;
            #endif
            
            if(!moved) 
            {
                gReveal('skateboard-reach');
                failCheck('It\'s clear back against the wall of the house.
                    Even when you stretch your arm as far as you can, the
                    skateboard is still at least three feet out of reach. ');
            }
        }
    }
    dobjFor(Move)
    {
        preCond = inherited + objHeld
    }
    
    dobjFor(MoveWith)
    {
        verify()
        {
            if(moved)
                illogicalAlready('You don\'t need to use {the iobj/him} to
                    move the skateboard. ');
        }
        check()
        {
            if (gIobj == bambooStick) failCheck ('You nudge the skateboard
                this way and that with the stick, but doing so brings it
                no closer to you. Possibly something just as long, but with
                an end better suited for grasping, would work better. ');
            
            if(gIobj != rake)
                failCheck('You can\'t reach {the dobj/him} with
                    {that iobj/him}. ');
        }
        action()
        {
            "By groping with {the iobj/him} under the porch, you manage to drag
            {the dobj/him} close enough to get your hands on it --- and then
            you pick it up. Success! ";
            achievement.awardPointsOnce();
            moveInto(gActor);
        }
    }
    achievement: Achievement { +4 "retrieving your skateboard from under the porch" }
    
    notifyInsert(obj, newCont)
    {
        if(obj == pot && newCont == self)
            setSuperclassList([Surface, TravelPushable]);
    }
    
    notifyRemove(obj)
    {
        if(obj == pot)
            setSuperclassList([Platform, Vehicle]);
    }
    
    cannotTakeMsg = '{You/he} {can\'t} pick up {the dobj/him} while the pot\'s
        on it. '
    
    cannotPushTravelMsg = 'It would be simpler to pick it up and carry it. '
    
    iobjFor(PutOn)
    {
        action()
        {
            if(gDobj == pot && meetsObjHeld(gActor))
                tryImplicitAction(Drop, self);
            
            inherited;
        }
    }
    
    useSpecialDesc = (location.ofKind(getOutermostRoom))
    specialDesc()
    {
        if(me.isIn(self))
            "You are <<me.posture.participle>> on your skateboard. ";
        else
            "Your skateboard is lying on <<getOutermostRoom.roomFloor.theName>>.";
    }
    
    remoteSpecialDesc(pov)
    {
        if(!elf.isIn(self))
          "Your skateboard is lying on <<getOutermostRoom.roomFloor.theName>>
          <<getOutermostRoom.inRoomName(pov)>>. ";
    }
    
    remoteDesc(pov)
    {
        if(pot.isIn(self))
            "There's a large pot <<elf.isIn(pot) ? 'containing ' + elf.aName :
              ''>> on it. ";
        else
            "You can't make out the details, but you're pretty sure it's yours.
            ";
    }
    
    beforeTravel(traveler, connector)
    {
        if(traveler == self && !getOutermostRoom.ofKind(OutdoorRoom))
            failCheck(cannotRideIndoorsMsg);
    }
    
    cannotRideIndoorsMsg = 'Your mother has finally more or less convinced you
        that skateboarding indoors is a bad idea. (The six months\' allowance
        it took to pay for the broken crockery drove the point home.) '
    
    sightSize = medium
    
    beforeMovePushable(traveler, connector, dest) 
    {
        "<.p>You push <<theName>> ";
        if(connector.ofKind(ThroughPassage))
            "through <<connector.theName>>.<.p>";
        else if(dest==backYard && getOutermostRoom==backPorch)
            "down the ramp.<.p>";
        else if(dest==backPorch && getOutermostRoom==backYard)
            "up the ramp.<.p>";
        else
            "out of <<getOutermostRoom.destName>>.<.p>";
    }
    
    describeMovePushable(traveler, connector)
    {
        "<.p>\^<<theName>> comes to rest
        <<getOutermostRoom.inRoomName(me)>>.<.p>";
    }
;

+ Decoration 'few sorry weeds/weed' 'weeds'
    "Your limited horticultural expertise doesn\'t run to identifying the
    varieties. So far as you\'re concerned they\'re just weeds, the sort of
    plants your father spends hours trying to eliminate from your back yard. "
    notImportantMsg = 'You never did have any time for weeds. '
    isPlural = true
;

+ holes: Container, CustomFixture
    'various sized hole/holes/pit/pits/excavations' 'holes'
    "Holes of various sizes have been dug across the lawn, seemingly at random,
    as if some demented fortune-seeker were desperately seeking buried treasure.
    The more prosaic explanation, given the state of the back yard, is that the
    holes are due to energetic but incompetent gardening efforts. "
    disambigName = 'excavations'
    bulkCapacity = 50
    maxSingleBulk = 50
    isPlural = true
    canMatchIt = true
    contentsLister: thingContentsLister 
    {
        showListPrefixWide(itemCount, pov, parent)
        {
            "There's ";
        }
        showListSuffixWide(itemCount, pov, parent)
        {
            " in one of the holes. ";
        }
    }
    descContentsLister: thingDescContentsLister
    {
        showListPrefixWide(itemCount, pov, parent)
        {
            "One of the holes contains ";
        }
    }
    lookInLister: thingLookInLister
    {
        showListEmpty(pov, parent)
        {
            "There's nothing of interest in any of the holes. ";
        }
        showListPrefixWide(itemCount, pov, parent)
        {
            "One of the holes contains ";
        }
    }
    cannotTakeMsg = 'A hole really isn\'t the kind of thing you can move around.
        After all a hole is the <i>absence</i> of material, and ... well,
        never mind, I\'m sure you get the point. '
    cannotJumpOverMsg = 'Jumping over the holes would doubtless provide hours of
        harmless amusement, but you have more pressing things to attend to, and
        you don\'t want Mrs. Pepper to return to find you playing hopscotch in
        her back yard. '
    cannotEnterMsg = 'Although the holes are of various sizes, none of them is
        large enough to contain a person. '
    
    dobjFor(FillWith) maybeRemapTo
    (gTentativeIobj && gTentativeIobj.indexWhich({x:x.obj_==rake}),
                                   MoveWith, gravel, IndirectObject)
    
    iobjFor(PourInto)  {   verify() {}   }
    
;

+ gravel: Decoration 'mounds (of) dirt (and) piles/mounds/dirt/gravel/soil' 
    'dirt and gravel'
    "Mounds of dirt and gravel strew the back yard alongside the holes from
    which they were dug. "
    
    notImportantMsg = 'The dirt doesn\'t look at all interesting. '
    
    lookInDesc = "A quick look turns up nothing of interest concealed in the
        mounds of dirt, and you see nothing to indicate that a longer search
        would be any more rewarding. "
    isMassNoun = true
    canMatchThem = true
    
    dobjFor(MoveWith)
    {
        verify() {}
        check()
        {
            if(gIobj != rake)
                failCheck('You won\'t get very far trying to shift the dirt
                    with {that iobj/him}. ');
        }
        action()
        {
            "You make a desultory attempt to rake some of the dirt back into
            one of the holes, but the effort soon tires you and the exercise
            seems pointless; Mrs. Pepper hasn\'t asked you to fix her garden
            and she\'d be too mean to pay you if you did. ";
        }
            
    }
;

//------------------------------------------------------------------------------
// The garage.

garage: Room 'Garage'
    "The interior of the garage smells dank and musty. A little sunlight leaks
    in through cracks and knotholes, and a pile of clutter occupies one
    corner. A workbench runs along the wall. "
    vocabWords = 'garage'
    west = garageDoorInside
    south: NoTravelMessage { "You can't get the main garage door open. " }
    out asExit(west)    
    #ifdef __DEBUG
    verifyDobjPow() { logicalRank(150, 'location'); }
    #endif
;

+ Decoration 'main wide (garage) door*doors' 'main garage door'
    "It's wide enough for a car to drive through. "
    notImportantMsg = 'Whatever you do to it, you can\'t get it to open. '
;

+ SimpleOdor 'dank musty smell' 'dank musty smell'
    "The interior of the garage smells dank and musty. "
;

+ cracks: Decoration 'knot cracks/knotholes/holes' 'cracks and knotholes'
    "Sunlight leaks into the garage in streaks and spots through the cracks and
    knotholes. "
    isPlural = true
;

+ Vaporous 'streaks/spots/sunlight' 'sunlight'
    desc = (cracks.desc)
    isMassNoun = true
    canMatchThem = true
;

+ Decoration 'dusty dust (and) cobwebs cardboard discarded 
    pile/cobwebs/clutter/cartons/junk/dust/boxes' 'junk'
    "The pile of clutter in the corner seems to consist mostly of discarded
    cardboard cartons, which are covered with dust and cobwebs. "
    
    dobjFor(LookIn) 
    {
        verify() {}
        check() {}
        action() 
        {
            "You poke around among the cartons for a minute, but there seems to
            be nothing of any interest lurking there. ";
        }
    }
;

+ garageDoorInside: Door ->garageDoorOutside 
    '(garage) side door*doors' 'garage side door'
    "It\'s an ordinary door. "
;

+ rake: Thing 'filthy long handled long-handled (metal) tines/rake' 'rake'
    "This is the long-handled rake that Mrs.\ Pepper tried to claw you with. Its
    long metal tines are filthy. "
    iobjFor(MoveWith)
    {
        verify() {}
    }
    iobjFor(AttackWith) {
        verify() {}
    }
    initSpecialDesc = "A long-handled rake leans against the wall. "
    sightSize = medium
    bulk = 75
;

+ workbench: Surface, Fixture 'stained nicked workbench/bench' 'workbench'
    "The workbench is built along one wall of the garage. It\'s rather stained
    and nicked, as if someone used it at one time, but now it\'s
    <<((toolbox.location == self) || (screwdriver.location == self)) ? 'mostly'
      : 'entirely'>> bare of tools. "
    bulkCapacity = 500
    maxSingleBulk = 200
;

++ toolbox: OpenableContainer 'metal tool toolbox/box/dents/dribbles/paint' 'toolbox'
    "The toolbox is made of metal, and has a hinged lid. Except for a few dents
    and dribbles of paint, it appears to be in good condition. "
    sightSize = medium
    bulkCapacity = 40
    maxSingleBulk = 40
    bulk = 50
;

+++ screwdriver: Thing 'flat-bladed flat bladed screwdriver/tool*tools' 
    'screwdriver'
    "It\'s a very ordinary flat-bladed screwdriver. "
    bulk = 7
    iobjFor(UnscrewWith) {  verify() {} }
    iobjFor(ScrewWith)   {  verify() {} }
    iobjFor(AttachWith)  {  verify() {} }
    iobjFor(TurnWith)    {  verify() {} }
    iobjFor(MoveWith) 
    {
        verify() {}
        check() {
            if (gDobj != glassJar) failCheck ('Moving {the dobj/him} with
                the screwdriver is a novel idea, but it\'s not easy to see how
                you would do that. ');
        }
    }
    cannotDrinkMsg = 'This screwdriver is a tool, not a cocktail. Besides,
        it\'ll be another seven or eight years before you\'re allowed vodka in
        your orange juice. '
;

//------------------------------------------------------------------------------
// The backporch is simply a place to travel through.

backPorch: Room 'Back Porch'
    "The back porch is a plain wooden structure, with no railing or decoration.
    It looks as if it was tacked onto the house as an afterthought. It\'s
    several feet above the ground, with a ramp leading up to one side. The
    back door leading into the house is just to the south. "
    south = backDoorOutside
    north = backYard
    in asExit(south)
    out asExit(north)
    down asExit(north)
    vocabWords = 'back porch'
    
    iobjFor(PutOn) remapTo(PutOn, DirectObject, roomFloor)
    
    cannotLookUnderMsg = 'You can\'t see under the porch from here. '
    getFacets = [porchFromBackYard]
    yellCheck()
    {
        if(gActor.canSee(neighbour))
            failCheck('If you want to attract {the neighbour/she}\'s attention
                it would be better to do so from the back yard. ');
        else
            inherited;
    }
;

+ Enterable ->backDoorOutside '(back)/rear/house' 'house'
    "From inside the porch, you can't see much of the house apart from its back
    door. "
;

++ backDoorOutside: LockableWithKey, Door 'back door*doors' 'back door'
    keyList = [keyBunch]
    knownKeyList = [keyBunch]

    dobjFor(Open) {
        check() {
            gReveal('door-tried');
            inherited;
        }
    }
;

//------------------------------------------------------------------------------
// The garden next door, which you can't actually visit.

nextDoorGarden: OutdoorRoom 'The Garden Next Door'

    remoteDesc(pov)
    {
        "Over the fence you can see that the garden next door is very well
        tended. Growing in one corner is a spectacular patch of bamboo shoots. ";
    }
    vocabWords = '(her) next door door\'s well tended well-tended neighbour\'s
        adjoining garden next neighbor\'s garden/(yard)/(door)'
    sightSize = large
    cannotJumpOverMsg = &needJetPackMsg
    actorInName = 'in the garden next door'
    owner = neighbour
    dobjFor(Enter)
    {
        action() { replaceAction(Climb, backyardFence); }
    }
;

+ bamboo: Fixture 'spectacular tall thick green bamboo 
    cluster/clump/patch/shoots/stand/bamboo'
    'patch of bamboo shoots'
    "The cluster of bamboo is tall, thick, and green. "
    sightSize = large
;

DistanceConnector [backYard, nextDoorGarden]
    transSensingThru(sense) 
    { 
        return sense == sight ? transparent : distant; 
    }
;

DistanceConnector [backPorch, backYard];

bambooStick: Thing 'leafy thin supple bamboo stick/sprigs/leaves/leaf/sprig'
    'bamboo stick'
    "The stick of bamboo is thin and supple. It\'s about four feet long, and a
    few green sprigs of leaves are sprouting from it. "
    bulk = 55
    
    isKnown = true 
    possiblyGive()
    {
        if(gRevealed('bamboo-request'))
        {
            "<.q>One good turn deserves another!<./q> she adds with a grin.
            <.q>Now that you\'ve returned my brooch, I\'d be happy to cut you
            a piece of bamboo!<./q>\b
            She crosses her garden and returns a few moments later bearing a
            stick of bamboo, which she hands to you over the fence. <.q>Well,
            I believe I\'ve done enough gardening for this afternoon,<./q> she
            finishes. <.q>I\'m going to go inside now and phone Gladys to tell
            her all about how a nice young person found my brooch and returned
            it to me.<./q> With that, she strips off her gardening gloves and
            trots off toward her back door. ";
            bambooStick.moveIntoForTravel(me);
            achievement.awardPointsOnce();
            neighbour.moveIntoForTravel(nil);
        }  
    }
    iobjFor(MoveWith)
    {
        verify() {}
        check() 
        {
            if (gDobj != skateboard) 
                failCheck('There\'s probably no reason to poke at that with the
                    bamboo stick. ');
        }
    }
    achievement: Achievement { +3 "getting the bamboo stick from Mrs.\ Dogwood" }
;
//------------------------------------------------------------------------------
// An object to represent Mrs Pepper's absence

mrsPepper: MultiLoc, Unthing 'mrs mrs. old pepper/woman/lady/witch' 'Mrs. Pepper'
    notHereMsg()
    {
        if(oldCar.isIn(nil))
            return 'She\'s not here. She\'s driven away. ';
        
        if(gRevealed('door-slam'))
            return 'She isn\'t here. She\'s slammed the door on you. ';
        
        return 'She\'s not here. ';
    }
    initialLocationClass = Room
    vocabLikelihood = 10
    isProperName = true
    isHer = true
    isGone = nil
;

/* 
 *   When Mrs Dogwood is in scope, the parser will automatically choose her 
 *   in preference to this UnThing, and without disambiguation prompt.
 */

MultiLoc, Unthing 'mrs. mrs dogwood/neighbor/neighbour/woman' 'Mrs. Dogwood'
    'She\'s not here. '
    initialLocationClass = Room
    isHer = true
//    vocabLikelihood = 10

;

/* A number of ubiquitous outdoor objects */


MultiLoc, Vaporous 'air' 'air'
    "You can't actually see it; you only know it's there because you're still
    breathing. "
    
    dobjFor(Take)
    {
        verify()
        {
            if(gAction.getEnteredVerbPhrase.startsWith('pick'))
                illogical(notWithIntangibleMsg);
        }
        action()
        {
            "You take a few deep breaths. ";
        }
    }
    
    notWithIntangibleMsg = 'You may as well try to push a cloud or ride a
                    moonbeam. '
    
    dobjFor(Smell)
    {
        verify(){}
        action() { replaceAction(SmellImplicit); }            
    }
    
    dobjFor(Feel)
    {
        verify() {}
        action()
        {
            "It is neither warm enough nor cold enough nor blustery enough to
            produce any tactile sensation. ";
        }
    }
    soundDesc = "The air is quite still today, and therefore quite silent. "
    lookInDesc = "The air is quite transparent. "
    initialLocationClass = Room
;

MultiLoc, Distant 'dazzling bright sun/disc/disk' 'sun'
    "The sun is a dazzling disk lazily making its way across the
    southwestern sky. It's far too bright to stare at. "
    initialLocationClass = OutdoorRoom
    tooDistantMsg(obj)
    {
        return 'The sun is about ninety-three million miles too far away. ';
    }
;

modify defaultSky
    vocabWords = 'clear blue - '
    desc = "The sky is a clear blue, without a cloud in sight. "
    tooDistantMsg(obj)
    {
        return 'Your aspirations would seem to be way above your abilities. ';
    }
    
;