/* cvrdln.c
 *************************************************************************/

/* 
 * Optional readline interface.  Provides all the usual editing stuff 
 * plus history, with completion on some of the common adventure verbs.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#ifdef HAVE_LIBREADLINE
#include <readline.h>
#include <history.h>
#endif

#ifdef HAVE_LIBREADLINE
static char *line_read = NULL;
#else
static char line_read[BUFSIZ];
#endif

#ifdef HAVE_LIBREADLINE
/* Completion word list.  NOTE - only the general words known in   */
/* all adventure games are here.  No special words are given away. */
static char *word_list[] = {
    "look", "help", "info", "get", "drop", "lock", "rub",
    "find", "inventory", "say", "quit", "score", "brief",
    "throw", "go", "pay", "break", "drink", "eat", "read",
    "tie", "untie", "feed", "pour", "kill", "cut", "fill",
    "touch", "save", "restore", NULL
};

/* Word completion function */
static char *
adv_word_match(const char *text, int state)
{
    static int list_index, len;
    char *name = NULL;
    char **cpp;

    /* If starting from scratch, initialize */
    if (!state) {
	list_index = 0;
	len = strlen(text);
    }

    /* Return the next matching name from the word list */
    for (cpp = &word_list[list_index++]; *cpp != NULL; cpp++, list_index++)
	if (!strncmp(*cpp, text, (size_t) len))
            break;

    if (*cpp != NULL) {
	name = (char *) malloc(strlen(*cpp)+1);
	strcpy(name, *cpp);
    }

    return name;
}
#endif

/* Return pathname of history file */
char *
history_file(void)
{
    static char path[BUFSIZ];
    sprintf(path, "%s/.crystal", getenv("HOME"));
    return path;
}

/* Initialize readline */
void
initline()
{
#ifdef HAVE_LIBREADLINE
    /* Set completion function */
    rl_completion_entry_function = adv_word_match;

    /* Initialize history functions */
    using_history();

    /* Read old history (if any) */
    read_history(history_file());
#endif
}

/* Get an input line */
char *
getline(char *prompt)
{
#ifdef HAVE_LIBREADLINE
    char *old_line = line_read;

    /* Get new line */
    line_read = readline(prompt);
    
    /* Save it on history if it's worth it */
    if (line_read && *line_read)
	if (!old_line || strcmp(old_line, line_read) != 0)
	    add_history(line_read);

    /* Free old line (if any) */
    if (old_line != NULL) free(old_line);
#else
    /* Get line */
    fputs(prompt, stdout);
    if (fgets(line_read, BUFSIZ, stdin) == NULL) return NULL;

    /* Remove trailing newline */
    line_read[strlen(line_read)-1] = '\0';
#endif

    return line_read;
}

/* Clean up readline */
void
endline()
{
#ifdef HAVE_LIBREADLINE
    /* Write history to file */
    if (write_history(history_file()))
	fprintf(stderr, "\nFailed to write history to %s: %s\n",
		history_file(), strerror(errno));
#endif
}
