#charset "us-ascii"

/*
 *   Copyright (c) 1999, 2002 by Michael J. Roberts.  Permission is
 *   granted to anyone to copy and use this file for any purpose.  
 *   
 *   This is a starter TADS 3 source file.  This is a complete TADS game
 *   that you can compile and run.
 *   
 *   To compile this game in TADS Workbench, open the "Build" menu and
 *   select "Compile for Debugging."  To run the game, after compiling it,
 *   open the "Debug" menu and select "Go."
 *   
 *   This is the "advanced" starter game - it has only the minimum set of
 *   definitions needed for a working game.  If you would like some more
 *   examples, create a new game, and choose the "introductory" version
 *   when asked for the type of starter game to create.  
 */

/* 
 *   Include the main header for the standard TADS 3 adventure library.
 *   Note that this does NOT include the entire source code for the
 *   library; this merely includes some definitions for our use here.  The
 *   main library must be "linked" into the finished program by including
 *   the file "adv3.tl" in the list of modules specified when compiling.
 *   In TADS Workbench, simply include adv3.tl in the "Source Files"
 *   section of the project.
 *   
 *   Also include the US English definitions, since this game is written
 *   in English.  
 */
#include <adv3.h>
#include <en_us.h>
#include <SquareCircle.h>

/*
 *   Our game credits and version information.  This object isn't required
 *   by the system, but our GameInfo initialization above needs this for
 *   some of its information.
 *   
 *   IMPORTANT - You should customize some of the text below, as marked:
 *   the name of your game, your byline, and so on.  
 */
versionInfo: GameID
    name = 'Square Circle'
    byline = 'by Eric Eve'
    htmlByline = 'by <a href="mailto:eric.eve@hmc.ox.ac.uk">
                  Eric Eve</a>'
    version = '1.2 (Post-Competition Release, December 2004)'
    authorEmail = 'Eric Eve <eric.eve@hmc.ox.ac.uk>'
    desc = 'What is your crime? Why do you feel both guilty and unjustly punished?
     What has happened to your memory? How will you draw a square circle and get
      out of your prison? What will you find then? '
    htmlDesc = 'What is your crime? Why do you feel both guilty and unjustly punished?
     What has happened to your memory? How will you draw a square circle and get
      out of your prison? What will you find then? '

    showCredit()
    {
        /* show our credits */
        inherited;
        
         "\bTADS 3 Library and language by Michael J. Roberts.\n
         Cquotes (c) 2002, 2004 Stephen Granade\n
         
         ncDebugActions (c) 2002 Nikos Chantziaras (not incorporated
          into the release build, but an invaluable aid for development
          and testing)\b
         Alpha-testing: Antonia Barke\n
         Beta-testing: Kristian Andresen, Paul Boucher, Nikos Chantziaras, Michel Nizette, 
         Andreas Sewe, Phil Swinbank and Mike Sousa (a great team who have contributed
         enormously to the development of this game)\b
         Freeware License text by Roger Firth.\b
         Finally, thanks to everyone who reviewed the competition version of this
         game or otherwise provided feedback on it, and especially to Jess Knoch
         and Paul O'Brian for sending me copies of their transcripts. Studying all
         these revealed numerous things that needed improving! ";

        /* 
         *   The game credits are displayed first, but the library will
         *   display additional credits for library modules.  It's a good
         *   idea to show a blank line after the game credits to separate
         *   them visually from the (usually one-liner) library credits
         *   that follow.  
         */
        "\b";
    }
    showAbout()
    {
        "<.p>        
        <i>Square Circle</i> was my entry 
         in the 2004 Annual IF Competition, in which it was placed fifth .
         This post-competition release aims to address the most obvious and fixable
         flaws that came to light in the competition reviews; most of the changes
         from the competition version are relatively minor tweaks, many of them to the 
         hint system, but hopefully they will add up to a reasonably enhanced playing
         experience. But should you discover any surviving bugs, please do let me
         know (email me at eric dot eve at hmc dot ox dot ac dot uk).\b
                
        It is fairly easy to get yourself killed, but if you do you can 
        always UNDO. It is not meant to be possible to get into an unwinnable situation
        without some fairly clear indication of your blunder. In any case, if
        you attempt an action that is likely to have dire consequences, the
        game will normally warn you (the first time), and will only let you go ahead 
        with it if you persist.\b
                      
        It is perfectly possible to win the game without attaining the maximum
        score. This is no cause for concern, it just means that there are other
        (mostly quite trivial) tasks you could have performed that would have
        earned some bonus points.\b
        
        For context-specific hints, type <A HREF='HINTS'>HINTS</A>.\n
        For general instructions, type <A HREF='INSTRUCTIONS'>INSTRUCTIONS</A>.\n
        For guidance on special (non-standard) interface features in this game,
        type <A HREF='FEATURES'>FEATURES</A>. (First time players, especially
        players who have not played many TADS 3 games, are <i>strongly</i> advised
        to look at this.)\n
        This game is Freeware; for license details type <A HREF='LICENSE'>LICENSE</A>.<.p>";
    }
;

/*
 *   Define the player character.  The name of this object is not
 *   important, but note that it has to match up with the name we use in
 *   the main() routine to initialize the game, below.
 *   
 *   Note that we aren't required to define any vocabulary or description
 *   for this object, because the class Actor, defined in the library,
 *   automatically provides the appropriate definitions for an Actor when
 *   the Actor is serving as the player character.  Note also that we
 *   don't have to do anything special in this object definition to make
 *   the Actor the player character; any Actor can serve as the player
 *   character, and we'll establish this one as the PC in main(), below.  
 */
me: Person
    /* the initial location */
    location = nil
    maxBulk = 10
    maxWeight = 10
    bulkCapacity = 100
    consulting = nil
    asleep = true
    pcDesc = "So far as you can tell without the aid of a mirror, you look much as you 
    always did, apart from <<clothing()>>. "
    clothing()
    {
      if(blueCoveralls.isWornBy(self))
        "the crumpled faded blue coveralls in which someone has dressed you";
      else if(oldClothes.isWornBy(self))
        "the disgusting old clothes you're wearing";
      else
        "the fact that you're completely naked";
    } 
    goToSleep()
    {
      local response;
      if(isIn(cellBed))
      {
        if(posture != lying)
          tryImplicitAction(LieOn, cellBed);
        response = rand ('You manage to doze off, but when you wake up again nothing much 
          seems to have changed.', 'When you drift off to sleep you have a nightmare about
          being trapped in a strange prison; then you wake up to find it\'s true. ',
          'You lie still for some time but sleep eludes you. ', 'Your snoring
          wakes you up again.' );
          asleep = true;
      }
      else
        response = rand('This doesn\'t seem quite the time or the place for a snooze. ',
           'Don\'t you have more important things to do? ', 'Dreaming about
           square circles won\'t help -- best stay awake. ', 'You shut your eyes
           and start counting sheep, but then remember that you\'re not in bed. ');
           
     say(response);
    }
    verifyDobjTalkTo { 
      if(gActor == self)
         illogicalSelf('You\'d better not start talking to yourself! '); }   
;

+ blueCoveralls : Wearable 'crumpled faded worn (blue) (prison) coveralls/uniform' 
   'faded blue coveralls'
   "They're rather worn, and obviously second-hand, but basically serviceable. No one
   could call them fashionable, and they're probably meant to be some kind of prison
   uniform. "
   isPlural = true
   wornBy = me
   dobjFor(Examine)
   {
     /* override the preference for objects that are worn or held so this
      * isn't annoyingly chosen as a default. 
      */
     verify() { logicalRank(80, 'not held'); }
   }
   dobjFor(Doff)
   {
     check()
     {
       if(!oldClothes.isIn(gActor))
       {
         "You're not wearing anything under the coveralls, and since on reflection
         you decide it isn't such a great idea to go around in your birthday suit you
         decide to keep them on for now. ";
         exit;
       }
       else if(!gActor.getOutermostRoom.private)
       {
         "You'd like to go somewhere more private before stripping off. ";
         exit;
       }
     }
   }
   dobjFor(Wear)
   {
     check()
     {
       if(oldClothes.isWornBy(gActor))
       {
         "You're already dressed. ";
         exit;
       }
     }
   }
   bulk = 5
   cannotWriteOnMsg = &uselessToScribbleOnMsg
   cannotDrawOnMsg = &uselessToScribbleOnMsg  
   
;

class Hand : Component
  isQualifiedName = true
  desc = "Both your hands are fairly smooth; by the look of them they haven't
   been used for much heavy manual labour. "
  iobjFor(RubWith)
  {
    verify() 
     {
       if(gIobj == gDobj)
        illogicalSelf('You can\'t rub {the dobj/him} with itself. ');
     }
    check() {}
    action() 
    {
      defaultReport('You rub {the dobj/him} vigorously with {the iobj/him}, 
       but without any obvious effect. ');
    }
    preCond = []
  }
  iobjFor(AttackWith) remapTo(Attack, DirectObject)
  iobjFor(DigWith) { preCond = [] }
  iobjFor(TurnWith) { preCond = [] }
  iobjFor(ScrewWith) { preCond = [] }
  iobjFor(UnscrewWith) { preCond = [] }
;

+ leftHand : Hand '(your) (my) left hand*hands' 'your left hand'  
  meetsObjHeld(actor) { return nil; }
  iobjFor(DigWith) { verify() {} }
;

+ rightHand : Hand '(your) (my) right hand*hands' 'your right hand'  
  iobjFor(RubWith)
  {
    verify()  {  logicalRank(110, 'default rubber');  }    
  }  
  iobjFor(DigWith)
  {
    verify() { logicalRank(110, 'default digger'); }    
  }
  iobjFor(TurnWith) { verify() { logicalRank(110, 'default'); } }
  iobjFor(ScrewWith) { verify() { logicalRank(110, 'default'); } }
  iobjFor(UnscrewWith) { verify() { logicalRank(110, 'default'); } }
  meetsObjHeld(actor) { return nil; }
;


+ feet : Component '(right) (left) bare foot/feet/toes' 'feet'
  "<<boots.isWornBy(gPlayerChar) ? 'They\'re inside a pair of boots. '
   : 'They look like pretty ordinary feet, except that they\'re bare. '>>"
   isPlural = true
   theName = 'your feet'
   aName = (theName)
   cannotTakeMsg = 'You\'re already quite attached to them. '
   iobjFor(PutOn)
   {
    verify()
    {
     if(gDobj != boots) 
       illogical('You really don\'t want to put {that dobj/him} on your feet. '); 
     else if(boots.isWornBy(gPlayerChar)) 
       illogicalAlready('You\'re already wearing them. ');
    }
    action() { replaceAction(Wear, gDobj); }
    
   }   
   meetsObjHeld(actor) { return nil; }
;


gameMain : GameMainDef
  initialPlayerChar = me
  showIntro()
  {
     cls();
     
     libGlobal.isTextOnlyInterpeter = (systemInfo(SysInfoInterpClass) ==  SysInfoIClassText);
//     libGlobal.canRestore = true;
     "<CENTER><b>SQUARE CIRCLE</b></CENTER>\b 
        Guilt -- remorse -- fear. These three emotions remain clear, even though everything
        else has dissolved into amorphous fog. You have done something terribly wrong, 
        and you are being punished for it -- or are you? There is guilt, and there is
        punishment, but the relation between the two seems confused.\b
        You dream fitfully, unable clearly to separate fantasy from memory. Somewhere
         in the muddle of dream and memories are conflicting images, of a peaceful
         hillside picnic with -- someone important, someone close, someone
         you're sharing deep and important things with -- but that someone
         won't quite come into focus, and then the scene shifts to
         sudden terror -- panic -- gunfire -- blood -- a futile attempt to escape --
         a surreal trial and a bizarre sentence, a peculiar punishment for an unreal crime.
         Pain and poignant pleasure, hope and terrifying despair, joy and unbearable loss mingle 
         in these memories -- or are they just dreams? Either way your sleep seems strangely 
         restless, as your fuddled brain struggles to make sense of the jumbled images.\b
         Then, groggily, you slowly wake up, totally disoriented, at first not knowing
         where you are, let alone how you got here, or even who you are.\b
         Slowly, painfully, you grope your way to full consciousness. It is, at least,
          all too clear <i>where</i> you are, and even if you can remember little else, the guilt,
         fear and remorse remain all too plain.\b\b\b<HR>\b";
         "<CENTER>[First time users may wish to type <b>about</b> at the command prompt]</CENTER>\b";
         "<CENTER>[Or to see the <b>about</b> text now press <b>a</b> -- 
          to restore a saved game press <b>r</b> -- 
            press any other key to go straight to the game]</CENTER>";
         local ans = inputManager.getKey(nil, nil).toLower;   

         cls();        
         if(ans is in ('a'))
         {           
           versionInfo.showAbout();                    
         }
         if (ans == 'r')
         {
           if(RestoreAction.askAndRestore())
              cls();
           else
             "\bRestore failed -- beginning game from start.\b";
         }
         if(gPlayerChar.location == nil)
         {
            gPlayerChar.moveIntoForTravel(cellBed);
            gPlayerChar.posture = lying;
         }         
  }
  
  showGoodbye()
  {
      "<.p>Thanks for playing!\b";
  }

  setAboutBox()
  {
  "<ABOUTBOX><CENTER>
     <body bgcolor=silver>
     <h1>Square Circle</h1>\b      
      by\b
     Eric Eve\b
     Version <<versionInfo.version>>\b
     </CENTER></ABOUTBOX>";     
  }
  allVerbsAllowAll = nil
;



   /* The power object keeps track of whether the electrical power in the prison
    * compound is on or off. It starts on.
    */

power : object
  on = true
;

   /* The daylight object keeps track of the brightness of outside light, which
    * can vary with the time of day, becoming 2 at dusk and 0 at night.
    */

enum dawn, noon, dusk, night;

daylight : object
  time = noon
  brightness
  {
    switch(time)
    {
      case night: 
        return 0;
      case noon:
      case dusk:
        return 3;
      default:
        return 2;
    }
  }
;

modify OutdoorRoom
  brightness = (daylight.brightness)
;

  /*
   *  From now on this file contains the prison and its immediate environs, together
   *  with their associated objects.
   */


cellFloor : defaultFloor 'solid bare concrete slab/floor' 'floor'
  desc = "The floor is a solid slab of bare concrete. "
  feelDesc = "The concrete floor feels rough, cold and hard.
  <<boots.isWornBy(gPlayerChar) ? '' : 'It also feels harsh on your
  bare feet. '>>"  
;

cellSouthWall : RoomPart 'grim dull grey gray south s wall*walls' 'south wall'
  "It's an unrelieved expanse of grey. "
;

cellNorthWall : RoomPart 'grim dull grey gray north n wall*walls' 'north wall'
  "Apart from the door, it's an blank expanse of depressing grey. "
;

cellEastWall : RoomPart 'grim dull grey gray east e wall*walls' 'east wall'
  "Only the bookcase against it relieves the dull expanse of depressing grey. "
;

cellWestWall : RoomPart 'grim dull grey gray west w wall*walls' 'west wall'
  "The bed rests against the west wall of the cell, which is otherwise just
   another dull expanse of grey. "
;

cell : Room 'Cell' 'the cell'
    "There is little in this aggressively spartan cell apart from <<desk.moved ?
     nil : 'an old wooden desk,'>> an uncomfortable bed, and a small wooden
     bookcase fixed to one wall. All four of the walls have been painted the 
     same depressingly dull grey colour, seemingly designed to starve the senses and sap 
     the spirit; each of them is utterly unrelieved by even a hint of decoration, let 
     alone a window; <<bulb.isIn(lightFitting) ?
     'indeed the only light comes from a naked light bulb dangling from the ceiling overhead' 
     : (lightCube.isIn(self) ? 'the light cube casts strange shadows on the walls'
     : 'the only available light seeps in through the open door')>>. 
     The sole exit is to the north, <<cellDoor.isOpen ? 'through a sturdy iron door
      that\'s now open' : 'but that\'s blocked by a solid iron door that looks
      emphatically closed'>>."
    north = cellDoor    
    out asExit(north)
    brightness = (cellDoor.isOpen ? 2 : min(0, vestibule.brightness - 1))
    roomParts = [cellFloor, defaultCeiling, cellNorthWall, cellSouthWall,
      cellEastWall, cellWestWall]
    private = true
    cannotGoMsg = 'Funnily enough, the wall seems to be in the way. '
    roomBeforeAction()
   {
     if(gActionIs(LookThrough)) 
       return;
     if(gActionIs(Examine) && (gDobj not in (note, redBook, greenBook, blueBook)))
       return;
     if(gActionIs(TravelVia) || gActionIs(LookUnder))
       return;  
     #ifdef __DEBUG
     if(gActionIs(Snarf))
       return; 
     #endif  
     if(gDobj != nil && !gDobj.isIn(cellBed) && gActor.isIn(cellBed) 
        && gDobj not in (cellBed, cellWestWall) )
     {
		 tryImplicitAction(Stand);
     }
     if(gDobj == cellDoor && gActor.location is in (desk, swivelChair))
     {
         tryImplicitAction(GetOffOf, gActor.location);
     }      
   } 
   roomAfterAction()
   {
     if(gActionIs(Yell))
      "No one takes any notice. ";
   }
;

+ cellBed : ComplexContainer, CustomImmovable, Bed  
  'plain (iron) utilitarian black black-painted bed/bedstead*furniture' 'bed'
  "The bed is strictly utilitarian, being little more than a thin mattress on a plain, 
   black-painted iron bedstead. "
   subUnderside : ComplexComponent, Underside { maxSingleBulk = 4 }   
   down asExit(out)
   up asExit(out)
   dobjFor(JumpOff)
  {
    verify() { if(!gActor.isIn(self))
                   illogicalNow('{You/he} {is}n\'t on the bed. '); }
    action()
    {
      replaceAction(GetOutOf, self);
    }
  }
  cannotTakeMsg = 'The bed is firmly bolted to the floor. '
  cannotPushTravelMsg = (cannotTakeMsg)
  moveToNoEffectMsg = (cannotTakeMsg)
  iobjFor(MoveTo) remapTo(PutUnder, DirectObject, self)
  makeStandingUp()
  {
    inherited;
    if(gActor.asleep)
    {
      "{You/he} yawn{s}, stretch{es}, and rise{s} to {its/her} feet. ";
      gActor.asleep = nil;
    }
  }
  lookInLister: DescContentsLister, BaseSurfaceContentsLister {
     showListEmpty(pov, parent)
     {
         /* show a default message indicating the surface is empty */
         gMessageParams(parent);
         defaultDescReport('{You/he} see{s} nothing on top of {the parent/him}. ');
     }
   }
 ;


++ mattress : Underside, Component 'thin mattress' 'mattress'
   "The mattress is covered in some well-worn fabric with a faded pattern of broad
    black and white stripes. "
   feelDesc = "The mattress feels hard and lumpy. "  
   smellDesc = "Frankly, the mattress could do with a wash. "
   maxSingleBulk = 1
   maxBulk = 5
   dobjFor(Lift) asDobjFor(LookUnder)
   iobjFor(PutOn) remapTo(PutOn, DirectObject, cellBed)
   dobjFor(SitOn) remapTo(SitOn, cellBed)
   dobjFor(LieOn) remapTo(LieOn, cellBed)
   dobjFor(StandOn) remapTo(StandOn, cellBed)
   dobjFor(Search) {
     action(){       
       mainReport('Well, running your hands over it suggests there\'s nothing much
        but stuffing inside. ');
     }     
   }
   cannotTakeComponentMsg(obj) { return 'It\'s bulky, awkward to carry, and of poor quality;
    on futher consideration {you/he} decide{s} it\'s better left on the bed. '; }
   cannotPutComponentMsg(obj) { return 'There doesn\'t really seem any good reason for not
    leaving it on the bed. '; } 
   cannotMoveMsg() { return cannotPutComponentMsg(self); } 
;

+++ Component 'faded (black) well-worn (and) (white) broad stripes/pattern/fabric' 'stripes'
    "The pattern of black and white stripes on the mattress fabric looks
     aggressively institutional. "
    isPlural = true
    bulk = 0
;

++ tinBox : Fillable, CanRub, KeyedContainer, Hidden 
   'narrow gold (empty) small black tin strip/strips/box/base/lid*boxes' 'small tin box'
   "The box is painted black all over, apart from a narrow gold strip round its
    base and another on its lid. "
   subLocation = &subUnderside
   keyList = [brassKey]
   initiallyLocked = true
   lockedStatusObvious = nil
   dobjFor(Open)
   {
     action()
     {
       inherited;
       achievement.awardPointsOnce();
     }
   }
   iobjFor(PutOn)
   {
     verify() { }
     action()
     {
       "{The dobj/he} falls straight off it. ";
       gDobj.moveInto(location);
     }
   }
   achievement : Achievement { +1 "finding the iron key " }
   replacement = tinSandBox
   bulkCapacity = 3
   bulk = 3
   sightSize = small
   inflammable = nil
;

+++ ironKey : Hidden, CanRub, Key 'large iron key/loop*keys' 'large iron key'
  "It's about six inches long, with a large loop at one end and a pair of
    large teeth at the other. "
  verifyDobjTake() { logicalRank(120, 'portable'); }    
  dobjFor(Turn)
  {
    verify() {}
    check() {}
    action()
    {
      if(location.ofKind(Keyhole))
        {
          "The key turns smoothly in the lock. ";
          cellDoor.makeLocked(!cellDoor.isLocked);
        }
      else if(isIn(rustyLock))         
        replaceAction(UnlockWith, rustyGate, self);        
      else
        "That achieves little. ";
    }
  }
  dobjFor(TurnWith)
  {
    verify() 
    { 
      if(!(location.ofKind(Keyhole) || location.ofKind(rustyLock)))
        illogicalNow('That would achieve nothing. ');
    }
    check()
    {
      if(isIn(rustyLock))
      {
        reportFailure('The key won\'t turn; trying to force it might break it. ');
        exit;
      }
    }
    action()
    {
      replaceAction(Turn, self);
    }
  }
 getFacets() { return [filedIronKey]; }
 dobjFor(RubWith)
 {
   action()
   {
     if(gIobj == file)
     {
       "Using the metal implement you file away at the teeth of the key until you're
        satisfied they look sufficiently different. ";
        filedIronKey.moveInto(location);
        moveInto(nil);
     }
     else if(gIobj.ofKind(Hand) && gVerbName == 'file')
       "You really can't file a metal object with your bare hands. ";
     else
      "Rubbing the key with {the iobj/him} doesn't seem to achieve very much. ";
   }
 }
 dobjFor(Pick) maybeRemapTo(gActor.canSee(brassKey), Take, self)
 discovered = true
 inflammable = nil
 shouldNotBreakMsg = 'That would be counter-productive. '
 sightSize = small
;

++++ ironKeyTeeth : Component 'large pair/teeth' 'teeth'
  "The iron key has a pair of large teeth at the business end. "
  dobjFor(RubWith) remapTo(RubWith, ironKey, IndirectObject)
  isPlural = true
  sightSize = small
;

+ lightFitting : OutOfReach, Immovable
   'plain (light) white coated white-coated fitting/wire/connector/cable/socket' 'light fitting'
   "The light fitting is simply a plain white connector at the end of a piece of white-coated 
    wire dangling from  the ceiling. "
   canObjReachContents(obj) { 
     return ((obj.isIn(desk) || obj.isIn(sandBox) || obj.isIn(guardChair)) && 
        obj.posture == standing) || gActionIs(PutUnder); 
    }
   iobjFor(PutIn)
   {
     verify()
      { if(gDobj not in (bulb, silverDisc) ) 
           illogical('You can\'t put that in the light fitting. '); 
      }
   }
   /*
    *  From the very start, testers tried putting, moving, or pushing things
    *  under the light, so I attempted to add rudimentary handling for this
    *  that made a bit more sense than the default library handling. Frankly,
    *  the mechanism used here is a bit of a kludge, but it handles this case
    *  well enough.
    *
    *  The main idea is that the desk starts off under the light, and always
    *  ends back under the light if its pushed back into the cell. So, while
    *  the desk is under the light, nothing else can be put there. If something
    *  bulky such as the chair is put under the light while the desk is not
    *  in the cell, it is pushed out of the way when the desk is pushed back into
    *  the cell.
    *
    *  None of this is actually necessary to the mechanics of the game; it simply
    *  provides more meaningful responses to actions the player might try.
    */ 
      
   iobjFor(PutUnder)
   {
     verify() { }     
     check() {
       if(underList.indexOf(gDobj)) {
         reportFailure('{The dobj/he} is already under the light fitting. ');
         exit;
       }
       if(desk.isIn(cell)) {
         reportFailure('You can\'t; the desk is in the way. ');
         exit;
       }
     }
     action()
     {
       if(!gDobj.ofKind(NonPortable) && !gDobj.ofKind(TravelPushable)){
        if(gDobj.location != cell) gDobj.moveInto(cell);
        "{You/he} position{s} {the dobj/him} under {the iobj/him}. ";
        underList += gDobj;
       } 
     }  
   }
   iobjFor(MoveTo) remapTo(PutUnder, DirectObject, self)
   underList = static new Vector(10)
   actionDobjPull {    
      reportFailure ('Tugging the cable gently has no effect; tugging it harder
        might be unwise. '); 
    }
   cannotMoveMsg = 'It\'s fixed to the ceiling. '
   cannotTakeMsg = (cannotMoveMsg) 
   afterAction()
   {
     if(underList.indexOf(gDobj) && gDobj.location != cell)
       underList.removeElement(gDobj);
   }
;

++ bulb : Fragile, LightSource 'naked light 100W bulb/lightbulb' 'light bulb'
  "<<isIn(lightFitting) ? 'The light bulb hangs from a plain fitting that is little 
    more than a short length of cable with a connector on the end.' : ''>>
    It appears to be a 100W bulb. "
   initSpecialDesc = ""
   dobjFor(Take)
   {

     action()
     {
       local oldLoc = location;
       if(isLit && isIn(lightFitting))
       {
         if(wouldBeDark && !warnedOfDark)
         {
           warnedOfDark = true;
           "Your hand reaches out for the bulb, but then you hesitate a moment
            when it occurs to you that removing the bulb from its fitting will
            almost certainly plunge the room into total darkness. ";
            exit;
         }
       }
       inherited;
       if(!isIn(lightFitting) && oldLoc == lightFitting)
       {        
         "The bulb is searingly hot to the touch, but you just manage to remove
         it from its fitting. As you do so, the bulb goes out. ";
         makeLit(nil);
       }
     }
   }
   dobjFor(PutIn)
   {
     action()
     {
       if(gIobj == lightFitting)
       {
         if(power.on)
         {
           "As you put the bulb into the fitting it lights once more. ";
           makeLit(true);
         }
         else
           "You replace the bulb in the fitting but it remains unlit. ";
         moveInto(lightFitting);
         
       }
     }
   }
   dobjFor(DrawOn)
   {
     verify() {}
     preCond = [penHeld, touchObj]
     check() 
       { 
         "You try drawing on the bulb but the pen doesn't make much of a mark. 
         If you want a curved surface perhaps you'd better try something else. ";
         exit;
       }
   } 
   iobjFor(ThrowAt)
   {
     action()
     {
       "{The dobj/he} miss{es} the bulb and falls harmlessly to the floor. ";
        gDobj.moveInto(getOutermostRoom);
        exit;
     }
   }   
   iobjFor(PutUnder) maybeRemapTo(isIn(lightFitting), PutUnder, DirectObject, lightFitting)
   iobjFor(MoveTo) maybeRemapTo(isIn(lightFitting), MoveTo, DirectObject, lightFitting)
   brightnessOn = (power.on ? 3 : 0)
   warnedOfDark = nil
   wouldBeDark()
   {
     local testResult;
     local wasLit = isLit;
     makeLit(nil);
     testResult = !gActor.canSee(self);
     makeLit(wasLit);
     return testResult;
   }
   feelDesc = "It feels <<isLit ? 'very hot -- almost too hot to touch and certainly
    too hot to hold' : 'quite smooth'>>. "
   notAWeaponMsg = 'That wouldn\'t be too bright. ' 
   sightSize = small
   bulk = 2
;

+ desk :  ComplexContainer, TravelPushable, Platform 
  'battered old wooden desk/top*furniture' 'desk'
  "It's a battered old wooden desk with a single, shallow drawer<<drawer.isOpen ?
   ' (which is currently open)' : ''>>. The top is quite badly
   scratched, three of the legs look chipped, and the fourth is slightly askew. Overall it 
   looks like the sort of furniture that could have been
   thrown out half a century ago from the office of a recently-demoted junior clerk in a 
   long-defunct department. <<isInHole ? 'Currently, one of its legs is wedged in a
    pothole. ' : ''>> "
   bulk = 20
  dobjFor(Open) remapTo(Open, drawer)
  dobjFor(Close) remapTo(Close, drawer)
  dobjFor(LookIn) remapTo(LookIn, drawer)
  dobjFor(Unlock) remapTo(Unlock, drawer)
  dobjFor(UnlockWith) remapTo (UnlockWith, drawer, IndirectObject)  
  isInHole = nil
  initializeThing()
  {
    inherited;
    lightFitting.underList.append(self);
  }
  describeMovePushable(traveler, connector)
     { 
         isInHole = nil;
         if (gActor.isPlayerChar)
           "<.p>You push <<theName>> into <<desk.location.destName>>. ";  
         if (desk.location == cell)
         {
           foreach(local cur in lightFitting.underList)
           {
             if(cur.bulk > desk.subUnderside.maxSingleBulk && cur != self)
             {
               "The desk knocks <<cur.theName>> aside. ";
               lightFitting.underList.removeElement(cur);
             }
           }
           lightFitting.underList.append(self);
         }        
     }
  beforeMovePushable(traveler, connector, dest) 
  {
    inherited(traveler, connector, dest);   
    if(dest != cell && lightFitting.underList.indexOf(self))
       lightFitting.underList.removeElement(self); 
    "{You/he} push{es} the desk out of <<traveler.location.destName>>. ";
  }   
  dobjFor(GetOutOf)
  {
   action()
   {
    if(!canBeSeenBy(gActor))
    {
      "Trying to get off the desk in the dark you stumble and fall.\b
       Oh dear! You seem to have broken your neck!\b";
       endGame(ftDeath);
    }
    else 
      inherited;
   }
  } 
  dobjFor(StandOn)
  {
    action()
    {
      "The battered old desk creaks a bit as you clamber onto it, but it seems to take
       your weight. <<isInHole ? 'With one leg in the pothole, however, the desk is a 
       bit unstable and wobbles as you shift your weight. ' : ''>> ";
      inherited;
    }
  }
  dobjFor(JumpOff)
  {
    verify() { if(!gActor.isIn(self))
                   illogicalNow('{You/he} {is}n\'t on the desk. '); }
    action()
    {
      replaceAction(GetOutOf, self);
    }
  }

   dobjFor(PutUnder) {
     verify() {  delegated Thing; }
     check() {  delegated Thing; }
     action() {  delegated Thing; }
   } 
  
  cannotMoveMsg = 'Please be more specific about where you want to push the desk. ' 
  cannotTakeMsg = 'The desk is too heavy to lift. ' 
  shouldNotBreakMsg = 'You give the desk a good thump, but it hurts you more than
   it hurts the desk. '
  uselessToAttackMsg = (shouldNotBreakMsg)
  specialDesc = "An old wooden desk stands <<locName>><<isIn(cell) ? 
   ', directly under the light fitting' : ''>><<isInHole ? ', with one leg in a pothole' : ''>>. "
  locName {
    if(location == smallYard)
      return 'in the small yard, near the gate';
    else
      return 'in the middle of ' + location.destName;
  } 
  specialDescOrder = 100    
  initSpecialDesc { if(gActionIs(LookUnder)) specialDesc; }
  subUnderside : ComplexComponent, Underside {
     maxSingleBulk = 4 
     alwaysListOnMove = true
     contentsListedSeparately = true     
  }  
  subSurface : ComplexComponent, Surface { }
;

class PaperSurface : Surface
   iobjFor(PutOn)
   {
     verify()
     {
       if(isHeldBy(gActor))
         illogicalNow('{The iobj/he} can\'t support anything while {you/he} {is}
           carrying it. ');
       else
         inherited;
     }
   }
   dobjFor(Take)
   {
     action()
     {
       if(contents.length>0)
         foreach(local obj in contents)
           if(!obj.ofKind(NonPortable))
             tryImplicitAction(Take, obj);
       inherited;
     }
   }
   bulkCapacity = 1
   tooLargeForContainerMsg(obj, cont)
     {
         gMessageParams(obj, cont);
         return '{The obj/he} {is} larger than {the cont/him} and would
          totally conceal it, so {you/he} decide{s} this might be unwise. ';
     }

;


++ note : CanRub, Readable, DrawingSurface, PaperSurface, Bendable 
   'small original paper note' 'small note'
  "The note reads:\b <q>You have been imprisoned here for crimes we are not prepared
   to divulge, since by becoming a felon you have forfeited your right to know.
   We have determined that the appropriate punishment for your particular offence is that
   you be kept in confinement until you draw a square circle. When you have completed
   this simple task, show your square circle to the prison guardian who may then, at
   his discretion, allow you out of the prison compound.
   If you fail in this task you will never be allowed out -- but this cannot be 
   regarded as a great loss.\b
   You should, of course, be aware that we make no guarantee that this task can be completed 
   with the materials that happen to be at hand in your cell -- you\'re being punished and we
   feel no obligation to make things easy for you. For the same reason your food supply
   has been strictly rationed, and should you fail to complete your task before it
   runs out, you will simply starve. This will be no worse than you deserve.\b
   Suffer well, heinous felon!\b
   (Signed) The Committee for Rational Public Order</q>"     
   maxDoodles = 10
   subLocation = &subSurface
   isInInitState = (!described)
   initDesc = "<<desc>>\bThe note leaves you feeling strangely uneasy -- and not just because
    of its harsh tone and seemingly impossible challenge. What is this nameless crime for which
    you have been condemned? Even though your memories refuse to focus, you can't shake
    off a sense of guilt, but you can't connect guilt and crime. "
   notAWeaponMsg = 'Perhaps a limp lettuce leaf would be more effective. '
   disambigName = 'original note'
   sightSize = small
   toks = nil   
;

++ Component 'scratches' 'scratches'
   "The scratches on the top of the desk form no particular pattern that you can
    discern, and don't seem to have got there deliberately, they are rather a sign
    of decades of wear, tear and casual abuse. "
    sightSize = small
    bulk = 0
;

++ drawer : KeyedContainer, Component 'shallow drawer' 'drawer'
   "The shallow drawer runs under the central part of the desk top. "
   keyList = [brassKey]
   initiallyLocked = true
   cannotOpenLockedMsg = '{The dobj/he} seem{s} to be locked. <.reveal DeskTried>'
   dobjFor(Pull) asDobjFor(Open)
   dobjFor(Push) asDobjFor(Close)
   dobjFor(Move){
     verify() {}
     action()
     {
       if(isOpen)
         replaceAction(Close, self);
       else
         replaceAction(Open, self);
     }
   } 
   verifyPutInInterior()
   {
     if(gDobj is in (desk, self))
       illogical('You can\'t put {the dobj/him} in itself. ');
     inherited;
   }
  shouldNotBreakMsg = 'Although the desk looks pretty battered, its
   drawer turns out to be pretty robust, and you fail to make any
   impression on it. '
  uselessToAttackMsg = (shouldNotBreakMsg)
;

+++ pen : CanRub, Hidden 'thick marking pen/marker' 'marking pen'
  "It's a thick marking pen for drawing in black. "
  dropped = nil
  discovered = true
  discover()
  {
    unpen.moveInto(nil);
    inherited;
  }
  dobjFor(WriteOn) { 
   verify() { illogicalSelf('{You/he} can\'t write on the pen with the pen. '); }
  }
  dobjFor(DrawOn) {
   verify() { illogicalSelf('{You/he} can\'t draw on the pen with the pen. '); }
  }
  dobjFor(Take)
  {
    action()
    {
      if(dropped || tinBox.discovered)      
       inherited;
      else
      {
        unpen.moveInto(gActor.location);
        moveInto(desk.subUnderside);
        dropped = true;
        discovered = nil;
        
        "As you take the pen out of the drawer it slips out of your fumbling grasp,
         falls to the floor and rolls out of sight under the desk. ";      
      }
    }
  }
   notAWeaponMsg = 'The pen may be mightier than the sword, but it
     will hardly prove so for that purpose. '
   sightSize = small
;

+++ unpen : Unthing 'thick marking pen/marker' 'marking pen'
    notHereMsg = 'The marking pen rolled out of sight after you dropped it. '
    location = nil
;

++ Component 'chipped askew (desk) leg*legs' 'legs'
  "Three of the desk's legs are chipped, and one looks slightly askew. " 
  isPlural = true
;

+ swivelChair : Chair 
   'plain (red) (office) straight-backed swivel chair/seat/back*chairs furniture' 
   'swivel chair'
  "It's a plain straight-backed office swivel chair, upholstered in red. The upholstery
   on the seat is worn so thin in places that it looks in imminent danger of tearing,
   and there's a dark-coloured stain on the back. "
  initSpecialDesc = "A swivel chair sits <<desk.isIn(location) ? 'behind the desk'
   : 'in the middle of the cell'>>. "
  
  allowedPostures = [sitting]
  cannotStandOnMsg = 'The swivel chair looks too wobbly to stand on. ' 
  dobjFor(Turn)
  {
    verify() {}
    action()
    {
      "{You/he} swivel{s} the chair back and forth, but the novelty soon wears off. ";
    }
  } 
  dobjFor(PushTravel)
  {
    verify() { }
    check() { }
    action() { return delegated TravelPushable.actionDobjPushTravel; }
  }
  movePushable(traveler, connector)
     {
         /* move me to the traveler's new location */
         moveIntoForTravel(traveler.location);
 
         /* describe what we're doing */
         describeMovePushable(traveler, connector);
     }

  describeMovePushable(traveler, connector)
     { 
         if (gActor.isPlayerChar)
           "You push <<theName>> into <<location.destName>>. ";
     }
  beforeMovePushable(traveler, connector, dest) 
  {
    inherited(traveler, connector, dest);
    "{You/he} push{es} the chair out of <<traveler.location.destName>>. ";
  }    
  pushTravelerClass = PushTraveler
  canPushTravelVia(connector, dest) { return true; }
  explainNoPushTravelVia(connector, dest) { }

  moveNoEffectMsg = 'Please be more specific about where you want to ' + gVerbName + ' it. '
  pushNoEffectMsg = (moveNoEffectMsg)
  pullNoEffectMsg = (moveNoEffectMsg)
  dobjFor(MoveTo){
    action() {
      if(gIobj.ofKind(Underside) || gIobj.ofKind(ComplexContainer))
        replaceAction(PutUnder, self, gIobj);
      else
        "It's probably better off where it is. ";
    }
  } 
  iobjFor(AttackWith)
  {
    verify() {}       
  }
  bulk = 10
  weight = 10
;

++ Decoration '(coffee) dark dark-coloured stain/coffee-stain' 'stain'
  "It looks like an old coffee-stain. "
  bulk = 0
  sightSize = small
;

++ Component 'red worn thin threadbare fabric/upholstery' 'upholstery'
   "It's worn very thin. "
   shouldNotBreakMsg = 'Inflicting any more damage on the threadbare
    fabric would be an act of pointless vandalism. '
   bulk = 0 
   weight = 0
   dobjFor(Pull) asDobjFor(Break)
   dobjFor(SitOn) remapTo(SitOn, swivelChair)
   feelDesc = "It feels as threadbare as it looks. "
;


+ bookcase : CustomFixture, Container 
  'small (book) low stained wooden pine case/bookshelf/shelf/bookcase*shelves' 
  'small bookcase'
  "The low bookcase is made of stained pine, and has just a couple of shelves. "
  maxSingleBulk = 5
  cannotTakeMsg = 'The bookcase is fixed firmly to the wall. '
  nothingBehindMsg = '{You/he} can\'t look behind the bookcase; it\'s
    fixed to the wall. '
;

+ cellDoor : LockableWithKey, Door 'heavy solid iron (battleship) (cell) gray grey door' 'door'
  "It's painted an uncompromising battleship grey, but is otherwise featureless
   apart from a small handle and a large keyhole. "
  keyList = [ironKey, filedIronKey]
  keyIsPlausible(key)
  {
    return key is in (ironKey, filedIronKey);
  }
  lockStatusObvious = nil
  uselessToAttackMsg = 'Ouch! That hurt (you, not the door)! '
  shouldNotBreakMsg = 'You can\'t break it -- it\'s solid iron. '
  thruDesc = "Through the open door you see part of the vestibule. "  
  darkIfClosed
  {
     local stat = isOpen, dark;
     makeOpen(nil);
     dark = !gActor.isLocationLit();
     makeOpen(stat);
     return dark;
  }
  dobjFor(Close)
  {
    check() 
    {
      inherited;
      if(darkIfClosed)
      {
        reportFailure('It occurs to you that closing the door will leave you
          completely in the dark, so you wisely decide against it. ');
        exit;
      }
    }
  }
;

class Handle: Component 'small plain round (door) handle/knob/doorknob' 'small handle'
  "The handle is a plain round knob towards the right-hand edge of the door. "
  
  dobjFor(Turn)
  {
    verify() {}
    action()
    {
      "You turn the handle";
      if(location.isOpen)
        " but since the door is already open, it is a rather pointless operation. ";
      else if(location.isLocked) {
        if(location.lockStatusObvious) 
          ", but since the door is locked this doesn't achieve much. ";
        else {
          " but find that the door is locked. ";
           location.lockStatusObvious = true;
          }  
        }
      else
      {
        " and open the door. ";
        location.makeOpen(true);
      }
      
    }
  }
  dobjFor(TurnWith)
  {
    verify() { }
    action() {
      replaceAction(Turn, self);
    }
  }
  cannotPutUnderMsg = 'Putting {the dobj/him} under {the iobj/him} would serve no useful purpose. '
;

++ Handle
  dobjFor(Pull) remapTo(Open, location)
  dobjFor(Push) remapTo(Close, location)
;

class Keyhole : Component, RestrictedContainer 'large keyhole/hole/lock' 'keyhole'
  "The keyhole is just under the handle. "
  validContents = [ironKey, filedIronKey]
  dobjFor(LookThrough)
  {
    action()
    {
      if(contents.length > 0)
        "{You/he} can see nothing through the keyhole while a key
          is in it. ";
      else if(cellDoor.isOpen)
        "You see nothing through the keyhole but a patch of grey wall. ";
      else if(lookThroughTo.wouldBeLitFor(gActor))
         lookThroughDesc;
      else
        "It's too dark on the other side of the door to see anything
         through the keyhole. ";
    }
  }  
  lookThroughTo = vestibule
  lookThroughDesc = ""
  dobjFor(Pick)
  {
    verify() { }
    action()
    {
      "Even assuming you had the right tools, you don't have the haziest notion
       how to pick a lock. ";
    }
  }
;

++ keyhole : Keyhole
   lookThroughDesc = "You can just make out a vestibule on the other side of the
          door. <<desk.isIn(vestibule) ? 'Apart from the desk the' : 'The'>>
          only features in view are an exit to the north and
          a notice next to it reading,\b<q>STRICTLY NO EXIT\nwithout a 
          square circle</q><.reveal vestibule-thru-keyhole>"
   lookThroughTo = vestibule

;

+ Hidden, Distant 'green-painted vestibule/(walls)' 'vestibule'
  desc()
  {
    if(cellDoor.isOpen && vestibule.wouldBeLitFor(gActor))
      "You can't see much of the vestibule from here, apart from a general
      impression of green-painted walls. ";
    else if(cellDoor.isOpen)
      "The vestibule is too dark to make out from here. ";
    else
      "You can't see much of the vestibule through the solid iron door! ";
  }

  dobjFor(Enter) remapTo(TravelVia, cellDoor)
  discovered = (vestibule.seen || gRevealed('vestibule-thru-keyhole'))
;


vestibuleNorthWall : RoomPart 'dark vile green north n wall*walls' 'north wall'
  "Apart from its being painted in a vile shade of dark green, there is little of note
   about the north wall except the passage and the sign next to it. "
;

vestibuleSouthWall : RoomPart 'dark vile green south s wall*walls' 'south wall'
  "There's nothing remarkable about it apart from its vile shade of dark green
   and the cell door. "
;

vestibuleEastWall : RoomPart 'dark vile green east e wall*walls' 'east wall'
  "It has been painted a vile shade of dark green. Roughly in the centre of
   the wall is a narrow doorway over which has been fixed a large circle. "
;

vestibuleWestWall : RoomPart 'dark vile green west w wall*walls' 'west wall'
  "It's dark green, with an archway roughly in the centre and a large square
   fixed over the archway. "
;

vestibule : Room 'Vestibule' 'the vestibule'
  "The vestibule is a large bare room, the walls of which are painted in one of those 
  vile shades of dark green normally associated with the more repressive kind of
  educational establishment. A large square has been fixed over an archway to the west,
  while an equally large circle adorns the 
  narrower doorway to the east. Next to the northern exit is a large sign. "
  south = cellDoorOutside
  west = squareRoom
  east : OneWayRoomConnector
  {
    -> circleRoom
    travelBarrier : PushTravelBarrier
    {
      canPushedObjectPass(obj) { return (obj != desk); }
      explainTravelBarrier(traveler)
     {
        "The doorway is too narrow for <<traveler.obj_.theName>> to fit through. ";
     }
    }
    noteTraversal(traveler)
    {
      gReveal('CircleRoom');
    }
    
  }
  north = wayOut
  out asExit(north)
  in asExit(south)
  roomParts = [defaultFloor, defaultCeiling, vestibuleNorthWall, vestibuleSouthWall,
    vestibuleEastWall, vestibuleWestWall]
  brightness = (power.on ? 3 : wayOut.brightness - 1)
  private = true
  
;

+ cellDoorOutside : LockableWithKey, Door ->cellDoor 'green (cell) door' 'door'
  "The outside of the cell door is painted in the same revolting shade of green as
   the rest of the vestibule. It has a small handle. "
   keyList = [ironKey, filedIronKey]
   keyIsPlausible(key) { return keyList.indexOf(key) != nil; }
   dobjFor(Push) asDobjFor(Open)
   dobjFor(Pull) asDobjFor(Close)
   thruDesc = "{You/he} see{s} part of the cell. "
;

++ Handle
  desc = "The handle is a plain round knob towards the left-hand edge of the door. "
  dobjFor(Pull) remapTo(Close, location)
  dobjFor(Push) remapTo(Open, location)
;

++ Keyhole
  lookThroughDesc = "You can just make out a cell on the other side of the door,
   but you can't see much more than a patch of grey wall <<desk.isIn(cell) ? 
   'and the corner of a desk' : ''>>."
   lookThroughTo = cell
;

+ Enterable ->cellDoorOutside 'cell' 'cell'
  "<<cellDoorOutside.isOpen ? 'What you can see of it through the door looks pretty
   drab' : 'It\'s on the other side of the door, which is closed, so you can\'t
   see it'>>. "
;

+ Enterable ->wayOut 'northern short exit/corridor/passage' 'exit'
  "The exit is a short corridor leading north. "
  dobjFor(Exit) asDobjFor(Enter)
;

+ Fixture, Readable 'large sign' 'large sign'
  "The sign reads:\b STRICTLY NO EXIT\nwithout a square circle"
;

+ Decoration 'cheap large white plywood fixed square' 'large white square'
  "It appears to have been cut out of a cheap sheet of plywood and painted white. It's
   stuck to the wall above the archway to the west. "
   notImportantMsg = 'You\'ve got far better things to do than fiddle with pieces of
    cheap plywood. '
;

+ Decoration 'large black plastic fixed circle' 'large black circle'
  "The circle looks like it's made of black plastic. It's been stuck over the doorway
  to the east. "
   notImportantMsg = 'Maybe you should try something more productive. That circle
    is only a cheap plastic decoration, after all.'
;

+ EntryPortal ->squareRoom 'broad west arch/archway' 'archway'
  "The broad archway leads through to another room to the west. "
;

+ EntryPortal ->(vestibule.east) 'narrow east doorway' 'doorway'
  "The narrow doorway leads east. "
;

+ guardianPaperNoise : SimpleNoise 'occasional sound/rustling/paper' 'sound of rustling paper'
  "You hear the occasional sound of rustling paper up the passage to the north. "
;

squareFloor : defaultFloor 'floor' 'floor'
  "The floor is covered with a chequer pattern of black and yellow squares. "
;

+ Component 'chequer pattern black (yellow) square*squares*patterns' 'black and yellow squares'
  "The black and yellow squares forming the chequer pattern are about ten
   inches to a side. "
  isPlural = true
;

squareNorthWall : defaultNorthWall
  desc = "The north wall has been painted in loud green and orange squares. "
;

+ Component 'loud green (and) orange square*squares patterns' 'green and orange squares'
  "The loud green and orange squares on the north wall vary in size from about
   four to eighteen inches a side. "
   isPlural = true
;

squareSouthWall : defaultSouthWall
  desc = "The south wall is covered with a pattern of large purple squares
   and small pink ones. "
;

+ Component 'offset large purple pattern/square*squares patterns' 'large purple squares'
  "The large purple squares on the north wall are about fourteen inches to a
   a side. They're arranged in an offset pattern that allows them to tessellate
   with the small pink ones. "
   isPlural = true
;

+ Component 'small pink square*squares patterns' 'small pink squares'
  "The small pink squares, about four inches to a side, fill in the spaces
   between the large purple squares on the south wall. "
   isPlural = true
;


squareWestWall : defaultWestWall '(w) west wall/spirals*walls' 'west wall'
  desc = "The west wall makes you feel dizzy: it's painted in stark black and white
   but you have a hard time deciding whether the pattern is a sequence of concentric
   squares or interlocking black and white square spirals. "
;

squareEastWall : defaultEastWall
  desc = "The east wall is covered with a series of large squares, each painted a different
  loud colour and each the same size as the large square doorway. "
;

squareCeiling : defaultCeiling
  desc = "The ceiling is mercifully a plain white, but is patterned with a number
   of projecting cubes. "
;
  
+ Component 'white decorative projecting -*cubes' 'projecting white cubes'
  "The cubes projecting from the ceiling give it a curiously irregular look,
   since they are a variety of sizes. They have no obvious function, and seem
   to be purely decorative. "
   isPlural = true   
;

squareRoom : DarkRoom 'Square Room' 'the square room'
  "Everything about this room is square: the shape of the room itself, the decorative
   patterns on the walls (painted in singularly tasteless colours), and the chequer 
   pattern on the floor. The only way out is to the east. "
  east = vestibule
  out asExit(east)
  roomParts = [squareFloor, squareNorthWall, squareSouthWall, squareEastWall,
    squareWestWall, squareCeiling]   
  private = true    
;

+ rubberSheet : CanRub, DrawingSurface, Bendable, Platform 
  'square rubber sheet/mat' 'square rubber sheet'
   "It's about eighteen inches square and a sixteenth of an inch thick. "
  initSpecialDesc = "A square sheet of rubber lies in one corner. "
  dobjFor(Pull) asDobjFor(Stretch)
  dobjFor(Stretch)
  {
    verify() { logicalRank(140, 'elastic'); }
    action()
    {
      "You stretch the rubber sheet";
      distortExtras();
    }
  }
  dobjFor(Bend)
  {
    verify() { logicalRank(140, 'elastic'); }
    action()
    {
      local verb = gAction.getOrigTokenList[1][1].toLower;
      "You <<verb>> the rubber sheet";
      distortExtras();
    } 
  }
  distortExtras()
  {    
    if(doodles != nil)
    {
      ", distorting everything you've drawn on the sheet, but ";
      local circle = doodles.indexOf(t_circle);
      local square = doodles.indexOf(t_square);
      if(square || circle)
      {
        if(circle)    
          "the circle doesn't become square";
        if(circle && square)
          " and ";      
        if(square)
          "the square doesn't become round";
      }
      else
        "nothing on it resembles a square circle";        
    }
  ".";
  }
  checkActorOutOfNested(allowImplicit)
     {
       if(gActionIs(TravelVia) && gDobj==blueDoor && !blueDoor.isOpen)
       {
         nestedAction(Open, blueDoor);           
       }
       if(!whiteCoat.isIn(courtyard))         
         inherited(allowImplicit);
         
     }
  out = noTravelOut
  notAWeaponMsg = 'It would be stretching it too far to call it a weapon. '
  dobjFor(StandOn)
  {
     preCond = [touchObj]
     verify() {
       if(gActor.isIn(self) && gActor.posture == standing)
         illogicalAlready('{You/he} {is} already standing on {the dobj/him}. ');
//       if(self.isIn(gActor))
//         illogicalNow('You\'re holding it. ');
    }
    
    action()
    {
      if(isIn(gActor))
        tryImplicitAction(Drop, self);
      if(isIn(gActor))
      {
        reportFailure('You can\'t stand on it while you\'re holding it. ');
        exit;
      }
      inherited;
    }
  }
  bulk = 3
;

+ cubeFitting : OutOfReach, Fixture, Surface 'short thin brass length/chain' 'chain'
  "This short length of thin brass chain hangs down from the ceiling, to which
   it is attached. "
   canObjReachContents(obj) 
   { 
     gReveal('CubeOutOfReach');     
     return ((obj.isIn(desk) || obj.isIn(sandBox) || obj.isIn(guardChair)) && 
        obj.posture == standing) || gActionIs(PutUnder);     
    
   }
   validContents = [lightCube]
   iobjFor(PutIn) remapTo(PutOn, DirectObject, self)
   iobjFor(PutOn)
   {
     check() {
       if(gDobj <> lightCube)
       {
         reportFailure('{You/he} can\'t put {the dobj/him} on the chain. ');
         exit;
       }
     }
   }
   iobjFor(PutUnder)
   {
     verify() {  }
     action() {
       if(gDobj == desk)
         "The desk is already under the chain that holds the light cube. ";
       else if(desk.isIn(squareRoom))
       {
         "Since the desk is already under it you put {the dobj/him} on the desk. ";
         gDobj.moveInto(desk.subSurface);
         exit;
       }
       else if(gDobj.location <> squareRoom)
       {
         "You put {the dobj/him} on the floor, roughly under the place where the
          the light cube hangs. ";
         gDobj.moveInto(squareRoom);
       }
       else
         "You push {the dobj/him} roughly under the chain for the light cube. ";
       
     }
   }
;

++ lightCube : Fragile, LightSource 'transparent light cube*cubes' 'light cube'
  "It's about nine inches to a side, and glows with a steady bright white light. "
  initSpecialDesc = "The light for this room comes from a transparent cube suspended
   from the ceiling. " 
   specialDescOrder = 10 
   dobjFor(Take)
   {
     action()
     {
       inherited;
       achievement.awardPointsOnce();
     }
   } 
   iobjFor(PutUnder) maybeRemapTo(isIn(cubeFitting), PutUnder, DirectObject, cubeFitting)
   achievement : Achievement { +1 "getting the light cube " }
   bulk = 3
   weight = 2
   feelDesc = "It feels smooth and cool to the touch. "
   notAWeaponMsg = '{You/he} do{es}n\'t want to risk damaging {the iobj/him}. '
   inflammable = nil   
;

+ whiteBox : Fillable, ComplexContainer 'large portable white square cardboard lid/box*boxes' 
  'large square white cardboard box'
  "It's a large white square cardboard box. "
  initSpecialDesc = "A large white square box sits in the middle of the floor. "
  specialDescOrder = 2000
  subSurface : ComplexComponent, Surface { 
    iobjFor(PutOn)
    {
      action() 
      {
        if(lexicalParent.subContainer.isOpen)
          tryImplicitAction(Close, lexicalParent.subContainer);
        inherited;    
      }      
    }
  }
  subContainer : ComplexComponent, OpenableContainer
    { 
      iobjFor(PutIn)
      {
        action()
        {
          if(gDobj != sand) inherited;
        }
      }
      bulkCapacity = (lexicalParent.bulkCapacity)
      dobjFor(Open)
      {
        action() {
          local surfcont = lexicalParent.subSurface.contents;
          if(surfcont.length > 0)
          {
            "(first ensuring there's nothing on the top of the box) ";
             foreach(local cur in surfcont)
               tryImplicitAction(Take, cur);
          }
          inherited;
        }
      }
    }
  dobjFor(StandOn)
  {
    verify() { illogical('The white box looks far too flimsy to bear your weight. '); }
  }
  dobjFor(Take)
  {
    action()
    {
      if(foodNote.isIn(subSurface) && !foodNote.moved)
      {
        "As you pick the white box up the piece of paper that was resting on it falls off and
         flutters onto the floor. ";
         foodNote.moveInto(location);
      }
      inherited;
    }
  }
  dobjFor(FillWith)
  {
    verify() {}
    preCond = [touchObj, objOpen]
    check() 
    {
      if(subContainer.contents.length > 0)
      {
        "It would be a good idea to take everything out of the box before trying to
          fill it with anything else. ";
        exit;  
      }
    }
    action()
    {
      inherited;
      if(sandBox.isIn(gActor))
      {
        "As {you/he} do{es} so, it becomes too heavy to hold, so {it actor/he}
         put{s} it down<<sandBox.getOutermostRoom() == smallYard ? ' next to the gate' : ''>>. ";
         sandBox.moveInto(gActor.location);
      }
    }
  }
  isOpen = (subContainer.isOpen)
  replacement = sandBox
  bulk = 10
  bulkCapacity = 9  
  isObviousContainer = true    
;

foodNote: CanRub, Readable, DrawingSurface, PaperSurface, Bendable 
 '(square) piece/paper/writing' 'square piece of paper'
  "There is some writing on the piece of paper that reads:\n
  <q>This box contains your entire food ration for the period of your imprisonment, 
   so you may wish to ration yourself carefully. 
   Do not complain if you become hungry: remember
   you are being punished and you are <i>meant</i> to suffer.</q> "   
   location = whiteBox.subSurface 
   sightSize = small
;

  /* The Fillable class works in conjunction with the SandFilled class defined
   * further on in this file. The idea is that an object that can be filled with
   * sand is replaced with another object representing its sandfilled equivalent
   * when filling with sand takes place. With the wisdom of hindsight it may have
   * been better to implement this using different states of the same object.
   */


class Fillable : object
  dobjFor(FillWith)
  {    
    verify() {}
    preCond = [touchObj, objOpen]
    check()
    {
      if(contents.length > 0)
      {
        "You'll have to empty <<theName>> before you can fill it with something else. ";
        exit;
      }
    }
    action()
    {

      "You fill <<theName>> with sand. ";
      if(replacement != nil)
      {
        local loc = location;
        moveInto(nil);
        replacement.makeOpen(true);
        replacement.moveInto(loc);        
        replacement.mySand.bulk = bulkCapacity;
        
      }
      
    }
  }
  replacement = nil
  getFacets 
  {
    if(replacement != nil)
      return [replacement];
    else
      return [];
  }
  iobjFor(PutIn)
      {
        action()
        {
          if(gDobj != sand) inherited;
        }
      }
  dobjFor(Pick) asDobjFor(Take)    
  notAWeaponMsg = 'If that\'s your idea of boxing you\'d better think again. '
  cannotWriteOnMsg = &uselessToScribbleOnMsg
  cannotDrawOnMsg = &uselessToScribbleOnMsg  
;


yellowBox : Fillable, OpenableContainer 'small yellow square cardboard box*boxes' 
  'small square yellow cardboard box'
  "It's a small square cardboard box, yellow in colour. " 
  location = whiteBox.subContainer
  replacement = yellowSandBox
  bulk = 6
  bulkCapacity = 5
  
;

+ greenBox : CanRub, Fillable, OpenableContainer 
  'tiny green square cardboard box*boxes' 'tiny square green cardboard box'
  "It's about six inches square. "
  dobjFor(Open)
  {
    action()
    {
      inherited;
      sandwich.seen = true;      
    }
  }
  replacement = greenSandBox
  bulk = 2
  bulkCapacity = 1
  sightSize = small
;

++ sandwich : CanRub, Food 'small (square) stale tasty ham slice/ham/sandwich/meal/food' 'ham sandwich'
  "The sandwich looks just a little stale, but otherwise edible. "
  initSpecialDesc = "Inside <<location.theName>> is a solitary small square ham sandwich; presumably
   it's someone's idea of a square meal.<<makeFound()>> "
  makeFound() { found = true; } 
  found = nil
  isInInitState = (!found)
  bulk = 1 
  dobjFor(Eat)
  {
    action()
    {
      "You eat the sandwich. It's not the freshest you've ever eaten but it still tastes
        good enough to you -- too bad there aren't any more. ";
       inherited;
    }
  }
  smellDesc = "There's a faint salty smell. "
  tasteDesc = "The ham's a bit salty and the bread's a tad stale, but otherwise
    it tastes just great. "
  lookInDesc = "There\'s not much in the sandwich but a slice of ham. "  
  sightSize = small
  cannotOpenMsg = 'Opening the ham sandwich reveals, as you might expect, a 
   slice of ham. Having made this momentous discovery you put the sandwich
   back together again. '
;

circleFloor : Floor 'concentric floor/rings/circles' 'floor'
  "The only noticeable oddity about the floor is that it's patterned with a
   series of concentric circles. "
;

circleRoom : DarkRoom 'Round Room' 'the round room'
  "This room is circular in shape with a pattern of concentric circles on the floor.
   The wall is partially decorated with abstract circular patterns, and partly
   with a schematic portrayal of the planets of a solar system going round a circular
   sun in circular orbits. Above is a plain, white domed ceiling.
   The only way out is to the west. "
   west = vestibule
   out asExit(west)
   roomParts = [circleFloor, circularWall, domedCeiling]
   private = true
;

+ ExitPortal ->vestibule 'way out/exit/door/doorway' 'exit'
  "The narrow doorway leads out to the west. "
  brightness = (vestibule.brightness == 3 ? 1 : 0)
;

+ Unthing 'sun' 'sun'
  "The sun is not actually represented anywhere in the room; if it were, it would
   be near the centre of the room, at the centre of the planetary orbit described
   on the wall. "
;

  /*
   * The globe and its associated objects are fairly complex, since together they
   * need to handle the various attempts (successful and near misses) to solve the
   * square circle puzzle.
   */

+ globe : Fragile, DrawingSurface 'large old antique globe' 'globe'
  "It's a large globe, maybe two feet in diameter, representing the planet. It must be
   quite an antique, though, since whoever made it can only have had the vaguest
   understanding of geography south of the equator. <<equator.linesDrawn ?
   equator.lineDesc() : nil>> <<meridian.linesDrawn ? meridian.lineDesc() : nil>>"
   initSpecialDesc = "An old globe rests against the wall. "
   squareCircleMsg = 'How do you propose to do that? Please be more specific. '
   roundSquareMsg = (squareCircleMsg)
   doodle(obj)
   {
     switch(obj){  
     case t_line:
       "There are all sorts of places a line could be drawn on the globe;
        please specify which you have in mind. "; break;
     case t_square:
       "Drawing a square on the surface of a sphere requires a bit
        of thought; please specify precisely how you intend to do it. "; break;   
     case t_circle:
       "Drawing a circle at some random point on the globe might not achieve
        much; you might want to be more specific. "; break;
     case t_circumference:
       "Well, there are several ways you could do that; perhaps you should
        specify which you have in mind. "; break;   
     case t_lines:
       "Try drawing things one at a time. "; break;    
     default:
       "Random doodling on the surface of the globe is, of course, perfectly
        possible, but it occurs to you that it is unlikely to prove fruitful.
        You accordingly pause to consider whether a more focused action would
        be more profitable. "; break;
     }  
   }
   bulk = 9
   weight = 4
   dobjFor(Turn) {
     verify() { }
     action() { "{You/he} spin{s} the globe round a few times. "; }
   }
   dobjFor(DrawOn) {
     action() {
       if(gTopic.getBestMatch is in (equator, t_circumference, t_greatCircle))
         replaceAction(DrawOn, equator, t_line);
       else if(gTopic.getBestMatch == meridian 
         || gTopic.getTopicText.toLower.find('meridian') != nil)
         replaceAction(DrawOn, meridian, t_line);  
       else if(gAction.getOrigTokenList.length > 5)
         "That seems a bit too complicated; something in the form
          DRAW X ON Y should be sufficient. ";          
       else
         inherited;    
     }
   }   
  iobjFor(PutOn)
  {
    verify() { }
    action()
    {
      "There's no flat surface on the globe, so as soon as you put {the dobj/him} on
       it, {it dobj/he} slip{s} off and falls onto the floor. ";
       gDobj.moveInto(getOutermostRoom);
    }
  }
  squareCircleDrawnOn = nil
  notAWeaponMsg = 'The situation does not call for global conflict. '

;

++ Component '(north) (south) pole/poles' 'poles'
  "The globe has two poles: a north pole at the top and a south pole
   at the bottom. "
   isPlural = true
   cannotDrawOnMsg = 'A pole is only a point, too small to draw anything on. '
   cannotWriteOnMsg = 'How small can you write? Not small enough to write anything
    on a point, at any rate, and a pole is only a point. '
;

class GreatCircle : DrawingSurface, Component
 linesDrawn = 0
  lineDesc()
  {
    "A heavy black black line runs ";
    switch(linesDrawn)
    {
      case 1: "one quarter of the "; break;
      case 2: "half "; break;
      case 3: "three quarters of the "; break;
      case 4: "all the "; break;
    } 
    "way round <<theName>>. <<linesDrawn > 1 ? 'A closer look reveals the line to
      be made up of ' + spellInt(linesDrawn) + ' individual lines.' : nil>> ";
  }
  doodle(obj)
  {    
     local drawingLine = true;
     if(obj is in (t_circle, t_square, t_circumference, t_greatCircle, equator))
     {
       drawingLine = nil;
       if(linesDrawn == 0)
         "You ponder for a moment how you might go about this, then you make a start ";
       else
         "You continue drawing the <<gTopic.getTopicText>> ";
     }
     
     if(obj is in (t_line, t_circle, t_greatCircle, t_square, t_circumference, equator))
       {
         if(linesDrawn < 4)
           "<<drawingLine ? 'You draw' : 'by drawing'>> a ";
         switch(linesDrawn++)
         {
           case 0: "line a quarter of the way round <<theName>>. "; break;
           case 1: "second line running a further quarter of
                    the way round <<theName>>. "; break;
           case 2: "a third line running a further quarter of
                    the way round <<theName>>. "; break;
           case 3: "fourth line, completing the circle round <<theName>>. ";                    
                    if(globe.squareCircleDrawnOn == nil){
                      squareCircleAchievement.circle = theName;
                      squareCircleAchievement.awardPointsOnce();
                      globe.squareCircleDrawnOn = self;
                      globe.initializeVocabWith('square circle');
                     }
                    break;
           default: "You have already drawn a complete circle round <<theName>>. ";
                    linesDrawn--; break;
         }
       }
       else if(obj == t_lines)
         "Try drawing them one at a time. ";
       else if(gAction.getOrigTokenList.length > 6)
         "Try something a bit simpler; something in the form
          DRAW X ON Y should be sufficient. ";
       else
         "You\'re not sure how to go about that. ";
      
  }
  squareCircleMsg = 'Nice idea -- but how?'
  roundSquareMsg = (squareCircleMsg)
;

++ equator : GreatCircle 'great equator/(circle)' 'equator'
  "The equator encompasses the circumference of the globe. <<linesDrawn ? lineDesc : nil>> "  
  sightSize = small
;

++ meridian :  GreatCircle, Component, Topic 'meridian' 'meridian'
  desc() {
     "There are several meridians on the globe, stretching from pole to pole. ";
     if(linesDrawn) {
       "You have drawn on one of them. ";
       lineDesc();
     }  
  }

  theName = (aName)
  dobjFor(DrawOn)
  {
    action()
    {
      if(gTopic.getBestMatch is in (t_line, t_circle, t_greatCircle, t_square))
      {
        if(linesDrawn == 0)
        "One meridian seems as good as any other for this purpose, so you choose
         one at random and start to draw. ";
        else if(linesDrawn > 3)
        {
          "You have already drawn a complete circle round one meridian; there's
           no need to start another. ";
           exit;
        }    
        else
          "(Continuing on the same meridian as before)\n";
      }
      inherited;
    }
  }
  sightSize = small
;

++ SecretFixture  
  noun = '*'
  matchNameCommon(origTokens, adjustedTokens)
  {
    if((listContains(adjustedTokens, 'meridian') ||
      listContains(adjustedTokens, 'meridians') || 
      listContains(adjustedTokens, 'meridi'))
      && !listContains(adjustedTokens, 'on'))
        return meridian;
    else
      return nil;
  }
;

++ squareCircleAchievement : Achievement +2 
   "drawing four lines round <<circle>> of the globe. "
   circle = 'the equator'
;
  
++ Component 'northern (n) (north) hemisphere*hemispheres' 'northern hemisphere'
  "The northern hemisphere is the portion of the globe lying about the equator;
   it seems to be reasonably accuratly depicted, so far as you can tell. "
   dobjFor(DrawOn) 
   {
     verify() {}
     action() { globe.doodle(gTopic.getBestMatch); }
   }
;

++ Component 'southern (s) (south) hemisphere*hemispheres' 'southern hemisphere'
  "The southern hemisphere is the portion of the globe lying below the
   equator; it seems to be so poorly depicted that whoever produced it can
   have had very little understanding of sub-equatorial geography. "
   dobjFor(DrawOn) 
   {
     verify() {}
     action() { globe.doodle(gTopic.getBestMatch); }
   }
;


+ roundTable : ComplexContainer,  Fragile, Heavy, DrawingSurface 
  'large round glass glass-topped table table/top' 'glass table'
  "The glass-topped table has a single stainless steel leg supporting it at the centre.
   <<silverDisc.isIn(self) ? 'The leg seems to be secured to the table by being screwed through
   the top, but a small silver disc covers the thread. ' : 'The naked thread of the screw
    fixing the leg to the top protrudes a short way through the centre. ' >>"
  initSpecialDesc = "A large, round, glass-topped table stands in the middle of the room. "
  subSurface: ComplexComponent, Surface { }
  subUnderside: ComplexComponent, Underside 
    { 
       maxSingleBulk = 7 
       material = glass
       iobjFor(PutUnder)
       {
         action()
         {
           inherited;
           if(lexicalParent.doodles != nil && gDobj == lightCube)
           {
             "Placing the light cube under the glass table top causes everything you
              have drawn on the latter (namely ";
              lexicalParent.listDoodles();              
              ") to be projected onto the domed ceiling. But as you stare up at the ceiling, 
               although the projection onto the
               hemispherical surface subtly alters the shape of what you've drawn,               
              nothing strikes you as being simultaneously square and circular. ";
           }
         }
       }
    }
  dobjFor(DrawOn)
  {
    action()
    {
      inherited;
      if(lightCube.location == subUnderside)
      {
        local obj = doodles[doodles.length];
        "As you draw <<obj.theName>> the light of the cube shining through the
         glass table projects it onto the hemispherical ceiling above, <<obj == t_square
         ? 'making it look almost round' : 'slightly distorting and blurring it'>>
          in the process. ";
      }       
      
    }
    
  }  
  dobjFor(WriteOn)
  {
    action()
    {
      inherited;
      if(lightCube.location == subUnderside)
      {
        "As you write, a weirdly distorted version of your letters is 
         projected onto the ceiling above. ";
      }
    }
  }
  
  cannotStandOnMsg = 'You\'re not sure whether the glass top will take your weight. '
  cannotSitOnMsg = (cannotStandOnMsg)
  cannotPushTravelMsg = 'It\'s too heavy to push anywhere. '
  maxDoodles = 20
  material = glass
;

++ silverDisc : Component '(small) silver metallic disc/disk' 'small silver disk'
   "It's silver in colour, plainly metallic, and about three quarters of an inch
    in diameter. "
  dobjFor(Turn) maybeRemapTo(isIn(roundTable), Unscrew, self)
  dobjFor(Unscrew)
  {
    verify() { if(!isIn(roundTable)) illogicalNow('It isn\'t screwed to anything. '); }
    action()
    {
      "You unscrew the small disc from the top of the table, taking it as you do so. ";
      setSuperclassList([Thing]);
      moveInto(gActor);
      roundTable.initializeVocabWith('thread');
    }
   
  }  
  dobjFor(TurnWith)
  {
    verify() { if(!isIn(roundTable)) inherited; }
    action() { replaceAction(Unscrew, self); }
  }
  dobjFor(UnscrewWith)
  {
    verify() { if(!isIn(roundTable)) inherited; }
    action() { replaceAction(Unscrew, self); }
  }
  dobjFor(PutIn)
  {
    action()
    {
      if(gIobj == lightFitting)
      {
        "You push the small silver disc into the light fitting";
        if(power.on)
        {
           power.on = nil;
           ". There is a flash and a bang as the disc shorts out the electric power,
            and all the lights in the building go out. You get a small shock but
            somehow escape electrocution, probably because your arm jerks your hand
            away in reflex";
            guardian.addToAgenda(powerCutAgenda);
        }
        ", then the disc falls back into your hand. ";
      }
      else 
        inherited;
    }
  }
  inflammable = nil
;

++ PermanentAttachment, Component 'stainless steel leg' 'leg'
   "The single leg supports the centre of the table, and is screwed through the glass
   table top. The bottom of the leg is fixed to a sturdy black base. "  
   baseCannotDetachMsg = 'The leg and base appear to be welded firmly together. '
;

+++ PermanentAttachmentChild, Component 'sturdy black base' 'sturdy black base'
   "The base is about eighteen inches in diameter (it is, of course,
     circular) and rests firmly on the ground. "
;

circularWall : Fixture 
  '(circular) abstract (spherical) solar planet/orbit/wall/patterns/system*planets*walls*orbits' 
   'wall'
  "The wall of this chamber forms a continuous circle, broken only by the doorway to
   the west. Various abstract circular patterns are drawn on the wall, but a little
   higher up is a view of spherical planets orbiting a sun notionally situated in
   the centre of the room. "
   cannotWriteOnMsg = &uselessToScribbleOnMsg
   cannotDrawOnMsg = &uselessToScribbleOnMsg
;

domedCeiling : defaultCeiling 'great domed white hemispherical ceiling/dome' 'ceiling'
  "The ceiling forms a great white dome<<projections>>."
  projections()
  {
    if(lightCube.isIn(roundTable.subUnderside) && roundTable.doodles != nil)
    { 
      ", on which is projected ";
      roundTable.listDoodles;
      
      if(roundTable.doodles.indexOf(t_square))
      ". The square projected onto the hemispherical dome looks almost round, as
        its four sides becomes arcs in the projection<.reveal projected-square>";
      if(roundTable.writing > '' )
      "<<roundTable.doodles.length > 1 ? 'Amongst these figures' :
       'Beside this figure' >> you can make out ";
    }  
    if(lightCube.isIn(roundTable.subUnderside) && roundTable.writing > '')
    {
      if(roundTable.doodles == nil)
        ", on which are projected ";
      "distorted versions of the words <q><<roundTable.writing>></q>";      
    }
  }
;  

wayOut : Room 'Corridor' 'the corridor'
  "This short, white-painted corridor leads from the vestibule to the south to an open exit 
   into a courtyard to the north. It's a pretty sterile, featureless space apart from a small
   panel on the wall next to the door. "
   south = vestibule
   north : OneWayRoomConnector {
     -> courtyard
     travelBarrier : PushTravelBarrier {
       canPushedObjectPass(obj) { return !guardian.isIn(wayOut); }
       explainTravelBarrier(traveler)
       {
         "The guardian blocks your path. <q>No square circle, no exit,</q> he
          insists. ";  
       }
     }
   }
   out asExit(north)
   brightness = (power.on ? 3 : 2)
;

+ Enterable ->vestibule 'vestibule' 'vestibule'
  "The vestibule is at the southern end of the corridor; you can't see much of
  it from here. "
;

+ guardChair : Chair, CustomFixture 
  'large gleaming thick black mock-leather stainless (steel) leather chair/upholstery/frame*chairs'
   'large chair'
  "The chair is of modern design, comprising a gleaming stainless steel
   frame with thick black mock-leather upholstery. <<ofKind(CustomFixture) ? 'A closer
     look at the frame reveals that the chair is bolted firmly to the floor. ' : ''>>"
   initSpecialDesc = "A large chair has been placed next to the exit. "
   specialDesc  {
     if(ofKind(CustomFixture))
       initSpecialDesc;
     else
       "There's a large black chair here. ";
   }   
   cannotTakeMsg {
      if(guardian.isIn(self))
        return 'You can\'t: quite apart from anything else, the guardian\'s sitting in it.';
      else   
        return 'The chair is firmly bolted to the floor. ';
   }
   cannotUnscrewMsg = '{You/he} can\'t quite manage that with your bare hands. '
   cannotMoveMsg { return cannotTakeMsg; }
   dobjFor(PushTravel)
   {
     verify() {
       if(ofKind(CustomFixture))
        illogicalNow(cannotTakeMsg);
     }
     action() {
       "Lacking castors, the guard chair is a bit awkard to push around. If {you/he}
        want{s} to move it, it'd probably be easier to pick it up and carry it. ";
     }
   }
   dobjFor(UnscrewWith)
   {
     verify() 
      { if(!ofKind(CustomFixture)) 
         illogicalAlready(alreadyUnscrewedMsg); }
     check()
      {
        if(screwList.indexOf(gIobj)==nil)
        {
          reportFailure('{You/he} can\'t unscrew it with {that iobj/him}. ');
          exit;
        }
      }
     action()
     {
        "With some effort, {you/he} eventually manage{s} to unscrew the chair
          from the floor. ";
        setSuperclassList([Chair]);  
     } 
   } 
   dobjFor(Unscrew)
   {  
     verify() {
       if(!ofKind(CustomFixture)) 
         illogicalAlready(alreadyUnscrewedMsg);
       else
         inherited;
     }     
   }
   screwBack {
     if(!guardChair.isIn(wayOut))
       {
         "There's nothing here to screw the chair to. ";         
       }
       else
       {
         "{You/he} screw{s} the guard's chair back into place. ";
          setSuperclassList([Chair, CustomFixture]);
          if(location != wayOut) moveInto(wayOut);
       }
   }   
   screwList = [file]
   alreadyUnscrewedMsg = 'The chair has already been unscrewed. '
   down asExit(out)
   dobjFor(Screw) remapTo(Screw, chairBolts)
   dobjFor(ScrewWith) remapTo(ScrewWith, chairBolts, IndirectObject)
   bulk = 10
   weight = 12
;

++ Unthing 'castor/castors' 'castors'
  "The large steel chair doesn\'t have any castors. "
  bulk = 0
;

++ chairBolts : Screwable, Component '(chair) bolt/series/bolts/nuts' 'bolts'
   isPlural = true
   screwList = [file]
   makeScrewed(stat)
   {
     if(stat)     
       guardChair.screwBack;      
     else
       guardChair.actionDobjUnscrewWith;
   }
   disambigName = 'chair bolts'
   bulk = 0
   isScrewed = (guardChair.ofKind(CustomFixture))
   sightSize = small
;

+ ExitPortal '(open) exit' 'exit' "The interesting exit is the one to the north, through
   which you catch a tantalizing glimpse of sunlight and greenery. "
   connector = static (wayOut.north)
   dobjFor(Enter) remapTo(TravelVia, connector)
   brightness = 1
;

+ Distant 'sunlight/courtyard/concrete' 'courtyard'
  "From here the chief impression is of sunlight playing on concrete, with just
   a glimpse of greenery through the gates on the far side of the courtyard. "
  brightness = 1
  dobjFor(Enter) remapTo(North)
;

+ Distant 'glimpse/greenery' 'greenery'
  "There's just a glimpse of greenery on the far side of the courtyard, through
   the gates opposite the exit. "
   brightness = 1
;

+ Distant 'tall black pair/gates' 'tall black gates'
  "A pair of tall black gates <<gates.isOpen ? 'stand open on' : 'enclose'>>
   the far side of the courtyard. "
   isPlural = true
;

+ smallPanel : IndirectLockable, Openable, Fixture 'small panel' 'small panel'   
   "The panel is about five inches long by three wide, with a small nut
     at one end. "
   initiallyLocked = true
   cannotUnlockMsg = 'The panel seems to be secured by a nut. '
   cannotLockMsg = (cannotUnlockMsg)
   makeOpen(stat)
   {
     inherited(stat);
     if(stat) "Opening the panel reveals a large red button. ";
   }
   makeLocked(stat)
   {
     if(stat && isOpen)
       tryImplicitAction(Close, self);
     inherited(stat);
   }
   dobjFor(LookBehind)
   {
     preCond = [objVisible, objOpen]
     verify {}
     action {
       "Behind the open panel is a large red button. ";
     }
   }
;

class Screwable : object
 cannotUnscrewMsg = '{You/he} can\'t unscrew {the dobj/him} with {your} bare hands. '
   cannotScrewMsg = '{You/he} can\'t screw {it dobj/him} with {your} bare hands. '
   cannotTurnMsg = '{You/he} can\'t turn {it dobj/him} with {your} bare hands. '
   makeScrewed(stat)
   {
     isScrewed = stat;
     if(unlockObj != nil)
        unlockObj.makeLocked(stat);
     "{You/he} <<stat ? nil : 'un' >>screw{s} <<theName>>. ";
   }
   isScrewed = true
   dobjFor(Fasten) asDobjFor(Screw)
   dobjFor(Unfasten) asDobjFor(Unscrew) 
   dobjFor(TakeWith) remapTo(UnscrewWith, self, IndirectObject)
   dobjFor(UnlockWith) remapTo(UnscrewWith, self, IndirectObject)
   dobjFor(LockWith) remapTo(ScrewWith, self, IndirectObject)
   dobjFor(TurnWith)
   {
     verify() {}
     check()
     {
       if(screwList.indexOf(gIobj)==nil)
       {
         reportFailure('{You/he} can\'t turn it with {that iobj/him}. ');
         exit;
       }
     }
     action() { makeScrewed(!isScrewed); }
   }
   dobjFor(UnscrewWith)
   {
     verify() { if(!isScrewed) illogicalAlready('{It dobj/he} {is} already unscrewed. '); }
     check() 
     {
       if(screwList.indexOf(gIobj)==nil)
       {
         reportFailure('{You/he} can\'t unscrew it with {that iobj/him}. ');
         exit;
       }
     }
     action() { makeScrewed(nil); }
   }
   dobjFor(ScrewWith)
   {
     verify() { if(isScrewed) illogicalAlready('{It\'s dobj} already screwed tight. '); }
     check() 
     {
       if(screwList.indexOf(gIobj)==nil)
       {
         reportFailure('{You/he} can\'t screw it with {that iobj/him}. ');
         exit;
       }
     }
     action() { makeScrewed(true); }
   }
   dobjFor(Screw)
   {
     verify() { }
     action() {
       if(file.screwRank > 100)
          askForIobj(ScrewWith);
       else
          mainReport(cannotScrewMsg);
     }
   }
   dobjFor(Unscrew)
   {
     verify() { }
     action() {
       if(file.screwRank > 100)
          askForIobj(UnscrewWith);
       else
          mainReport(cannotUnscrewMsg);
     }
   }
   dobjFor(Turn)
   {
     verify() { }
     action() {
       if(file.screwRank > 100)
          askForIobj(TurnWith);
       else
          mainReport(cannotTurnMsg);
     }
   }
   screwList = []
   unlockObj = nil
   
;

++ smallNut : Screwable, Component 'small nut' 'small nut'
   "The small nut is situated at one end of the panel. "
   unlockObj = smallPanel
   screwList = [file]
   cannotTakeComponentMsg(loc)
   {
     if(isScrewed)
       return 'It seems to be pretty firmly fixed to the panel. ';
     else
       return '{You/he} {has} unscrewed the nut from the wall, but there\'s
        still a long thread passing through the panel, and since the
        nut is of no use to you there seems little point going to
        the bother of trying to remove it. ';
   }
  bulk = 0
;

+ buttonContainer : SecretFixture, OpenableContainer
  isOpen = (smallPanel.isOpen)
;

++ redButton : Button, Fixture, EventList 'large big alarm red button/alarm' 'big red button'
   "The big red button is marked: <FONT COLOR=RED>EMERGENCY ALARM</FONT>"
   specialDesc = "A large red button is mounted on the wall behind 
     the open panel. "
   daemonID = nil
   daemon()   {  doScript();  }
   eventList =
   [
     new function {
       guardian.moveIntoForTravel(avenue);
       guardian.setCurState(guardianPistol);
       whiteCoat.moveIntoForTravel(avenue);       
       },
       
     new function {
       whiteCoat.moveIntoForTravel(driveway); 
       sirenNoise.start;      
     } ,
     
     new function {                    
       whiteCoat.scriptedTravelTo(courtyard); 
       whiteCoat.killPlayer();
       guardian.moveIntoForTravel(driveway);
     } ,
     
     new function {
       whiteCoat.scriptedTravelTo(wayOut);
       whiteCoat.killPlayer();
       guardian.moveIntoForTravel(courtyard);
     } ,
     
     new function { 
       redButton.daemonID.removeEvent();
       redButton.daemonID = nil;
       sirenNoise.stop();
       whiteCoat.startKillDaemon();
     }  
   ]
   dobjFor(Push)
   {
     check()
     {
       if(whiteCoat.canSee(self))
       {
         "<q>That won't be necessary.</q> the white-coated man assures you.<.p>";
         exit;
       }  
     }
     action()
     {
       if(daemonID==nil)
       {
         "Alarm bells start ringing and a siren starts wailing. ";          
         curScriptState = 1;
         daemonID = new Daemon(self, &daemon, 1);  
         if(gRevealed('guard-watch')) 
         {
           guardianAchievement.addToScoreOnce(2);       
           gReveal('guard-moved');
         }
         if(hermit.curState == hermitRoadWaiting)
           hermit.setCurState(hermitFollowing);
       }
       else
         "Nothing else happens. ";
     }     
   }
;

newspaper : Readable 'rational main news/newspaper/paper/headlines' 'newspaper'
   @guardian
   "It's the <i>Rational News.</i>
   The main headline reads:\n<q>ANOTHER MONTH OF ORDER -- GOVERNMENT HEADS FOR
    POLL TRIUMPH</q>   
   Further down the page a smaller headline declares
   <q>Production Targets Met for 29th Month in Succession</q><.p>"  
   cannotWriteOnMsg = 'The guardian isn\'t about to let you do that. '
   cannotDrawOnMsg = (cannotWriteOnMsg)
   sightSize = small
;

 /*
  *  Several of the rooms that follow are going to be interconnected by DistanceConnectors.
  *  This means that the walls of several rooms may be visible at once, so that, for
  *  example, the command EXAMINE N WALL may refer to a number of different possible
  *  north walls. The CustomWall class provides a better disambigName for such
  *  situations.
  */
  
class CustomWall : RoomPart
  disambigName {
    if(isIn(gActor.getOutermostRoom))
      return name;
    else
      return name + ' of the ' + roomName;
  }
  sightSize = large 
;

courtyardNorthWall : CustomWall 'high north n concrete wall/(courtyard)*walls' 'north wall'
  "The north wall runs along the outer edge of the courtyard, and is pierced
   by a pair of tall gates. "
   cannotClimbMsg = 'Not only does the north wall look pretty difficult to climb,
    it\'s also utterly futile to make the attempt when there\'s a perfectly
    good gate. '  
   roomName = 'courtyard'
;

courtyardEastWall : CustomWall 'high east e concrete wall/(courtyard)*walls' 'east wall'
  "The east wall is just featureless concrete, apart from a blue door. "
  cannotClimbMsg = 'Someone more athletic than you might manage to scale
   the wall, but there\'s nothing to get a purchase on. '  
  roomName = 'courtyard' 
;

courtyardWestWall : CustomWall 'high west w concrete wall*walls' 'west wall'
  "A thick green bush grows against the west wall, which is otherwise just more
   featureless concrete. "
  cannotClimbMsg = 'Quite apart from any other difficulties, the bush growing
   against the wall frustrates all your attempts to climb it; the bush is too flimsy
   to support your weight, but quite thick enough to prevent your getting any
   purchase on the wall. ' 
   roomName = 'courtyard'
;

courtyardSouthWall : CustomWall 'south s wall/(courtyard)*walls' 'south wall'
  "The prison block effectively forms the south wall of the courtyard. "
  cannotClimbMsg = 'The prison block presents a blank wall, giving
  you nothing to get a purchase on. '
  roomName = 'courtyard'
;

courtyardGround : Floor 'hard rough concrete (courtyard) ground/floor' 'rough concrete ground'
  "It's just a bare expanse of concrete. "
  feelDesc = "<<boots.isWornBy(gPlayerChar) ? 'You bend down to touch it; strangely enough,
   it feels just like concrete' : 'The rough ground feels hard and
   harsh on your bare feet'>>. "
   disambigName = 'courtyard ground'
   location = courtyard
;

courtyard : Room 'Prison Courtyard' 'the prison courtyard'
  "The prison courtyard is enclosed by high concrete walls on three sides and the prison
   block on the fourth.<<boots.isWornBy(gPlayerChar) ? '' : ' The ground is also concrete,
   and feels hard and rough on your bare feet. '>> A pair of gates 
   <<gates.isOpen ? ' stands invitingly open to the 
   north, giving access to' : 'block the way north, but through them you can see'>>
   a sweeping drive running across a broad lawn. To the south a narrow passage leads
   back into the prison block, while a blue door is set into the east wall. Just
   by this door is a large steel panel, which is also blue. Against
   the west wall grows a thick green bush. "
   north = gates
   south = wayOut
   in asExit(south)
   east = blueDoor
   west : NoTravelMessage { "Even if you could force your way through the foliage,
    the wall forms an inpenetrable barrier. " }
   roomParts = [courtyardGround, courtyardEastWall, courtyardWestWall,
    courtyardSouthWall, courtyardNorthWall, defaultSky]
   roomBeforeAction()
   {
//     if(gActor.location==self) return;
     local loc = gActor.location;
     if(gActionIs(Out) && loc != self)
     {
       replaceAction(GetOffOf, loc);
       exit;
     }
     if(!gActionIs(Examine) && !gActionIs(Drop)
        && gDobj != nil && gDobj not in (blueDoor, steelHandle, loc, 
      largeSteelPanel, redSwitch, bolts) && !gDobj.isIn(gActor) &&
      !gDobj.isIn(loc) && loc != self)
      tryImplicitAction(GetOffOf, loc);
      
    
   }   
   roomAfterAction()
   {
      if((gActionIs(Examine) || gActionIs(Search))
       && gDobj == courtyardGround && file.isIn(bush))
        nestedAction(LookUnder, bush);  
   }
   atmosphereList : StopEventList {
    [
      'It feels good to be out in the open, but you\'d really like to get
       as far away from here as you can, in case they change their mind about letting
        you go. ',
      nil
    ]
   }   
;

+ bush : Underside, Fixture 'thick green bush/foliage/greenery/bushes' 'thick green bush'
   "The bush grows to a height of about four feet down the entire length of
   the west wall. Its foliage grows thick and green, forming almost as effective
   a barrier as the wall behind<< trappedFile.location != nil ?
     ', but ' + metallicMsg : nil>>. "
   metallicMsg {
    metalSeen = true;    
    return '{you/he} catch{es} a glimpse of something metallic among the greenery';
   }
   actionDobjLookIn
   {
     if(trappedFile.location != nil)     
       "\^<<metallicMsg>>. ";     
     else
       inherited;
   }
   metalSeen = nil
   shouldNotBreakMsg = 'It\'s far too thick for you to make a significant
     impact on it. '
   dobjFor(Push) asDobjFor(Shake)
   dobjFor(Pull) asDobjFor(Shake)
   dobjFor(Move) asDobjFor(Shake)
   dobjFor(Shake)
   {
     preCond = [objVisible, touchObj]
     verify() {}
     action() { 
         if(trappedFile.location <> nil)
         {
           "As you shake the bush you hear a metallic <i>clunk</i> as something
             hits the ground. ";
           file.makePresent();
           trappedFile.moveInto(nil);
         }
         else
           "Shaking the bush has very little effect beyond rustling its foliage. "; 
       }
   }  
   nothingBehindMsg = 'The only thing behind the bush is the wall. '
   cannotClimbMsg = (courtyardWestWall.cannotClimbMsg)
   sightSize = large
;

++ file : CanRub, PresentLater, Hidden
   '(small) metal metallic steel multipurpose rough something
    implement/file/strip/spanner/wrench/tool/head/edge/thing/metallic' 
   'small metal implement'
  "It looks like some kind of multipurpose tool, comprising a steel strip with
    a small spanner head at one end and a rough edge at the other. "
  iobjFor(ScrewWith) { 
    verify() { logicalRank(screwRank, 'screw tool');} 
    action() { screwRank = 150; }
  }
  iobjFor(UnscrewWith) 
  { 
     verify() { logicalRank(screwRank, 'screw tool');} 
     action() { screwRank = 150; }
  }
  iobjFor(TurnWith) 
  { 
     verify() { logicalRank(screwRank, 'screw tool'); } 
     action() { screwRank = 150; }
  }
  iobjFor(CutWith) { verify() {} }
  discover()
  {
    gActor.setIt(self);
    inherited;
  }
  sightSize = small
  inflammable = nil
  screwRank = 100
  bulk = 2
  weight = 3
; 

+ trappedFile : Decoration 'something metallic something/thing/object/glimpse' 'something metallic'
  "It's hard to make out clearly, but it looks like it's caught somewhere deep in the
  bush. "
  notImportantMsg = 'You can\'t get at it through the thick foliage. '
  sightSize = small
;

+ Unthing 'something metallic something/thing/object/glimpse' 'something metallic'
  notHereMsg = 'It\'s no longer there. '
;

+ Enterable ->wayOut 'ugly grey concrete single-storey prison block' 'block'
  "The prison block is an ugly, grey concrete single-storey building forming
  the southern side of the courtyard. " 
  sightSize = large
  remoteDesc(pov)
  {
    if(pov.isIn(prisonRoof))
      "The prison block is the ugly, grey concrete building on the roof
       of which you're currently <<gPostureName>>. ";
    else
      desc;
  }
;

+ Distant 'sweeping drive/driveway' 'sweeping drive'
  "The drive runs north from the gates across a broad lawn. "
  sightSize = large
;

+ Distant 'broad green freshly-mown freshly-cut lawn' 'broad lawn'
  "From what you can see of the lawn through the gates, it looks freshly-cut. It
  stretches out on either side of the driveway, an incongruous decoration to 
  the front of a prison. "   
  sightSize = large
;

+ gates : Door 'black wrought tall iron pair/gate/gates' 'pair of gates'
  "The gates are about six feet high and made of wrought iron painted black. "
  isPlural = true  
  initiallyOpen = true
  canTravelerPass(traveler)
  {
    return !(traveler==gPlayerChar && (!letPlayerPass || gRevealed('squarecircle-statement')));
  }
  explainTravelBarrier(traveler)
  {
     if(gRevealed('squarecircle-statement'))
       "You decide against another encounter with the gentleman in the white coat. ";
     else
     {
      "You walk up to the gates, then pause, peering out into the driveway beyond.
       There's no one in sight, you could easily walk on, and yet you wonder why
       they're letting you go so easily. Somehow it almost seems <i>too</i> easy.<.p>";
       letPlayerPass = true;
     }
  }
  letPlayerPass = nil
  dobjFor(LookThrough)
  {
    action() {
      "Peering through the gates you see a driveway leading north across a 
       freshly-cut lawn. ";     
    }
  }
  sightSize = large
  cannotClimbMsg = 'Climbing the gates would be utterly pointless when
   you can walk straight through them. '
;

+ blueDoor : Door -> alleyDoor 'blue (steel) painted door' 'blue door'
   "It seems to be made of steel, painted blue, and has a small steel
    handle. "
   warned = nil
   panelDesc = "There's a large steel panel, also painted blue, mounted on the 
      wall just next to it. "
   basicExamine()
   {
     inherited;
     if(gActor.isIn(courtyard)) panelDesc;
   }
   warning(stat)
   {
     if(!warned && power.on && redSwitch.isOn && stat)
     {
       "As you start to turn the handle you hesitate, some instinct warning
        you that things might not be quite so simple. ";
       warned = true;
       exit;
     }
   }
   dobjFor(Open)
   {
     check()
     {
       if(steelHandle.touchHandle)
         inherited;
       else
         exit;
     }
   }
   makeOpen(stat)
   {
     warning(stat);
     if(stat) 
     {
       if(gActor.isIn(rubberSheet))
         "The insulation of the rubber sheet allows you to touch the
          handle without getting a shock, so you open the blue door.<.p>";
       if(!redSwitch.isOn || !power.on)
          achievement.awardPointsOnce();
       else
       {
          "Alarm bells start to ring, causing a tall, balding man in a white
           coat to come racing into the courtyard.\b
           <q>You! Stop! Wait right there!</q> he cries.<.p>";
           whiteCoat.moveInto(courtyard);
           whiteCoat.killPlayer();
           exit;
       }
     }
     inherited(stat);
     gReveal('blue door open');
   }
   
   dobjFor(OpenWith) remapTo(TurnWith, steelHandle, IndirectObject)
   
   achievement : Achievement { +1 "opening the blue door without setting off the alarm" }
   sightSize = large
   remoteDesc(pov)
   {
     if(pov.isIn(courtyardWall))
       "You can't really see much of the blue door from here, since you're
        standing on the wall in which it's set. ";
   }
;

++ steelHandle : Component 'small steel (door) handle' 'handle'
  "The small steel handle is firmly fixed to the door, at about three feet from
  the ground. "
  touchHandle()
  {
    if(gActor.location == courtyard && rubberSheet.isIn(courtyard) 
       && gRevealed('blue door open'))
    {
      if(rubberSheet.isIn(gActor))
        tryImplicitAction(Drop, rubberSheet);
      if(rubberSheet.location == courtyard)
        tryImplicitAction(StandOn, rubberSheet);
    }
    if(gActor.isIn(rubberSheet) || !power.on || gActor.isIn(desk) || boots.isWornBy(gActor))
      return true;
    else
    {
      "Touching the steel handle gives you an electric shock that causes you
       to jerk your hand away at once.<.reveal electrified handle> ";
      return nil; 
    }
  } 
  dobjFor(Feel)
  {
    action()
    {
      if(touchHandle)
        "The handle seems smooth to the touch. ";
    }
  }
  dobjFor(Turn)
  {
    verify() {}
    action()
    {
      if(touchHandle)
      {
        blueDoor.warning(!blueDoor.isOpen);
        "You turn the handle";
        if(!blueDoor.isOpen)
        {
          " and open the door. ";
           blueDoor.makeOpen(true);
        }
        else
          ", but the door's already open this doesn't achieve much. ";        
      }  
    }
  }
  iobjFor(PutRound) {
    action() {
      if(gDobj.ofKind(Bendable))
      {
        reportAfter('But it may be possible to turn the handle with it.<.p>');
        
      }
    }
  }
  dobjFor(TurnWith)
  {
    verify() {
      if(blueDoor.isOpen) illogicalAlready('The blue door is already open. ');
    }
    check() {
      if(!gIobj.ofKind(Bendable)){
        "That doesn\'t work. ";
        exit;
      }
    }
    action() {
      "{The iobj/he} insulate{s} you from any current flowing in the handle, so that
       you are able to turn the handle and open the door. ";
       blueDoor.makeOpen(true);
    }
  }
  dobjFor(Pull)
  {
    verify() {}
    action() {
      if(touchHandle())
        replaceAction(Close, blueDoor);  
    }     
  }
  dobjFor(Push) remapTo(Open, blueDoor)
  sightSize = small
  
;

+ largeSteelPanel : IndirectLockable, Openable, Fixture 'large blue steel panel' 
   'large steel panel'
  "The large steel panel << isOpen ? '(behind which is a large red 
   switch) ' : '' >>is bolted to the wall. "
   initiallyLocked = true
   cannotUnlockMsg = 'The panel is locked with a series of bolts. '
   cannotLockMsg = (cannotUnlockMsg)
   makeOpen(stat)
   {
     inherited(stat);
     if(stat) "Opening the panel reveals a large red switch behind. ";
   }
   makeLocked(stat)
   {
     if(stat && isOpen)
       tryImplicitAction(Close, self);
     inherited(stat);
   }
   dobjFor(Unscrew) remapTo(Unscrew, bolts)
   dobjFor(UnscrewWith) remapTo(UnscrewWith, bolts, IndirectObject)
   dobjFor(LookBehind)
   {
     preCond = [objVisible, objOpen]
     verify  {}
     action  {
       "Behind the open panel is a large red switch. ";
     }
   }
;

++ bolts : Screwable, Component '(panel) bolt/series/bolts/nuts' 'bolts'
   "The bolts are screwed through one end of the panel. "
   verifyDobjUnscrewWith() {
     if(!isScrewed) illogicalNow('They\'re already unscrewed as far as 
      they need to be. ');
   }   
   isPlural = true
   screwList = [file]
   unlockObj = largeSteelPanel
   cannotTakeComponentMsg(loc)
   {
       if(isScrewed)
         return 'They\'re attached to the panel. ';
       else
         return 'Although they\'re no longer fastening the panel to the
          wall, their threads still pass through the panel, and since
          they\'re really quite stiff to turn and they\'re of absolutely 
          no use to you, there\'s really no point in trying to remove them. ';       
    }    
  disambigName = 'panel bolts'
  sightSize = small
;
 
+ Container, SecretFixture 'cavity' 'cavity'  
  isOpen = (largeSteelPanel.isOpen)
;

++ redSwitch : Switch, Fixture 'large alarm red switch/alarm' 'red switch'
  "It's labeled LARM, but the label looks as if part of it has been broken
   off at the left-hand end. The switch is currently
   o<<isOn ? 'n' : 'ff'>>. "
  isOn = true    
  specialDesc = "Behind the open panel next to the blue door is a large
   red switch. "
  specialDescOrder = 10
  sightSize = small
  dobjFor(Pull) asDobjFor(Flip)
  dobjFor(Push) asDobjFor(Flip)
  dobjFor(Move) asDobjFor(Flip)
;

++ Fixture 'broken (alarm) (larm) (left) (left-hand) label/end' 'broken label'
  "The label just above the large red switch reads
  <FONT COLOR=RED>LARM</FONT>, but it seems to be broken off
  at the left-hand end. "
;

driveway : OutdoorRoom 'Broad Drive' 'the broad drive'
  "The driveway runs across a freshly-cut lawn towards a row of tall trees lining 
   an inviting avenue to the north. Next to the courtyard gates to the south stands
   a small white building. "     
   travelerArriving (traveler, origin, connector, backConnector)    
   {     
      inherited(traveler, origin, connector, backConnector);
      if(traveler == gPlayerChar)
       {
         "As you stop to get your bearings, the door of the small white hut flies
        open, and a tall, balding man in a white coat comes out. ";
         
         whiteCoat.moveIntoForTravel(driveway);
         whiteCoat.killPlayer;
       }         
   }   
  south = courtyard
  north = avenue
;

+ whiteHut : Decoration 'small freshly-painted windowless flat-roofed white hut/building' 
  'small white hut'
  "The small white building, really just a hut, stands next to the gates into the courtyard. 
   It is windowless, freshly-painted and flat-roofed. "
   remoteDesc(pov)
   {
     if(pov.isIn(hutRoof))
       "You can't make out much of the hut from here, because you're standing
        on its roof.";
     else
       desc;
   }
   sightSize = large
;

+ courtyardGates : Door ->gates 'black wrought iron (courtyard) pair/gate*gates' 'pair of gates'
  "A pair of black wrought iron gates lead into the courtyard beyond. "
  isPlural = true
  sightSize = large
;

+ PathPassage 'broad grey gray gravel driveway/drive' 'driveway'
  "The grey gravel driveway leads from the gates to the south across a freshly-cut
   lawn towards the avenue of trees to the north. "
  destination = avenue
  sightSize = large
;

+ Decoration 'freshly-cut lawn/garden' 'lawn'
  "The freshly-cut lawn seems strangely out of place in front of a prison courtyard,
   as if trying to disguise what lies beyond by presenting the appearance of an
   idyllic garden. It stretches out on either side of the broad driveway running
   north and south. "
   sightSize = large
;

+ Distant 'avenue/(trees)' 'avenue'
  "The avenue to the north is a continuation of the broad driveway. Tall trees -- conifers
   by the look of them -- line both sides of the avenue. "
   sightSize = large
;

+ Distant 'tall trees/conifers' 'trees'
  "The tall trees lining the avenue to the north look like conifers. They almost completely
   obscure the view to northeast and northwest. "
  isPlural = true
  sightSize = large
;




avenue : OutdoorRoom 'Avenue' 'the avenue'
  "The driveway continues northwards towards a gate, while the avenue runs southwards
   past rows of tall trees. Beyond the trees to the south a small white building stands
   next to a courtyard entrance. "
  north = prisonGate2
  south : OneWayRoomConnector {
    -> driveway
    warned = nil
    canTravelerPass(traveler) 
     { return traveler != gPlayerChar || (warned && !gRevealed('squarecircle-statement'))
        || guardian.isIn(avenue); }
    explainTravelBarrier(traveler)
    {
      if(gRevealed('squarecircle-statement'))
        "You realize you'll almost certainly run into that white-coated fellow
         again if you walk into that driveway, and you decide you've had quite
         enough of him. ";
      else
      {
        warned = true;
        "You take one step forward then stop; the driveway looks as if it leads
        straight back to the prison from which you have just escaped, and you
        wonder if you really want to go back there. ";
      }
    }
  }
  cannotGoMsg = 'The trees are too densely packed. '
;

+ prisonGate2 : Door 'gate' 'gate';

+ Distant 'small freshly-painted white building/hut' 'small white building'
  "You can't make out much detail from here, apart from the fact that it looks freshly-painted
  and that it stands next to a courtyard entrance. "
;

+ Distant 'rather grim courtyard entrance' 'courtyard entrance'
  "The courtyard entrance is a long way to the south, at the end of the driveway that extends
   beyond the avenue leading through the trees. "
;

+ Decoration 'tall rows/trees/conifers/avenue' 'tall trees'
  "Rows of tall trees, mainly conifers, line both sides of the avenue, obscuring the view to
   both east and west. "
  isPlural = true
;

+ PathPassage 'drive/driveway' 'driveway'
  "The driveway leads south across a well-cut lawn towards a courtyard entrance. "  
  dobjFor(TravelVia)
  {
    action() { replaceAction(South); }
  }
;

+ Distant 'freshly well-cut well cut lawn/grass' 'lawn'
  "The lawn stretches out on either side of the driveway. It looks freshly cut
   and peversely inviting, unlike the rather grim courtyard beyond. "
;

class Unclimbable : object
 dobjFor(Climb)
  {
    verify() {}
    action()
    {
      "<<cantClimbMsg>>";
    }
  }
  cantClimbMsg = 'You cannot climb it. '
;
 
alleyNorthWall : Unclimbable, CustomWall 'north n high red brick wall/(alley)*walls' 'north wall'
  "It's made of red bricks and is about seven feet high. "
  cantClimbMsg = 'It looks too high too climb. '  
  roomName = 'alley'
;


alleySouthWall : Unclimbable, CustomWall 
  'south s high concrete prison block (alley)/wall*walls' 'south wall'
  "The blank concrete wall is actually part of the north wall of the prison block. "
  cantClimbMsg = 'You scrabble ineffectually at the wall but totally fail to climb
   so much as an inch of it. '
  roomName = 'alley' 
;

alleyGround : defaultGround '(narrow) gravel (alley) path/ground' 'ground'
  "The ground here is a narrow gravel path, with a prominent pothole towards
   the eastern end. "
   disambigName = 'alley ground'
;

+ potHole : Fillable, Container '(pot) pothole/hole' 'pothole'
  "There is a squarish pothole in the ground to one side of the alley, 
    about six or seven inches a side. It is slightly irregular in shape
    in that the eastern edge of the hole is vertical, while the western
    side slopes a little. <<holeSand.isIn(self) ?
    'It has, however, been completely filled with sand. ' : nil >>"    
  bulkCapacity = 2  
  sightSize = small  
  sandFill(obj)
  {
     "You pour the sand into the hole";
     if(obj.bulk > bulkCapacity)
       ", spilling the surplus";
     ". ";
     holeSand.moveInto(self);
     return true;
  }
  iobjFor(PourInto)
  {
    verify() 
      { 
        if(getBulkWithin() >= bulkCapacity)
          illogicalNow('{The iobj/he} is already full.' );
      }
    check()
    {
      if(getBulkWithin() > 0)
      {
        "That would bury the existing contents of the pothole, and you decide you'd
          rather not do that. ";
        exit;
      }
      if(desk.isIn(alley))
      {
        "You'll need to move the desk out of the way before you can do that. ";
         exit;
      }
    }
  }
  iobjFor(PutIn)
  {    
    check() {
      inherited;
      if(desk.isInHole)
       {
         reportFailure('The desk leg is in the way. ');
         exit;
       }
    }
    action()
    {
      if(!gDobj.ofKind(Sand))
        inherited;
    }
  }
  dobjFor(FillWith) remapTo(PutIn, IndirectObject, self);
;

holeSand : Fixture '(hole) sand' 'sand in hole'
  "The sand completely fills the pothole. "
  bulk = static (potHole.bulkCapacity)
;

alley : OutdoorRoom 'Narrow Alley' 'the narrow alley'
  "This narrow alley runs along a narrow gravel path between a high brick wall to the north 
   and a concrete wall to the south. There is a steel door at the western end of the alley, which 
   continues towards a small yard to the east. "
  vocabWords = 'narrow alley'
  roomParts = [alleyGround, alleySouthWall, alleyNorthWall, defaultSky]  
  west = alleyDoor
  east : OneWayRoomConnector
  {
    -> smallYard    
    travelBarrier : PushTravelBarrier
    {
      explained = nil
      canPushedObjectPass(obj) 
      {
        return(obj != desk || greenSandBox.isIn(potHole) || holeSand.isIn(potHole));
      }
      explainTravelBarrier(traveler)
     {        
        if(greenBox.isIn(potHole))
          "The front desk leg almost passes over the pothole, but the green box
           isn't firm enough to take its weight, so the leg gets stuck in the 
           hole and you can't push the desk any further east. ";
        else if(desk.isInHole)
           "{You/he} push{es} the desk as hard as {it actor/he} can, but to no
           avail: the leg won't budge out of the pothole. ";
        else if(explained) {
           "Although you try to avoid the pothole, the
            narrowness of the passage and the lie of the ground combine to
            make it impossible, so the desk leg still ends up trapped in 
            the hole, and you still can\'t push the desk any further east. ";
            desk.isInHole = true;
         }
        else
        {
          "One of the desk legs gets stuck in a pothole, and try as you might,
           you can't push the desk any further east. ";
           explained = true;
           desk.isInHole = true;
        }
     }
    }
  }
  
;



+ alleyDoor : Door 'blue steel door' 'steel door'
  "The steel door is painted blue. "
  okayOpenMsg = '{You/he} open{s} the blue steel door. '
  travelBarrier : PushTravelBarrier
  {
    canPushedObjectPass(obj) 
      {
        return(obj != desk || !desk.isInHole);
      }
    explainTravelBarrier(traveler)
     {
        desk.isInHole = nil;
        "By pushing with all your might you just manage to get the trapped
         desk leg to ride up the shallower side of the pothole. "; 
     }
  }
  makeOpen(stat)
  {
    inherited(stat);
    if(stat && redSwitch.isOn && power.on)
    {
      "An alarm bell starts ringing. ";
      whiteCoat.moveIntoForTravel(driveway);
      whiteCoat.scriptedTravelTo(courtyard);
      whiteCoat.startKillDaemon();  
      if(!whiteCoat.canSee(gPlayerChar))
      {
        whiteCoat.addPendingAction (true, whiteCoat, WaitAction);
        whiteCoat.addToAgenda(conditionalEastAgenda);
      }
    }
  }
  
;

++ Component 'plain steel metal (door) handle' 'handle'
   "It's just a plain steel door handle. "
   disambigName = 'plain steel door handle'
   dobjFor(Turn) remapTo(Open, alleyDoor)
;

+ Decoration 'red (almost) brown brick/bricks' 'red bricks'
  "Red bricks, some weathered almost to brown, make up the north wall of
   this narrow alley. A close examination of the bricks
   reveals nothing of much interest, however; they're just the kind of plain,
   ordinary bricks you'd expect to see a wall made of. "  
   isPlural = true 
;

+ Enterable ->smallYard 'small yard' 'small yard'
  "The small yard is at the eastern end of the alley. From here it appears to
   be totally enclosed. "
;

smallYardNorthWall : CustomWall 'north high concrete (n) wall/(yard)*walls' 'north wall'
  "It's just a high concrete wall. "
  cannotClimbMsg = (smallYardWestWall.cannotClimbMsg)
  roomName = 'yard'
;

smallYardSouthWall : CustomWall 'south (s) wall/(yard)*walls' 'south wall'
  "The high concrete wall to the south is pierced by a tall rusty gate overlooking
    a large open field. "
  cannotClimbMsg = 'You see no point in trying to climb the unclimbable south
   wall when there\'s a perfectly good gate. '
  roomName = 'yard'
;

smallYardWestWall : CustomWall 'bleak blank concrete west (w) prison wall/(yard)/block*walls' 
   'west wall'
  "The west wall of the yard is actually part of the east wall of the prison block,
   a bleak, blank concrete wall. To the north of the prison block a narrow alley runs
   west. "
   cannotClimbMsg = 'It\'s too high, and offers nothing to get a purchase on. '
   roomName = 'yard'
;

smallYardEastWall : CustomWall 'high concrete east (e) wall/(yard)*walls' 'east wall'
  "It's just a high concrete wall. "
  cannotClimbMsg = (smallYardWestWall.cannotClimbMsg)
  roomName = 'yard'
;


smallYard : Room 'Small Yard' 'the small yard'
  "This small yard is totally enclosed apart from a narrow alley running west and
   a tall gate to the south. A large pile of sand rests in the corner. "
   
   west = alley
   south = rustyGate
   roomParts = [defaultGround, defaultSky, smallYardNorthWall, smallYardSouthWall,
     smallYardWestWall, smallYardEastWall]
;

+ Decoration 'corner' 'corner'
   "One corner of the yard is taken up with a large pile of sand. "
;

+ sand : Fixture 'large fine yellow (some) pile/sand' 'pile of sand'
  "The pile of fine yellow sand fills one corner of the yard. "
  dobjFor(Take)
  {
    verify() {}
    action()
    {
      "You take a handful of sand, but it rapidly slips away between your fingers. ";
    }
  }
  dobjFor(PutIn)
  {
    verify() {}
    preCond = [touchObj]
    check() {}
    action() 
    {
           
      if(gIobj == whiteBox.subContainer)
        replaceAction(FillWith, whiteBox, self);
      else if(gIobj.ofKind(Fillable))
        replaceAction(FillWith, gIobj, self);
      else
        "You pick up handful of sand to put in {the iobj/him}, but what
         doesn't slip through your fingers is blown away on the breeze. ";
           
    }
  }
  iobjFor(FillWith)
  {
    verify() {}
  }
  iobjFor(PourOnto)
  {
    verify() {}
    check()
    {
      if(!gDobj.ofKind(Sand))
      {
        reportFailure('You can\'t pour that here. ');
        exit;
      }
    }
  }
  dobjFor(DigWith)
  {
    verify() { }
    action() { "Digging in the pile of sand reveals nothing but a whole
                    lot more sand. "; }
  }
  sandFill(obj)
  {
    "You pour <<obj.theName>> back onto the pile of sand. ";
    return true;
  }
  
  
  cannotStandOnMsg = 'This proves fairly futile, as the sand shifts
    beneath your weight, and you keep sliding back to the ground. '
  cannotClimbMsg = (cannotStandOnMsg)
  cannotMoveMsg = 'It would take too long to move the whole pile of sand,
    and even if you managed it, it would serve no useful purpose. '
  nothingInsideMsg = '{You/he} find{s} nothing in the pile of sand except more sand. '
  sightSize = large
  theName = (gActionIs(PutIn) ? 'sand from the pile' : inherited )
;

  /*
   * The gate from the small yard to the field is represented by two objects
   * forming its two sides. We define RustyFieldGate class to define the handling
   * common to both sides of the gate.
   */
   
class RustyFieldGate : LockableWithKey, Door
 dobjFor(Climb)
  {
    verify() {}
    action()
    {
      if(isOpen)
        "Trying to climb the gate while it's open would be a pretty
          futile exercise. ";
      else
         replaceAction(TravelVia, gateConnector);
    }
  }
  keyList = [filedIronKey]
    
  lockOrUnlockAction(lock)
  {
    if(gIobj == ironKey)    
       reportFailure('The iron key fits the lock but won\'t quite turn in it.
         <.reveal GateTried>');    
    else
      inherited(lock);
  }
  
  keyIsPlausible(key)
  {
    return key is in (ironKey, filedIronKey);
  }
  dobjFor(LookThrough)
  {
    action() { thruDesc; }      
  }
  makeOpen(stat)
  {
    if(trough.isIn(field))
    {
      "{You/he} can\'t <<stat ? 'open' : 'close'>> <<theName>>; the trough is
       in the way. ";
      exit;
    }
    inherited(stat);
    if(stat)
      fieldGate.hasBeenOpened = true;
  }
  cannotJumpOverMsg = '{You/he} {is}n\'t that athletic; {you/he} leap{s} as high as
   {it actor/he} can, but {it actor/he} come{s} nowhere near to clearing the gate. '
  travelBarrier : PushTravelBarrier {     
     explainTravelBarrier(traveler)
     {
         "\^<<traveler.obj_.theName>> won't fit through the gate. ";
     }

  } 
;

+ rustyGate : RustyFieldGate ->fieldGate 'tall rusty (iron) black gate' 'rusty gate'
  "The gate, which is about six feet high, is made of rusty iron, from which most of 
   the black paint is peeling off. "
  thruDesc = "Through the gate you see a small open field. <<trough.isIn(field)
   ? 'A large wooden trough sits in the field, right next to the other side of the
   gate. ' : ''>> "
   basicExamine()
   {
     inherited;
     fieldDesc;
   }
   fieldDesc = "On the other side of the gate you can see a large open
    field."
   dobjFor(Push) asDobjFor(Open)
   dobjFor(Pull) asDobjFor(Close)
   shouldNotBreakMsg = 'It looks too sturdy for that to be a feasible option'
;

++ Component 'peeling (black) paint' 'black paint'
  "The black paint is starting to peel off the gate, revealing quite a bit
   of rust beneath. "
   bulk = 0
   sightSize = small
;

++ Component 'patches/rust' 'rust'
  "Several patches of rust can be seen on the gate where the paint has peeled
   away. The gate itself still looks pretty sound, however."
   bulk = 0
   sightSize = small
;

++ rustyLock : RestrictedContainer, Component 'lock/keyhole' 'lock'
   "The lock is not so rusted that it shouldn't work; from the size of the 
    keyhole it's obviously meant to take a fairly substantial key. "
    validContents = [ironKey, filedIronKey]
    dobjFor(Unlock) remapTo(Unlock, rustyGate)
    dobjFor(UnlockWith) remapTo(UnlockWith, rustyGate, IndirectObject)
    dobjFor(Lock) remapTo(Lock, rustyGate)
    dobjFor(LockWith) remapTo(LockWith, rustyGate, IndirectObject)
    dobjFor(Pick)
    {
      verify() {}
      action() { "You don't know how to. "; }
    }
;

+ Distant 'large open field' 'large open field'
  "The field on the far side of the gate stretches some way to the south and
   looks as if it opens out round the back of the prison block. "
   dobjFor(Enter) remapTo(South)
;

+ Distant '(east) distant bare concrete prison slab/concrete/wall/block' 'prison block'
  "Through the gate you can just make out the east wall of the prison block,
   a slab of bare concrete, jutting out into the field. " 
   dobjFor(Examine)
     {
         verify()
         {            
             logicalRank(60, 'x decoration');
         }
     } 
  disambigName = 'distant prison block'   
;

+ Enterable ->alley 'narrow alley' 'narrow alley'
  "The alley runs west from here, along the north side of the prison block. "
;

 /* Note that the original iron key is replaced with the filedIronKey object
  * once it has been rubbed with the implement.
  */

filedIronKey : Key 'large (filed) iron key*keys' 'large iron key'
  "It's about six inches long, with a large loop at one end and a pair of
    large teeth at the other. One of the teeth has been filed down slightly. "  
   dobjFor(Turn)
   {
     verify() { 
       if(location is in(rustyLock, rustyFieldLock) || location.ofKind(Keyhole)) 
          logicalRank(120, 'key in lock');
       else 
          inherited;
     }
     action() {
       if(isIn(rustyLock) || isIn(rustyFieldLock)) 
         replaceAction(UnlockWith, location, self);
       else if(location.ofKind(Keyhole))
        {
          "The key turns smoothly in the lock. ";
          cellDoor.makeLocked(!cellDoor.isLocked);
        }      
      else
        "That achieves little. ";
    }
  }  
       
   
  dobjFor(RubWith)
  {
    action()
    {
      if(gIobj == file)
        "The iron key has already been filed enough -- more filing would probably
         be counterproductive. ";
      else if(gIobj.ofKind(Hand) && gVerbName == 'file')
        "You can't file the key with you bare hands -- not that it needs filing. ";
        
    }
  }    
  dobjFor(Pick) maybeRemapTo(gActor.canSee(brassKey), Take, self)
  sightSize = small
  inflammable = nil
;

+ Component 'filed pair/teeth' 'teeth'
  "The iron key has a pair of large teeth at the business end; one of them has
   been filed down slightly. "
   isPlural = true
   sightSize = small
;


gateConnector : OneWayRoomConnector   
   destination = field
   noteTraversal(traveler)
   {
     "By standing on <<traveler.location.theName>> you just manage to clamber over 
      the gate into the field beyond.<.p>";
      traveler.rememberTravel(smallYard, field, fieldGate);
      rustyGate.rememberTravel(smallYard, traveler, field);
      fieldGate.rememberTravel(field, traveler, smallYard);
   }
   canTravelerPass (traveler) 
   {
     return traveler.location is in (desk, guardChair, sandBox);
   }
   explainTravelBarrier (traveler) 
   {
     "The gate is just too high too climb over. ";
   }
;

sandBox : SandFilled, Heavy, Platform 'large square white box*boxes' 'large square white box'
  "The large white box is full of sand. "
  weight = 15
  bulkCapacity = 40
  specialDesc = "A large white box has been placed just next to the gate. "
  useSpecialDesc() { return isDirectlyIn(smallYard); }
  altObject = whiteBox
  roomLocation = (location != nil ? location.roomLocation : nil)  
  dobjFor(StandOn)
  {
    preCond = static inherited + objClosed;
  }
  dobjFor(Open)
  {
    verify() {
      inherited;
      if(gActor.isIn(self))
        illogicalNow('{You/he} can\'t open the box while {it actor/he} {is}
          standing on it. ');
    }
    
    action() {      
      if(contents.length > 1)
      {
        "(first ensuring there's nothing on the top of the box) ";
         foreach(local cur in contents)
           if(!cur.ofKind(Sand))
           tryImplicitAction(Take, cur);
      }
      inherited;
    }
      
  }
  dobjFor(GetOutOf)
  {
    preCond { return delegated Platform.preCondDobjGetOutOf; }
  }
  dobjFor(Board)
  {
    preCond { return inherited Platform + objClosed; }
  }
  iobjFor(PutOn)
  {
    preCond = static inherited + objClosed;
  }  
;

+ Sand bulk = 20;

yellowSandBox : SandFilled, Thing 'small yellow box*boxes' 'small square yellow box'

  weight = 7
  dobjFor(StandOn)
  {
    verify {illogical('The yellow box is too small to stand on. '); }
  }
  altObject = yellowBox
;

+ Sand;

greenSandBox : SandFilled, CanRub, Thing 'tiny green square cardboard box*boxes' 
  'tiny green cardboard box'
  
  weight = 3  
  dobjFor(StandOn)
  {    
    verify {illogical('The green box is far too tiny to stand on. '); }
  }
  altObject = greenBox
  sightSize = small
;

+ Sand;

tinSandBox : SandFilled, Thing 'small black tin box*boxes' 'small tin box'
   
   altObject = tinBox 
   bulkCapacity = 3
   bulk = 3 
   sightSize = small
;

+ Sand;

/* The SandFilled class defines the objects that are the sand-filled equivalents of
 * the boxes. The original boxes are held in the altObject property. Emptying the
 * SandFilled box is equivalent to taking the sand out of it, which in turn calls
 * the swap method to replace the sand-filled box with its empty equivalent.
 */

class SandFilled : Openable
  desc = "<<altObject.desc()>> <<isOpen ? 'It\'s full of sand. ' :''>>"
  mySand = nil
  altObject = nil
  swap()
  {
    if(altObject.ofKind(Thing))
    {
      local loc = location;
      moveInto(nil);
      altObject.moveInto(loc);      
    } 
  }
  actionDobjOpen()
  {
    "Opening <<theName>> reveals it to be full of sand. ";
    inherited;
  }
  dobjFor(Empty) 
  {
    preCond = [objOpen, touchObj]
    verify() { }
    action()
    {
      replaceAction(Take, mySand);
    }
  }
  iobjFor(PutIn)
  {
    verify() { illogicalAlready('{The iobj/he} is already full of sand. '); }
    preCond = [objVisible, touchObj, objOpen]
  }
  iobjFor(PourInto) asIobjFor(PutIn)
  dobjFor(FillWith) remapTo(PutIn, IndirectObject, self)
  getFacets() {
    if(altObject != nil)
      return [altObject];
    else
      return inherited;
  }
  dobjFor(Pick) asDobjFor(Take)
  bulk = (altObject == nil ? 0 : altObject.bulk)
  bulkCapacity = (altObject == nil ? 0 : altObject.bulkCapacity)
  cannotWriteOnMsg = &uselessToScribbleOnMsg
  cannotDrawOnMsg = &uselessToScribbleOnMsg
  descContentsLister: openableContentsLister {
    showListPrefixWide(itemCount, pov, parent)
     {
         "\^<<parent.openStatus>>, and on it <<itemCount==1 ? 'is' : 'are'>> ";
     }

    
  }
;

/* The Sand class represents the contents of a SandFilled box. It is designed to be
 * nested within a SandFilled box and then initializes itself and its container to
 * point to each other. When the PC removes the Sand from a SandFilled box some scattered
 * sand must go somewhere, so the Sand object creates a new ScatteredSand object in
 * its room location, unless one is already present, and then calls the swap method
 * on its container to replace it with its empty version
 */
 

class Sand : Component 'boxed sand' 'boxed sand'
   "The tightly packed sand fills the box. "
   container = nil
   initializeThing
   {
     inherited;
     if (location != nil)
     {
       container = location;
       container.mySand = self;
     }
   }
   dobjFor(Take)   
   {
     verify() { logicalRank(110, 'boxed');}
     check() {}
     action() 
     {
     "You empty the sand out of <<container.theName>>, scattering it as you do so. ";
     local obj;
     local s;
     foreach(obj in roomLocation.contents)
       if (obj.ofKind(ScatteredSand))
       {
         s = obj;
         break;
       }
     if(s==nil && roomLocation != sand.location)
     {
       s = new ScatteredSand;       
       s.moveInto(roomLocation);
       roomLocation.isSandy = true;
     } 
     if(container != nil)
       container.swap;
    }  
  } 
  dobjFor(TakeFrom)
  {
    verify() {}
  }
  dobjFor(PutIn)
  {
     preCond = [touchObj]
     verify() { logicalRank(70, 'boxed'); }
     action()
     {
       if(gIobj.propDefined(&sandFill))
       {
         if(gIobj.sandFill(self))
           container.swap;
       }
       else
         replaceAction(Take, self);
     }
  }
  dobjFor(PourInto) remapTo(PutIn, self, IndirectObject)
  dobjFor(PourOnto) 
  {
    preCond = [touchObj, objVisible]
    verify() { }
    action() 
    {
      if(container != nil && !container.isOpen)
        if(!tryImplicitAction(Open, container))
          exit;
      if(gIobj.ofKind(Surface) || gIobj.ofKind(Floor))
        replaceAction(Take, self);
      else
        replaceAction(PutIn, self, gIobj);
    }
  }
  dobjFor(PutOn) remapTo(PourOnto, self, IndirectObject)
  inflammable = nil
  disambigName = ('sand in ' + container.theName)
  notAContainerMsg = 'The sand is too tightly packed. '  
;

modify Floor
  iobjFor(PourOnto)
  {
    verify() {}
  }
;


/* ScatteredSand represents the sand scattered from a SandFilled box that
 * has been emptied in a particular location.
 */

class ScatteredSand : Unthing '(scattered) sand' 'scattered sand'
  notHereMsg = 'The sand you scattered out of the box has dispersed too far and wide
    to be of any use -- you certainly can\'t recover it. '  
;

sirenNoise : MultiLoc, SimpleNoise 'alarm/siren' 'alarm'
  "Alarm bells are ringing and a siren is wailing. "
  isAmbient = nil
  start()
  {
    foreach(local cur in soundLocList)
      moveIntoAdd(cur); 
  }
  
  stop()
  {
    if(gPlayerChar.canHear(self))
      "The alarm is silenced. ";
    foreach(local cur in soundLocList)
      moveOutOf(cur); 
      
  }
  soundLocList = [wayOut, vestibule, cell, squareRoom, courtyard, driveway,
    alley, smallYard, circleRoom]    
;



newFoodNote : Readable, DrawingSurface, PaperSurface 
 'bright small brief yellow note message/piece/paper*pieces notes' 'yellow piece of paper'
 "It's a small, bright yellow piece of paper, on which a message has been written. "
  readDesc = "The message reads: <q>The enclosed sandwich is the last thing you will
  ever have to eat unless and until you come to your senses and give us the answer
  we want. Please note that starvation is not a pleasant death, so you may well wish
  to rethink your stubborn obstinacy.</q>"
;
  
  
prisonHutCeiling : OutOfReach, RoomPart 'roof/ceiling' 'roof'
  "Apart from the skylight, there's nothing remarkable about it. "
  canObjReachContents(obj)
  {
    return (obj.isIn(prisonHutChair) && obj.posture==standing);
  }
;

prisonHut : Room 'Cramped Wooden Hut' 'the cramped wooden hut'
  "The inside of this hut is almost totally bare, apart from a door to the north.
   There are no windows, but light streams in through a skylight in the roof. "
   north = prisonHutDoor
   up = skylight   
   roomParts = static inherited - defaultCeiling + prisonHutCeiling
;

+ prisonHutDoor : LockableWithKey, Door 'wooden door' 'door'
;

+ prisonHutChair : Chair 'straight-backed uncomfortable wooden chair*chairs' 'wooden chair'
  "It's one of those straight-backed wooden chairs that seem to have been
   designed with the sole object of maximimising the discomfort of anyone
   rash enough to sit on them. "
  initSpecialDesc = "The sole piece of furniture here is a wooden chair. "
  cannotMoveMsg = 'There\'s not many places to ' + gVerbName + ' it in
   this tiny hut. ' 
  obviousPostures = [sitting, standing] 
  down asExit(out)
;

++ hutNote : DrawingSurface, Readable, PaperSurface 
   'small rectangular paper piece/paper/note/writing' 
    'small piece of paper'
   "It's a small, rectangular piece of paper, about five inches by three, 
    which looks like it could have been torn from a notebook. There's some
    writing on it. <<isHeldBy(gActor) ? 'It looks like there might be some
    writing on the other side as well. ' : ''>>"
    readDesc {
      if(isHeldBy(gActor))
        "On one side";
      else
        "On the note";
      " is written, <q>You will be left here for an indefinite but sufficiently
        long time to get <i>very</i> hungry and thirsty. You would do well
        to use the time to consider your position and resolve to be more
        co-operative on our return.</q> ";
      if(isHeldBy(gActor))
       "\bOn the other side is drawn a circle, then a square, and finally
         the numbers 5, 9 and 6.";
    }
   bulk = 0    
   initSpecialDesc 
   {
       if(described)
        "There's a small piece of paper on the chair. ";
       else  
       "A small piece of paper has been carefully placed on the chair,
       as if to ensure you don't miss it. ";
   }
   specialDescBeforeContents = nil
   verifyDobjExamine() { logicalRank(120, 'more likely'); } 
   disambigName = 'small rectangular piece of paper'
   sightSize = small
;  
  
+++ Component 'other (both) side/sides/(note)/(piece)/(paper)/circle/square' 
  'the other side of the note'
  dobjFor(Examine)
  {
    action() {
      if(!hutNote.isHeldBy(gActor))
        tryImplicitAction(Take, hutNote);
      replaceAction(Examine, hutNote);  
    }
  }
  dobjFor(Read)
  {
    
    action() {
      if(!hutNote.isHeldBy(gActor))
        tryImplicitAction(Take, hutNote);
      replaceAction(Read, hutNote);  
    }
  }
;

+ skylight : OutOfReach, IndirectLockable, Door 'square skylight/frame' 'skylight'
  "It's set in the roof, and looks about eighteen inches square. At one end
   of it is a thick steel ring, which is evidently used to lock it. "
  canObjReachContents(obj)
  {
    return (obj.isIn(prisonHutChair) && obj.posture == standing);
  }
  canTravelerPass(traveler)
  {
    return traveler.isIn(prisonHutChair) && traveler.posture == standing;
  }
  explainTravelBarrier(traveler)
  {
    "You can't reach the skylight from here. ";
  } 
  connectorStagingLocation = nil
  dobjFor(TravelVia)
  {
    preCond = [standOnChair, touchObj, objOpen]    
  }
  dobjFor(Climb) asDobjFor(GoThrough)
  isLocked = (padlock.isIn(skylightRing))
  cannotUnlockMsg = 'It looks like {you/he} need{s} to remove the padlock.<.reveal padlock> '
  nothingThroughMsg = 'You can see little through the skylight apart from a patch
   of blue sky. '
  shouldNotBreakMsg = 'You can\'t see how to. '
;

++ skylightRing : Component, RestrictedContainer 'thick steel ring/latch' 'ring'
   "The thick ring is attached firmly to the end of the skylight. An equally
    thick steel latch attached to the frame <<padlock.isIn(self) ? 'passes over
    the ring and is held in place by a padlock, thereby locking the skylight'
    : 'hangs open'>>. "
    
   validContents = [padlock]
   shouldNotBreakMsg = 'It\'s too solid; you can\'t. '
;

+++ padlock : IndirectLockable, Openable, Thing
  'small steel padlock/lock' 'small steel padlock'
  "The most obvious feature of this small steel padlock is its black
   circular dial. "
  dobjFor(Take)
  {
    check() {
      if(isIn(skylightRing) && !isOpen)
      {
        reportFailure('You\'ll have to open the padlock first.<.reveal padlock> ');
        exit;
      }
    }
  }
  dobjFor(Lock) asDobjFor(Close)
  makeOpen(stat)
  {
    inherited(stat);
    if(!stat)
    {
      makeLocked(true);
      "The padlock emits a sharp click as it snaps shut and relocks. ";
    }
  }
  isListedInContents = (!isIn(skylightRing))  
  shouldNotBreakMsg = 'The padlock proves too strong to break. '
  sightSize = small
  cannotUnlockMsg = 'It looks as if you\'ll need to find the right combination. '
  inflammable = nil
;

++++ padlockDial:  NumberedDial, Component 'black circular dial' 'dial'
  "It's a black circular dial with white numbers round the edge. It looks like
   it can be turned to any number from 0 to 99<.reveal padlock>. "
   minSetting = 0
   maxSetting = 99
   curSetting = '51' 
   makeSetting(val)
   {
     inherited(val);
     if(val != combination[combState])
       combState = 1;
     else     
       combState++;
     if(combState > combination.length)
     {
       "As you turn the dial to <<val>> the padlock springs open.<.reveal padlock-open> ";
       padlock.makeLocked(nil);
       padlock.makeOpen(true);
       combState = 1;
     } 
     if(val != nonCombination[nonCombState])
       nonCombState = 1;
     else
       nonCombState++;
     if(nonCombState > nonCombination.length)
     {
       nonCombState = 1;
       "The padlock remains resolutely locked when you turn the dial to <<val>>, but
        then you never really expected it to be that straightforward. ";
     }
   }
   combination = ['25', '81', '36']
   nonCombination = ['5', '9', '6']   
   combState = 1
   nonCombState = 1
   cannotMoveMsg = 'It\'s a dial: you turn it. '  
   sightSize = small    
;

standOnChair : PreCondition
 checkPreCondition(obj, allowImplicit)
     {
       if(allowImplicit && (!gActor.isIn(prisonHutChair) || gActor.posture != standing))
       {
         tryImplicitAction(StandOn, prisonHutChair);
       }
       if(!gActor.isIn(prisonHutChair) || gActor.posture != standing)
       {
         reportFailure('You need to be standing on the chair. ');
         exit;
       }
     }
;

hutRoofFloor : Floor 'flat wooden (hut) roof/floor/ground' 'roof'
  "The hut has a flat wooden roof, with a skylight set in it. "
  sightSize = large
  location = hutRoof
;

Occluder, DistanceConnector
  [hutRoof, driveway]
  occludeObj(obj, sense, pov)
     {
         if(obj is in (defaultSky, sun, clouds) )
           return nil;
                      
         
         if(pov.isIn(driveway))
         {
           return obj.isIn(hutRoof);
         }
         return nil;
     }

;


hutRoof : OutdoorRoom 'Roof of Hut' 'the roof of the hut'
  "From the roof of the hut you can see northwards up the drive to an avenue
   of trees, and southwards into the prison courtyard below. The roof of the
   hut is almost level with the top of the courtyard wall, so you could
   step right onto it. Roughly in the middle of the roof is a<<roofLight.isOpen
    ? 'n open' : ''>> skylight. " 
  down = roofLight
  south :TravelMessage {
    -> courtyardWall
    "Overcoming your dislike of heights -- especially as experienced from the 
     top of a narrow wall -- you step up onto the wall, and work your way
     gingerly round to the northeastern corner, and from there to the top
     of the east wall of the courtyard. "
  }
  roomParts = [hutRoofFloor, defaultSky]
  roomBeforeAction()
  {
    if(gActionIs(Jump))
    {
      "It's too far down to the ground; you can't afford a twisted ankle, and anyway
       you might land straight in the arms of the wit in a white-coat and his
       pal the guardian goon with a gun. ";
       exit;
    }
  }
  actorInPrep = 'on'
;

+ roofLight: IndirectLockable, Door ->skylight 'skylight' 'skylight'
  isLocked = (skylight.isLocked)
  nothingThroughMsg = 'Through the skylight you can just make out the interior of a small hut,
   with a straight-backed wooden chair almost directly beneath the skylight. '
;

+ Fixture 'courtyard wall' 'courtyard wall'
  "The courtyard wall runs parallel to the southern edge of the
   hut roof for a few feet, then turns south towards the prison block.
   The wall is just about broad enough to walk on if you don't have too bad a head
   for heights. "
  dobjFor(Enter) remapTo(South)
  dobjFor(StandOn) remapTo(South)
  dobjFor(Board) remapTo(South)
;

DistanceConnector [courtyardWall, driveway ];
DistanceConnector [courtyardWall, hutRoof];
DistanceConnector [courtyardWall, alley];

Occluder, DistanceConnector [courtyardWall, courtyard]
  occludeObj(obj, sense, pov)
     {
         if(obj is in (defaultSky, sun, clouds))
           return nil;
         
         if(pov.isIn(courtyardWall))
           return((obj.ofKind(Distant) && obj.isIn(courtyard)) 
            || obj.isIn(largeSteelPanel) ||
            obj == largeSteelPanel || obj.isIn(blueDoor) 
            || obj is in (alleyDoor, whiteHut, alley) ) ;
         else
            return (obj is in (courtyardWall, alley) || obj.isIn(courtyardWall)); 
             
         
     }


;

courtyardWallGround : Floor 'ground/floor/top/wall' 'top of the wall'
  "It's about six inches wide -- not nice to walk or stand on. "
;

courtyardWall : OutdoorRoom 'Courtyard Wall' 'the courtyard wall'
 "The top of the courtyard wall runs parallel to the southern edge of the hut
  roof just below, and then turns south towards the prison block. To the north
  you can see across the roof of the hut into a drive, with an avenue beyond. To
  the west you look down into the prison courtyard, while to the south the
  wall runs all the way to the roof of the prison. To the east you can
  look down a narrow alley. "
 roomBeforeAction
 {

   if(gActionIs(Drop))
   {
     "On second thoughts, you decide that this is a seriously bad place
      to start dropping things. There's no telling where they might land,
      and they could very easily break. ";
      exit;
   }
   if(gActionIs(Jump) || gActionIs(JumpOff))
   {
     "No way. It's bad enough having to stand on this wall. Walking along it
      is a trial. The very thought of jumping on it (let alone jumping off
      it) makes you feel physically ill. ";
      exit;
   }
   if(gActionIs(Yell))
   {
     "Not wishing to draw attention to yourself, you decide to whisper instead. ";
     exit;
   }
   if(gActionIs(LieOn))
   {
     "There's nowhere to lie here - the top of the wall is too narrow. ";
     exit;
   }
 }
 north : TravelMessage {
     -> hutRoof
     "You grit your teeth and force yourself to edge your way to the northeast
      corner of the wall. Once there you advance more briskly along the top
      of the south wall until, a few paces along, you can leap down, with
      considerable belief, onto the roof of the hut. "
  }
 south : TravelMessage {
      -> prisonRoof
      "Faulteringly you work your way south until, with enormous relief, you
       can step onto the firm, flat roof of the prison building. "
  }
 down : NoTravelMessage {
   "It's too far down to the ground - you might break something if you try to jump
     down and the wall has no handholds or footholds to allow a safe descent
     climbing down. ";
   }
  vocabWords = '(courtyard) wall'  
  sightSize = large
  actorInPrep = 'on'
  dobjFor(ClimbDown) remapTo(Down)
  roomParts = [courtyardWallGround, defaultSky]
;

+ Distant '(prison) courtyard' 'prison courtyard'
  "The courtyard is an expanse of concrete enclosed by the wall you're now
  <<gPostureName>> on, a concrete building to the south, and walls on the other 
  two sides. A thick bush grows against the west wall, while in the north wall
  is a pair of tall iron gates. "
;

+ Distant 'narrow alley' 'narrow alley'
  "The narrow alley runs east from the courtyard wall alongside the wall of
   the prison compound. "
;


Occluder, DistanceConnector [prisonRoof, field]
  occludeObj(obj, sense, pov)
     {
         if(obj is in (defaultSky, sun, clouds) )
           return nil;
           
         if(pov.isIn(prisonRoof))
           return (obj is in (drainpipe, fieldNorthWall, fieldWestWall, westFieldForest,
            courtyardSouthWall, alleySouthWall, smallYardWestWall) || obj.isIn(courtyardSouthWall)
            || (obj.ofKind(Distant) && obj.isIn(smallYard))
            || obj.ofKind(Enterable) || obj == alley);
         else
           return obj.isIn(prisonRoof);         
     }
;
DistanceConnector [prisonRoof, courtyard];
DistanceConnector [prisonRoof, westField] name = 'connector';

DistanceConnector [prisonRoof, alley];
DistanceConnector [prisonRoof, smallYard];



prisonRoofGround : Floor '(prison) flat dirty grey roof/ground/floor/expanse' 'prison roof'  
  "It's just a flat, dirty grey expanse. "
  location = prisonRoof
;

prisonRoof : OutdoorRoom 'Prison Block Roof' 'the roof of the prison block'
  "The prison roof affords an excellent view of the fields to south and east, and of the forest
   beyond them to the east. Along the southern boundary of this field runs a tall wooden fence, 
   on the far side which a swollen river dashes along from 
   under an old stone bridge to the southwest. 
   Over this bridge runs a tarmac road that continues
   along the west side of the field and then past the west side
   of the prison compound.
   To the north the prison roof overlooks a courtyard immediately below, the courtyard
   being enclosed by a wall that someone with a fair head for heights might just be able to 
   walk along the top of. On the other side of this wall a narrow
   alley leads into a small enclosed yard to the east. To complete this
   clockwise circuit the top of a black drainpipe can just
   be seen peeping over the eastern edge of the roof. "
 actorInPrep = 'on' 

 north : TravelMessage {
    -> courtyardWall 
    "With some reluctance, you force yourself to leave the comparitive safety of the
     roof and step out on top of the narrow wall, telling yourself not to look down. ";
 }
 down : TravelConnector {
   dobjFor(TravelVia)
   {
    action(){
     if(gRevealed('drainpipe-descended'))
       replaceAction(TravelVia, roofPipe);
     else
       "You certainly can't jump down from here, you might damage something
       (probably yourself). ";
     }  
   }
   isConnectorApparent(origin, traveler) { return gRevealed('drainpipe-descended'); }
 }
 roomParts = [prisonRoofGround, defaultSky]
;

+ roofPipe : StairwayDown ->drainpipe 'black top/drainpipe/pipe' 'drainpipe'
  "The drainpipe runs down the side of the building. "
  noteTraversal(traveler)
  {
    "You manage to lower yourself safely down the drainpipe to the ground below. 
    <.reveal drainpipe-descended>";
  }
;

+ Distant 'swollen river' 'river'
  "The river, swollen by recent floods, runs along the far side of the fence towards
   the forest to the east and a bridge to the west. "
;

+ Distant 'old stone delapidated bridge' 'bridge'
  "The river rushes under an old stone bridge to the southwest. It's difficult to
   make out from here, but the bridge looks a bit delapidated. "
;

+ Distant 'large open field' 'large open field'
  "Most of the large open field lies to the south of the prison block on which
   you're currently <<gPostureName>>. The field is bounded by a fence to the south,
   a road to the west, and a forest to the east. A small part of the field lies
   directly to the east of the prison block. "  
;

+ Decoration '(prison) compound/block' 'prison compound'
  "The prison compound comprises the block you're now <<gPostureName>> on, the
   courtyard just to the north of it, and the area around it to west and north,
   including the driveway beyond the courtyard gates. "
  notImportantMsg = 'That hardly seems a practicable suggestion. '
;

+ Distant '(prison) courtyard' 'prison courtyard'
  "The courtyard is an expanse of concrete enclosed by the building you're now
  <<gPostureName>> on and walls on the other three sides. A thick bush grows
  against the west wall, while there's a blue door in the east wall and a pair
  of gates in the north wall. "
;

+ Distant 'small enclosed yard' 'small enclosed yard'
  "The yard, on the eastern side of the building on which you're now <<gPostureName>>,
   is only a few square yards in extent. It's entirely enclosed by high concrete walls
   apart from an alley leading off to the west and a gate to the south. The northeast
   corner of the alley is occupied by a pile of sand. "
;

+ Distant 'narrow alley' 'narrow alley'
  "The narrow alley runs alongside the north wall of the prison block, and a high
   brick wall that runs parallel to it. At its east end the alley opens out into
   a small yard, while at its west end is a door through the east wall
   of the courtyard. "
;   

+ Distant 'tarmac road' 'road'
  "The road runs along the western edge of the field and the disappears out of
   sight across the old stone bridge. "
;
