/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 *      djp     In real life:   David J. Picton
 *
 * Modification History (this file)
 *
 * 24-Aug-99    bjs     Pre-release version 0.00
 *              bjs     Added 550-point extensions for beta release
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *              djp     * Cleaned up a few bugs in the code for throwing
 *                        objects at the Djinn.
 *              djp     * Implemented a heredesc for Slime and Basilisk and
 *                        removed these from the room description.
 *              djp     * Removed redundant location code from
 *                        floatingdecoration objects.
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version
 *                      * Added 'safe' travel method where appropriate
 *                      * Changed the bottomless pit code.  Any food and
 *                        water should be used up!
 *                      * Changed In_Safe to be described by a heredesc
 *                        method.
 *                      * Added iswavetarget property for cases when the
 *                        rod is waved at a bridge or chasm etc.
 *                      * Improved the description of the niche if objects
 *                        are put in it.
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version:
 *                      * In the 550-point and 701-point games, the player
 *                        sinks if trying to carry the clam over the quicksand.
 *                      * For the 701-point game, we require that the plate
 *                        be carried in hand, not (say) in the sack.
 *              
 * 20-Dec-00    djp     Version 2.02: bugfix release
 *                      Changes in this version:
 *                      * Corrected the Balcony.reservoir method for the
 *                        701-point game. 
 *                      * Changes to allow the gong to be hit with an object.
 *                      * Fixed misnamed exithints property for By_Figure
 *
 * 15-May-01    djp     Version 2.04: bugfix release
 *
 * 8-Jun-01     djp     Version 2.05: bugfix release
 *                      Changes in this version:
 *                      * Corrected the code for throwing objects at the
 *                        Djinn (including special handling for liquids)
 *
 *              djp     Version 2.08: bugfix release
 *                      * Made Volcano_Below into a distantItem.
 *                      * Added a method for leaving the fog.
 *                      * Added methods to enter the pentagram.
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      * Fix to cantReachRoom method in the location of
 *                        the safe (when taking the rug or some nestedroom 
 *                        object, otherRoom comes out as the object itself.)
 *                      * Directions have been added for entering the safe,
 *                        based on some arbitrary choices.
 *                      * Safe door objects have been added, with support for
 *                        pulling, pushing, opening and closing.
 *                      * Adjustments to the safe code so that things look
 *                        right if we view the closed safe with the crystal
 *                        ball.
 *                      * Updated the safe to use the new
 *                        Internal_Room class.
 *                      * Corrected methods for looking in or on the Stone,
 *                        which is now a chairitem.
 *                      * Changed the code for the fog to see if the room
 *                        is lit by any light source, using the darkroom
 *                        islit property.
 *                      * Made provision for location methods to depend on 
 *                        an actor other than the player.
 *                      * Allowed NPCs into the Vault and In_Safe.  (NPCs
 *                        can enter the safe but only when it is open to the
 *                        Vault.  NPCs are always allowed out to the Vault.)
 *
 * 31-May-03    djp     Version 2.10: minor bugfixes and enhancements
 *                      * Enhanced code for attempting to open the safe.
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *
 * 23-Aug-03    djp     Version 2:12 
 *                      * Set global.triggered_alert flag after a security
 *                        alert.
 *
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *                      Changes in this version
 *                      * Defined a class for Stone (for duplicate
 *                        Sandstone Chambers in the 701+ point game)
 *                      * For the 580-point game: enhancements to the code 
 *                        for In_Safe to acknowledge an attempt to pull the
 *                        door closed from the inside, using the suction
 *                        cups.
 *                      * Changes to Catacombs enterroom/leaveroom methods to
 *                        improve efficiency (a loop over all items has been
 *                        replaced by a loop over room contents).
 *
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *
 * 28-Jun-05    djp     Version 3.11: Enhancements and bugfixes
 *
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Addition of room properties for areas where drinking
 *                        wine is forbidden.  wino_wsbridge is set beyond the
 *                        wheat-stone bridge, and wino_quicksand is set beyond
 *                        the quicksand.
 *        
 * 21-Nov-10    djp     Version 3.21
 *                      * Corrected the code to set the wino_wsbridge
 *                        property in In_Safe, if appropriate
 *                      * Changes to pentagram code to accommodate the new
 *                        'posture' property (standing/sitting/lying)
 *                      * The onroom property of the pentagram is now set to
 *                        nil (so that the player will be said to be 'in' the
 *                        pentagram)
 */

/* Additional rooms for 550-point version */

/* Note on compound words.  Compound word constructions were used in the
past to circumvent problems when verbs were used as adjectives; for example,
'actor, west' wouldn't parse correctly if an item defined 'west' as an
adjective.   This fix is no longer needed, and has been backed out from
version 2.00 onwards.

For the benefit of people using TADS 2.4.0 and later, we now also avoid
the use of adjectives which are used as a noun by other objects.
A bugfix in the interpreter allows the use of adjectives in these
circumstances, and this caused some confusion; for example, 'bird' could
refer to the cage in the absence of the bird, and 'rod' could refer to the
crystal bridge in the absence of the rod.
 */

/*
 * A class for the ice tunnels. Since the NPCs can't use magic words in
 * this game, they will become trapped upon entering the maze. Hence,
 * I've made them NoNPC, although they weren't in the original. -BJS
 */
class CCR_ice_tunnel: NoNPC
    game550 = true
    sdesc = "Ice Tunnels"
    ldesc = {  local xn := 0, xls := 0;
        I(); "You are in an intricate network of ice tunnels.  ";
        // List exit tunnels:

        // Store the exits which lead somewhere in the number xls.
        // Let xn be the number of exits available.
        if(proptype(self, &north) = 2) { xls += 1; xn++; }
        if(proptype(self, &south) = 2) { xls += 2; xn++; }
        if(proptype(self, &east) = 2) { xls += 4; xn++; }
        if(proptype(self, &west) = 2) { xls += 8; xn++; }
        if(proptype(self, &sw) = 2) { xls += 16; xn++; }
        if(proptype(self, &nw) = 2) { xls += 32; xn++; }
        if(proptype(self, &ne) = 2) { xls += 64; xn++; }
        if(proptype(self, &se) = 2) { xls += 128; xn++; }

        if(xn = 0) "Uh, oh!  There are no exits";

        if(xn = 1) { local xi := 1; // xi is direction counter.
            "The only exit is to the ";
            while(xls<>0) {  // print the list of exits.
            if (xls%2 = 1)  // if xls is odd, the exit in
                self.printdir(xi); // direction xi exists.
            xls /= 2;  // lop off last bit of exitlist,
            xi++;     // and go to next direction.
            }
        }

        if(xn = 2) { local fxp, xi := 1;
            "There are exits to the ";
            while(xls<>0) {  // print the list of exits.
            if (xls%2 = 1) { // if xls is odd, the exit in
                self.printdir(xi);  // direction xi exists.
                if (not fxp) " and ";
                fxp := true;  // fxp indicates whether the
            }        // first exit has been printed.
            xls /= 2;  // lop off last bit of exitlist,
            xi++;    // and go to next direction.
            }
        }

        if(xn > 2) { local nxp := 0, xi := 1;
            "There are exits to the ";
            while(xls<>0) {  // print the list of exits.
            if (xls%2 = 1) { // if xls is odd, the exit in
                self.printdir(xi); // direction xi exists.
                nxp++;   // nxp is number of exits printed.
                if (not (nxp = xn)) ", ";
                if (nxp = xn-1) "and ";
            }
            xls /= 2;   // lop off last bit of exitlist,
            xi++;      // and go to next direction.
            }
        }
        ".  ";
    }
    printdir(dir) = {
        switch(dir) {
            case 1: "north"; break;
            case 2: "south"; break;
            case 3: "east"; break;
            case 4: "west"; break;
            case 5: "southwest"; break;
            case 6: "northwest"; break;
            case 7: "northeast"; break;
            case 8: "southeast"; break;
        }
    }
    myhints = [ Icehint ]
;

N_Of_Reservoir: darkroom, NoNPC
    game550 = true
    // In the 701-point game we use Sword_Point instead
    game701 = nil
    sdesc = "North Edge of Reservoir"
    ldesc = {
        I(); "You are at the northern end of a large underground
        reservoir.  Across the water to the south, a dark passage
        is visible.  Another passage leads north from here.  Large
        clawed tracks are visible in the damp ground, leading from
        the passage into the water.";
    }
    north = Warm_Room
    passage = Warm_Room
    warm = Warm_Room
    balcony = Balcony
    south = {
         local actor := getActor(&travelActor);
         Reservoir.doCross(actor);
         return nil;
    }
    cross = (self.south)  // BJS: added
    across = (self.south) //   ""  ""
    exithints = {
        if (global.game701)return [N_Of_Reservoir, &south];
        else return [Sword_Point, &south];
    }
;
Brass_Gong: fixeditem
    game550 = true
    sdesc = "large brass gong"
    ldesc = "It's just a large brass gong fastened to the wall
             of the room."
    heredesc = {P(); I(); "There is a large brass gong fastened to the wall
            here.";}
    location = N_Of_Reservoir
    location701 = Sword_Point
    noun = 'gong'
    adjective = 'large' 'brass'
    verDoAttack(actor) = {} // That's how "HIT GONG" gets parsed!
    verDoAttackWith(actor,io) = {}
    doAttack(actor) = {
        if(Turtle.location = self.location) {
            ">BONNNNNGGGGGGGGGG<"; P();
            "Darwin the Tortoise blinks in surprise at the
            noise, but does nothing.";
        }
        else { ">BONNNNNGGGGGGGGGG<"; P();
            I(); "A hollow voice says, \"The GallopingGhost
            Tortoise Express is now at your service!\""; P();
            I(); "With a swoosh and a swirl of water, a large
            tortoise rises to the surface and paddles
            over to the shore near you.  The message,
            \"I'm Darwin - ride me!\" is inscribed on
            his back in ornate letters.";
            Turtle.moveInto(self.location);
        }
    }
    doAttackWith(actor,io) = {self.doAttack(actor);}
;
Turtle_Tracks: fixeditem
    game550 = true
    sdesc = "clawed tracks"
    ldesc = "Large clawed tracks are visible in the damp ground,
             leading from the passage into the water."
    noun = 'track' 'tracks'
    adjective = 'clawed' 'damp' 'large'

    location = N_Of_Reservoir
    location701 = Sword_Point
;

Warm_Room: darkroom, NoNPC
    game550 = true
    sdesc = "Small Warm Chamber"
    ldesc = {I();
        "You are in a small chamber with warm walls.  Mist drifts
        into the chamber from a passage entering from the south
        and evaporates in the heat.  Another passage leads out to
        the northeast.";
    }
    south = {
        if (global.game701) return Sword_Point;
        else return N_Of_Reservoir;
    }
    reservoir = {return self.south;}
    ne = Balcony
    balcony = Balcony
    exithints = {
        if (global.game701) return [Sword_Point, &south];
        else return [N_Of_Reservoir, &south];
    }
;

Balcony: room, NoNPC
    game550 = true
    sdesc = "Treasure Room Balcony"
    ldesc = { I();
        "You are in a high balcony carved out of solid rock
        overlooking a large, bare chamber lit by dozens of
        flickering torches.  A rushing stream pours into
        the chamber through a two-foot slit in the east
        wall and drains into a large pool along the north
        side of the chamber.  A small plaque riveted to
        the edge of the balcony reads, \"You are looking
        at the Witt Company's main treasure room,
        constructed by the famous architect Ralph Witt
        in 4004 B.C., and dedicated to the proposition
        that all adventurers are created equal (although
        some are more equal than others).  NO ADMITTANCE
        VIA THIS ENTRANCE!\"  A small, dark tunnel leads
        out to the west.";
    }
    west = Warm_Room
    out = Warm_Room
    warm = Warm_Room
    reservoir = {
        if (global.game701) return Sword_Point;
        else return N_Of_Reservoir;
    }
    jump = { return broken_neck.death; }
    exithints = {
        if (global.game701) return [Sword_Point, &reservoir];
        else return [N_Of_Reservoir, &reservoir];
    }
;

WittTreasSign: fixeditem, readable
    game550 = true
    sdesc = "small plaque"
    ldesc = "It says,
        \"You are looking at the Witt Company's main
        treasure room, constructed by the famous
        architect Ralph Witt in 4004 B.C., and
        dedicated to the proposition that all
        adventurers are created equal (although
        some are more equal than others).  NO
        ADMITTANCE VIA THIS ENTRANCE!\""
    noun = 'plaque' 'sign'
    adjective = 'small' 'riveted'
    location = Balcony
;

WheatStoneBridge: floatingdecoration
    iswavetarget = true // magic can be worked by waving the rod at it ...
    game550 = true
    exists = nil
    sdesc = "wheat-stone bridge"
    ldesc = "The bridge is a fragile-looking arch of wheat-colored stone,
        leading across the volcanic gorge."
    loclist = [] // initially doesn't exist
    heredesc = { P(); I();
        "There is a wheat-colored
        stone bridge arching over the gorge."; }
    noun = 'bridge'
    adjective = 'wheat' 'wheat-colored' 'wheat-coloured' 'wheatstone' 
                'wheat-stone' 'magic' // Some adjectives deleted for TADS 2.4.0+.

    verDoCross(actor) = {}
    doCross(actor) = {
        if (self.exists and actor = Bear) "The archway looks too
            fragile to support the bear.";
        else if (self.exists) { // unreachable in 350-point mode.
            if (actor.isIn(At_Breath_Taking_View)) {
                if (Bear.isfollowing)
                    "That archway looks pretty fragile - you'd
                    better leave the bear here.";
                else { // Note that it is not necessary to wear
                       // the ring, since the original version
                       // had no wear verb. (You do have
                       // to wear the ring in the 660-point
                       // version, though.)
                    if(mithril_ring.location = actor) {
                        "As you approach the center of the archway,
                        hot vapors saturated with brimstone drift
                        up from the lava in the gorge beneath your
                        feet.  The mithril ring ";
                        if (mithril_ring.isworn) "on your finger";
                        else "in your hand";
                        " quivers and glows, and the fumes eddy away
                        from the bridge without harming you.";
                        P();
                        actor.travelTo(Valley_Faces);
                        }
                        else { "As you approach the center of the
                            archway, hot vapors saturated with
                            brimstone drift up from the lava in
                            the gorge beneath your feet.  You are
                            swiftly overcome by the foul gasses
                            and, with your lungs burned out, fall
                            off of the bridge and into the gorge.";

                            die();
                        }
                    }
                }
                else {
                if(mithril_ring.location = actor) {
                    "As you approach the center of the archway,
                    hot vapors saturated with brimstone drift
                    up from the lava in the gorge beneath your
                    feet.  The mithril ring in your hand
                    quivers and glows, and the fumes eddy away
                    from the bridge without harming you.";
                    P();
                    if (sceptre.isInside(actor)) { I();
                        "As you reach the center of the bridge,
                         a ghostly figure appears in front of
                         you.  He (?) stands at least eight
                         feet tall, and has the lower body of
                         an enormous snake, six arms, and an
                         angry expression on his face.
                         \"You'll not have my sceptre that
                         easily!\" he cries, and makes a
                         complex magical gesture with his
                         lower right arm.  There is a brilliant
                         flash of light and a vicious >crack<,
                         and the bridge cracks and plummets
                         into the gorge.";
                        // Zap the bridge ...
                        self.exists := nil;
                        self.moveLoclist([]);
                        // drop the sceptre at Valley_Faces
                        sceptre.moveInto(Valley_Faces);
                        // but drop the rest of the player's
                        // items at Breath_Taking_View so he
                        // can get across again
                        actor.moveInto(At_Breath_Taking_View);
                        // Zap the player

                        die();
                    }
                    else actor.travelTo(At_Breath_Taking_View);
                }
                else { "As you approach the center of the
                    archway, hot vapors saturated with
                    brimstone drift up from the lava in
                    the gorge beneath your feet.  You are
                    swiftly overcome by the foul gasses
                    and, with your lungs burned out, fall
                    off of the bridge and into the gorge.";

                    die();
                }
            }
        }
        else {
            "I'm afraid "; if (actor = Bear) "he"; else "I";
            " can't go that way - walking on red-hot
            lava is contrary to union regulations (and is bad
            for your health anyhow). ";
        }
    }
;


Valley_Faces: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    isbonus = true
    sdesc = "South End of Valley of Faces"
    ldesc = { I();
        "You are standing at the southern end of a long valley
        illuminated by flickering red light from the volcanic gorge
        behind you.  Carved into the walls of the valley are an
        incredible series of stone faces.  Some of them look down
        into the valley with expressions of benevolence that would
        credit a saint;  others glare with a malice that makes the
        heart grow faint.  All of them are imbued with a fantastic
        seeming of life by the shifting and flickering light of the
        volcano.  The entire far end of the valley is taken up by
        an immense carving of a seated figure;  its exact form
        cannot be seen from here due to the uncertainty of the light.";
    }
    north = By_Figure
    south = { 
        local actor := getActor(&travelActor);
        WheatStoneBridge.doCross(actor); 
        return nil; 
    }
    cross = { return self.south; }
    gorge = { return self.south; }
    exithints = [ At_Breath_Taking_View, &south ]
;
Stone_Faces: fixeditem
    game550 = true
    sdesc = "stone faces"
    ldesc = "Some of them look down into the valley with expressions
        of benevolence that would credit a saint;  others glare
        with a malice that makes the heart grow faint.  All of
        them are imbued with a fantastic seeming of life by the
        shifting and flickering light of the volcano."
    noun = 'face' 'faces'
    adjective = 'stone'
    location = Valley_Faces
;


By_Figure: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "North End of Valley of Faces"
    ldesc = { I();
        "You are standing at the north end of the Valley of the
        Stone Faces.  Above you, an incredible bas-relief statue of an
        immense minotaur has been carved out of the rock.  At least
        sixty feet high, it sits gazing down at you with a faint but
        definite expression of amusement.  Between its feet and the
        floor is a rock wall about ten feet high which extends across
        the entire north end of the valley.  ";
        Rock_Wall.described;
    }
    south = Valley_Faces
    nw = { if(Rock_Wall.has_crumbled) return South_Fog;
        else return self.noexit;
    }
    left = { return self.nw; }
    ne = { if(Rock_Wall.has_crumbled) return Winding_Pass;
        else return self.noexit;
    }
    right = { return self.ne; }
    north = { if(Rock_Wall.has_crumbled) return South_Basilisk;
        else return self.noexit;
    }
    middle = { return self.north; }
    exithints = [ South_Fog, &nw,   Winding_Pass, &ne,
                   South_Basilisk, &north ]
;
Rock_Wall: fixeditem
    game550 = true
    sdesc = "rock wall"
    ldesc = { if(self.has_crumbled) "Dark tunnels lead northeast, north,
        and northwest through the crumbled wall.";
        else "The rock wall extends across the entire north
            end of the valley.";
    }
    has_crumbled = nil
    noun = 'wall' 'rock'
    adjective = 'rock' 'stone'
    location = By_Figure
    described = { if(self.has_crumbled) "Dark tunnels lead northeast,
        north, and northwest."; }
;
Statue: fixeditem
    game550 = true
    sdesc = "immense statue"
    ldesc = "The statue is gazing at you with an amused expression on its
        face."
    noun = 'statue' 'minotaur' 'bas-relief' 'relief'
    adjective = 'immense' 'minotaur' 'bas-relief' 'relief' 'bas'
    location = By_Figure
;

South_Fog: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "South End of Foggy Plain"
    ldesc = { I(); "You are standing at the southern end of what appears
        to be a large room filled with multicolored fog.
        The sides and far end of the room cannot be seen due
        to the thickness of the fog - it's a real pea-souper
        (even to the color in places!).  A passage leads
        back to the south;  a dull rumbling sound issues
        from the passage.";
    }
    south = By_Figure
    north = { Fog.dir_to_go := rand(8); // initialize the fog.
        return Foggy_Plain;
    }
;

Foggy_Plain: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Foggy Plain"
    ldesc = {} // dealt with by Fog heredesc
    myhints = [ Foghint ]

    // The numbers are for the Fog.ldesc method.
    north = { return Fog.wander(3); }
    south = { return Fog.wander(4); }
    east = { return Fog.wander(1); }
    west = { return Fog.wander(2); }
    sw = { return Fog.wander(7); }
    nw = { return Fog.wander(8); }
    se = { return Fog.wander(6); }
    ne = { return Fog.wander(5); }
;
Fog: fixeditem
    game550 = true
    sdesc = "fog"
    // DJP - added a heredesc
    heredesc = {P(); I(); self.ldesc;}
    ldesc =  {
        if(self.location.lamplit) switch(rand(8)) { // DJP
            case 1: "You are standing, badly befuddled, in a
                    pale purple fog."; break;
            case 2: "You are wandering around in the middle
                    of a bright red fog."; break;
            case 3: "You are lost in the midst of a thick,
                    pea-green fog."; break;
            case 4: "You are trying to find your way through
                    a dense coal-black fog."; break;
            case 5: "You are lost in the heart of a strange
                    yellow fog."; break;
            case 6: "You are standing, badly bedazzled, in a
                    day-glow orange fog."; break;
            case 7: "You are hunting your way through a
                    shimmering magenta fog."; break;
            case 8: "You are somewhere in the center of a
                    weird, pearly pink fog."; break;
        }
        else switch (self.dir_to_go) {

            /* direction to go. Should match the values
                    in the direction properties for the
                    Foggy_Plain. */

            case 1: "A faint glow of light is visible
                     through the fog to the east."; break;
            case 2: "A glimmer of light is visible
                     through the fog in the west."; break;
            case 3: "A glow of light is visible through
                     the fog to the north."; break;
            case 4: "A faint shimmer of light is
                     visible to the south."; break;
            case 5: "A flickering light is visible through
                     the fog in the northeast."; break;
            case 6: "A dim light is visible in the
                     southeast."; break;
            case 7: "A dim glow of light is visible
                     in the southwest."; break;
            case 8: "A dim flickering light is visible
                     through the fog in the northwest.";
                     break;
            default: "Whoops! The direction hasn't been set yet!";
        }
        "\n";
    }
    dir_to_go = 0 // should be set when fog is entered or wander() is
                  // called.
    noun = 'fog'
    location = Foggy_Plain
    adjective = 'thick' 'pea' 'soup' 'pea-soup' 'multicolored' 'multicoloured'
    verDoUnboard(actor) = {
        "You'll have to tell me how to do that. ";
    }
    wander(dir) = {
        local actor := getActor(&travelActor);
        local i, dest;
        if((not Foggy_Plain.lamplit) and (dir=dir_to_go)) dest := Plain_Center;
        else dest := Foggy_Plain; // Stay on the plain.

        if(brass_lantern.islit and
        brass_lantern.isIn(Foggy_Plain) and
        not brass_lantern.isIn(actor)) {
            "You're just
            wasting your batteries leaving the
            lamp on like that.\n";
            return Foggy_Plain;
        } // Since you can't lose a lit lamp in the fog.

        dir_to_go:=rand(8); /* reset proper direction
                        Doing it in both cases eliminates
                        the need to do it when the player
                        leaves the center. */

        for (i:=firstobj(item); i<>nil; i:=nextobj(i, item)) {
            // Any items lying around are lost.
            // DJP - exclude fixed items.
            if (i.isfixed or i.location <> Foggy_Plain ) continue;
            i.moveInto(nil);
            "The fog is so dense that I don't
            think you'll be able to find ";
            i.thedesc;
            " again.\n";
        }

        return dest;  // Go wherever is necessary.

    }
;

Plain_Center: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Foggy Plain by Cairn of Rocks"
    ldesc = { I(); "You are standing in a fog-filled room next to a tall
        cairn of glowing rocks.  An opening in the cairn leads
        down to a dark passage.";
    }
    // No need to reset the fog, since that was done when we got here.
    north = Foggy_Plain
    south = Foggy_Plain
    east = Foggy_Plain
    west = Foggy_Plain
    nw = Foggy_Plain
    sw = Foggy_Plain
    se = Foggy_Plain
    ne = Foggy_Plain
    down = Nondescript
;

Nondescript: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Nondescript Chamber"
    ldesc = { I(); "You're in a small, nondescript chamber.  A dark
        passage leads up and to the south, and a wide but
        low crawl leads north.";
    }
    south = Plain_Center
    north = By_Pentagram
    up = Plain_Center
    pentagram = By_Pentagram
;

By_Pentagram: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Pentagram Room"
    ldesc = { I(); "You're in a small room with a very smooth rock floor,
        onto which has been marked a pentagram.  A low crawl
        leads out to the west, and a crack in the rock leads
        north.  ";
    }
    west = Nondescript
    north = Chimney
    out = Nondescript
    nondescript = Nondescript
    crack = Chimney
    chimney = Chimney
;
Floor_Pentagram: fixeditem, beditem
    game550 = true
    sdesc = "pentagram"
    ldesc = { "It's just a pentagram marked on the floor.  ";
        self.heredesc;
        if (itemcnt( self.contents )) {
            P(); I(); "In "; self.thedesc; " you see ";
            listcont( self ); ". ";
        }
    }
    heredesc = {
        if(Djinn.location = self) {
            P(); I();"There is a twelve-foot
            djinn standing in the center of the pentagram,
            glowering at you.  ";
        }
    }
    location = By_Pentagram
    noun = 'pentagram'
    reachable = ([]+self)
    onroom = nil // say IN the pentagram, not ON
    verGrab (obj) = {
        if(Djinn.isIn(self)) "Right now, the pentagram's magic barrier 
            prevents you from taking any objects it contains. ";
    }
    out = {return self.location;}
    verDoOpen(actor) = {}
    doOpen(actor) = {
        if (Djinn.isIn(self))
        { P();
        "The pentagram's magical barrier sparks fitfully and goes down.  The
        Djinn stretches gratefully and smiles at you.  \"AGAIN, MY THANKS,\"
        he says.  \"MY ADVICE TO YOU WILL TAKE THE FORM OF A HISTORY LESSON.
        WHEN RALPH WITT, THE ARCHITECT AND CONSTRUCTOR OF THIS CAVE, WAS VERY
        YOUNG, HE BECAME VERY INCENSED THAT HIS NAME WAS AT THE END OF THE
        ALPHABET.  HE FELT (FOR SOME REASON) THAT THE LETTER W BELONGED NEAR
        THE BEGINNING OF THE ALPHABET, AND THAT ALL OF THOSE \"UPSTART LETTERS
        WHICH UNFAIRLY USURPED THE BEST PLACES\" SHOULD BE FORCED INTO EXILE
        AT THE END OF THE ALPHABET.  HIS INSTINCT FOR MATTERS MAGICAL AND
        MYSTICAL LED HIM TO APPLY THIS STRANGE BELIEF INTO THE CAVE'S
        STRUCTURE WHEN HE EXCAVATED IT.  YOU HAVEN'T YET BEEN AFFECTED BY HIS
        STRANGE HABITS, BUT YOU SHOULD REMEMBER THIS.  FAREWELL, AND GOOD
        LUCK.\"  With that, the Djinn evaporates into a cloud of smoke and
        drifts rapidly away.\n";
        if(not phugggVerb.isused)
             // Don't mention this if it's already been used.
             notify(Djinn, &phugggtell, Djinn.phugggtime);

        Djinn.moveInto(nil);
        }
        else "The pentagram is empty - there's nothing to
            let out!  ";
    }
    verIoPutIn(actor) = {
        if (Djinn.isIn(self))
            "You can't put things in the pentagram right now. ";
        else inherited.verIoPutOn(actor);
    }
    ioPutIn(actor, dobj) = {
        "You have set "; dobj.thedesc; " down in the center of the
        pentagram.  ";
        dobj.doPutOn(actor, self);
    }
    verIoThrowAt(actor) = {}
    verIoThrowTo(actor) = {}
    ioThrowTo(actor, dobj) = {
        self.ioThrowAt(actor, dobj);
    }
    ioThrowAt(actor, dobj) = {
        local o := dobj;
        if (isclass(dobj,contliquid)) o := dobj.mycont;
        if (not Djinn.isIn(self)) {
            "There's not much point in throwing anything
            at an empty pentagram. ";
            return;
        }
        if (o = glass_vial) {
            o.doThrowAt(actor,Djinn);
            return;
        }
        // DJP - we only drop the axe, so we add a
        // comment about catching or dodging it on the
        // rebound.
        "The "; o.sdesc; " rebounds harmlessly
            from the pentagram's magic force
            field. ";
            if(isclass(o,weapon)) {
            "You dodge it as it bounces back towards
            you. "; o.moveInto(droploc(actor));
            }
            else {
            "You catch "; if (o.isThem) 
                "them as they bounce ";
            else
                "it as it bounces "; 
            "back towards you. ";
            }
        P();
        "It's just as well - the djinn doesn't seem
        dangerous. ";
    }
    verDoSiton(actor) = {
        if (Djinn.isIn(self)) {
            "You can't enter the pentagram right now.  ";
        }
        else pass verDoSiton;
    }
    verDoStandon(actor) = {
        if(actor.posture != 'standing') return;
        if (actor.location = self)
        {
            "%You're% already on "; self.thedesc; "! ";
        }
    }
    doStandon(actor) = {
        "Okay, %you're% now standing "; 
        if (self.onroom) "on "; else "in "; 
        self.thedesc; ".   Nothing unexpected happens.";
        actor.moveInto(self);
        actor.posture := 'standing';
    }
    verIoPutOn(actor) = {
        self.verIoPutIn(actor);
    }
    ioPutOn(actor,dobj) = {
        self.ioPutIn(actor,dobj);
    }
    verDoBoard(actor) = {self.verDoStandon(actor);}
    doBoard(actor) = {self.doStandon(actor);}
    verDoEnter(actor) = {self.verDoStandon(actor);}
    doEnter(actor) = {self.doStandon(actor);}
    isdroploc = true   // Items are dropped on the pentagram
    default_posture = 'standing'
;

Chimney: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "End of Crack, at Bottom of Chimney"
    ldesc = { I(); "The crack in the rock ends here, but a narrow
        chimney leads up.  You should be able to climb it.";
    }
    south = By_Pentagram
    up = In_Tube
    climb = In_Tube
    tube = In_Tube
    pentagram = By_Pentagram
    crack = By_Pentagram
;

In_Tube: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Lava Tube at Top of Chimney"
    ldesc = { I(); "You're at the top of a narrow chimney in the rock.
        A cylindrical tube composed of hardened lava leads
        south.";
    }
    south = Tube_Slide
    down = Chimney
    climb = Chimney
    chimney = Chimney
    tube = Tube_Slide
    slide = Tube_Slide
;

Tube_Slide: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Steep Slide in Tube"
    ldesc = { I(); "The lava tube continues down and to the south, but
        it becomes very steep - if you go down it you
        probably won't be able to get back up.";
    }
    north = In_Tube
    south = { "Wheeeeee......     >oof<"; P(); return South_Fog; }
    down = self.south
    slide = self.south
    chimney = Chimney
    tube = Chimney
;

South_Basilisk: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Rough, Narrow Passage"
    ldesc = { I(); "You are in a narrow and rough passage running
        north and south.  A dull rumbling sound can be
        heard from the south.";
    }
    south = By_Figure
    north = { // stir the basilisk, but not the statue.
        if(Basilisk.isIn(self)) "The basilisk stirs restlessly
            and grumbles in its sleep as you pass, but
            it does not awaken."; P();
        return North_Basilisk;
    }
    exithints = [ North_Basilisk, &north ]
;

North_Basilisk: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "North of Basilisk"
    ldesc = { I(); "You're in a rough passage to the north of the
        basilisk's den.";
    }
    south = {
        local actor := getActor(&travelActor);
        if (Basilisk.isIn(self)) { // Do neither if it's a statue.
            if (metal_plate.location =  actor) {
                Basilisk.petrified;
                self.playerstoned:=nil;
            }
            else {
                Basilisk.petrifier;
                self.playerstoned:=true;
            }
        } // Note that a dead player should not be moved, since
          // it won't occur until after the resurrection.
        if(not self.playerstoned) return South_Basilisk;
        else return nil;
    }
    north = Basilisk_Fork
    exithints = [ South_Basilisk, &south ]
;

Basilisk_Fork: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Fork by Steps"
    ldesc = { I(); "The passage here enters from the south and divides,
        with a wide tunnel exiting to the north and a set of
        steps leading downward.";
    }
    north = Peelgrunt
    south = North_Basilisk
    down = On_Steps
    steps = On_Steps
    peelgrunt = Peelgrunt
;

Peelgrunt: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Peelgrunt Room"
    ldesc = { I(); "You are in the Peelgrunt room.";}
    south = { // Can't walk through the safe door.
        if (In_Safe.isopen) { "The safe's door is blocking the exit
            passage - you'll have to close it to get out
            of here.";
            return nil;
        }
        else return Basilisk_Fork;
    }
    out = { return self.south; }
    fork = { return self.south; }
    in = {
        if (In_Safe.isopen) return In_Safe; 
        else "The safe's door is closed, and you can't get in!";
        return nil;
    }
    safe = {return self.in;} // DJP
    west = {return self.in;} // DJP - logically the direction should be
                             // east or west; arbitrary choice made for west.
    // DJP - added a cantReachRoom
    cantReachRoom(otherRoom) = {
        if (toplocation(otherRoom) = In_Safe) {
            "You can't reach that from outside the safe. ";
        }
        else pass cantReachRoom;
    }
    exithints = [ Basilisk_Fork, &out,  In_Safe, &in ]
;

On_Steps: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "On the Steps"
    ldesc = { I(); "You are on a long, spiral set of steps leading
        downwards into the earth.";
    }
    up = Basilisk_Fork
    down = Steps_Exit
    steps = Steps_Exit
;

Steps_Exit: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Exit on Steps"
    ldesc = { I(); "A small tunnel exits from the steps and leads north.
        The steps continue downwards.";
    }
    north = Fake_Y2
    up = On_Steps
    down = Storage
    steps = Storage
    // exit = Fake_Y2   (since "exit" clashes with the keyword.)
;

Storage: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Storage Room"
    ldesc = { I(); "You're in what was once a storage room.  A set of
        steps leads up.";
    }
    up = Steps_Exit
    steps = Steps_Exit
;

Fake_Y2: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "At \"Y2\"?"
    ldesc = {
        local actor := getActor(&verbActor);
        I(); "You are in a large room, with a passage to the
        south, a passage to the west, and a wall of broken
        rock to the east. There is a large \"Y2\" on ";

        if (actor.isIn(Fake_Y2Rock))
             "the rock you are sitting on.";
        else
             "a rock in the room's center.";
    }
    plugh = Volcano_Platform
    south = Steps_Exit
    east = Fake_Jumble_Of_Rock
    wall = Fake_Jumble_Of_Rock
    broken = Fake_Jumble_Of_Rock
    west = { Catacombs.roomnumber := 1; return Catacombs; }
    plover = Volcano_Platform

    hollowvoice = {
        if (not parserGetMe().isIn(self))return;
        if (rand(100) <= 25) {
             P(); I(); "A hollow voice says, \"Plugh.\"\n";
        }
    }

;
Fake_Y2Rock: fixeditem, chairitem, readable
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "\"Y2\" rock"
    ldesc = "There is a large \"Y2\" painted on the rock."
    noun = 'rock'
    adjective = 'y2'

    location = Fake_Y2

    plugh = { return self.location.plugh; }

    onroom = true   // We set ON the rock, not IN it.

    //
    // We want the player to be able to pick things in the
    // room up while sitting on the rock.
    reachable = {return self.location.allcontents;}
    // This makes the contents of containers accessible as well:
    canReachContents = true
;

Fake_Jumble_Of_Rock: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Jumble of Rock"
    ldesc = {
        I(); "You are in a jumble of rock, with cracks
        everywhere.";
    }
    down = Fake_Y2
    y2 = Fake_Y2
    up = { Catacombs.roomnumber := 1; return Catacombs; }
;

// Note how the catacombs are implemented!  There is really only one room,
// but it looks like 19 rooms to the player.  The relevant properties and
// methods are:
//
// roomnumber: The catacomb number (which in the 550-point game goes up
// if you choose the correct direction or down for the wrong direction.)
// oldnumber: The old value of roomnumber (for the leaveroom method).
//
// leaveroom: This method is now called by the standard leaveRoom method, and
// causes objects in the room are moved into Elsewhere.  They are 'tagged'
// with the room number so they can be reinstated when the same .
//
// enterroom: This method is called by the standard enterRoom method.
// It updates oldnumber and reinstates the correct room contents.

Catacombs: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Catacombs"
    ldesc = { I(); "You are in the catacombs.  Enchanted tunnels lead
        in all directions.";
        if(global.debug)
        "  You are in room number <<self.roomnumber>>.";
    }
    roomnumber = 1
    oldnumber = 1
    maxnumber = 0   // highest number seen by player
    leaveroom() = { // called whenever the player leaves one room.
        local i,o,c,l;
        c := self.contents;
        l := length(c);
        for (i:=1; i <= l; i++) {
            o := c[i];
            if (o.isfixed) continue; // DJP - exclude fixed items
            // Move everything out so it doesn't show up
            // in the next room. Only alters objects directly
            // in the catacombs of course, not in the player.
            //
            o.moveInto(Elsewhere);
            o.catac_room_num := self.oldnumber;
        }
    }
    leaveRoom(actor) = {
        self.leaveroom;
        pass leaveRoom;
    }
    enterroom() = {
        local i,o,c,l; // Update oldnumber and reinstate items
        c := Elsewhere.contents;
        l := length(c);
        // DJP - make the room look 'unseen' if it should be.
        if (self.roomnumber > self.maxnumber) {
             self.isseen := nil;
             self.maxnumber := self.roomnumber;
        }
        self.oldnumber := self.roomnumber;
        for (i:=1; i <= l; i++) {
            o := c[i];
            if (o.isfixed) continue; // DJP - exclude fixed items
            //
            // Move everything with the appropriate room
            // number into the new room.
            //
            if(o.catac_room_num = self.roomnumber)
                o.moveInto(self);
        }
    }
    enterRoom(actor) = {
        self.enterroom;
        pass enterRoom;
    }

    // For the 550-point game, the directions through the Catacombs
    // are fixed.  Note that this is not the case for the 660-point
    // game, which will need rather different coding.  The 'correct'
    // moves through the catacombs are chosen by the player on his
    // first trip through them!

    north = {  // Direction to next room varies with the room number
        if (self.roomnumber = 8 or self.roomnumber = 10 or
        self.roomnumber = 19) self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        if (self.roomnumber = 20) return Fake_Y2;
        // in last room, leads out.
        return self;     // leads to previous room in catacombs
    }
    south = {
        if (self.roomnumber = 1 or self.roomnumber = 4 or
        self.roomnumber = 9 or self.roomnumber = 18)
        self.roomnumber++;
        else self.roomnumber--;
        return self;
    }
    east = {
        if (self.roomnumber = 11) return West_Audience;
        if (self.roomnumber = 12 or self.roomnumber = 15)
        self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    west = {
        if (self.roomnumber = 6) self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    nw = {
        if (self.roomnumber = 7 or self.roomnumber = 3)
                       self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    ne = {
        if (self.roomnumber = 14) self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    se = {
        if (self.roomnumber = 13 or self.roomnumber = 16)
                       self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    sw = {
        if (self.roomnumber = 2 or self.roomnumber = 11)
                       self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    down = {
        if (self.roomnumber = 5 or self.roomnumber = 17)
                       self.roomnumber++;
        else if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }
    up = {
        if (self.roomnumber <> 1) self.roomnumber--;
        return self;
    }   // None of the rooms use up as the passage to the next room!
    back = {
        "The directions are particularly confusing in here, and
        the rooms all look alike.  You'll have to tell me how to go
        back. ";
        return nil;
    }
;

Elsewhere: darkroom, NoNPC // So compiler won't complain about lack of exits.
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Elsewhere in Catacombs"
    ldesc = "A repository for objects removed from the Catacombs.
        You should never actually get here."
;

West_Audience: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "West Audience Hall"
    ldesc = { I(); "You are standing at the west end of the royal
        Audience Hall.  The walls here are composed
        of the finest marble, and the floor is built
        of slabs of rare onyx and bloodstone.  The
        ceiling is high and vaulted, and is supported
        by pillars of rare Egyptian red granite; it
        gives off a nacreous glow that fills the
        entire chamber with a light like moon-light
        shining off of polished silver.";
    }
    west = Catacombs  // roomnumber should still be 11.
    east = East_Audience
;

East_Audience: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "East Audience Hall"
    ldesc = {
        I(); "You are at the eastern end of the Audience Hall.
        There is a large dais rising out of the floor
        here;  resting upon the dais is a strange-looking
        throne made out of interlocking bars and rods of
        metal."; P();
        if ( Skeleton.isIn(self) ) {
            I(); "Resting on
            the throne (\"sitting\" isn't really the
            right word) is an incredible skeleton.  It
            is fairly humanoid from the waist up (except
            for its incredible size and four extra arms);
            below that, it resembles the body of a giant
            python, and is wrapped in and around the bars
            and rods of the throne.  Clutched in one bony
            hand is a long sceptre, ornately encrusted
            with sapphires!!";
        }
    }
    west = West_Audience
;
Serpent_Throne: fixeditem  // Not to be confused with the Throne of the
    game550 = true         // Mountain King.
    sdesc = "throne"
    ldesc = "The throne is composed of several interlocking bars and rods,
        as though its owner had a serpentine body."
    noun = 'throne'
    adjective = 'strange' 'strange-looking' 'metal'
    location = East_Audience
    verDoEnter = { "It's not built for humans."; }
    verDoBoard = { "It's not built for humans."; }
;
Skeleton: fixeditem, qcontainer
    game550 = true
    sdesc = "skeleton"
    ldesc = "Below the waist, the skeleton resembles that of a giant
        python, and is wrapped carefully through the throne.  Above
        the waist, it resembles the skeleton of a giant human with
        six arms.  In one of these arms is a long, sapphire-encrusted
        sceptre!"
    location = East_Audience
    noun = 'skeleton' 'python'
    adjective = 'strange' 'skeletal' 'python' 'six-armed'
  // 'python' is okay since this is the only thing in the game with that name,
  // right?
    verIoPutIn(actor) =
    {"I don't know how to put anything into <<self.thedesc>>. ";}
;
Winding_Pass: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Winding Passage"
    ldesc = { I(); "You are in a winding passage which enters from
        the northwest, loops around several times, and
        exits to the north.";
    }
    north = Golden
    golden = Golden
    nw = By_Figure
;

Golden: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Golden Chamber"
    ldesc = { I(); "You are in a chamber with golden walls and a high
        ceiling.  Passages lead south, northeast, and
        northwest.";
    }
    south = Winding_Pass
    nw = Translucent
    ne = Arabesque
    arabesque = Arabesque
    translucent = Translucent
;

Translucent: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Translucent Room"
    ldesc = { I(); "You are in a large room whose walls are composed of
        some translucent whitish material.  The room is
        illuminated by a flickering reddish glow shining
        through the southern wall.  A passage leads east.";
    }
    east = { // Summon goblins.
        local actor := getActor(&travelActor);
        if(not self.gobs_called and bracelet.isInside(actor)) {
            self.gobs_called := true; // Don't summon them twice.
            Goblins.summon(Golden);
        }
        return Golden;    // In either case.
    }
    out = { return self.east; }
    golden = { return self.east; }
    exithints = [ Golden, &east ]
;

Arabesque: darkroom, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Arabesque Room"
    ldesc = { I(); "You are in a small room whose walls are covered
        with an elaborate pattern of arabesque figures
        and designs.";
    }
    south = Golden
    out = Golden
    golden = Golden
;

Volcano_Platform: room, NoNPC
    game550 = true
    wino_wsbridge = true // can only exit via stone bridge
    sdesc = "Platform above Volcano"
    ldesc = { I(); "You are precariously perched on a tiny platform
                    suspended in midair.  Two thousand feet below
                    you is the mouth of a very active volcano,
                    spewing out a river of hot lava.";
    }
    plugh = Fake_Y2
    plover = Fake_Y2
    north = { "EEEEEEEEAAAAAAAAAAAaaaaahhhhhhhh...........
             >sizzle<";
        die();
        return nil;
    }
    south = { return self.north; }
    east = { return self.north; }
    west = { return self.north; }
    nw = { return self.north; }
    sw = { return self.north; }
    ne = { return self.north; }
    se = { return self.north; }
    down = { return self.north; }
    jump = { return self.north; }
    climb = { return self.north; }
;
Volcano_Below: fixeditem, distantItem
    sdesc = "active volcano"
    ldesc = "Two thousand feet below you is the mouth of a very
        active volcano, spewing out a river of hot lava."
    noun = 'volcano' 'lava' 'river'
    adjective = 'active' 'hot' 'lava'
    location = Volcano_Platform
;

Sandstone_Chamber: darkroom
    game550 = true
    sdesc = "Sandstone Chamber"
    ldesc = { I(); "You are in a small chamber to the east of the Hall
        of Mists.  The walls are composed of rough red
        sandstone.  There is a large, cubical chunk of rock
        in the center of the room.";
    }
    west = In_Hall_Of_Mists
    out = In_Hall_Of_Mists
    mists = In_Hall_Of_Mists
    ana2 = Blue_Sandstone_Chamber
;

class Stoneclass: fixeditem, chairitem, owntrans // DJP added chairitem etc.
    sdesc = "chunk of rock"
    ldesc = { "It's a large, cubical chunk of rock sitting in the center
        of the room.";
    }
    noun = 'rock' 'stone' 'chunk' 'block'
    verDoLookin(actor) = {}
    doLookin(actor) = {self.doSearch(actor);}
    doLookon(actor) = {self.doSearch(actor);}
    adjective = 'large' 'cubical'
    //
    // We want the player to be able to pick up objects while sitting on the 
    // rock.
    reachable = {return self.location.allcontents;}
    // This makes the contents of containers accessible as well:
    canReachContents = true
;

Stone: Stoneclass
    game550 = true
    ldesc = { 
        inherited.ldesc;
        if (singing_sword.isStuck) 
             "  There is a
             sword plunged deep into the rock, singing quietly.";
    }
    location = Sandstone_Chamber
    doSearch(actor) = {
        local itc := itemcnt(self.contents);
        if (singing_sword.isStuck) { 
             "There is a sword plunged deep into the rock, singing quietly.  ";
             if(itc > 0) "You also see ";
        }
        else {
             if(itc = 0) "There's nothing on the rock. ";
             else "On the rock you see ";
        }
        if(itc > 0) {
             listcont(self);". ";
        }
    }
    ana2 = Blue_Stone
;

Morion: darkroom
    game550 = true
    sdesc = "Morion Room"
    ldesc = { I();
        "You are in a small room.  The walls are composed of
        a dark, almost black form of smoky quartz; they
        glisten like teeth in the lamp-light. ";
        if (global.game701)
            "The exits are the passage to the south through
            which you entered, and a narrow passage leading northeast. ";
        else
            "The only exit is the passage to the south through which you
            entered. ";
    }
    south = In_Hall_Of_Mt_King
    out = In_Hall_Of_Mt_King
    ne = {
        if (global.game701) return Throne_Room;
        else pass ne;
    }

    NPCexit1 = {
        if (global.game701) return Throne_Room;
        else return nil;
    }

    exithints = {
        if (global.game701) return [Throne_Room, &ne];
        else pass exithints;
    }
;

Vault: darkroom
    game550 = true
    sdesc = "Vaulted Room"
    ldesc = { I(); "You are in a room with a high, vaulted ceiling.  A
        tunnel leads upwards and to the north. ";
    }
    north = {
        if (In_Safe.isopen) "The safe's door is blocking the exit
            passage - you'll have to close it to get out
            of here.";
        else return In_Hall_Of_Mt_King;
        return nil;
    }
    up = { return self.north; }
    in = {
        if (In_Safe.isopen) return In_Safe; 
        else "The safe's door is closed, and you can't get in!";
        return nil;
    }
    safe = {return self.in;} // DJP
    east = {return self.in;} // DJP - logically the direction should be
                             // east or west; arbitrary choice made for east.

    // No need for an NPC exit to do this, since the room
    // is closed to NPCs.

    // DJP - added a cantReachRoom
    cantReachRoom(otherRoom) = {
        if (toplocation(otherRoom) = In_Safe) {
            "You can't reach that from outside the safe. ";
        }
        else pass cantReachRoom;
    }
    exithints = [ In_Safe, &in,  In_Hall_Of_Mt_King, &up ]
    myhints = [ Vaulthint ]

    // Let NPCs out of here.
    NPCexit1 = In_Hall_Of_Mt_King 
    // NPCs can enter the safe, but only if it opens to the Vault.
    NPCexit2 = {
        if(In_Safe.out = self) return In_Safe;
        else return nil;
    }
;

In_Safe: darkroom, Internal_Room, floatingdecoration
    // DJP - added Internal_Room class and fixed location method.
    // DJP - added detailed description of walls and doors, added provision
    //       for viewing when closed (using sapphire in 701-point game)
    // Don't allow wine to be drunk if the safe has been entered from beyond
    // the Wheatstone Bridge. 
    wino_wsbridge = {
        if(out) {
            if(out.wino_wsbridge) return true;
            else return nil;
        }
        else return nil;
    }       
    game550 = true      // This safe is not to be confused with Safe
    sdesc = "In the Safe"
    multisdesc = "safe" // Used for actions on multiple objects
    bothdoors = nil // If true, the player has noticed that there are two 
                    // doors.
    // The description of the safe as a decoration
    // differs from its room description, and there is a special method
    // for 'x safe'
    inldesc = {
        I(); 
        "You are inside the safe.  ";
        switch (self.out) {
            case nil:
                if(self.bothdoors) "Both doors are firmly shut. ";
                else "The door is firmly shut.  So firmly, in fact, that
                     you can only just make out its outline in the 
                     wall. ";
                break;
            case Vault:
                if(self.bothdoors) "The west door is open. ";
                else "The door is open and exits to the west. ";
                break;
            case Peelgrunt:
                if(self.bothdoors) "The east door is open. ";
                else "The door is open and exits to the east. ";
                break;
        }
    }
    outldesc = {
        if (not self.isopen) "It's a large metal safe, only one wall of
             which is visible.  ";
        else "It's a large metal walk-in safe, the door of which has been swung
            open.  "; 
    }
    doInspect(actor) = {
        if (actor.isIn(self)) SafeWalls.doInspect(actor);
        else pass doInspect;
    }
    heredesc = {
        P();
        I(); "A massive walk-in safe takes up one entire wall.  ";
        if (In_Safe.isopen) "Its door has been swung open and blocks
            the exit passage.";
        else if (In_Safe.is_fused) "It is closed, and there are signs
            of melting around the edges of its door.";
        else "It is tightly closed, and has no handle, lock, or
            keyhole.";
    }
    out = nil // Initially. When the safe is opened, it will change to
              // the appropriate room.
    loclist = [ Peelgrunt  Vault ]
    location = {
        local actor := getActor(&currentActor);
        if ( actor.isIn(self) ) return self.out; // IMPORTANT: prevent
                                                 // infinite loop.
        else pass location;
    }
    // Added directions, based on an arbitrary choice.
    east = {
        if (self.out = Peelgrunt) return Peelgrunt;
        else pass east;
    }
    west = {
        if (self.out = Vault) return Vault;
        else pass west;
    }
    password = nil  // Initially. When sceptre is taken, it changes to
                    // the number for the appropriate word.
    opens = { ">ker-THUNK<\t>screeeeeeeeeech<";
        P(); "The (somewhat rusty) safe is now open.";
        self.isopen := true;
        self.out := self.location;
    }
    closes = {
             ">Creeeeeeeeeeeeeeeeeek<    >ker-CHUNK!<";
             P();  "The safe is now closed. ";
             self.isopen := nil;
             self.out := nil; // for viewing with crystal ball
    }
    melts = {P();
        ">bong<\t\t\t\t\t\tThe very air quivers with sound as though\n";
        "\ \ >bong<\t\t\t\t\tsomeone, somewhere in the distance, has struck\n";
        "\t\ >bong<\t\t\t\t\tthree powerful blows on an immense brass gong.\n";
        P(); "Smoke trickles out from around the edges of the
        safe's door, and the door itself glows red with
        heat for a moment."; P();
        "A hollow voice says,  \"This is a Class 1 security
        alarm.  All cave security forces go to Orange Alert.
        I repeat - Orange Alert.\"";
        self.is_fused := true;
        // Add flag so that we can test whether an alert has been issued,
        // before the blob summoning code is executed.  This flag is reset
        // in the die() routine.
        global.triggered_alert := true;
        notify(Blob, &summon, 2); // Two turns before blob is
                                  // actually summoned.
    }
    noun = 'safe' 'door' // for open door 
    adjective = 'large' 'metal' 'walk-in' // DJP added walk-in
    verDoClose(actor) = {
        if (not In_Safe.isopen) "The safe is already closed!";
    }
    verDoOpen(actor) = {
        if (In_Safe.isopen) 
             "The safe is already open! ";
        else if (In_Safe.is_fused) 
            "The door to the safe seems to be fused shut - I can't open it. ";
        else 
            "The door to the safe has no keyhole, dial,
            or handle - I can't figure out how to open it! ";
    }
    verDoUnlock(actor) = {self.verDoOpen(actor);}
    verDoLock(actor) = {self.verDoClose(actor);}
    doLock(actor) = {self.doClose(actor);}
    verDoLockWith(actor,io) = {
        if (self.isopen) {
            "I think you could lock this safe simply by closing it. ";
        }
        else "It's already locked.";
    }

    verDoUnlockWith(actor,io) = {self.verDoOpen(actor);}
    verDoOpenWith(actor,io) = {
        if ((io = cups) and not self.isopen) 
            "You pull the safe door with all your might, but 
            it remains firmly shut. ";
        else self.verDoOpen(actor);
    }
    verDoCloseWith(actor,io) = {
        self.verDoClose(actor);
    }
    doClose(actor) = {
        if (actor.isIn(self)) {
            if (cups.isVisible(actor))
                self.doCloseWith(actor,cups);
            else
                "There is no handle on the inside of the safe door, nor any 
                other way to get a firm enough grip on it.  You'll have to 
                leave the safe before shutting it. ";
        }
        else self.closes;    
    }
    doCloseWith(actor,io) = {
        if (actor.isIn(self) and (io = cups) and self.isopen) 
            "Even with the suction cups, you find that you are unable to 
            pull the door hard enough to close it fully.  You'll have to 
            leave the safe before shutting it. ";
        else self.doClose(actor);  
    }
    verDoPush(actor) = {
        if (actor.isIn(self)) 
            "You'll have to push from the
            other side - there's no handle on the inside 
            of the safe door, nor any other
            way to get a firm enough grip on it from here. ";
        else
            self.verDoClose(actor);
    }
    doPush(actor) = {
        self.closes;
    }

    verDoPull(actor) = {
        if (actor.isIn(self)) "There is no handle on the
             inside of the safe door, nor any other
             way to get a firm enough grip on it.  You'll have
             to leave the safe before shutting it. ";
        else self.verDoOpen(actor);
    }
    verDoPullWith(actor,io) = {
        if (io != cups) {
             "I'm not sure how that would help. ";
             return;
        }
        if (actor.isIn(self)) 
             self.verDoCloseWith(actor,io);
        else
             self.verDoOpenWith(actor,io);
    }
    doPullWith(actor,io)= {
        if (actor.isIn(self))
             self.doCloseWith(actor,io);
        else
             self.doOpenWith(actor,io);
    }
    verIoPutOn(actor) = {}
    ioPutOn(actor,dobj) = {
        local outhideStatus := outhide(true);
        self.verDoPullWith(actor,dobj);
        if (outhide(outhideStatus)) {
            self.verDoPullWith(actor,dobj);
            return;
        }
        else self.doPullWith(actor,dobj);
    }
    verIoAttachTo(actor) = {}
    ioAttachTo(actor,dobj) = {
        self.ioPutOn(actor,dobj);
    }

    isopen = nil   // Closed at start of game.
    is_fused = nil   // In case wrong password is used.

    doSearch(actor) =
    {
        if (itemcnt(self.contents) <> 0) {
            "There's a list of words hanging on one wall, and the safe also 
            contains ";
            listcont(self); ". ";
        }
        else {
            "It's empty,  except for the list on 
            the wall. ";
        }
    }

    doEnter(actor) = { 
        actor.travelTo(self.location,&in);
    }

    // Special reachability check.  From outside the safe, we can only 
    // reach objects which were put in there from the same side.
    special_cantreach(obj,actor,chain) = {
        local topcont, actor_side, obj_side;
        // check that the actor is outside the safe and that the object
        // to be tested isn't the room itself
        if (not actor.isIn(self) and obj <> self) {
            topcont := chain[2]; /* top-level container of obj */
            if(toplocation(actor) <> topcont.metoploc) return true;
        }
        return nil;
    }
    enterRoom(actor) = {
        delword(In_Safe,&noun,'door');
        if (self.out=Vault) {
             SafeDoor_W.moveInto(self);
             if (self.bothdoors) SafeDoor_E.moveInto(self);
        }
        else {
             SafeDoor_E.moveInto(self);
             if (self.bothdoors) SafeDoor_W.moveInto(self);
        }
        pass enterRoom;
    }
    leaveRoom(actor) = {
        addword(In_Safe,&noun,'door');
        SafeDoor_W.moveInto(nil);
        SafeDoor_E.moveInto(nil);
        pass leaveRoom;
    }

    // Let NPCs out into the Vault.  (Peelgrunt is off-limits).
    NPCexit1 = Vault
;

class SafeDoor_inside: fixeditem
    noun = 'door'
    plural = 'doors'
    ldesc = {
        if (self.isopen) {
            caps(); self.thedesc; " is open, but I don't think you'll be
            able to close it from inside the safe. ";
        }
        else {
            caps(); self.thedesc; " is firmly closed, and I can't figure
            out any way to open it. "; 
        }
    }
    isopen = (In_Safe.out = self.myout)
    verDoOpen(actor) = {
        if(self.isopen)In_Safe.verDoOpen(actor);
        else "I can't figure out any way to open <<self.thedesc>>.";
    }
    verDoClose(actor) = {
        if(self.isopen)In_Safe.doClose(actor);
        else "It's already closed. ";
    }
    verDoCloseWith(actor,io) = {
        if(self.isopen)In_Safe.doCloseWith(actor,io);
        else "It's already closed. ";
    }
    verDoPush(actor) = {
        if(self.isopen)In_Safe.verDoOpen(actor);
        else "You push <<self.thedesc>> with all your might, but it remains
        firmly shut. ";
    }
    verDoPull(actor) = {
        if(self.isopen)In_Safe.doClose(actor);
        else "There's no point - it opens outwards. ";
    }
    verDoPullWith(actor,io) = {
        if(self.isopen or (io != cups))In_Safe.verDoPullWith(actor,io);
        else "There's no point - it opens outwards. ";
    }
    doPullWith(actor,io) = {
        In_Safe.doPullWith(actor,io);
    }
    verIoPutOn(actor) = {}
    ioPutOn(actor,dobj) = {
        if(self.isopen or (dobj != cups))In_Safe.ioPutOn(actor,dobj);
        else "There's no point - it opens outwards. ";
    }
    verIoAttachTo(actor) = {}
    ioAttachTo(actor,dobj) = {
        if(self.isopen or (dobj != cups))In_Safe.ioAttachTo(actor,dobj);
        else "There's no point - it opens outwards. ";
    }
    verDoEnter(actor) = {}
    verDoBoard(actor) = {}
    doBoard(actor) = {self.doEnter(actor);}
;
SafeDoor_W: SafeDoor_inside

    game550 = true
    sdesc = {if (In_Safe.bothdoors) "west door"; else "door";}
    adjective = 'west' 'w' 'western'
    myout = Vault
    // kludge to get round 'push west door' being interpreted as
    // (push west) door, not push (west door).
    verDoMoveW(actor) = {}
    doMoveW(actor) = {
        local outhideStatus := outhide(true);
        self.verDoPush(actor);
        if(outhide(outhideStatus)) {
            /* verification failed - run again showing the message */
            self.verDoPush(actor);
        }
        else
            /* verification OK - do a simple push */
            self.doPush(actor);
    }    
    doEnter(actor) = {actor.travelTo(self.location,&west);}
;

SafeDoor_E: SafeDoor_inside

    game550 = true
    sdesc = {if (In_Safe.bothdoors) "east door"; else "door";}
    adjective = 'east' 'e' 'eastern'
    myout = Peelgrunt
    // kludge to get round 'push east door' being interpreted as
    // (push east) door, not push (east door).
    verDoMoveE(actor) = {}
    doMoveE(actor) = {
        local outhideStatus := outhide(true);
        self.verDoPush(actor);
        if(outhide(outhideStatus)) {
            /* verification failed - run again showing the message */
            self.verDoPush(actor);
        }
        else
            /* verification OK - do a simple push */
            self.doPush(actor);
    }    
    doEnter(actor) = {actor.travelTo(self.location,&east);}
;

List: fixeditem, readable // BJS: Added this object, to reveal the
                          // unused keywords.
    game550 = true
    sdesc = "list of words"
    heredesc = { P(); I(); "A list of words hangs on one wall."; }
    // DJP - make 'x list' the same as 'read list'
    ldesc = {
        local i; "It says: \"zorton snoeze knerl klaetu blerbi\"";
        if (In_Safe.password) {
             "\nThere is a marking next to \"";
             // Now print the vault keyword, using its sdesc.
             for(i:=firstobj(VaultKeyVerb); i;  i:=nextobj(i, VaultKeyVerb))
                 if(i.wordnum = In_Safe.password) i.sdesc;
             ".\"\n";
        }
    }
    location = In_Safe
    noun = 'list' 'words' 'word'
    adjective = 'word' 'words'
    verDoTake(actor) = {
        "The list is attached so firmly to the wall that
        it won't come off.";
    }
;

SafeWalls: fixeditem, floatingItem
    game550 = true
    sdesc = "walls"
    ldesc = { 
        I();"The inner walls of the safe are almost featureless";
        if (In_Safe.out = nil) // only when viewed with sapphire
            ", but close examination reveals the outlines of doors in the 
            east and west walls.  Both are firmly shut. ";
        else {
            ", but a large door has been opened in the ";
            if (In_Safe.out = Peelgrunt) "east"; else "west";
            " wall.  At first glance the ";
            if (In_Safe.out = Peelgrunt) "west"; else "east";
            " wall looks completely blank, but close examination reveals 
            the outline of a second door, identical to the first - but firmly 
            closed. ";
        }
        In_Safe.bothdoors := true;
        SafeDoor_E.moveInto(In_Safe);
        SafeDoor_W.moveInto(In_Safe);
    }
    noun = 'walls' 'wall'
    adjective = 'north' 'south' 'east' 'west'
    doCount = {"As you might expect, the safe has four walls. ";}
    // seen inside safe only
    location = {
        local actor := getActor(&currentActor);
        if (actor.isIn(In_Safe)) return In_Safe; 
        else return nil;
    }
    locationOK = true
    noun = 'walls'
;

Corridor_1: darkroom
    game550 = true
    sdesc = "Wide, North-and-South Corridor"
    ldesc = { I(); "You are standing in a wide, north-and-south corridor.";
    }
    north = Corridor_2
    south = In_Hall_Of_Mt_King
;

Corridor_2: darkroom
    game550 = true
    sdesc = "Bend in Wide Corridor"
    ldesc = { I(); "You are standing at a bend in a wide corridor which
        runs to the east and west.  To the east, the corridor
        turns left and continues north; to the west, it turns
        left and continues south.";
    }
    north = Tool_Room
    south = Corridor_1
    west = Corridor_1
    east = Tool_Room
    sw = Corridor_1  // BJS: Added.
    ne = Tool_Room   //  "" "" ""
;

Tool_Room: darkroom
    game550 = true
    sdesc = "Tool Room"
    ldesc = { I(); "You are in a small, low-ceilinged room with the words,
        \"Witt Company Tool Room - Melenkurion division\"
        carved into one of the walls.  A wide corridor runs
        south from here.";
    }
    south = Corridor_2
    out = Corridor_2
;
ToolMessage: fixeditem, readable
    game550 = true
    sdesc = "carving on the wall"
    ldesc = {
        "The message carved into the wall reads, \"Witt Company
        Tool Room - Melenkurion division.\"";
    }
    noun = 'message' 'carving' 'writing' 'words'
    adjective = 'carved'
    location = Tool_Room
;

Corr_Divis: darkroom
    game550 = true
    sdesc = "Division in Passage"
    ldesc = { I(); "You are at a division in a narrow passage.  Two spurs
        run east and north;  the main passage exits to the
        south.";
    }
    north = Spherical_Room
    south = In_Hall_Of_Mt_King
    east = In_Cubicle
;

In_Cubicle: darkroom
    game550 = true
    sdesc = "Dank Cubicle"
    ldesc = { I(); "You are in a small, dank cubicle of rock.  A small
        passage leads back out to the south;  there is no
        other obvious exit.";
    }
    south = Corr_Divis
    out = Corr_Divis
    corridor = Corr_Divis
;

Spherical_Room: darkroom
    game550 = true
    sdesc = "Spherical Room"
    ldesc = { I(); "You're in a large, completely spherical room with
                    polished walls.  A narrow passage leads out to the
                    north.";
    }
    north = Corr_Divis
    out = Corr_Divis
    corridor = Corr_Divis
;

Glassy_Room: darkroom
    game550 = true
    sdesc = "Large Room with Glassy Walls"
    ldesc = { I(); "You're standing in a very large room (which however
                    is smaller than the Giant room) which has smooth,
                    glassy-looking walls.  A passage enters from the
                    south and exits to the north.";
    }
    south = At_Recent_Cave_In
    north = { if (Ogre.exists) "The ogre growls at you and refuses to
                                let you pass.";
              else return Sorc_Lair;
              return nil;
    }
    lair = { return self.north; }
    NPCexit1 = { return Sorc_Lair; } // As usual, NPCs get by with no trouble.
    exithints = [ Sorc_Lair, &north ]
;

Sorc_Lair: room
    game550 = true
    isbonus = true
    sdesc = "Sorcerer's Lair"
    ldesc = { I(); "This is the Sorcerer's Lair.  The walls are covered
                    with exotic runes written in strange, indecipherable
                    scripts;  the only readable phrase reads \"noside
                    samoht\".  Strange shadows flit about on the walls,
                    but there is nothing visible to cast them.
                    Iridescent blue light drips from a stalactite far
                    above, falls towards the floor, and evaporates before
                    touching the ground.  A deep, resonant chanting
                    sound vibrates from deep in the ground beneath your
                    feet, and a whispering sound composed of the echoes
                    of long-forgotten spells and cantrips seeps from the
                    walls and fills the air.  Passages exit to the east
                    and west.";
    }
    west = Glassy_Room
    east = Brink_North
;
Sorc_Runes: fixeditem, readable
    game550 = true
    sdesc = "exotic runes"
    ldesc = {
            "The walls are covered with exotic runes written in strange,
            indecipherable scripts;  the only readable phrase reads
            \"noside samoht\".";
    }
    noun = 'runes' 'rune' 'words'
    adjective = 'indecipherable' 'exotic'
    location = Sorc_Lair
;

Brink_North: darkroom
    game550 = true
    sdesc = "Brink of Bottomless Pit"
    ldesc = { I(); "You are standing on the brink of what appears to
                    be a bottomless pit plunging down into the bowels
                    of the earth.  Ledges run around the pit to the
                    east and west, and a passage leads back to the
                    north.";
    }
    north = Sorc_Lair
    west = Brink_South
    east = Brink_East
    lair = Sorc_Lair
    jump = { 
        local actor := getActor(&travelActor);
        Bott_Pit.plunge(actor); return nil; 
    }
;

Brink_South: darkroom
    game550 = true
    sdesc = "South Edge of Bottomless Pit"
    ldesc = { I(); "You are standing at the south end of a ledge running
                    around the west side of a bottomless pit.  The ledge
                    once continued around to the east side of the pit,
                    but was apparently obliterated by a rock-slide years
                    ago.  A cold wind blows out of a tunnel leading to
                    the southeast.";
    }
    north = Brink_North
    se = Ice_Room
    jump = { 
        local actor := getActor(&travelActor);
        Bott_Pit.plunge(actor); 
        return nil; 
    }
;

Brink_East: darkroom
    game550 = true
    sdesc = "East Side of Bottomless Pit"
    ldesc = { I(); "You are standing on the eastern side of a bottomless
                    pit.  A narrow ledge runs north towards a
                    dimly-visible passage;  the ledge once continued
                    south of this point but has been shattered by falling
                    rock.  A narrow crack in the rock leads northeast.";
    }
    north = Brink_North
    ne = Crack_1
    crack = Crack_1
    jump = { 
        local actor := getActor(&travelActor);
        Bott_Pit.plunge(actor); 
        return nil; 
    }
;
Bott_Pit: floatingdecoration
    game550 = true
    sdesc = "bottomless pit"
    ldesc = "It's a deep-looking pit."
    loclist = [ Brink_North  Brink_East  Brink_South ]
    noun = 'pit'
    adjective = 'bottomless' 'deep' 'looking' 'deep-looking'
    verDoEnter(actor) = {}
    doEnter(actor) = { self.plunge(actor); }
    plunge(actor) = {
            local eaten;
            "You have jumped into a bottomless pit.
            You continue to fall for a very long time. ";
            if(brass_lantern.islit and brass_lantern.isIn(actor))
            {
                    "First, your lamp runs out of power and goes
                    dead. ";
            }


            // Zap the lamp if it's lit (wherever it is)
            if (brass_lantern.islit) brass_lantern.setlife(0);
            // Since we're mentioning hunger and thirst, consume
            // any accessible food and drink.  The code to do this is a 
            // little complex - in the 701-point game, the items may be
            // in closed containers and we must open them first.

            // Start by opening the sack if we have it (it might contain the
            // keys to unlock the chest)
            if (sack.isIn(actor)) {
                sack.isopen := true;
            }
            // Open the chest if it is unlocked.
            if (treasure_chest.isIn(actor) and not treasure_chest.islocked) {
                treasure_chest.isopen := true;
            }
            // Unlock the chest if we can.
            if (treasure_chest.isIn(actor) and treasure_chest.islocked
            and set_of_keys.isInside(actor) and not set_of_keys.isInside
            (treasure_chest)) {
                if(set_of_keys.location != actor) 
                    set_of_keys.moveInto(actor);
                treasure_chest.isopen := true;
                treasure_chest.islocked := nil;
            }
            // Try opening the sack again (in case it was hidden inside
            // the chest)
            if (sack.isIn(actor) and not sack.isopen) {
                sack.isopen := true;
            }
            if (tasty_food.isIn(actor)){
                tasty_food.moveInto(nil);
                eaten := true;
            }
            if (honeycomb.isIn(actor)){
                honeycomb.moveInto(nil);
                eaten := true;
            }
            if (mushrooms.isIn(actor)){
                mushrooms.moveInto(nil);
                mushrooms.regrow; // Regrow mushrooms straight away due to
                                  // lapse of time.
            }
            if (mushroom.isIn(actor)){
                mushroom.moveInto(nil);
                mushroom.regrow;  // Regrow mushroom straight away due to
                                  // lapse of time.
            }
            if (bottle.isIn(actor) and bottle.haswater)bottle.empty;
            if (flask.isIn(actor) and flask.haswater)flask.empty;
            if (cask.isIn(actor) and cask.haswater)cask.empty;
            if (bottle.isIn(actor) and bottle.haswine)bottle.empty;
            if (flask.isIn(actor) and flask.haswine)flask.empty;
            if (cask.isIn(actor) and cask.haswine)cask.empty;

            // Mention hunger only if we didn't manage to eat any
            // substantial food items (mushrooms don't count).
            "Later, you die of ";
            if (not eaten) "hunger and ";
            "thirst.\n";
            die();
    }
;

Crack_1: darkroom
    game550 = true
    sdesc = "Narrow, Twisting Crack"
    ldesc = { I(); "You are following a narrow crack in the rock which
                    enters from the southwest, turns and twists somewhat,
                    and exits to the southeast.";
    }
    sw = Brink_East
    se = Crack_2
;

Crack_2: darkroom
    game550 = true
    sdesc = "North End of Tight Passage"
    ldesc = { I(); "You are standing at the northern end of a rather
                    tight passage.  A narrow crack in the rock leads
                    west.";
    }
    west = Crack_1
    south = { 
        local actor := getActor(&travelActor);
        if (Slime.exists) { 
            Slime.doCross(actor); 
            return nil; 
        }
        else return Crack_3;
    }
    cross = { return self.south; }  // BJS: added these.
    across = { return self.south; }  /* They aren't appropriate   *
               verbs to use if the slime is dead, but never mind. */
    NPCexit1 = { if (Slime.exists) return nil;
                    else return Crack_3;
    }       // NPCs can't cross the slime.
    exithints = [ Crack_3, &south ]
;

Crack_3: darkroom
    game550 = true
    sdesc = "South End of Tight Passage"
    ldesc = { I(); "You are at the southern end of a tight passage.
                    A hands-and-knees crawl continues to the south.";
    }
    north = Crack_2   // The slime must be dead by now.
    south = Crack_4
    crawl = Crack_4
;

Crack_4: darkroom
    game550 = true
    sdesc = "Very Small Chamber"
    ldesc = { I(); "You are in a very small chamber.  A narrow crawl
                    leads north."; }
    north = Crack_3
    out = Crack_3
    crawl = Crack_3
;

Ice_Room: darkroom
    game550 = true
    sdesc = "Ice Room"
    ldesc = { I(); "You are in the Ice room.  The walls and ceiling here
                    are composed of clear blue glacial ice;  the floor
                    is fortunately made of rock and is easy to walk
                    upon.  There is a passage leading to the northwest,
                    and a slide of polished ice leading downwards to the
                    east - if you were to slide down it you probably
                    couldn't get back up.";
    }
    nw = Brink_South
    down = { "Wheeeeee...... OOF!"; P(); return Slide_Base; }  // Note that
    east = { return self.down; }   // NPCs won't use these exits. If they
    slide = { return self.down; } // did, they would become trapped.
    thurb = Ice_Cave_Exit    // The exit was originally one-way, but I've
;                               // made it two-way for consistency with the
                           // other words.
Slide_Base: darkroom, NoNPC
    game550 = true
    sdesc = "Bottom of Icy Slide"
    ldesc = { I(); "You're at the entrance to an extensive and intricate
                    network of ice tunnels carved out of solid ice.  A
                    slippery slope leads upwards and north, but you
                    cannot possibly climb up it.  Other passages lead
                    south and northwest.";
    }
    north = { "The icy slide is far too steep and slippery to climb.";
                    return nil; }
    up = { return self.north; }
    slide = { return self.north; }
    climb = { return self.north; }
    south = Ice_2a
    nw = Ice_4
    exithints = [ Ice_Room, &north ]
;

Ice_1: darkroom, CCR_ice_tunnel
    north = Ice_1a
    west = Ice_2
;
Ice_1a: darkroom, CCR_ice_tunnel
    south = Ice_1
;
Ice_2: darkroom, CCR_ice_tunnel
    east = Ice_1
    west = Ice_3
    north = Ice_2a
;
Ice_2a: darkroom, CCR_ice_tunnel
    north = Slide_Base
    south = Ice_2
;
Ice_3: darkroom, CCR_ice_tunnel
    east = Ice_2
    north = Ice_3a
;
Ice_3a: darkroom, CCR_ice_tunnel
    south = Ice_3
;
Ice_4: darkroom, CCR_ice_tunnel
    east = Slide_Base
    west = Ice_5
;
Ice_5: darkroom, CCR_ice_tunnel
    ne = Ice_4
    south = Ice_6
;
Ice_6: darkroom, CCR_ice_tunnel
    north = Ice_5
    south = Ice_7
    west = Ice_9
;
Ice_7: darkroom, CCR_ice_tunnel
    north = Ice_6
;
Ice_8: darkroom, CCR_ice_tunnel
    north = Ice_9
;
Ice_9: darkroom, CCR_ice_tunnel
    east = Ice_6
    south = Ice_8
    north = Ice_10
;
Ice_10: darkroom, CCR_ice_tunnel
    south = Ice_9
    nw = Ice_11
;
Ice_11: darkroom, CCR_ice_tunnel
    east = Ice_10
    west = Ice_12
;
Ice_12: darkroom, CCR_ice_tunnel
    ne = Ice_11
    south = Ice_12a
    west = Ice_15
;
Ice_12a: darkroom, CCR_ice_tunnel
    north = Ice_12
    south = Ice_13
;
Ice_13: darkroom, CCR_ice_tunnel
    north = Ice_12a
;
Ice_14: darkroom, CCR_ice_tunnel
    north = Ice_15a
;
Sculpt_Niche: fixeditem, qcontainer   // BJS: added this object,
    game550 = true                    // put the sculpture in it.
    sdesc = "niche"
    heredesc = { P(); I();
      if(sculpture.isIn(self) and self.isqcontainer) {
        "A finely carved sculpture of ";
        sculpture.animdesc;
        " is resting in a niche melted out of the icy wall of the
          tunnel!";
      }
      else "There is a niche here, melted out of the icy wall of the
        tunnel.";
    }
    ioPutIn(actor,dobj) = {
            if(dobj.islong or dobj.islarge or dobj.ishuge) {
                    "It won't fit! ";
                    return;
            }
            else inherited.ioPutIn(actor,dobj);
            if (dobj.location = self) isqcontainer := nil;
    }
    noun = 'niche'
    adjective = 'melted'
    location = Ice_14
;
Ice_15: darkroom, CCR_ice_tunnel
    east = Ice_12
    south = Ice_15a
    nw = Ice_16
;
Ice_15a: darkroom, CCR_ice_tunnel
    south = Ice_14
    north = Ice_15
;
Ice_16: darkroom, CCR_ice_tunnel
    east = Ice_15
    west = Ice_17
;
Ice_17: darkroom, CCR_ice_tunnel
    ne = Ice_16
    south = Ice_18
;
Ice_18: darkroom, CCR_ice_tunnel
    north = Ice_17
    south = Ice_19
    west = Ice_21
    nw = Ice_22
;
Ice_19: darkroom, CCR_ice_tunnel
    north = Ice_18
    west = Ice_20
;
Ice_20: darkroom, CCR_ice_tunnel
    east = Ice_19
    north = Ice_21
;
Ice_21: darkroom, CCR_ice_tunnel
    east = Ice_18
    south = Ice_20
;
Ice_22: darkroom, CCR_ice_tunnel
    se = Ice_18
    nw = Ice_23
;
Ice_23: darkroom, CCR_ice_tunnel
    east = Ice_22
    west = Ice_24
;
Ice_24: darkroom, CCR_ice_tunnel
    ne = Ice_23
    south = Ice_25
    west = Ice_29
;
Ice_25: darkroom, CCR_ice_tunnel
    north = Ice_24
    south = Ice_26
    west = Ice_28
    nw = Ice_28a
;
Ice_26: darkroom, CCR_ice_tunnel
    north = Ice_25
    nw = Ice_27
;
Ice_27: darkroom, CCR_ice_tunnel
    se = Ice_26
    north = Ice_28
;
Ice_28: darkroom, CCR_ice_tunnel
    east = Ice_25
    south = Ice_27
;
Ice_28a: darkroom, CCR_ice_tunnel
    se = Ice_25
    north = Ice_29
;
Ice_29: darkroom, CCR_ice_tunnel
    east = Ice_24
    south = Ice_28a
    nw = Ice_Cave_Exit
;
Ice_Cave_Exit: room, NoNPC // lit by the letters. BJS
    game550 = true
    sdesc = "Small, Icy Chamber"
    ldesc = { I();
        "You are in a small chamber melted out of ice.  Glowing
        letters in midair spell out the words
        \"This way out\".";
    }
    east = Ice_29
    thurb = Ice_Room
;
Glow_Letter: fixeditem, readable
    game550 = true
    sdesc = "glowing letters"
    ldesc = {
        "Glowing letters in midair spell out the words
        \"This way out\".";
    }
    noun = 'letters' 'words'
    adjective = 'glowing' 'midair'
    location = Ice_Cave_Exit
;

Coral_Passage: darkroom
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "Coral Passage"
    ldesc = { I();
        "You are in an arched coral passage which enters from
        the west, splits, and continues on to the east over
        a smooth and damp-looking patch of sand.  The fork
        in the passage once led to the south, but it is
        now completely blocked by debris.";
    }
    // DJP - corrected 'lead' to 'led'.  
    east = { 
        local actor := getActor(&travelActor);
        return QuickSand.doCross(actor); 
    } 
                                             // doCross should return
    cross = { return self.west; }            // the appropriate value.
    west = In_Arched_Hall
;       // Don't allow 'back' across the quicksand -BJS

QuickSand: floatingdecoration
    iswavetarget = true // magic can be worked by waving the rod at it ...
    game550 = true
    sdesc = "damp-looking sand"
    ldesc = "A smooth, damp-looking patch of sand."
    noun = 'sand' 'quicksand' 'patch'
    adjective = 'smooth' 'damp' 'damp-looking' 'wet' 'wet-looking'
    loclist = [ Coral_Passage  Coral_Pass_2 ]
    doCross(actor) = {
        // DJP - player should sink if carrying the clam/oyster
        if(giant_bivalve.isIn(actor)) self.isHard := nil;
        if(self.isHard) {
            self.isHard := nil;
            if (actor.isIn(Coral_Passage))
                 return Coral_Pass_2;
            else return Coral_Passage;
        }
        else if(self.sankBefore) "You know, I've heard
             of people who really fell in for the soft
             sell, but >glub< this >glub< is >glub<
             ridiculous!             >blop!<";
        else {
             "Hmmm... this sand is rather soft, and you're
             sinking in a little....  In fact, you're
             sinking in a lot!  Oh, no - it's
             QUICKSAND!!  HELP!!  HELP!!!  HELP!!
             >glub<   >glub<    >blurp<";
             self.sankBefore := true;
        }
        // DJP - removed redundant 'if' statement
        die();
        return nil;
    }
    isHard = nil
    sankBefore = nil
;

Coral_Pass_2: darkroom, NoNPC  // They can't cross the quicksand
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "Bend in Arched Coral Corridor"
    ldesc = { I();
        "You are at a bend in an arched coral passage;  the
        passage enters from the west over a patch of damp
        sand, turns, and continues north.";
    }
    north = Arch_Fork
    west = { 
        local actor := getActor(&travelActor);
        return QuickSand.doCross(actor); 
    } 
                                             // doCross should return
    cross = { return self.west; }            // the right value.
;     // Don't allow 'back' across the quicksand -BJS

Arch_Fork: darkroom, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "Fork in Arched Coral Passage"
    ldesc = { I();
        "You are at a fork in a high, arched coral passage.
        The main portion of the passage enters from the south;
        two smaller passages lead east and north.  The smell
        of salt water is very strong here.";
    }
    south = Coral_Pass_2
    north = Fourier
    east = Jonah
    jonah = Jonah
    fourier = Fourier
;

Jonah: darkroom, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "Entrance to Jonah Room"
    ldesc = { I();
        "You are standing at the entrance of the Jonah room,
        a cavernous hall with high ribbed walls.  The hall
        extends far to the south;  a coral passage leads
        west.";
    }
    west = Arch_Fork
    south = In_Jonah
;

In_Jonah: darkroom, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "South End of Jonah Room"
    ldesc = { I();
        "You are at the south end of the Jonah room.  Ahead
        of you, the way is barred by a large set of immense
        stalactites and stalagmites which intermesh like
        clenched teeth.  Nothing except blackness is visible
        between the stone formations.";
    }
    north = Jonah
    out = Jonah
;

Fourier: darkroom, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    Zarkalonroom = true
    sdesc = "Fourier Passage"
    ldesc = { I();
        "You are in the Fourier passage.  This is a long and
        highly convoluted passage composed of coral, which
        twists and turns like the path of an earthworm
        tripping on LSD.  The passage here enters from the
        northwest, convulses, and exits to the southwest
        (from which direction can be felt a cool and
        salty-smelling breeze).";
    }
    nw = Arch_Fork
    sw = Beach_Shelf
    ana = Blue_Fourier
;

Beach_Shelf: room, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    Zarkalonroom = true // Zarkalonized pendant required for transindection
    sdesc = "Shelf of Rock Above Beach"
    //
    // Note by D.J. Picton:
    // This area is different in the 660-point game.  So different,
    // in fact, that I've taken the liberty of coding it here, even
    // though the 660-point game isn't implemented yet!
    //
    // (The 660 and 770-point games move the alien scene to the Pentagram 
    // Room area.
    //
    // The 701-point game is based on the 550-point game so we keep the alien
    // scene here, neatly solving the problem of differing sea levels.
    //
    // The 660 and 770-point games do have an apparent sea level problem,
    // because the sea is said to be 1000ft below the surface and we don't
    // have the feeling of having climbed down that far to reach this room
    // (or the sewage outflow in the ante-room area). I say 'apparent',
    // because there could still be hidden space portals in the cave.)
    //
    ldesc = { 
        I();
        if(global.game660) {   
            "You are standing on a large shelf of sedimentary rock overlooking
            a sandy beach. The shelf is an extension of sheer chalk cliff which
            continues north and south for as far as the eye can see. Crudely
            carved steps lead down from the shelf to the beach, and a twisting
            coral passage exits to the west. "; 
        }
        // N.B. in the 770-point game this area lies at the foot of the
        // 1000ft cliff below the giant's picnic site.  There are more rooms
        // and the east and west directions are reversed!
        else {
            "You are standing on a large shelf of sedimentary rock
            overlooking a lava beach.  The shelf is an extension
            of an incredible cliff which extends north, south, and
            upwards for as far as the eye can see.  Crudely carved
            steps lead down from the shelf to the beach, and a
            twisting coral passage exits to the west. ";
            // special stuff for 701+ point game.
            if(global.game701p) {
                if (not self.isseen and not Blue_Beach_Shelf.isseen and not
                Blue_Beach.isseen) {
                    P(); I();
                    "In a flash of intuition, you notice something which
                    previous adventurers may have missed.  There's a zig-zag
                    pattern on the cliff which doesn't look like a natural
                    feature.  Then you realize what it is - the remains of
                    a stone path which has now been almost obliterated by
                    rockfalls. ";
                }
                else {
                    "A stone path once led up the cliff, but much of 
                    it has now been obliterated by rockfalls. ";
                }
            }
        }
    }
    west = Fourier
    down = Beach
    steps = Beach
    ana = Blue_Beach_Shelf
    myhints = {
        if(global.game701p) 
            return [Beach_Shelfhint];
        else
            return nil;
    }
;

Beach_Shelf_Path: fixeditem
    game701p = true  // 701+ point game only
    wino_quicksand = true // must cross quicksand to leave
    sdesc = "stone path"
    ldesc = "The rockfalls have destroyed most of the path.  The
        lowest intact section starts about 20 feet above you. "
    noun = 'path'
    adjective = 'stone'
    location = Beach_Shelf
;


Beach_Shelf_Cliff: floatingdecoration
    sdesc = "cliff"
    ldesc = {
        "It extends north, south and upwards as far as the eye can see. ";
        if (self.location.analevel = 1)
            "The cliff looks exactly as it did at Red level - but now a timber
            roadway has been built on the remains of the path, leading up the
            cliff. ";
        else if (global.game701p)
            "A stone path once led up the cliff face, but most of it has now
            been obliterated by rockfalls. ";
    }
    noun = 'cliff'
    loclist = [Beach_Shelf, Beach, Blue_Beach_Shelf, Blue_Beach]
;


Beach: room, NoNPC
    game550 = true
    wino_quicksand = true // must cross quicksand to leave
    Zarkalonroom = true // Zarkalonized pendant required for transindection
    isbonus = true
    sdesc = "Beach"
    ldesc = { 
        I();
        if(global.game660) {
            "You are standing on a short, rocky beach. To the west a sheer
            white cliff rises into the sky. Rugged and unclimbable outcrops of
            rock block out all view to the north and south. To the east, a
            narrow inlet of sea water laps gently upon the beach. Steps lead up
            the cliff to a shelf of rock. ";
        }
        else {
            "You are standing on a short, barren beach composed of
            hardened lava.  Rugged and unclimbable volcanic hills
            block all view to the north and south, and a seemingly
            infinite cliff fills the entire western hemisphere.
            To the east, a narrow inlet of ocean water laps gently
            upon the beach.  The scene is illuminated by the light
            of three small moons shining through the shimmering
            glow of an aurora that fills the entire sky with
            golden splendor.  Steps lead up the cliff to a shelf
            of rock. ";
        }
    }
    west = { Dinghy.seen_before := true;
             return Beach_Shelf;
    }
    up = { return self.west; }
    steps = { return self.west; }
    ledge = { return self.west; }
    exithints = [ Beach_Shelf, &west ]
    ana = Blue_Beach
;

Dinghy: fixeditem
    game550 = true
    sdesc = "shattered dinghy"
    ldesc = "The remains consist of little more than a few broken boards,
        upon one of which may be seen a crude sketch of a skull and
        two crossed thighbones (perhaps this dinghy was once owned
        by a cook?)"
    location = Beach
    noun = 'dinghy' 'boat' 'remains'
    adjective = 'shattered' 'dinghy'
    seen_before = nil  // The description appears only on the first visit.
    heredesc = {
        P(); I();
        if (not Dinghy.seen_before) {
            "Lying upon the beach
            are the shattered remains of what must once
            have been a dinghy.  ";
            Dinghy.ldesc;
        }
        else
            "The shattered remains of a dinghy lie
            forlornly on the beach.";
    }

    verDoRide(actor) = { "The dinghy is too badly damaged to be seaworthy."; }
    verDoLaunch(actor) = {self.verDoRide(actor); }
    verDoRow(actor) = {self.verDoRide(actor); }
    verDoEnter(actor) = {self.verDoRide(actor); }
    verDoBoard(actor) = {self.verDoRide(actor); }
;

Moons: floatingdecoration, distantItem
    sdesc = "moons"
    ldesc = "Yes - three moons!  The largest one looks about half
        the size of the Earth's moon and has a red tinge; the other
        two are much smaller and look blue. It appears that you have passed 
        through a space portal without realizing it ..."
    noun = 'moon' 'moons'
    adjective = 'small'
    loclist = [Beach_Shelf, Blue_Beach_Shelf, Beach, Blue_Beach,
              Zarkalon_Tunnel_Entrance, Blue_Zarkalon_Tunnel_Entrance,
              Zarkalon_Cliff_Top, Blue_Zarkalon_Cliff_Top,
              Zarkalon_Tower_Top, Blue_Zarkalon_Tower_Top]
    doCount(actor) = {
        "What planet do you think you're on?  I've already told you that
        there are three moons.  On second thoughts, that was a dumb
        question.  ";
        if(global.knowsgreenname)
            "You know that the Green-level elves call this planet 
            \"Ondralstir\" and the Blue-level humans call it \"Zarkalon\".
            I guess we'll stay with the Blue-level name. ";
        else if(BlueBoard1.isread)
            "I guess we'll stay with the Blue-level name for this planet:
            Zarkalon. ";
        else
            "I don't know the answer either... ";
    }
;
